package WebService::NetSuite::SuiteTalk;

use Modern::Perl '2011';    ## no critic (Modules::ProhibitUseQuotedVersion)

our $VERSION = '0.001';     # TRIAL VERSION
use utf8;
use Moo;
use MooX::HandlesVia;
use MooX::Types::MooseLike::Base qw(CodeRef HashRef InstanceOf);
use Class::Method::Modifiers 2.00 'fresh';
with 'WebService::NetSuite::SuiteTalk::Role::Connection';

has _calls => (
    is          => 'rw',
    isa         => HashRef [CodeRef],
    handles_via => 'Hash',
    handles     => {
        '_exists' => 'exists',
        '_create' => 'set',
    },
    default => sub { {} },
);

sub _call {
    my ( $self, $operation_name, $parameters_ref, @call_options ) = @_;
    $self->_operation_name($operation_name);

    # cache compiled client code
    if ( not $self->_exists($operation_name) ) {
        $self->_create(
            $operation_name => $self->wsdl->compileClient(
                $operation_name, transport => $self->transport,
            ),
        );
    }

    # make the call and either return or throw an exception
    my ( $answer, $trace ) = $self->_calls->{$operation_name}->(
        passport   => $self->passport,
        parameters => $parameters_ref,
        @call_options,
    );
    if ( not $answer ) {
        for ( $trace->errors ) { $_->throw }
    }
    return ( $answer, $trace );
}

sub BUILD {
    my $instance = shift;
    for my $method ( map { $_->name } $instance->wsdl->operations ) {
        ## no critic (Subroutines::ProhibitCallsToUndeclaredSubs)
        fresh $method => sub {
            my $self = shift;
            my %arg = 'HASH' eq ref $_[0] ? %{ $_[0] } : @_;
            return $self->_call( $method => \%arg );
        };
    }
    return;
}

1;

# ABSTRACT: NetSuite SOAP interface that caches compiled calls in memory

__END__

=pod

=encoding UTF-8

=for :stopwords Mark Gardner ZipRecruiter cpan testmatrix url annocpan anno bugtracker rt
cpants kwalitee diff irc mailto metadata placeholders metacpan

=head1 NAME

WebService::NetSuite::SuiteTalk - NetSuite SOAP interface that caches compiled calls in memory

=head1 VERSION

version 0.001

=head1 SYNOPSIS

    use WebService::NetSuite::SuiteTalk;
    my $suitetalk = WebService::NetSuite::SuiteTalk->new(
        account_id    => '12345678',
        email_address => 'netsuite_customer@example.com',
        password      => 'sekrit',
        xsi_auto      => 1,
    );
    my $answer_ref = $suitetalk->changePassword(
        changePassword => {
            currentPassword => 'sekrit',
            newPassword     => 'tirkes',
            newPassword2    => 'tirkes',
            justThisAccount => 'true',
        },
    );

=head1 DESCRIPTION

This class provides an in-memory lazy cache for compiled SOAP calls using
L<NetSuite SuiteTalk|http://www.netsuite.com/portal/platform/developer/suitetalk.shtml>
web services. The first call to any SOAP method uses
L<XML::Compile::WSDL11|XML::Compile::WSDL11> to compile and execute that method
against the specified NetSuite SuiteTalk service; subsequent calls can vary the
parameters but will use the same compiled code.

Available method names are dynamically loaded from the SuiteTalk WSDL file's
operations, and can be passed either a hash or reference to a hash with the
necessary parameters. In scalar context they return a reference to a hash
containing the results of the SOAP call; in list context they return the
results hashref and an L<XML::Compile::SOAP::Trace|XML::Compile::SOAP::Trace>
object suitable for debugging and exception handling.

If there is no result then an exception will be thrown.

As of this writing the following operations are published in the NetSuite
SuiteTalk API 2014.2 schema.  Consult NetSuite's Help Center for full
documentation on input and output parameters for each operation.

=head1 METHODS

=head2 add

=head2 addList

=head2 asyncAddList

=head2 asyncDeleteList

=head2 asyncGetList

=head2 asyncInitializeList

=head2 asyncSearch

=head2 asyncUpdateList

=head2 asyncUpsertList

=head2 attach

=head2 changeEmail

=head2 changePassword

=head2 checkAsyncStatus

=head2 delete

=head2 deleteList

=head2 detach

=head2 get

=head2 getAll

=head2 getAsyncResult

=head2 getBudgetExchangeRate

=head2 getConsolidatedExchangeRate

=head2 getCurrencyRate

=head2 getCustomizationId

=head2 getDataCenterUrls

=head2 getDeleted

=head2 getItemAvailability

=head2 getList

=head2 getPostingTransactionSummary

=head2 getSavedSearch

=head2 getSelectValue

=head2 getServerTime

=head2 initialize

=head2 initializeList

=head2 login

=head2 logout

=head2 mapSso

=head2 search

=head2 searchMore

=head2 searchMoreWithId

=head2 searchNext

=head2 ssoLogin

=head2 update

=head2 updateInviteeStatus

=head2 updateInviteeStatusList

=head2 updateList

=head2 upsert

=head2 upsertList

=for Pod::Coverage BUILD

=head1 OTHER METHODS AND ATTRIBUTES

Because this class consumes the
C<WebService::NetSuite::SuiteTalk::Role::Connection> role, it also has that
role's attributes and methods. For example, you can use the C<user_agent>
attribute to access the L<LWP::UserAgent|LWP::UserAgent> used to retrieve and
call the SuiteTalk service, or the C<wsdl> method to access the underlying
L<XML::Compile::WSDL11|XML::Compile::WSDL11> object.

=head1 SUPPORT

=head2 Perldoc

You can find documentation for this module with the perldoc command.

  perldoc WebService::NetSuite::SuiteTalk

=head2 Websites

The following websites have more information about this module, and may be of help to you. As always,
in addition to those websites please use your favorite search engine to discover more resources.

=over 4

=item *

MetaCPAN

A modern, open-source CPAN search engine, useful to view POD in HTML format.

L<http://metacpan.org/release/WebService-NetSuite-SuiteTalk>

=item *

Search CPAN

The default CPAN search engine, useful to view POD in HTML format.

L<http://search.cpan.org/dist/WebService-NetSuite-SuiteTalk>

=item *

AnnoCPAN

The AnnoCPAN is a website that allows community annotations of Perl module documentation.

L<http://annocpan.org/dist/WebService-NetSuite-SuiteTalk>

=item *

CPAN Ratings

The CPAN Ratings is a website that allows community ratings and reviews of Perl modules.

L<http://cpanratings.perl.org/d/WebService-NetSuite-SuiteTalk>

=item *

CPAN Forum

The CPAN Forum is a web forum for discussing Perl modules.

L<http://cpanforum.com/dist/WebService-NetSuite-SuiteTalk>

=item *

CPANTS

The CPANTS is a website that analyzes the Kwalitee ( code metrics ) of a distribution.

L<http://cpants.cpanauthors.org/dist/WebService-NetSuite-SuiteTalk>

=item *

CPAN Testers

The CPAN Testers is a network of smokers who run automated tests on uploaded CPAN distributions.

L<http://www.cpantesters.org/distro/W/WebService-NetSuite-SuiteTalk>

=item *

CPAN Testers Matrix

The CPAN Testers Matrix is a website that provides a visual overview of the test results for a distribution on various Perls/platforms.

L<http://matrix.cpantesters.org/?dist=WebService-NetSuite-SuiteTalk>

=item *

CPAN Testers Dependencies

The CPAN Testers Dependencies is a website that shows a chart of the test results of all dependencies for a distribution.

L<http://deps.cpantesters.org/?module=WebService::NetSuite::SuiteTalk>

=back

=head2 Bugs / Feature Requests

Please report any bugs or feature requests through the web
interface at
L<https://github.com/mjgardner/webservice-netsuite-suitetalk/issues>.
You will be automatically notified of any progress on the
request by the system.

=head2 Source Code

The code is open to the world, and available for you to hack on. Please feel free to browse it and play
with it, or whatever. If you want to contribute patches, please send me a diff or prod me to pull
from your repository :)

L<https://github.com/mjgardner/webservice-netsuite-suitetalk>

  https://github.com/mjgardner/webservice-netsuite-suitetalk.git

=head1 AUTHOR

Mark Gardner <mjgardner@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 by ZipRecruiter.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
