package IO::Iron::IronCache::Policy;

## no critic (Documentation::RequirePodAtEnd)
## no critic (Documentation::RequirePodSections)

use 5.010_000;
use strict;
use warnings FATAL => 'all';

# Global creator
BEGIN {
    use parent qw( IO::Iron::PolicyBase ); # Inheritance
}

# Global destructor
END {
}

=head1 NAME

IO::Iron::IronCache::Policy - Base package (inherited) for IronCache::Client package.

=head1 VERSION

version 0.10

=cut

our $VERSION = '0.10'; # VERSION: generated by DZP::OurPkgVersion

=head1 SYNOPSIS

	# new() in the inheriting sub class.

	sub new {
		my ($class, $params) = @_;
		my $self = IO::Iron::IronCache::Policy->new();
		# Add more keys to the self hash.
		my @self_keys = (
				'caches',        # References to all objects created of class IO::Iron::IronCache::Cache.
				legal_keys(%{$self}),
		);
		use Data::Dumper; print Dumper($self); print Dumper(\@self_keys);
		unlock_keys(%{$self});
		lock_keys_plus(%{$self}, @self_keys);
		my @caches;
		$self->{'caches'} = \@caches;
	
		unlock_keys(%{$self});
		bless $self, $class;
		lock_keys(%{$self}, @self_keys);
	
		return $self;
	}

=cut

use Log::Any  qw{$log};
use Hash::Util 0.06 qw{lock_keys unlock_keys};
use Carp::Assert::More;
use English '-no_match_vars';
use Params::Validate qw(:all);
use Carp;
use Try::Tiny;


=head1 METHODS

=head2 is_cache_name_alternatives

=cut

sub is_cache_name_alternatives {
    my $self = shift;
    my %params = validate(@_, { } ); # No parameters
    $log->tracef('Entering is_cache_name_alternatives(%s)', \%params);
    my $rval = !(
            defined $self->{'policy'}->{'definition'}->{'no_limitation'}
            && $self->{'policy'}->{'definition'}->{'no_limitation'} == 1
            );
    $log->tracef('Exiting is_cache_name_alternatives():%d', $rval);
    return $rval;
}

=head2 is_item_key_alternatives

=cut

sub is_item_key_alternatives {
    my $self = shift;
    my %params = validate(@_, { } ); # No parameters
    $log->tracef('Entering is_item_key_alternatives(%s)', \%params);
    my $rval = !(
            defined $self->{'policy'}->{'definition'}->{'no_limitation'}
            && $self->{'policy'}->{'definition'}->{'no_limitation'} == 1
            );
    $log->tracef('Exiting is_item_key_alternatives():%d', $rval);
    return $rval;
}

=head2 cache_name_alternatives

Return all possible cache name alternatives according to the current policy.

=over 8

[No parameters.]

=back

Return: list of cache name alternatives

Will throw NoIronPolicyException if there is no limit to the alternatives.

=cut

sub cache_name_alternatives {
    my $self = shift;
    my %params = validate(@_, { } ); # No parameters
    $log->tracef('Entering cache_name_alternatives(%s)', \%params);

    my @alternatives = $self->alternatives( 'required_policy' => 'name' );

    $log->tracef('Exiting cache_name_alternatives():%s', \@alternatives);
    return @alternatives;
}

=head2 item_key_alternatives

Return all possible item key alternatives according to the current policy.

=over 8

[No parameters.]

=back

Return: list of item key alternatives

Will throw NoIronPolicyException if there is no limit to the alternatives.

=cut

sub item_key_alternatives {
    my $self = shift;
    my %params = validate(@_, { } ); # No parameters
    $log->tracef('Entering item_key_alternatives(%s)', \%params);

    my @alternatives = $self->alternatives( 'required_policy' => 'item_key' );

    $log->tracef('Exiting item_key_alternatives():%s', \@alternatives);
    return @alternatives;
}

=head2 is_valid_cache_name

Check if the cache name is valid according to the policy. Return 1/0.

=over 8

=item name string to verify.

=back

=cut

sub is_valid_cache_name {
    my $self = shift;
    my %params = validate(
        @_, {
            'name' => { type => SCALAR, }, # cache name.
        }
    );
    $log->tracef('Entering is_valid_cache_name(%s)', \%params);
    my $validity = 1;
    try {
        $log->tracef('is_valid_cache_name:Enter try/catch.');
        $self->validate_cache_name('name' => $params{'name'});
    }
    catch {
        $log->tracef('is_valid_cache_name:Caught exception.');
        croak $_ unless blessed $_ && $_->can('rethrow'); ## no critic (ControlStructures::ProhibitPostfixControls)
        if ( $_->isa('IronPolicyException') ) {
            $log->tracef('Caught IronPolicyException.');
            $validity = 0;
        }
    };
    $log->tracef('Exiting is_valid_cache_name():%d', $validity);
    return $validity;
}

=head2 validate_cache_name

Same as above but if validation fails, throws IronPolicyException.
If valid, returns undef.

=over 8

=item name string to verify.

=back

=cut

sub validate_cache_name {
    my $self = shift;
    my %params = validate(
        @_, {
            'name' => { type => SCALAR, }, # cache name.
        }
    );
    $log->tracef('Entering validate_cache_name(%s)', \%params);
    $self->validate_with_policy(
            'policy' => 'name',
            'candidate' => $params{'name'});
    $log->tracef('Exiting validate_cache_name():[NONE]');
    return;
}

=head2 is_valid_item_key

Check if the item key is valid according to the policy. Return 1/0.

=over 8

=item key string to verify.

=back

=cut

sub is_valid_item_key {
    my $self = shift;
    my %params = validate(
        @_, {
            'key' => { type => SCALAR, }, # cache name.
        }
    );
    $log->tracef('Entering is_valid_item_key(%s)', \%params);
    my $validity = 1;
    try {
        $log->tracef('is_valid_item_key:Enter try/catch.');
        $self->validate_item_key('key' => $params{'key'});
    }
    catch {
        $log->tracef('is_valid_item_key:Caught exception.');
        croak $_ unless blessed $_ && $_->can('rethrow'); ## no critic (ControlStructures::ProhibitPostfixControls)
        if ( $_->isa('IronPolicyException') ) {
            $log->tracef('Caught IronPolicyException.');
            $validity = 0;
        }
    };
    $log->tracef('Exiting is_valid_item_key():%d', $validity);
    return $validity;
}

=head2 validate_item_key

Same as above but if validation fails, throws IronPolicyException.
If valid, returns undef.

=over 8

=item key string to verify.

=back

=cut

sub validate_item_key {
    my $self = shift;
    my %params = validate(
        @_, {
            'key' => { type => SCALAR, }, # cache name.
        }
    );
    $log->tracef('Entering validate_item_key(%s)', \%params);
    $self->validate_with_policy(
            'policy' => 'item_key',
            'candidate' => $params{'key'});
    $log->tracef('Exiting validate_item_key():[NONE]');
    return;
}


# INTERNAL METHODS
# For use in the inheriting subclass

# This is a late binding to inherited method get_policies:
sub _THIS_POLICY {
    my $self = shift;
    my %params = validate(
        @_, {
            # No parameters.
        }
    );
    return 'cache';
}


=head1 AUTHOR

Mikko Koivunalho, C<< <mikko.koivunalho at iki.fi> >>


=head1 BUGS

Please report any bugs or feature requests to C<bug-io-iron at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=IO-Iron>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.


=head1 SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc IO::Iron::IronCache::Policy


You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=IO-Iron>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/IO-Iron>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/IO-Iron>

=item * Search CPAN

L<http://search.cpan.org/dist/IO-Iron/>

=back


=head1 ACKNOWLEDGMENTS

Cool idea, "message queue in the cloud": http://www.iron.io/.
And well implemented.


=head1 LICENSE AND COPYRIGHT

Copyright 2013 Mikko Koivunalho.

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

Any use, modification, and distribution of the Standard or Modified
Versions is governed by this Artistic License. By using, modifying or
distributing the Package, you accept this license. Do not use, modify,
or distribute the Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made
by someone other than you, you are nevertheless required to ensure that
your Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service
mark, tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge
patent license to make, have made, use, offer to sell, sell, import and
otherwise transfer the Package with respect to any patent claims
licensable by the Copyright Holder that are necessarily infringed by the
Package. If you institute patent litigation (including a cross-claim or
counterclaim) against any party alleging that the Package constitutes
direct or contributory patent infringement, then this Artistic License
to you shall terminate on the date that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE, OR NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY
YOUR LOCAL LAW. UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR
CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, OR
CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE OF THE PACKAGE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


=cut

1; # End of IO::Iron::IronCache::Policy
