################################################################################
#
# $Project: /Convert-Binary-C $
# $Author: mhx $
# $Date: 2004/03/22 20:38:00 +0100 $
# $Revision: 55 $
# $Snapshot: /Convert-Binary-C/0.50 $
# $Source: /Makefile.PL $
#
################################################################################
# 
# Copyright (c) 2002-2004 Marcus Holland-Moritz. All rights reserved.
# This program is free software; you can redistribute it and/or modify
# it under the same terms as Perl itself.
# 
################################################################################

# Check which version of perl the user is running,
# and emit some warnings when appropriate.

BEGIN {
  if( $] < 5.006 ) {
    print STDERR <<ENDWARN;

--> WARNING: The version of perl you're using ($]) is very old.
-->
-->   Convert::Binary::C is intended to build cleanly on
-->   perl versions >= 5.6.0. However, there are some hacks
-->   to make the code compatible with older versions.
-->
ENDWARN
    if( $] == 5.005_03 ) {
      print STDERR "-->   Although the module should build without problems,\n",
                   "-->   you should try to update your perl installation.\n\n";
    }
    else {
      if( $] < 5.005_03 ) {
        print STDERR "-->   Chances aren't very good that the module will\n",
                     "-->   build successfully.\n";
      }
      else {
        print STDERR "-->   Chances are quite good that the module will\n",
                     "-->   build successfully.\n";
      }
      print STDERR <<ENDWARN;
-->
-->   You can try to build the module with this version of
-->   perl, but you should rather update your perl installation.

ENDWARN
    }
  }
}

# Use some standard modules

use ExtUtils::MakeMaker;
use File::Find;
use Config;

$MODULE = 'Convert::Binary::C';

# We need bison only when the source code is modified
# Actually, we need a recent version of bison ( >= 1.31 ),
# but this isn't checked here.

$BISON = $Config{bison} || 'bison';

# Where to look for includes

@INC_PATH = qw(
  ctlib
);

# All object files (without extension)

@OBJECT = qw(
  ctlib/byteorder
  ctlib/ctdebug
  ctlib/cterror
  ctlib/ctparse
  ctlib/cttype
  ctlib/fileinfo
  ctlib/parser
  ctlib/pragma
  ctlib/util/hash
  ctlib/util/list
  ctlib/util/memalloc
  ctlib/ucpp/assert
  ctlib/ucpp/cpp
  ctlib/ucpp/eval
  ctlib/ucpp/nhash
  ctlib/ucpp/lexer
  ctlib/ucpp/macro
  ctlib/ucpp/mem
);

unshift @OBJECT, $MODULE =~ /([^:]+)$/;

# Files additionally to be removed on 'make realclean'

@REALCLEAN = qw(
  ctlib/parser.output
  ctlib/pragma.output
  t/debug.out
);

@CLEAN = qw(
  $(OBJECT)
  t/*.vgo
);

@DEFINE = qw(
  ABORT_IF_NO_MEM
  NO_SLOW_MEMALLOC_CALLS
  UCPP_CONFIG
  UCPP_REENTRANT
);

# On AIX systems, this should be defined for ucpp
$^O eq 'aix' and push @DEFINE, qw( POSIX_JMP );

# Supported features, and flags to set when (e)nabled or (d)isabled

%FEATURES = (
  debug   => {
               enabled => $Config{ccflags} =~ /-DDEBUGGING\b/ ? 1 : 0,
               e_flags => [qw( CTYPE_DEBUGGING DEBUG_MEMALLOC DEBUG_HASH YYDEBUG=1 )],
               d_flags => [qw( NDEBUG )],
             },
  ieeefp  => {
               enabled => undef,
               e_flags => [qw( CBC_HAVE_IEEE_FP )],
               d_flags => [qw()],
             },
  ($Config{usethreads} || $Config{use5005threads} || $Config{useithreads}) ? (
    threads => {
                 enabled => 1,
                 e_flags => [qw( CBC_THREAD_SAFE )],
                 d_flags => [qw()],
               }
  ) : (),
  $Config{gccversion} ? (
  '$format-check' => {
               enabled => 0,
               e_flags => [qw( CTYPE_FORMAT_CHECK UTIL_FORMAT_CHECK )],
               d_flags => [qw()],
             },
  '$coverage' => {
               enabled => 0,
               e_flags => [qw()],
               d_flags => [qw()],
             }
  ) : (),
  '$mem-check' => {
               enabled => 0,
               e_flags => [qw( MEM_DEBUG DEBUG_MEMALLOC TRACE_MEMALLOC )],
               d_flags => [qw()],
             },
);

# Automatically generated files

%EXAMPLES = (
  map {
    my $x=$_;
    s/^bin/examples/;
    s/PL$/pl/;
    ($x => $_)
  } glob "bin/*.PL"
);

%TOKENIZER = (
  'ctlib/t_parser.pl' => ['ctlib/t_parser.c',
                          'ctlib/t_keywords.c',
			  'ctlib/t_ckeytok.c',
			  'ctlib/t_basic.c',
			 ],
  'ctlib/t_pragma.pl' =>  'ctlib/t_pragma.c',
  'ctlib/t_config.pl' => ['ctlib/t_config.c',
                          'ctlib/t_sourcify.c',
                         ],
);

%GENERATE = (
  %EXAMPLES,
  'ctlib/arch.pl' => 'ctlib/arch.h',
);

push @REALCLEAN, map { ref $_ ? @$_ : $_ } values %GENERATE;

%GENERATE = (
  %GENERATE,
  %TOKENIZER,   # files generated by Devel::Tokenizer::C must not be deleted
);

# Extract features/optimizations from the commandline arguments

@ARGV = map {
  my $myopt = 0;
  if( my($what, $feat) = /^(en|dis)able-(\S+)$/ ) {
    exists $FEATURES{$feat} or $feat = '$'.$feat;
    exists $FEATURES{$feat}
      or die "Invalid feature '$2'. Use one of [ @{[keys %FEATURES]} ].\n";
    $FEATURES{$feat}{enabled} = $what eq 'en';
    $myopt = 1;
  }
  elsif( /^help$/ ) {
    die <<ENDUSAGE;

USAGE: $^X Makefile.PL enable-feature disable-feature

  Available Features: @{[sort grep !/^\$/, keys %FEATURES]}
    development only: @{[sort grep /^\$/, keys %FEATURES]}

ENDUSAGE
  }
  $myopt ? () : $_
} @ARGV;

if( $FEATURES{'$coverage'}{enabled} ) {
  push @CLEAN, qw( *.gcov *.da *.bb *.bbg );
}

# run the IEEE check anyway
@ieee_fail = &check_ieee_fp;

if( defined $FEATURES{'ieeefp'}{enabled} ) {
  if( $FEATURES{'ieeefp'}{enabled} and @ieee_fail ) {
    print <<ENDMSG;
-----------------------------------------------------------------------
  My tests indicate that your machine does not store floating point
  values in IEEE format. However, you explicitly turned IEEE floating
  point support on. So I will trust you. You have been warned!
-----------------------------------------------------------------------
ENDMSG
  }
}
else {
  $FEATURES{'ieeefp'}{enabled} = !@ieee_fail;
  if( @ieee_fail ) {
    print <<ENDMSG;

--> !!!!!!!!!!   --------------------------------------------------
--> !! WHOA !!    You did not pass the IEEE floating point check!
--> !!!!!!!!!!   --------------------------------------------------
--> 
--> This means I've done a couple of very simple tests to see if your machine
--> is storing floating point numbers in IEEE format or not. From the results
--> I concluded that your machine does _NOT_ store floating point values in
--> IEEE format.
--> 
--> These are the values for which the IEEE test failed:
--> 
--> value            format   expected                  got
--> ---------------------------------------------------------------------------
ENDMSG

  my $hex = sub { join ' ', map { sprintf "%02X", $_ } unpack "C*", $_[0] };

  for my $t ( @ieee_fail ) {
    printf "--> %-15s  %-7s  %-24s  %-24s\n", $t->{value}, $t->{check},
           $hex->( $t->{expected} ), $hex->( $t->{got} );
  }

  print <<ENDMSG;
--> ---------------------------------------------------------------------------
--> 
--> If you're aware of the fact that your machine does not support IEEE
--> floating point, please ignore the junk above. You can suppress this
--> message by explicitly disabling the 'ieeefp' feature:
--> 
-->   $^X Makefile.PL disable-ieeefp
--> 
--> If you're sure that your machine has IEEE floating point support and the
--> tests are just complete crap, you can force IEEE support by explicitly
--> enabling the 'ieeefp' feature:
--> 
-->   $^X Makefile.PL enable-ieeefp

ENDMSG
  }
}

if( $Config{usemymalloc} eq 'y' and $FEATURES{threads}{enabled} ) {
  die <<ENDMSG;

--> !!!!!!!!!!!!!   ------------------------------------------
--> !! S T O P !!    Unsupported Perl configuration detected!
--> !!!!!!!!!!!!!   ------------------------------------------
-->
-->   You're using a multithreaded Perl binary that was configured
-->   with -Dusemymalloc=y. This configuration is very dangerous
-->   when using XS modules. Convert::Binary::C will work as long
-->   as you're not actually using threads, but there's a 100%
-->   chance that it will fail if you're using threads. Either
-->   disable threads in your binary or don't use Perl's malloc.
-->
-->   Huh? Don't understand anything of the above? Don't plan to
-->   use threads? Ok, just run
-->
-->     $^X Makefile.PL disable-threads
-->
-->   and build the module as usual.

ENDMSG
}

WriteMakefile(
  'NAME'            => $MODULE,
  'VERSION_FROM'    => 'lib/Convert/Binary/C.pm',
  'ABSTRACT'        => 'Binary Data Conversion using C Types',
  'AUTHOR'          => 'Marcus Holland-Moritz <mhx@cpan.org>',
  'OBJECT'          => join( ' ', map { "$_\$(OBJ_EXT)" } @OBJECT ),
  'INC'             => join( ' ', map { "-I$_" } @INC_PATH ),
  'EXE_FILES'       => ['bin/ccconfig'],
  'PL_FILES'        => \%GENERATE,
  'CONFIGURE'       => \&configure,
  'clean'           => { FILES => "@CLEAN" },
  'realclean'       => { FILES => "@REALCLEAN" },
);

#############################################################################

sub configure {

  # Configure and print information about features

  for( keys %FEATURES ) {
    my $f = $FEATURES{$_};
    $f->{enabled} and print /^\$/ ? "DEVELOPMENT-ONLY feature '$_' selected\n"
                                  : "Building with feature '$_'\n";
    push @DEFINE, @{$f->{enabled} ? $f->{e_flags} : $f->{d_flags} };
  }
  
  my $config = {
    'DEFINE' => join(' ', map("-D$_", @DEFINE)),
    'depend' => { find_depend( @INC_PATH ) },
  };

  $FEATURES{'$coverage'}{enabled} and
      $config->{'OPTIMIZE'} = '-g -fprofile-arcs -ftest-coverage';

  $config;
}

sub MY::c_o {
  package MY;
  my $c_o = shift->SUPER::c_o(@_);
  $c_o =~ s/^\s+\$\(CCCMD\).*$/$&\n\t\$(MV) \$(\@F) t_object.tmp\n\t\$(MV) t_object.tmp \$\@/mg;
  $c_o;
}

sub MY::constants {
  package MY;
  shift->SUPER::constants(@_).<<END

# Yacc to generate parser
YACC = $::BISON

# Options for valgrind
VALGRIND_OPTIONS = --leak-check=yes \\
		   --leak-resolution=high \\
		   --show-reachable=yes \\
		   --num-callers=50
END
}

sub MY::postamble {
  package MY;
  my $postamble = shift->SUPER::postamble(@_).<<END;

ctlib/parser.c: ctlib/parser.y
	\$(YACC) -v -p c_ -o \$*.c \$*.y

ctlib/pragma.c: ctlib/pragma.y
	\$(YACC) -v -p pragma_ -o \$*.c \$*.y
END

  if( $::Config{ccflags} =~ /-DDEBUGGING\b/ ) {
    my($sym) = $::MODULE =~ /^([^:]+)/;
    $postamble .= <<END;

valgrind: pure_all
	\@export PERL_DESTRUCT_LEVEL=2; \\
	echo PERL_DESTRUCT_LEVEL=\$\$PERL_DESTRUCT_LEVEL; \\
	for file in t/*.t; do \\
	  echo checking \$\$file with valgrind...; \\
	  valgrind \$(VALGRIND_OPTIONS) \$(PERL) -Mblib \$\$file >\$\$file.vgo 2>&1; \\
	  grep $sym \$\$file.vgo; \\
	done
END
  }
  else {
    $postamble .= <<END;

valgrind:
	\@echo Sorry, I need a debugging version of perl for valgrind.
END
  }

  $::FEATURES{'$coverage'}{enabled} and $postamble .= <<END;

coverage:
	\@for file in *.bb; do gcov \$\$file; done
END

  $postamble;
}

sub MY::test {
  package MY;
  my $test = shift->SUPER::test(@_);
  $::FEATURES{debug}{enabled} and
      $test =~ s!^test\s*:.*!$&$/\t\@\$(RM_F) t/debug.out!m;
  $test
}

sub MY::installbin {
  package MY;
  my $ibin = shift->SUPER::installbin(@_);
  my @ex = values %::EXAMPLES;
  $ibin =~ s!^pure_all\s*:+\s*!$&@ex !m;
  $ibin
}

sub MY::metafile {
  package MY;
  my $self = shift;
  eval { require YAML; 1 }
      or return $self->SUPER::metafile_target(@_);

  my $node = new YAML::Node {};

  $node->{name}              = $self->{DISTNAME};
  $node->{version}           = $self->{VERSION};
  $node->{license}           = 'perl';
  $node->{distribution_type} = 'module';
  $node->{generated_by}      = "$::MODULE $0";

  my $dump = YAML::Dump( $node );

  $dump =~ s/^(.*)$/\t\$(NOECHO) \$(ECHO) "$1" >>META.yml/gm;
  $dump =~ s/>>META\.yml/>META.yml/;

  return "metafile:\n$dump";
}

# The following routines will extract the include dependencies
# from the source files.

sub find_depend {
  my @inc_path = ('.', @_);
  my(%depend, %d);

  printf "Finding dependencies...\n";

  for( @inc_path ) {
    /\/$/ or $_ .= '/';
  }

  File::Find::find( sub {
    /\.(?:xs|[chy])$/ or return;
    $File::Find::dir =~ /^\.[\/\\]t[\/\\]/ and return; # exclude test directory
    my @incs;
    open FILE, $_ or die $!;
    while( <FILE> ) {
      my($inc,$base) = /^\s*#\s*include\s*"([^"]+\.\w+)"/ or next;
      for my $path ( @inc_path ) {
        if( -e "$path$inc" ) {
          push @incs, $File::Find::dir . "/$path" . $inc;
        }
      }
      for my $gen ( keys %GENERATE ) {
        push @incs, grep /\E$inc/, ( ref $GENERATE{$gen} ? @{$GENERATE{$gen}} : $GENERATE{$gen} );
      }
    }
    close FILE;
    return unless @incs;
    my $name = $File::Find::name;
    for( @incs, $name ) {
      s/\.[\\\/]//;
      s/^\.\/|\/\.(?=\/)//g;
      s/[^\/]+\/\.\.\///g;
    }
    @{$depend{$name}}{@incs} = (1)x@incs;
  }, '.' );

  for( @OBJECT ) {
    my $name = $_;
    for( qw( xs y c ) ) {
      -e "$name.$_" and $name .= ".$_" and last;
    }
    my %incs;
    rec_depend( $name, \%depend, \%incs );
    $d{"$_\$(OBJ_EXT)"} = join ' ', keys %incs;
  }

  %d;
}

sub rec_depend {
  my($f,$d,$i) = @_;
  my $h = $d->{$f};
  for( keys %$h ) {
    exists $i->{$_} and next; $i->{$_} = 1;
    exists $d->{$_} and rec_depend( $_, $d, $i );
  }
}

sub is_big_endian
{
  my $byteorder = $Config{byteorder}
               || unpack( "a*", pack "L", 0x34333231 );

  die "Native byte order ($byteorder) not supported!\n"
      if   $byteorder ne '1234'     and $byteorder ne '4321'
       and $byteorder ne '12345678' and $byteorder ne '87654321';

  $byteorder eq '4321' or $byteorder eq '87654321';
}

sub check_ieee_fp
{
  my @test = (
    {
      value  => '-1.0',
      double => pack( 'C*', 0xBF, 0xF0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 ),
      single => pack( 'C*', 0xBF, 0x80, 0x00, 0x00 ),
    },
    {
      value  => '0.0',
      double => pack( 'C*', 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 ),
      single => pack( 'C*', 0x00, 0x00, 0x00, 0x00 ),
    },
    {
      value  => '0.4',
      double => pack( 'C*', 0x3F, 0xD9, 0x99, 0x99, 0x99, 0x99, 0x99, 0x9A ),
      single => pack( 'C*', 0x3E, 0xCC, 0xCC, 0xCD ),
    },
    {
      value  => '1.0',
      double => pack( 'C*', 0x3F, 0xF0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 ),
      single => pack( 'C*', 0x3F, 0x80, 0x00, 0x00 ),
    },
    {
      value  => '3.1415926535',
      double => pack( 'C*', 0x40, 0x09, 0x21, 0xFB, 0x54, 0x41, 0x17, 0x44 ),
      single => pack( 'C*', 0x40, 0x49, 0x0F, 0xDB ),
    },
    {
      value  => '1.220703125e-4',
      double => pack( 'C*', 0x3F, 0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 ),
      single => pack( 'C*', 0x39, 0x00, 0x00, 0x00 ),
    },
  );

  my @fail;

  for my $t ( @test ) {
    my $s = pack 'f', $t->{value};
    my $d = pack 'd', $t->{value};
    unless( &is_big_endian ) { for( $s, $d ) { $_ = reverse $_ } }

    $s eq $t->{single} or push @fail,
        { value => $t->{value}, check => 'single', expected => $t->{single}, got => $s }; 

    $d eq $t->{double} or push @fail,
        { value => $t->{value}, check => 'double', expected => $t->{double}, got => $d }; 
  }

  return @fail;
}

