use v6;
use OWL::IRI;
use OWL::Axiom;
use OWL::Declaration;
use OWL::SubClassOf;
use OWL::ClassAssertion;

package OWL {

  class Ontology {

    constant @entity-types = < class data-type object-property data-property
                               annotation-property named-individual
                             >;

    # Will be set by new() when OWL needs an OWL::Ontology.
    #
    has OWL::IRI $iri;
    has OWL::IRI $version;

    # Entities. Key is a full-iri and value is its type which can be any of
    # @entity-types or an full-iri which must be an existing key.
    #
    has Hash $.entities;
    has Hash $.entity-objects;
    has Hash $.object-properties;
    has Hash $.data-properties;


    # Axioms. Key is a unique full-iri and value is an array axiom objects
    # about the iri.
    #
    has Hash $.axioms;

    # Extra information not used for logic reasoning
    #
    has Hash $.annotations;

    #---------------------------------------------------------------------------
    # The Owl object must be given. With this items can be found back
    #
    submethod BUILD ( Str :$iri, Str :$version-iri, :@imports ) {
      $!iri = OWL::IRI.check-iri($iri) if ?$iri;
      $!version = OWL::IRI.check-iri($version) if ?$version and ?$!iri;

      self.declaration(
        [ class => <owl:thing owl:nothing>,
          object-property => <owl:top-object-property
                              owl:bottom-object-property
                             >,
          data-property => <owl:top-data-property owl:bottom-data-property>,
          data-type => <rdfs:literal xsd:integer xsd:boolean xsd:string>,
#          data-type => <i>,
#          annotation-property => <i>,
        ]
      );
    }

    #---------------------------------------------------------------------------
    # Declare new values for some type of entity. It accepts an array of pairs.
    # If all declarations are ok it returns True otherwise it returns False and
    # stops at the failing declaration.
    #
    method declaration ( Array $declarations --> Bool ) {

      my Bool $declaration-ok = True;

      DECLARATIONS-LOOP: for @$declarations {

        # Check declaration type
        #
        my $type = .key;
        if OWL::Declaration.check-entity-type($type) {
          note "Type '$type' not recognized";
          $declaration-ok = False;
          last;
        }

        # Declare the values of the given type
        #
        for @(.value) -> $value {

          # Check if value of type resolves to an iri string
          #
          my $iri = OWL::IRI.check-iri($value);
          if ! ?$iri {
            note "Cannot declare '$value' as $type";
            $declaration-ok = False;
            last DECLARATIONS-LOOP;
          }

          # Check if declaration exists. Entry of given iri must exist in
          # $!axioms and the first is always the declaration.
          #
          if ?$!axioms{$iri.Str} {
            note "'$iri' Already declared as ", $!axioms{$iri}[0].get-type;
            $declaration-ok = False;
            last DECLARATIONS-LOOP;
          }

          # If not, store it
          #
          my OWL::Declaration $ax .= new();
          $ax.set( $type, $iri);
          $!axioms{$iri.Str} = [$ax];
say "Declare $iri as $type";
        }
      }

      return $declaration-ok;
    }

    #---------------------------------------------------------------------------
    # Functional interface:
    #   SubClassOf := 'SubClassOf' '(' axiomAnnotations subClassExpression superClassExpression ')'
    # This interface
    #   sub-class-of( [ superClassExpression => ( subClassExpression, ...), ...])
    #
    method sub-class-of ( Array $assertions --> Bool ) {

      my Bool $subclass-ok = True;

      SUBCLASS-LOOP: for @$assertions {
        my $class = .key;

        # Check the class name that it has
        #   a) class name has an iri
        #   b) that the class iri is registered
        #   c) that it is registered as a class.
        #
        my $class-iri = OWL::IRI.check-iri($class);
#say "SCO 0: $class-iri";
        if ! ?$class-iri {
          note "Class '$class' not recognized";
          $subclass-ok = False;
          last;
        }

        # Check if registered
        #
        if ! ?$!axioms{$class-iri.Str} {
          note "Class '$class' not declared";
          $subclass-ok = False;
          last;
        }

        # Check if registered as class
        #
        if ! self.check-if-type( $class-iri, :types(<class sub-class-of>)) {
          note "Class '$class' not declared as class or subclass";
          $subclass-ok = False;
          last;
        }

        # Define each value as a subclass
        #
        for @(.value) -> $value {

          # Check and get iri from the value
          #
          my $sub-class-iri = OWL::IRI.check-iri($value);
#say "$value --> $sub-class-iri'";
          if ! ?$sub-class-iri {
            note "Cannot subclass '$value' to $class-iri";
            $subclass-ok = False;
            last SUBCLASS-LOOP;
          }

          # Check if there is already the same subclass setting
          #
          for @($!axioms{$sub-class-iri.Str}) -> $axiom {

            if $axiom.get-type eq 'sub-class-of'
               and $axiom.sub-class-expression.get-iri.Str eq $sub-class-iri.Str
               and $axiom.super-class-expression.get-iri.Str eq $class-iri.Str {
 
              note "$sub-class-iri.already subclassed to $class-iri";
              $subclass-ok = False;
              last SUBCLASS-LOOP;
            }
          }

          my OWL::SubClassOf $ax .= new();
          $ax.set( $class-iri, $sub-class-iri);
          $!axioms{$sub-class-iri.Str}.push($ax);
say "$sub-class-iri' is a subclass of $class-iri";
        }
      }

      return $subclass-ok;
    }

    #---------------------------------------------------------------------------
    # Instance of. Keys are classes and must exist and values must be declared
    # in entities of any type. Values must be named-individuals. Method accepts an array of pairs.
    #
    method class-assertion ( Array $assertions --> Bool ) {

      my Bool $assertion-ok = True;

      ASSERTIONS-LOOP: for @$assertions {
        my $class = .key;

        # Check the class name that it has
        #   a) class name has an iri
        #   b) that the class iri is registered
        #   c) that it is registered as a class.
        #
        my $class-iri = OWL::IRI.check-iri($class);
#say "CA IRI:$class-iri";
        if ! ?$class-iri {
          note "Class '$class' not recognized";
          $assertion-ok = False;
          last;
        }

        # Check if registered
        #
        if ! ?$!axioms{$class-iri.Str} {
          note "Class '$class' not declared";
          $assertion-ok = False;
          last;
        }

#say "CA ax: ", $!axioms{$class-iri.Str};
        # Check if registered as class
        #
        if ! self.check-if-type( $class-iri, :types(<class sub-class-of>)) {
          note "Class '$class' not declared as class or subclass";
          $assertion-ok = False;
          last;
        }

        for @(.value) -> $value {

          # Check that the value name has an iri and is registered.
          #
          my $iri = OWL::IRI.check-iri($value);
          if ! ?$iri {
            note "IRI $value not recognized";
            $assertion-ok = False;
            last ASSERTIONS-LOOP;
          }

          if ! self.check-if-type( $iri, :types(<named-individual>)) {
            note "$iri is not a named-individual";
            $assertion-ok = False;
            last ASSERTIONS-LOOP;
          }

          # Check if there is already the same assertions setting
          #
          for @($!axioms{$iri.Str}) -> $axiom {
            if $axiom.get-type eq 'class-assertion'
               and $axiom.class-expression.get-iri.Str eq $class-iri.Str
               and $axiom.individual.get-iri.Str eq $iri.Str {

              note "$iri.already asserted to $class-iri";
              $assertion-ok = False;
              last ASSERTIONS-LOOP;
            }
          }

          my OWL::ClassAssertion $ax .= new();
          $ax.set( $class-iri, $iri);
          $!axioms{$iri.Str}.push($ax);

say "Assert '$iri' as a $class-iri";
        }
      }

      return $assertion-ok;
    }

    #---------------------------------------------------------------------------
    # Array of pairs where keys with one value each where key is a data
    # property and value a data value.
    #
    method data-property-range ( Array $ranges ) {

      for @$ranges {
        my $data-property = .key;
        my $data-property-iri = OWL::IRI.check-iri($data-property);
        if !?$data-property-iri {
          note "Data property '$data-property' not declared";
          next;
        }

        my $data-type = .value;
        my $data-type-iri = OWL::IRI.check-iri($data-type);
        if !?$data-type-iri {
          note "Data type '$data-type' not declared";
          next;
        }

        if !self.check-if-type( $data-type-iri, :types(<data-type>)) {
          note "Data type '$data-type-iri' not recognized";
          next;
        }

        $!data-properties{$data-property-iri} = $data-type-iri;
say "Data property $data-property-iri set to $data-type-iri";
      }
    }

    #---------------------------------------------------------------------------
    # Array of pairs. Keys are data properties and values are pairs of which
    # the key is a class and the value the value of the class in the proper type
    # of data-property-range.
    #
    method data-property-assertion ( Array $assertions ) {

      for @$assertions {
        my $data-property = .key;
        my $data-property-iri = OWL::IRI.check-iri($data-property);
        if ! ?$data-property-iri {
          note "Data property '$data-property' not declared";
          next;
        }

        # The value is also a pair of 
        my $assertion = .value;

        my $subject = $assertion.key;
        my $subject-iri = OWL::IRI.check-iri($subject);
        if ! ?$subject-iri {
          note "iri of $subject not defined";
        }

        if ! ?$!entities{$subject-iri} {
          note "$subject-iri not declared";
        }

        # Get and check type of value
        #
        my $subject-value = $assertion.value;
        # ...

        $!data-properties{$data-property-iri} = $subject-iri => $subject-value;
say "Data property $data-property-iri: $$subject-iri = $subject-value";
      }
    }

    #---------------------------------------------------------------------------
    # Array of pairs where keys with one value each where key is a data
    # property and value a data value.
    #
    method object-property-range ( Array $ranges ) {
return;
      for @$ranges {
        my $object-property = .key;
        my $object-property-iri = OWL::IRI.check-iri($object-property);
        if !?$object-property-iri {
          note "Data property '$object-property' not declared";
          next;
        }

        my $data-type = .value;
        my $data-type-iri = OWL::IRI.check-iri($data-type);
        if !?$data-type-iri {
          note "Data type '$data-type' not declared";
          next;
        }

        if !self.check-if-type( $data-type-iri, :types(<data-type>)) {
          note "Data type '$data-type-iri' not recognized";
          next;
        }

        $!object-properties{$object-property-iri} = $data-type-iri;
say "Data property $object-property-iri set to $data-type-iri";
      }
    }

    #---------------------------------------------------------------------------
    # Array of pairs. Keys are data properties and values are pairs of which
    # the key is a class and the value the value of the class in the proper type
    # of data-property-range.
    #
    method object-property-assertion ( Array $assertions ) {
return;
      for @$assertions {
        my $object-property = .key;
        my $object-property-iri = OWL::IRI.check-iri($object-property);
        if !?$object-property-iri {
          note "Data property '$object-property' not declared";
          next;
        }
      }
    }

    #---------------------------------------------------------------------------
    # 
    method check-if-type ( OWL::IRI $iri, :$types --> Bool ) {

#say "Check 0: $types, $iri";
      my Bool $check-ok = False;

      for @($!axioms{$iri.Str}) -> $axiom {
#say "Check 1: ", $axiom;
#say "Check 2: ", $axiom.get-type;
        $check-ok = $axiom.get-type ~~ any @$types;
        last if $check-ok;
      }

      return $check-ok;
    }
  }
}
