package Net::Shadowsocks;

# ABSTRACT: An asynchronous, non-blocking shadowsocks client and server module.
use 5.006;
use strict;
use warnings;
use Carp;
use Crypt::Cipher::RC6;
use Crypt::Mode::CBC;
use Crypt::Random qw( makerandom_octet );
use Digest::MD5;
use Mcrypt qw(ARCFOUR RIJNDAEL_128 RIJNDAEL_256);
use Mcrypt qw(:MODES);
use Mcrypt qw(:FUNCS);



our $VERSION = '0.7.0';


sub _EVP_BytesToKey($$$$$) 
{
    my ( $key_len, $iv_len, $salt, $data, $count ) = @_;
    my $md_buf = '';
    my $key    = '';
    if ( $data eq '' ) 
    {
        return $key;
    }
    my $addmd = 0;
    for ( ; ; ) 
    {
        my $md;
        $md = Digest::MD5->new;
        if ( $addmd++ > 0 ) 
        {
            $md->add($md_buf);
        }
        $md->add($data);
        if ( $salt ne '' ) 
        {
            $md->add_bits( $salt, 64 );
        }
        $md_buf = $md->digest();
        for ( my $i = 1 ; $i < $count ; $i++ ) 
        {
            $md->reset();
            $md->add($md_buf);
            $md_buf = $md->digest();
        }
        $key .= $md_buf;
        if ( length($key) >= $key_len ) 
        {
            $key = substr( $key, 0, $key_len );
            last;
        }
    }
    return $key;
}

sub _initialize_cipher($$) 
{
        my %ciphers = 
    (
        "rc4-md5"  => [Mcrypt::ARCFOUR,Mcrypt::STREAM,16,16],
        "rc4-sha" => [Mcrypt::ARCFOUR,Mcrypt::STREAM,16,20],
        "aes-128-cbc"  => [Mcrypt::RIJNDAEL_128,Mcrypt::CBC,16,16],
        "aes-128-cfb"  => [Mcrypt::RIJNDAEL_128,Mcrypt::CFB,16,16],
        "aes-128-ctr"  => [Mcrypt::RIJNDAEL_128,Mcrypt::CTR,16,16],
        "aes-128-ofb" => [Mcrypt::RIJNDAEL_128,Mcrypt::OFB,16,16],
        "aes-256-cbc"  => [Mcrypt::RIJNDAEL_256,Mcrypt::CBC,32,16],
        "aes-256-cfb"  => [Mcrypt::RIJNDAEL_256,Mcrypt::CFB,32,16],
        "aes-256-ctr"  => [Mcrypt::RIJNDAEL_256,Mcrypt::CTR,32,16],
        "aes-256-ofb"  => [Mcrypt::RIJNDAEL_256,Mcrypt::OFB,32,16],
        "rc6" => [undef,undef,16,16]
    );
    my $_method   = shift;
    my $_password = shift;

    my $_encryptor;
    my $_decryptor;
    my $_key;
    my $_iv;
    my $_nonce;
    if ( $_method eq 'rc6') 
    {
        $_key = Net::Shadowsocks::_EVP_BytesToKey( 16, 16, '', $_password, 1 );
        $_iv = makerandom_octet( Length => %ciphers.{$_method}.[3] );
        carp length($_key);
        $_encryptor = Crypt::Mode::CBC->new('RC6');
        $_decryptor = Crypt::Mode::CBC->new('RC6');
    }
    elsif(!defined($_method))
    {
        $_encryptor = Mcrypt::mcrypt_load(%ciphers.{$_method}.[0],'',%ciphers.{$_method}.[1] , '' );
        $_decryptor = Mcrypt::mcrypt_load(%ciphers.{$_method}.[0] ,'',%ciphers.{$_method}.[1], '' );
        $_key = _EVP_BytesToKey( %ciphers.{$_method}.[2], %ciphers.{$_method}.[3], '', $_password, 1 );
        $_iv = makerandom_octet( Length => %ciphers.{$_method}.[3] );
    }
    else
    {
        $_encryptor = Mcrypt::mcrypt_load(Mcrypt::ARCFOUR,'',Mcrypt::STREAM , '' );
        $_decryptor = Mcrypt::mcrypt_load(Mcrypt::ARCFOUR ,'',Mcrypt::STREAM, '' );
        $_key = _EVP_BytesToKey(16, 16, '', $_password, 1 );
        $_iv = makerandom_octet( Length => 16 );
        }
    return $_encryptor, $_decryptor, $_key, $_iv, $_nonce;
}


1;    # End of Net::Shadowsocks

__END__

=pod

=encoding UTF-8

=head1 NAME

Net::Shadowsocks - An asynchronous, non-blocking shadowsocks client and server module.

=head1 VERSION

version 0.7.0

=head1 SYNOPSIS

    Shadowsocks protocol client and server module.
    
    The following encryption methods are supported:

rc4-md5 rc4-sha aes-128-cbc aes-128-cfb aes-128-ofb aes-256-cbc aes-256-cfb aes-256-ofb

Experimental support of RC6 is also provided.

Experimental server TCP Fast Open is also supported if available on Mac OS X and Linux. 

Please also note that both the stock libmcrypt and stock Mcrypt module do not export CTR 
mode, So the aes-128-ctr and aes-256-ctr method will not work out of box. You have to use this 
  patched version of libmcrypt and Libmcrypt module from https://osdn.net/projects/mcrypt/
to get it to work.  

=head1 DESCRIPTION

Net::Shadowsocks is a Perl implementation of the shadowsocks (Chinese: 影梭)
protocol client and server , Shadowsocks is a secure transport protocol based on
SOCKS Protocol Version 5 (RFC 1928 ). 

Design goal of this module is to provide same level of transport security as TLS 
1.2, not more,not less. So it does not implement all the encryption methods 
supported by other Shadowsocks implementations if It is not also a TLS 1.2 cipher. 
To sum it up:

1. The following encryption methods are supported:

RC4-MD5 RC4-SHA AES-128-CBC AES-128-CFB AES-128-OFB AES-256-CBC 
AES-256-CFB AES-256-OFB

experimental support of RC6 is also provided.

2.The following ciphers deprecated by Shadowsocks are not supported: 
bf-cfb chacha20 salsa20

3.The following ciphers not recommended by Shadowsocks are not  supported: 
aes-192-cfb aes-192-ctr camellia-128-cfb camellia-192-cfb camellia-256-cfb 
chacha20-ietf 

4.The following ciphers recommended by Shadowsocks are not supported: 
aes-128-gcm aes-192-gcm aes-256-gcm chacha20-ietf-poly1305 
xchacha20-ietf-poly1305 

The following table shows TLS 1.2 cipher suite definitions and  equivalent
encryptions methods used in this module: 

TLS 1.2 Cipher Suite Definitions              Cipher                Encryption sMethod

TLS_RSA_WITH_RC4_128_MD5                RC4_128            RC4-MD5
TLS_RSA_WITH_RC4_128_SHA                 RC4_128            RC4-SHA
TLS_RSA_WITH_AES_128_CBC_SHA        AES_128_CBC    AES-128-CBC/CFB/OFB
TLS_RSA_WITH_AES_256_CBC_SHA        AES_256_CBC    AES-256-CBC/CFB/OFB
TLS_RSA_WITH_AES_128_CBC_SHA256  AES_128_CBC    AES-128-CBC/CFB/OFB
TLS_RSA_WITH_AES_256_CBC_SHA256  AES_256_CBC    AES-256-CBC/CFB/OFB

Please note TLS 1.2 has removed IDEA and DES cipher suites. and because of 
CVE-2016-2183,  http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2016-2183
, this module has removed all support for DES and 3DES ciphers.

Please also note that both the stock libmcrypt and stock Mcrypt module do not export CTR 
mode, So the aes-128-ctr and aes-256-ctr method will not work out of box. You have to use this 
  patched version of libmcrypt and Libmcrypt module from https://osdn.net/projects/mcrypt/
to get it to work.  

Project website https://osdn.net/projects/ssperl/

Net::Shadowsocks

=head1 VERSION

Version 0.7.0

=head1 AUTHOR

Li ZHOU, C<< <lzh at cpan.org> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-net-shadowsocks at rt.cpan.org>, or through
the web interface at L<http://rt.cpan.org/NoAuth/ReportBug.html?Queue=Net-Shadowsocks>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.

=head1 SUPPORT

You can find documentation for this module with the perldoc command.

perldoc Net::Shadowsocks

You can also look for information at:

=over 4

=item * RT: CPAN's request tracker (report bugs here)

L<http://rt.cpan.org/NoAuth/Bugs.html?Dist=Net-Shadowsocks>

=item * AnnoCPAN: Annotated CPAN documentation

L<http://annocpan.org/dist/Net-Shadowsocks>

=item * CPAN Ratings

L<http://cpanratings.perl.org/d/Net-Shadowsocks>

=item * Search CPAN

L<http://search.cpan.org/dist/Net-Shadowsocks/>

=back

=head1 ACKNOWLEDGEMENTS

=head1 LICENSE AND COPYRIGHT

Copyright 2017 Li ZHOU.

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

L<http://www.perlfoundation.org/artistic_license_2_0>

Any use, modification, and distribution of the Standard or Modified
Versions is governed by this Artistic License. By using, modifying or
distributing the Package, you accept this license. Do not use, modify,
or distribute the Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made
by someone other than you, you are nevertheless required to ensure that
your Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service
mark, tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge
patent license to make, have made, use, offer to sell, sell, import and
otherwise transfer the Package with respect to any patent claims
licensable by the Copyright Holder that are necessarily infringed by the
Package. If you institute patent litigation (including a cross-claim or
counterclaim) against any party alleging that the Package constitutes
direct or contributory patent infringement, then this Artistic License
to you shall terminate on the date that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
AND CONTRIBUTORS " AS IS ' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE, OR NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY
YOUR LOCAL LAW. UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR
CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, OR
CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE OF THE PACKAGE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=head1 AUTHOR

Li ZHOU <lzh@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2017 by Li ZHOU.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
