package Mojolicious::Plugin::GzipStatic;
use Mojo::Base 'Mojolicious::Plugin';
use Mojo::Util qw(monkey_patch);
use IO::Compress::Gzip 'gzip';

our $VERSION = '0.03';

sub register {
    my ($self, $app) = @_;

    monkey_patch 'Mojolicious::Static', serve => sub {
        my ($self, $c, $rel) = @_;

        return undef unless my $asset = $self->file($rel);
        my $headers = $c->res->headers;

        my $types = $c->app->types;
        my $type = $rel =~ /\.(\w+)$/ ? $types->type($1) : undef;
        if (defined($type)
            && $type =~ /text|xml|javascript|json/
            && ($c->req->headers->accept_encoding // '') =~ /gzip/i) {
            $c->res->headers->append(Vary => 'Accept-Encoding');

            $c->res->headers->content_encoding('gzip');
            gzip \$asset->slurp, \my $compressed;
            $asset = Mojo::Asset::Memory->new;
            $asset->add_chunk($compressed);
        }

        return !!$self->serve_asset($c, $asset) if $headers->content_type;

        # Content-Type
        $headers->content_type($type || $types->type('txt'));
        return !!$self->serve_asset($c, $asset);
    };
}

1;
__END__

=encoding utf8

=head1 NAME

Mojolicious::Plugin::GzipStatic - Mojolicious Plugin to to compress the static files before serving them.

=head1 SYNOPSIS

  # Mojolicious
  $self->plugin('GzipStatic');

  # Mojolicious::Lite
  plugin 'GzipStatic';

=head1 DESCRIPTION

L<Mojolicious::Plugin::GzipStatic> is a L<Mojolicious> plugin that monkey_patches 
L<Mojolicious::Static> to compress the static files before serving them.

See L<https://en.wikipedia.org/wiki/HTTP_compression> and
L<http://mojolicious.org/perldoc/Mojolicious/Guides/Rendering#Post-processing-dynamic-content>.

=head1 METHODS

L<Mojolicious::Plugin::GzipStatic> inherits all methods from
L<Mojolicious::Plugin> and implements the following new ones.

=head2 register

  $plugin->register(Mojolicious->new);

Register plugin in L<Mojolicious> application.

=head1 SEE ALSO

L<Mojolicious>, L<Mojolicious::Guides>, L<http://mojolicious.org>, L<Mojolicious::Static>, L<IO::Compress::Gzip>.

=cut
