package Regexp::Pattern::License::Parts;

use utf8;
use strictures 2;

=head1 NAME

Regexp::Pattern::License::Parts - Regular expressions for licensing sub-parts

=head1 VERSION

Version v3.0.27

=cut

our $VERSION = version->declare("v3.0.27");

=head1 SYNOPSIS

 use Regexp::Pattern; # exports re()
 my $re = re("License::Parts::mit_advertising");

=head1 DESCRIPTION

L<Regexp::Pattern::License::Parts> provides a hash of regular expression patterns
related to sub-parts of legal software licenses.

L<Regexp::Pattern> is a convention for organizing reusable regex patterns.

=head1 PATTERNS

=head2 Single part

Patterns each covering a single part, e.g. a clause or a disclaimer phrase.

=over

=item * ad_verbatim

=item * ad_always

=item * cc

=item * cc_by

=item * cc_cc0

=item * cc_nc

=item * cc_nd

=item * cc_sa

=item * cc_sp

=item * cc_url

=item * cc_url_pd

=item * derive_altermark

=item * derive_no_relicense

=item * derive_share

=item * discl

=item * discl_provide

=item * discl_disclaim

=item * name_no_ad

=item * origin_no_misrepresent

=item * origin_no_misrepresent_ack

=item * permission_granted_agree_fee

=item * permission_use_deal_sw

=item * permission_use_deal_materials

=item * permission_use_fee_agree

=item * permission_person_materials

=item * permission_person_materials

=item * permission_person_sw_doc

=item * promo_no_author

=item * provided_no_warranty

=item * repro_notice_cond_discl

=item * retain_notice

=item * retain_notice_cond_discl

=item * use_sw_doc_any

=item * version_later_para

=item * version_later_postfix

=item * version_number

=item * version_prefix

=back

=cut

my $D  = qr/[-–]/;     # dash
my $SD = qr/[ -–]/;    # space or dash

# permission clause
my $granted = qr/Permission is hereby granted/;

# purpose
my $to_use      = qr/to use, copy, modify, and distribute/;
my $to_deal_sw  = qr/to deal in the Software without restriction/;
my $to_deal_mat = qr/to deal in the Materials without restriction/;
my $any_purpose = qr/for any purpose/;
my $redist_src  = qr/Redistributions of source code/;
my $redist_bin  = qr/Redistributions in binary form/;
my $redist_any  = qr/Redistributions of any form whatsoever/;
my $promote     = qr/endorse or promote products derived from this software/;

# text
my $notice       = qr/the above copyright notice/;
my $cond         = qr/this list of conditions/;
my $discl        = qr/the following disclaimer/;
my $no_agreement = qr/without written agreement/;
my $no_permit    = qr/without specific prior written permission/;
my $ack          = qr/the following acknowledge?ment/;
my $product_by   = qr/This product includes software developed by/;

# money
my $no_charge = qr/free of charge/;
my $no_fee    = qr/without fee/;
my $no_fees   = qr/without license or royalty fees/;

# asset
my $sw_doc      = qr/this software and its documentation/;
my $sw_docfiles = qr/this software and associated documentation files/;
my $doc         = qr/the documentation/;
my $other_mat   = qr/other materials provided with the distribution/;
my $ad_mat
	= qr/All advertising materials mentioning features or use of this software/;

# agent
my $to_person = qr/to any person obtaining a copy/;
my $no_authors
	= qr/Neither the (?:names? .*|authors?) nor the names of(?: (?:its|their|other|any))? contributors/;

# origin
my $origin_no_misrepresent = qr/must not be misrepresented/;
my $origin_ack             = qr/an acknowledgment \(see the following\)/;

# forking
my $derive_altermark
	= qr/Altered source versions must be plainly marked as such/;

# invariant parts
my $invariant_notice = qr/This notice may not be removed or altered/;

our %RE = (
	ad_verbatim => {
		caption => 'advertisement (verbatim)',
		tags    => ['clause'],
		pat     => qr/$ad_mat must display $ack[^a-zA-Z]+$product_by/,
	},
	ad_always => {
		caption => 'advertisement',
		tags    => ['clause'],
		pat     => qr/$redist_any must retain $ack[^a-zA-Z]+$product_by/,
	},
	cc => {
		caption => 'Creative Commons qualifier',
		tags    => ['qualifier'],
		pat     => qr/(?:Creative Commons|CC)/,
	},
	cc_by => {
		caption => 'Creative Commons Attribution component',
		tags    => ['component'],
		pat     => qr/(?:Attribution)/,
	},
	cc_cc0 => {
		caption => 'Creative Commons CC0 component',
		tags    => ['component'],
		pat     => qr/(?:CC0|Zero|0)/,
	},
	cc_nc => {
		caption => 'Creative Commons NonCommercial component',
		tags    => ['component'],
		pat     => qr/(?:Non$SD?Commercial)/,
	},
	cc_nd => {
		caption => 'Creative Commons NoDerivatives component',
		tags    => ['component'],
		pat     => qr/(?:No$SD?Deriv(?:ative)?s)/,
	},
	cc_sa => {
		caption => 'Creative Commons ShareAlike component',
		tags    => ['component'],
		pat     => qr/(?:Share$SD?Alike)/,
	},
	cc_sp => {
		caption => 'Creative Commons Sampling Plus component',
		tags    => ['component'],
		pat     => qr/(?:Sampling$SD?Plus)/,
	},
	cc_url => {
		caption => 'Creative Commons URL',
		tags    => ['url'],
		pat     => qr"(?:(?:https?:?)?(?://)?creativecommons.org/licenses/)",
	},
	cc_url_pd => {
		caption => 'Creative Commons stem (public domain)',
		tags    => ['url'],
		pat => qr"(?:(?:https?:?)?(?://)?creativecommons.org/publicdomain/)",
	},
	derive_altermark => {
		caption => 'deriving (must plainly mark)',
		tags    => ['clause'],
		pat     => qr/$derive_altermark/,
	},
	derive_no_relicense => {
		caption => 'deriving (no relicense)',
		tags    => ['clause'],
		pat     => qr/may not be "?relicensed"?/,
	},
	derive_share => {
		caption => 'deriving (must share source)',
		tags    => ['clause'],
		pat =>
			qr/source code of the altered version of this software available/,
	},
	discl => {
		caption => 'disclaimer',
		tags    => ['component'],
		pat     => qr/(?:CC0|Zero|0)/,
		pat     => qr/THIS SOFTWARE IS PROVIDED/,
	},
	discl_provide => {
		caption => 'disclaimer (provides)',
		tags    => ['disclaimer'],
		pat     => qr/(?:\S+ ){1,15}PROVIDES? THIS SOFTWARE/,
	},
	discl_disclaim => {
		caption => 'disclaimer (disclaims)',
		tags    => ['disclaimer'],
		pat     => qr/(?:\S+ ){1,15}DISCLAIMS? ALL WARRANTIES/,
	},
	name_no_ad => {
		caption => 'no promo (advertising)',
		tags    => ['clause'],
		pat     => qr/without specific, written prior permission/,
	},
	origin_no_misrepresent => {
		caption => 'origin (no misrepresentation)',
		tags    => ['clause'],
		pat     => qr/$origin_no_misrepresent/,
	},
	origin_no_misrepresent_ack => {
		caption => 'origin (no misrepresentation, acknowledgement)',
		tags    => ['clause'],
		pat     => qr/$origin_no_misrepresent.*, $origin_ack/,
	},
	permission_granted_agree_fee => {
		caption => 'permission (granted, no agreement or fee)',
		tags    => ['clause'],
		pat     => qr/$granted, $no_agreement and $no_fees/,
	},
	permission_use_deal_sw => {
		caption => 'usage (deal, software, any purpose)',
		tags    => ['clause'],
		pat     => qr/$to_deal_sw/,
	},
	permission_use_deal_materials => {
		caption => 'usage (deal, materials, any purpose)',
		tags    => ['clause'],
		pat     => qr/$to_deal_mat/,
	},
	permission_use_fee_agree => {
		caption => 'permission (use, no fee or agreement)',
		tags    => ['clause'],
		pat     => qr/and without a written agreement/,
	},
	permission_person_materials => {
		caption => 'permission (granted, no charge, person, materials)',
		tags    => ['clause'],
		pat =>
			qr/$granted, $no_charge, $to_person of $sw_docfiles \(the "?Materials"?\)/,
	},
	permission_person_sw_doc => {
		caption => 'permission (granted, no charge, person, software)',
		tags    => ['clause'],
		pat =>
			qr/$granted, $no_charge, $to_person of $sw_docfiles \(the "?Software"?\)/,
	},
	promo_no_author => {
		caption => 'no author in promos',
		tags    => ['clause'],
		pat =>
			qr/(?:$no_authors may|The name.*? (?:may|must) not) be used to $promote $no_permit/,
	},
	provided_no_warranty => {
		caption => 'provided (no warranty)',
		tags    => ['clause'],
		pat     => qr/without express or implied warranty/,
	},
	repro_notice_cond_discl => {
		caption => 'reproduce in doc (notice, conditions, disclaimer)',
		tags    => ['clause'],
		pat =>
			qr/$redist_bin must reproduce $notice, $cond and $discl in $doc and\/or $other_mat/,
	},
	retain_notice => {
		caption => 'retain in source (notice)',
		tags    => ['clause'],
		pat     => qr/This notice may not be removed or altered/,
	},
	retain_notice_cond_discl => {
		caption => 'retain in source (notice, conditions, disclaimer)',
		tags    => ['clause'],
		pat     => qr/$redist_src must retain $notice,? $cond and $discl/,
	},
	use_sw_doc_any => {
		caption => 'usage (software and documentation, any purpose)',
		tags    => ['clause'],
		pat     => qr/$to_use $sw_doc $any_purpose/,
	},
	version_later_para => {
		caption => 'version "or later" (paragraph)',
		tags    => ['trait'],
		pat     => qr/Later versions are permitted/,
	},
	version_later_postfix => {
		caption => 'version "or later" (postfix)',
		tags    => ['trait'],
		pat =>
			qr/(?:and|or)(?: ?\(?at your option\)?)?(?: any)? (?:later|newer)(?: version)?/,
	},
	version_number => {
		caption => 'version number',
		tags    => ['component'],
		pat     => qr/\d(?:\.\d+)*/,
	},
	version_prefix => {
		caption => 'version prefix',
		tags    => ['component'],
		pat     => qr/[Vv](?:ersion |ERSION |\.? ?)/,
	},
);

=encoding UTF-8

=head1 AUTHOR

Jonas Smedegaard C<< <dr@jones.dk> >>

=head1 COPYRIGHT AND LICENSE

  Copyright © 2016 Jonas Smedegaard

This program is free software; you can redistribute it and/or modify it
under the terms of the GNU General Public License as published by the
Free Software Foundation; either version 3, or (at your option) any
later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License along
with this program. If not, see <https://www.gnu.org/licenses/>.

=cut

1;
