package App::Software::License; # git description: v0.03-7-gba62de5
# ABSTRACT: Command-line interface to Software::License
# KEYWORDS: license licence LICENSE generate distribution build tool

our $VERSION = '0.04';

use Moose;
use File::HomeDir;
use File::Spec::Functions qw/catfile/;
use Module::Runtime qw/use_module/;
use Software::License;

use namespace::autoclean;

with qw/MooseX::Getopt MooseX::SimpleConfig/;

#pod =head1 SYNOPSIS
#pod
#pod     software-license --holder 'J. Random Hacker' --license Perl_5 --type notice
#pod
#pod =head1 DESCRIPTION
#pod
#pod This module provides a command-line interface to Software::License. It can be
#pod used to easily produce license notices to be included in other documents.
#pod
#pod All the attributes documented below are available as command-line options
#pod through L<MooseX::Getopt> and can also be configured in
#pod F<$HOME/.software_license.conf> though L<MooseX::SimpleConfig>.
#pod
#pod =cut

#pod =attr holder
#pod
#pod Name of the license holder.
#pod
#pod =cut

has holder => (
    is       => 'ro',
    isa      => 'Str',
    required => 1,
);

#pod =attr year
#pod
#pod Year to be used in the copyright notice.
#pod
#pod =cut

has year => (
    is       => 'ro',
    isa      => 'Num',
);

#pod =attr license
#pod
#pod Name of the license to use. Must be the name of a module available under the
#pod Software::License:: namespace. Defaults to Perl_5.
#pod
#pod =cut

has license => (
    is      => 'ro',
    isa     => 'Str',
    default => 'Perl_5',
);

#pod =attr type
#pod
#pod The type of license notice you'd like to generate. Available values are:
#pod
#pod B<* notice>
#pod
#pod This method returns a snippet of text, usually a few lines, indicating the
#pod copyright holder and year of copyright, as well as an indication of the license
#pod under which the software is distributed.
#pod
#pod B<* license>
#pod
#pod This method returns the full text of the license.
#pod
#pod =for :stopwords fulltext
#pod
#pod B<* fulltext>
#pod
#pod This method returns the complete text of the license, preceded by the copyright
#pod notice.
#pod
#pod B<* version>
#pod
#pod =for :stopwords versioned
#pod
#pod This method returns the version of the license.  If the license is not
#pod versioned, this returns nothing.
#pod
#pod B<* meta_yml_name>
#pod
#pod This method returns the string that should be used for this license in the CPAN
#pod META.yml file, or nothing if there is no known string to use.
#pod
#pod =for Pod::Coverage run
#pod
#pod =cut

has type => (
    is      => 'ro',
    isa     => 'Str',
    default => 'notice',
);

has '+configfile' => (
    default => catfile(File::HomeDir->my_home, '.software_license.conf'),
);

has _software_license => (
    is      => 'ro',
    isa     => 'Software::License',
    lazy    => 1,
    builder => '_build__software_license',
    handles => {
        notice   => 'notice',
        text     => 'license',
        fulltext => 'fulltext',
        version  => 'version',
    },
);

sub _build__software_license {
    my ($self) = @_;
    my $class = "Software::License::${\$self->license}";
    return use_module($class)->new({
        holder => $self->holder,
        year   => $self->year,
    });
}

around BUILDARGS => sub {
    my $orig = shift;
    my $self = shift;
    my $args = $self->$orig(@_);
    $args->{license} = $args->{extra_argv}->[0]
        if @{ $args->{extra_argv} };
    return $args;
};

around get_config_from_file => sub {
    my $orig = shift;
    my $ret;
    eval { $ret = $orig->(@_); };
    return $ret;
};

sub run {
    my ($self) = @_;
    my $meth = $self->type;
    print $self->_software_license->$meth;
}

__PACKAGE__->meta->make_immutable;

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

App::Software::License - Command-line interface to Software::License

=head1 VERSION

version 0.04

=head1 SYNOPSIS

    software-license --holder 'J. Random Hacker' --license Perl_5 --type notice

=head1 DESCRIPTION

This module provides a command-line interface to Software::License. It can be
used to easily produce license notices to be included in other documents.

All the attributes documented below are available as command-line options
through L<MooseX::Getopt> and can also be configured in
F<$HOME/.software_license.conf> though L<MooseX::SimpleConfig>.

=head1 ATTRIBUTES

=head2 holder

Name of the license holder.

=head2 year

Year to be used in the copyright notice.

=head2 license

Name of the license to use. Must be the name of a module available under the
Software::License:: namespace. Defaults to Perl_5.

=head2 type

The type of license notice you'd like to generate. Available values are:

B<* notice>

This method returns a snippet of text, usually a few lines, indicating the
copyright holder and year of copyright, as well as an indication of the license
under which the software is distributed.

B<* license>

This method returns the full text of the license.

=for :stopwords fulltext

B<* fulltext>

This method returns the complete text of the license, preceded by the copyright
notice.

B<* version>

=for :stopwords versioned

This method returns the version of the license.  If the license is not
versioned, this returns nothing.

B<* meta_yml_name>

This method returns the string that should be used for this license in the CPAN
META.yml file, or nothing if there is no known string to use.

=for Pod::Coverage run

=head1 AUTHOR

Florian Ragwitz <rafl@debian.org>

=head1 CONTRIBUTORS

=for stopwords Karen Etheridge Randy Stauner

=over 4

=item *

Karen Etheridge <ether@cpan.org>

=item *

Randy Stauner <rwstauner@cpan.org>

=back

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2009 by Florian Ragwitz.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
