#
#   $Id: 14_test_secant.t,v 1.4 2007/02/20 14:13:35 erwan Exp $
#
#   test Math::Polynom->secant
#

use strict;
use warnings;
use Test::More tests => 37;
use lib "../lib/";

use Math::Polynom;

sub alike {
    my($v1,$v2,$precision) = @_;
    if ( abs(int($v1-$v2)) <= $precision) {
	return 1;
    }
    return 0;
}

sub test_secant {
    my($p,$args,$want)=@_;
    my $precision = $args->{precision} || 0.1;
    my $v = $p->secant(%$args);
    ok(alike($v,$want,$precision), $p->stringify." ->secant(p0 => ".$args->{p0}.", p1 => ".$args->{p1}.", precision => $precision) = $want (got $v)");
}

my $p1 = Math::Polynom->new(2 => 1, 0 => -4);
is($p1->iterations,0,"p1->iterations=0 before search");
test_secant($p1, {p0 => 0.5, p1 => 1}, 2);
is($p1->iterations,4,"p1->iterations=3 after search");
test_secant($p1, {p0 => 0.5, p1 => 1}, 2);
test_secant($p1, {p0 => -5, p1 => -1}, -2);
test_secant($p1, {p0 => 0.5, p1 => 1}, 2);
test_secant($p1, {p0 => 0.5, p1 => 1, precision => 0.0001}, 2);
test_secant($p1, {p0 => 0.5, p1 => 1, precision => 0.0000001}, 2);
test_secant($p1, {p0 => 0, p1 => -100, precision => 0.0000001}, -2);
is($p1->iterations,15,"p1->iterations=14 after search");
test_secant($p1, {p0 => 0.5, p1 => 1}, 2);
is($p1->iterations,4,"p1->iterations=3 after simpler search");


my $p2 = Math::Polynom->new(5 => 5, 3.2 => 4, 0.9 => -2);  # 5*x^5 + 4*x^3.2 - 2*x^0.9
test_secant($p2, {p0 => 0.5, p1 => 1, precision => 0.000000000000001}, 0.6161718040343);

eval { test_secant($p2, {p0 => 0.5, p1 => 1, precision => 0.000000000000001}, 0.6161718040343); };
ok((!defined $@ || $@ eq ''),"secant() does not fails on polynom 2 with negative guess (newton_raphson would)");

my $p3 = Math::Polynom->new(2 => 1, 1 => -2, 0 => 1); # x^2 -2*x +1
test_secant($p3,{p0 => 0.5, p1 => 1},1);
test_secant($p3,{p0 => 500, p1 => -500},1);
test_secant($p3,{p0 => 100000, p1 => 99999},1);

# TODO: handle calculation overflows...
my $v;
eval { $v = $p3->secant(p0 => 100000000000000000, p1 => 999999999999999999999, max_depth => 1); };
ok((defined $@ && $@ =~ /reached maximum number of iterations/),"secant() fails when max_depth reached");
ok($p3->error_message =~ /reached maximum number of iterations/,"\$p3->error_message looks good");
is($p3->error,Math::Polynom::ERROR_MAX_DEPTH,"\$p3->error looks good");

# empty polynom error
my $p4 = Math::Polynom->new();
eval { $p4->secant(p0 => 0, p1 => 1); };
ok((defined $@ && $@ =~ /empty polynom/),"secant() fails on empty polynom");
ok($p4->error_message =~ /empty polynom/,"\$p4->error_message looks good");
is($p4->error,Math::Polynom::ERROR_EMPTY_POLYNOM,"\$p4->error looks good");

# a tuff one: the slope leads to a negative next try, while the polynom contains a root -> complex value
my $p5 = Math::Polynom->new(0.2 => 2, 0 => -1); # 2*x^0.2-1
eval {
    $v = $p5->secant(p0 => 0, p1 => 10);
};
ok((defined $@ && $@ =~ /not a real number/),"secant() fails on polynom 2*x^0.2 - 1");
ok($p5->error_message =~ /not a real number/,"\$p5->error_message looks good");
is($p5->error,Math::Polynom::ERROR_NAN,"\$p5->error looks good");

# fault handling
eval { $p4->secant(p0 => 0, p1 => 0); };
ok((defined $@ && $@ =~ /same value for p0 and p1/),"secant() fails when p0 == p1");

eval {$p1->secant(p0 => undef, p1 => 0); };
ok((defined $@ && $@ =~ /got undefined p0/),"p0 => undef");

eval {$p1->secant(p0 => 0, p1 => undef); };
ok((defined $@ && $@ =~ /got undefined p1/),"p1 => undef");

eval {$p1->secant(precision => undef); };
ok((defined $@ && $@ =~ /got undefined precision/),"precision => undef");

eval {$p1->secant(p0 => 'abc', p1 => 0); };
ok((defined $@ && $@ =~ /got non numeric p0/),"p0 => 'abc'");

eval {$p1->secant(p0 => 0, p1 => 'abc'); };
ok((defined $@ && $@ =~ /got non numeric p1/),"p1 => 'abc'");

eval {$p1->secant(precision => 'abc'); };
ok((defined $@ && $@ =~ /got non numeric precision/),"precision => 'abc'");

# this ugly polynom makes secant converge toward something that is not a root... (real life example)
my $p6 = Math::Polynom->new(
			    6.67123287671233 => 36808.3333333333,
			    4.4986301369863  => 29291.6666666667,
			    4.0027397260274  => 29291.6666666667,
			    5.33424657534247 => 35683.3333333333,
			    7.16712328767123 => 39583.3333333333,
			    0.161643835616438 => 25983.3333333333,
			    7.92054794520548 => 39583.3333333333,
			    8.83835616438356 => 42475,
			    0.580821917808219 => 25983.3333333333,
			    9.5041095890411 => 43000,
			    7.67123287671233 => 39583.3333333333,
			    0.915068493150685 => 25983.3333333333,
			    6.33424657534247 => 36808.3333333333,
			    8.16712328767123 => 42475,
			    5.41917808219178 => 35683.3333333333,
			    8.92054794520548 => 42475,
			    2.08767123287671 => 25216.6666666667,
			    10.3369863013699 => 45683.3333333333,
			    10.9232876712329 => 45683.3333333333,
			    8.67123287671233 => 42475,
			    7.5013698630137 => 39583.3333333333,
			    9.58904109589041 => 43000,
			    12.013698630137 => -4996365,
			    10.5890410958904 => 45683.3333333333,
			    6.41917808219178 => 36808.3333333333,
			    7.33424657534247 => 39583.3333333333,
			    3.08767123287671 => 25100,
			    0 => 25983.3333333333,
			    1 => 28400,
			    5.25205479452055 => 35683.3333333333,
			    5.0027397260274 => 35683.3333333333,
			    1.91780821917808 => 28400,
			    10.841095890411 => 45683.3333333333,
			    8.33424657534247 => 42475,
			    7.41917808219178 => 39583.3333333333,
			    10.4219178082192 => 45683.3333333333,
			    4.08767123287671 => 29291.6666666667,
			    0.246575342465753 => 25983.3333333333,
			    10.0082191780822 => 45683.3333333333,
			    0.413698630136986 => 25983.3333333333,
			    6.25205479452055 => 36808.3333333333,
			    0.665753424657534 => 25983.3333333333,
			    9.75616438356164 => 43000,
			    2.91780821917808 => 25216.6666666667,
			    0.832876712328767 => 25983.3333333333,
			    8.41917808219178 => 42475,
			    5.08767123287671 => 35683.3333333333,
			    7.25205479452055 => 39583.3333333333,
			    8.5013698630137 => 42475,
			    0.0849315068493151 => 25983.3333333333,
			    3.91780821917808 => 25100,
			    1.83561643835616 => 28400,
			    8.25205479452055 => 42475,
			    9.16986301369863 => 43000,
			    9.92328767123288 => 43000,
			    4.91780821917808 => 29291.6666666667,
			    10.6739726027397 => 45683.3333333333,
			    9.67397260273973 => 43000,
			    2.83561643835616 => 25216.6666666667,
			    9.42191780821918 => 43000,
			    10.5041095890411 => 45683.3333333333,
			    1.4986301369863 => 28400,
			    6.09041095890411 => 36808.3333333333,
			    1.24931506849315 => 28400,
			    3.83561643835616 => 25100,
			    9.33698630136986 => 43000,
			    1.58356164383562 => 28400,
			    10.2547945205479 => 45683.3333333333,
			    7.09041095890411 => 39583.3333333333,
			    1.33150684931507 => 28400,
			    2.24931506849315 => 25216.6666666667,
			    9.84109589041096 => 43000,
			    4.83561643835616 => 29291.6666666667,
			    1.66849315068493 => 28400,
			    2.58356164383562 => 25216.6666666667,
			    8.09041095890411 => 42475,
			    1.41643835616438 => 28400,
			    2.33150684931507 => 25216.6666666667,
			    3.24931506849315 => 25100,
			    10.1698630136986 => 45683.3333333333,
			    0.495890410958904 => 25983.3333333333,
			    10.7561643835616 => 45683.3333333333,
			    1.75068493150685 => 28400,
			    2.66849315068493 => 25216.6666666667,
			    3.58356164383562 => 25100,
			    2.4986301369863 => 25216.6666666667,
			    2.0027397260274 => 25216.6666666667,
			    9.09041095890411 => 43000,
			    5.58630136986301 => 35683.3333333333,
			    9.25479452054794 => 43000,
			    2.41643835616438 => 25216.6666666667,
			    3.33150684931507 => 25100,
			    4.24931506849315 => 29291.6666666667,
			    2.75068493150685 => 25216.6666666667,
			    3.66849315068493 => 25100,
			    4.58356164383562 => 29291.6666666667,
			    6.58630136986301 => 36808.3333333333,
			    3.41643835616438 => 25100,
			    4.33150684931507 => 29291.6666666667,
			    6.00547945205479 => 36808.3333333333,
			    1.16438356164384 => 28400,
			    3.75068493150685 => 25100,
			    4.66849315068493 => 29291.6666666667,
			    5.5013698630137 => 35683.3333333333,
			    7.58630136986301 => 39583.3333333333,
			    5.75342465753425 => 35683.3333333333,
			    4.41643835616438 => 29291.6666666667,
			    7.00547945205479 => 39583.3333333333,
			    1.08493150684932 => 28400,
			    2.16438356164384 => 25216.6666666667,
			    0.328767123287671 => 25983.3333333333,
			    4.75068493150685 => 29291.6666666667,
			    3.4986301369863 => 25100,
			    3.0027397260274 => 25100,
			    0.747945205479452 => 25983.3333333333,
			    8.58630136986301 => 42475,
			    5.83835616438356 => 35683.3333333333,
			    6.75342465753425 => 36808.3333333333,
			    8.00547945205479 => 42475,
			    3.16438356164384 => 25100,
			    5.16712328767123 => 35683.3333333333,
			    5.92054794520548 => 35683.3333333333,
			    6.83835616438356 => 36808.3333333333,
			    7.75342465753425 => 39583.3333333333,
			    9.00547945205479 => 43000,
			    10.0931506849315 => 45683.3333333333,
			    4.16438356164384 => 29291.6666666667,
			    5.67123287671233 => 35683.3333333333,
			    6.5013698630137 => 36808.3333333333,
			    6.16712328767123 => 36808.3333333333,
			    6.92054794520548 => 36808.3333333333,
			    7.83835616438356 => 39583.3333333333,
			    8.75342465753425 => 42475,
			    );

eval {  $p6->secant(p0 => 0.5, p1 => 1, max_depth => 50, precision => 0.001); };
ok((defined $@ && $@ =~ /doesn.t appear to be a root/),"secant() fails with not a root message");
ok($p6->error_message =~ /doesn.t appear to be a root/,"\$p6->error_message looks good");
is($p6->error,Math::Polynom::ERROR_NOT_A_ROOT,"\$p6->error looks good");


