NAME
====

Discogs::API - Provide basic API to Discogs

SYNOPSIS
========

```raku
use Discogs::API;
my $discogs = Discogs.new;

my $release = $discogs.release(249504);
```

DESCRIPTION
===========

Discogs::API provides a Raku library with access to the [Discogs](https://discogs.com) data and functions. It tries to follow the API as closely as possible, so the up-to-date [Discogs developer information](https://www.discogs.com/developers) can be used to look up matters that are unclear from thie documentation in this module.

One exception to this rule is that fieldnames in the JSON generated by Discogs that are using snake_case, are converted to use kebab-case in the Raku interface. So a field called `allows_multiple_values` in the JSON blob, will be accessible using a `allow-multiple-values` method in this module.

UTILITY METHODS
===============

new
---

```raku
my $discogs = Discogs.new(
  client   => $client,     # Cro::HTTP::Client compatible, optional
  token    => "xfhgh1624", # Discogs access token, default: none
  key      => "kahgjkhdg", # Discogs access key, default: none
  secret   => "454215642", # Discogs access secret, default: none
  currency => "EUR",       # default: "USD"
  per-page => 10,          # default: 50
;
```

Create an object to access the services that the Discogs API has to offer.

  * client - a Cro::HTTP::Client compatible client

One will be provided if not specified.

  * token - Discogs Access Token

A token needed to access certain parts of the Discogs API. See [https://www.discogs.com/settings/developers](https://www.discogs.com/settings/developers) for more information. Defaults to whatever is specified with the DISCOGS_TOKEN environment variable.

  * key - Discogs Access Key

A string needed to access certain parts of the Discogs API. See [https://www.discogs.com/developers#page:authentication](https://www.discogs.com/developers#page:authentication) for more information.

  * secret - Discogs Access Secret

A string needed to access certain parts of the Discogs API. See [https://www.discogs.com/developers#page:authentication](https://www.discogs.com/developers#page:authentication) for more information.

  * currency

A string indicating the default currency to be used when producing prices of releases in the Discogs Marketplace. It should be one of the following strings:

    USD GBP EUR CAD AUD JPY CHF MXN BRL NZD SEK ZAR

  * per-page

An integer indicating the default number of items per page that should be produced by methods that return objects that support pagination.

client
------

```raku
my $default = Discogs::API.client;  # the default client

my $client = $discogs.client;       # the actual client to be used
```

Return the default `Cro::HTTP::Client` object when called as a class method. That object will be used by default when creating a `Discogs::API` object. Intended to be used as a base for alterations, e.g. by overriding the `GET` method during testing.

Returns the actual object that was (implicitely) specified during the creation of the `Discogs::API` object when called as an instance method.

GET
---

```raku
my $content = $discogs.GET("/artists/108713", Discogs::API::Artist);
```

Helper method to fetch data using the Discogs API for the given URI, and interpret it as data of the given class. Returns an instance of the given class, or throws if something went wrong.

test-with
---------

```raku
my $discogs = Discogs::API.new.test-with($path);
```

Provide an alternate [Discogs::API](Discogs::API) object that can be used for testing. Instead of fetching the data from Discogs, it will look at the indicated path and map the URI to a file with the same name. So an URI like `"/artist/108713"` and a path `"foo/bar".IO` will look for a file called `"foo/bar/artist/108713.json"`, slurp that, and create the requested object out of that.

CONTENT METHODS
===============

artist
------

```raku
my $artist = $discogs.artist(108713);
```

Fetch the information for the given artist ID and return that in an [Discogs::API::Artist](Discogs::API::Artist) object.

artist-releases
---------------

```raku
my $artist-releases = $discogs.artist-releases(
  108713,         # the artist ID
  page     => 2,  # page number, default: 1
  per-page => 25, # items per page, default: object

  sort => "year", # sort on given key, default no sort
  sort-order => "desc", # sort order, default to "asc"
);
```

Fetch all of the releases of given artist ID and return them in pages in a [Discogs::API::ArtistReleases](Discogs::API::ArtistReleases) object.

  * page

An integer indicating the page to obtain the `ArtistRelease` objects of. Defaults to 1.

  * per-page

An integer indicating the maximum number of items per page to be produced. Defaults to what was (implicitely) specified with the creation of the `Discogs::API` object.

  * sort

A string indicating how the `ArtistRelease` objects to be returned. Defaults to no sorting. The following fields can be specified:

    year title format

  * sort-order

A string indicating the sort order of any sort action to be performed on the `ArtistRelease` objects to be returned. Defaults to "asc". The following fields can be specified:

    asc desc

community-release-rating
------------------------

```raku
my $rating = $discogs.community-release-rating(249504);

my $rating = $discogs.community-release-rating($release);
```

Fetch the information about the Discogs community rating for a given release and return that as an [Discogs::API::CommunityReleaseRating](Discogs::API::CommunityReleaseRating) object.

The release parameter can either be given as an unsigned integer, or as an [Discogs::API::Release](Discogs::API::Release) object.

label
-----

```raku
my $label = $discogs.label(1);
```

Fetch the information for the given label ID and return that in an [Discogs::API::Label](Discogs::API::Label) object.

label-releases
--------------

```raku
my $label-releases = $discogs.label-releases(
  1,              # the label ID
  page     => 2,  # page number, default: 1
  per-page => 25, # items per page, default: object
);
```

Fetch all of the releases of given label ID and return them in pages in a [Discogs::API::LabelReleases](Discogs::API::LabelReleases) object.

master-release
--------------

```raku
my $master-release = $discogs.master-release(1000);
```

Fetch the information for the given master release ID and return that in an [Discogs::API::MasterRelease](Discogs::API::MasterRelease) object.

master-release-versions
-----------------------

```raku
my $master-release-versions = $discogs.master-release-versions(
  1000,                 # the master release ID
  page     => 2,        # page number, default: 1
  per-page => 25,       # items per page, default: object

  format => "CD",       # limit to format, default no limit
  label  => "Foo",      # limit to label, default no limit
  released => 1992,     # limit to year, default no limit
  country => "Belgium", # limit to country, default no limit
  sort => "released",   # sort on given key, default no sort
  sort-order => "desc", # sort order, default to "asc"
);
```

Fetch all of the versions of a given master release ID and return them in pages in a [Discogs::API::MasterReleaseVersions](Discogs::API::MasterReleaseVersions) object. It supports the following optional named parameters:

  * page

An integer indicating the page to obtain the `MasterReleaseVersion` objects of. Defaults to 1.

  * per-page

An integer indicating the maximum number of items per page to be produced. Defaults to what was (implicitely) specified with the creation of the `Discogs::API` object.

  * format

A string indicating the `format` of the `MasterReleaseVersion` objects to be returned. Defaults to no limitation on format.

  * label

A string indicating the `label` of the `MasterReleaseVersion` objects to be returned. Defaults to no limitation on label.

  * released

An integer indicating the year of release of the `MasterReleaseVersion` objects to be returned. Defaults to no limitation on year.

  * country

A string indicating the `country` of the `MasterReleaseVersion` objects to be returned. Defaults to no limitation on country.

  * sort

A string indicating how the `MasterReleaseVersion` objects to be returned. Defaults to no sorting. The following fields can be specified:

    released title format label catno country

  * sort-order

A string indicating the sort order of any sort action to be performed on the `MasterReleaseVersion` objects to be returned. Defaults to "asc". The following fields can be specified:

    asc desc

release
-------

```raku
my $release = $discogs.release(249504, currency => "EUR");
```

Fetch the information for the given release ID and return that in an [Discogs::API::Release](Discogs::API::Release) object. Optionally takes a named `currency` parameter that should have one of the supported currency strings. This defaults to the value for the currency that was (implicitely) specified when creating the `Discogs::API` object.

search
------

Perform a general search in the Discogs database, optionally searching specific parts. Returns a [Discogs::API::SearchResults](Discogs::API::SearchResults) object.

```raku
my $search = $discogs.search(
  "nirvana",              # optional, general query
  page     => 2,          # page number, default: 1
  per-page => 25,         # items per page, default: object

  type => "release",      # optional search for type only
  title => "nits - urk",  # optional artist - release search query
  release-title => "urk", # optional search for to release name
  credit => "kurt",       # optional search for credits
  artist => "nirvana",    # optional search for artist name
  anv => "nirvana",       # optional search for artist name variation
  label => "Foo",         # optional search for label
  genre => "rock",        # optional search for genre
  style => "grunge",      # optional search for style
  country => "belgium",   # optional search for country
  year => 1992,           # optional search for year of release
  format => "CD",         # optional search for format
  catno => "DGCD-24425",  # optional search for catalog number
  barcode => "7 2064-24425-2 4", # optional search for barcode
  track => "smells like", # optional search for title of track
  submitter => "milKt",   # optional search for username of submitter
  contributor => "liz",   # optional search for username of contributor
);
```

  * page

An integer indicating the page to obtain the `Discogs::API::SearchResult` objects of. Defaults to 1.

  * per-page

An integer indicating the maximum number of items per page to be produced. Defaults to what was (implicitely) specified with the creation of the `Discogs::API` object.

  * query

The string to be searched for.

  * type

A string to determine which main fields to be searched. Should be one of:

    release master artist label

  * title

Special formatted string to search for an artist / release title combination. The hyphen indicates the separation, so e.g. "nirvana - nevermind".

  * release-title

Search for given string as title of a release only.

  * credit

Search for given string as credit for a release only.

  * artist

Search for given string as main name of artist only.

  * anv

Search for given string as alternative name variation of artist only.

  * label

Search for given string as name of label only.

  * genre

Search for given string as name of genre only.

  * style

Search for given string as name of style only.

  * country

Search for given string as name of country only.

  * year

Search for given year of release only.

  * format

Search for given string as format only.

  * catno

Search for given string as catalog number only.

  * barcode

Search for given string as barcode only.

  * track

Search for given string as track title only.

  * submitter

Search for given string as the username of a submitter only.

  * submitter

Search for given string as the username of a contributor only.

user-release-rating
-------------------

```raku
my $rating = $discogs.user-release-rating(249504, "username");

my $rating = $discogs.user-release-rating($release, "username");

my $rating = $discogs.user-release-rating(249504, $user);

my $rating = $discogs.user-release-rating($release, $user);
```

Fetch the information about the rating for a given release and a username and return that as a [Discogs::API::UserReleaseRating](Discogs::API::UserReleaseRating) object.

The release parameter can either be given as an unsigned integer, or as an [Discogs::API::Release](Discogs::API::Release) object. The user parameter can either be given as a string, or as an [Discogs::API::User](Discogs::API::User) object.

ADDITIONAL CLASSES
==================

In alphatical order:

Discogs::API::Artist
--------------------

  * data-quality

String indicating the quality of the data.

  * id

The artist ID.

  * images

A list of [Discogs::API::Image](Discogs::API::Image) objects for this artist.

  * members

A list of [Discogs::API::Member](Discogs::API::Member) objects of this artist.

  * name

String with the main name of the artist.

  * namevariations

A list of strings with alternate names / spellings of the artist.

  * profile

A string with a profile of the artist.

  * releases-url

The URL to fetch all of the releases of this Artist using the Discogs API.

  * resource-url

The URL to fetch this object using the Discogs API.

  * uri

The URL to access information about this artist on the Discogs website.

  * urls

A list of URLs associated with this artist.

Discogs::API::ArtistReleases
----------------------------

This object is usually created as part of a [Discogs::API::ArtistReleases](Discogs::API::ArtistReleases) object.

  * artist

A string with the name of the artist of this release.

  * community-in-collection

An unsigned integer indicating the number of people in the Discogs community that have this release.

  * community-in-wantlist

An integer indicating the number of people in the Discogs community that want to have this release.

  * format

A string indicating the format of this release.

  * id

The ID of this release.

  * label

The ID of the associated `Discogs::API::Label` object.

  * resource-url

The URL to obtain the information about this release using the Discogs API.

  * role

A string indicating the role of this release compared to its [Discogs::API::MasterRelease](Discogs::API::MasterRelease) object.

  * stats

An [Discogs::API::Stats](Discogs::API::Stats) object with user and community statistics. It is probably easier to use the short-cut methods `community-in-collection`, `community-in-wantlist`, `user-in-collection`, `user-in-wantlist`.

  * status

A string indicating the status of the information about this release.

  * thumb

A URL for a thumbnail image associated with this release.

  * title

A string with the title of this release.

  * type

A string indicating the type of release, e.g. "master".

  * user-in-collection

A boolean indicating whether the current user has this release.

  * user-in-wantlist

A boolean indicating whether the current user wants to have this release.

  * year

An unsigned integer for the year this release was released.

Discogs::API::ArtistReleases
----------------------------

Retrieves a list of all [Discogs::API::ArtistRelease](Discogs::API::ArtistRelease) objects that were made by the given artist ID, and pagination settings.

  * first-page

Returns the first page of the information of this object, or `Nil` if already on the first page.

  * first-page-url

The URL to fetch the data of the **first** page of this object using the Discogs API. Returns `Nil` if the there is only one page of information available.

  * items

An integer indicating the total number of [Discogs::API::LabelRelease](Discogs::API::LabelRelease) objects there are available for this artist.

  * last-page

Returns the last page of the information of this object, or `Nil` if already on the last page.

  * last-page-url

The URL to fetch the data of the **last** page of this object using the Discogs API. Returns `Nil` if already on the last page.

  * next-page

Returns the next page of the information of this object, or `Nil` if already on the last page.

  * next-page-url

The URL to fetch the data of the **next** page of this object using the Discogs API. Returns `Nil` if already on the last page.

  * page

An integer indicating the page number of this object.

  * pages

An integer indicating the number of pages of information available for this object.

  * pagination

The [Discogs::API::Pagination](Discogs::API::Pagination) object associted with this object. Usually not needed, as its information is available in shortcut methods.

  * per-page

An integer representing the maximum number of items on a page.

  * previous-page

Returns the previous page of the information of this object, or `Nil` if already on the first page.

  * previous-page-url

The URL to fetch the data of the **previous** page of this object using the Discogs API. Returns `Nil` if already on the first page.

  * releases

A list of [Discogs::API::ArtistRelease](Discogs::API::ArtistRelease) objects.

Discogs::API::ArtistSummary
---------------------------

  * anv

A string with the artist name variation.

  * id

The artist ID.

  * join

A string indicating joining.

  * name

A string with the name.

  * resource-url

The URL to fetch the full artist information using the Discogs API.

  * role

A string indicating the role of this artist.

  * tracks

A string indicating the tracks on which the artist participated.

Discogs::API::CatalogEntry
--------------------------

An object that describes entities in the Discogs database that are also referred to as `Label`s. Usually created indirectly by other objects.

  * catno

A string with the identifying catalog number.

  * entity-type

An unsigned integer with a description of the type of this entity.

  * entity-type-name

A string with the name of this entity.

  * id

The numeric ID of this catalog entry.

  * name

The name of this catalog entry.

  * resource-url

The URL to fetch the full information of this catalog entry using the Discogs API.

Discogs::API::Community
-----------------------

Usually obtained indirectly from the `community` method on the [Discogs::API::Release](Discogs::API::Release) object. These methods can also be called directly on the [Discogs::API::Release](Discogs::API::Release) object, as these are also provided as shortcuts.

  * contributors

A list of [Discogs::API::User](Discogs::API::User) objects of contributors to the community information of this release.

  * data-quality

A string describing the quality of the data of this release.

  * have

An integer indicating how many community members have this release.

  * rating

A rational number indicating the rating the members of the community have given this release.

  * status

The status of the information about this release in the community.

  * submitter

The [Discogs::API::User](Discogs::API::User) object for the submitter of this release.

  * want

An integer indicating how many community members want to have this release.

Discogs::API::CommunityReleaseRating
------------------------------------

The rating of the Discogs community for a specific release.

  * rating

A rational number indicating the rating.

  * release-id

An unsigned integer for the ID of the release.

Discogs::API::FilterFacet
-------------------------

An object usually created as part of the [Discogs::API::MasterReleaseVersions](Discogs::API::MasterReleaseVersions) object.

  * allows-multiple-values

A Bool indicating whether more than one value is allowed in `values`.

  * id

The ID.

  * title

The title.

  * values

A list of one or more values.

Discogs::API::Filters
---------------------

An object usually created as part of the [Discogs::API::MasterReleaseVersions](Discogs::API::MasterReleaseVersions) object.

  * applied

A hash indicating which [Discogs::API::FilterFacet](Discogs::API::FilterFacet)s have been applied.

  * available

A hash of unsigned integer indicating how many entries are available.

Discogs::API::Format
--------------------

An object that describes the format of a release. Usually created by other objects.

  * descriptions

A list of strings describing this format.

  * name

The name of this format.

  * qty

An unsigned integer indicating the number of copies that are available in this format in the Discogs Marketplace.

Discogs::API::Identifier
------------------------

A generic object created by other objects.

  * type

A string indicating the type.

  * value

A string indicating the value.

Discogs::API::Image
-------------------

An object describing an image in the Discogs database. Usually created by other objects.

  * height

The height of the image in pixels.

  * resource-url

The URL to access this image on the Discogs image website.

  * type

String with the type for this image: either "primary" or "secondary".

  * uri

The URL to access this image on the Discogs image website.

  * uri150

The URL to access a 150x150 pixel version of the image on the Discogs image website.

  * width

The width of the image in pixels.

Discogs::API::Label
-------------------

The `Label` object represents a label, company, recording studio, location, or other entity involved with [Discogs::API::Artist](Discogs::API::Artist)s and [Discogs::API::Release](Discogs::API::Release)s. Labels were recently expanded in scope to include things that aren't labels – the name is an artifact of this history.

  * contact-info

A string with contact information for this label.

  * data-quality

A string describing the quality of the data of this label.

  * id

The ID of this label.

  * images

A list of [Discogs::API::Image](Discogs::API::Image) objects for this label.

  * name

A string with the name of this label.

  * profile

A string with a profile about this label.

  * releases-url

A URL to retrieve all the [Discogs::API::Release](Discogs::API::Release) objects associated with this label using the Discogs API.

  * resource-url

The URL to obtain the information about this label using the Discogs API.

  * sublabels

A list of [Discogs::API::SubLabel](Discogs::API::SubLabel) objects describing subdivisions of this label.

  * uri

A URL to see the information of this label on the Discogs website.

  * urls

A list of URLs related to this label.

Discogs::API::LabelRelease
--------------------------

This object is usually created as part of a [Discogs::API::LabelReleases](Discogs::API::LabelReleases) object.

  * artist

A string with the name of the artist of this release.

  * catno

A string with the catalog number of this release.

  * format

A string with the format of this release.

  * id

An unsigned integer for the ID of this release.

  * resource-url

A URL to get the full release information of this release using the Discogs API.

  * status

The status of the information about this release in the community.

  * thumb

A URL for a thumbnail image for this release.

  * title

A string for the title of this release.

  * year

An unsigned integer indicating the year this release was released.

Discogs::API::LabelReleases
---------------------------

Retrieves a list of all [Discogs::API::LabelRelease](Discogs::API::LabelRelease) objects that are versions of a given master release ID, and pagination settings.

  * first-page

Returns the first page of the information of this object, or `Nil` if already on the first page.

  * first-page-url

The URL to fetch the data of the **first** page of this object using the Discogs API. Returns `Nil` if the there is only one page of information available.

  * items

An integer indicating the total number of [Discogs::API::LabelRelease](Discogs::API::LabelRelease) objects there are available for label.

  * last-page

Returns the last page of the information of this object, or `Nil` if already on the last page.

  * last-page-url

The URL to fetch the data of the **last** page of this object using the Discogs API. Returns `Nil` if already on the last page.

  * next-page

Returns the next page of the information of this object, or `Nil` if already on the last page.

  * next-page-url

The URL to fetch the data of the **next** page of this object using the Discogs API. Returns `Nil` if already on the last page.

  * page

An integer indicating the page number of this object.

  * pages

An integer indicating the number of pages of information available for this object.

  * pagination

The [Discogs::API::Pagination](Discogs::API::Pagination) object associted with this object. Usually not needed, as its information is available in shortcut methods.

  * per-page

An integer representing the maximum number of items on a page.

  * previous-page

Returns the previous page of the information of this object, or `Nil` if already on the first page.

  * previous-page-url

The URL to fetch the data of the **previous** page of this object using the Discogs API. Returns `Nil` if already on the first page.

  * releases

A list of [Discogs::API::LabelRelease](Discogs::API::LabelRelease) objects.

Discogs::API::MasterRelease
---------------------------

The MasterRelease object represents a set of similar [Discogs::API::Release](Discogs::API::Release)s. Master releases have a "main release" which is often the chronologically earliest.

  * artists

A list of [Discogs::API::ArtistSummary](Discogs::API::ArtistSummary) objects for this master release.

  * data-quality

A string describing the quality of the data of this master release.

  * fetch-main-release

Fetch the main [Discogs::API::Release](Discogs::API::Release) of this main release using the Discogs API.

  * fetch-most-recent-release

Fetch the most recent [Discogs::API::Release](Discogs::API::Release) of this main release using the Discogs API.

  * genres

A list of strings describing the genres of this master release.

  * id

The ID of this master release.

  * images

A list if [Discogs::API::Image](Discogs::API::Image) objects associated with this master release.

  * lowest-price

The lowest price seen for any of the releases of this master release on the Discogs Marketplace, in the currency that was (implicitely) specified when the [Discogs::API](Discogs::API) object was made.

  * main-release

The ID of the [Discogs::API::Release](Discogs::API::Release) object that is considered to be the main release.

  * main-release-url

The URL to access the data of the main release using the Discogs API.

  * most-recent-release

The ID of the [Discogs::API::Release](Discogs::API::Release) object that is considered to be the most recent release.

  * most-recent-release-url

The URL to access the data of the most recent release using the Discogs API.

  * num-for-sale

An integer indicating the number of copies of any release of this main release, that are for sale on the Discogs Marketplace.

  * resource-url

The URL to obtain the information about this master release using the Discogs API.

  * styles

A list of strings describing the styles of this master release.

  * title

A string with the title of this master release.

  * tracklist

A list of [Discogs::API::Track](Discogs::API::Track) objects describing the tracks of this master release.

  * uri

A URL to see the information of this master release on the Discogs website.

  * versions-url

A URL to fetch the [Discogs::API::MasterReleaseVersion](Discogs::API::MasterReleaseVersion) objects for this master release using the Discogs API.

  * videos

A list of [Discogs::API::Video](Discogs::API::Video) objects associated with this master release.

  * year

An integer for the year in which this master release was released.

Discogs::API::MasterReleaseVersion
----------------------------------

This object is usually created as part of the [Discogs::API::MasterReleaseVersions](Discogs::API::MasterReleaseVersions) object.

  * catno

The catalog number of the associated [Discogs::API::CatalogEntry](Discogs::API::CatalogEntry) object.

  * community-in-collection

An unsigned integer indicating the number of people in the Discogs community that have this release.

  * community-in-wantlist

An unsigned integer indicating the number of people in the Discogs community that want to have this release.

  * country

A string indicating the country of this release.

  * format

A string indicating the format of this release.

  * id

The ID of this release.

  * label

The ID of the associated `Discogs::API::Label` object.

  * major-formats

A string indicating the major formats in which this release is available.

  * released

An unsigned integer indicating the year that this release was released.

  * resource-url

The URL to obtain the information about this release using the Discogs API.

  * stats

An [Discogs::API::Stats](Discogs::API::Stats) object with user and community statistics. It is probably easier to use the short-cut methods `community-in-collection`, `community-in-wantlist`, `user-in-collection`, `user-in-wantlist`.

  * status

A string indicating the status of the information about this release.

  * thumb

A URL for a thumbnail image associated with this release.

  * title

A string with the title of this release.

  * user-in-collection

An unsigned integer indicating whether the current user has this release.

  * user-in-wantlist

An unsigned integer indicating whether the current user wants to have this release.

Discogs::API::MasterReleaseVersions
-----------------------------------

Retrieves a list of all [Discogs::API::MasterReleaseVersion](Discogs::API::MasterReleaseVersion) objects that are versions of a given master release ID, and pagination settings.

  * filter-facets

A list of [Discogs::API::FilterFacet](Discogs::API::FilterFacet) objects associated with this object.

  * filters

A list of [Discogs::API::Filter](Discogs::API::Filter) objects associated with this object.

  * first-page

Returns the first page of the information of this object, or `Nil` if already on the first page.

  * first-page-url

The URL to fetch the data of the **first** page of this object using the Discogs API. Returns `Nil` if the there is only one page of information available.

  * items

An integer indicating the total number of [Discogs::API::MasterReleaseVersion](Discogs::API::MasterReleaseVersion) objects there are available for this master release.

  * last-page

Returns the last page of the information of this object, or `Nil` if already on the last page.

  * last-page-url

The URL to fetch the data of the **last** page of this object using the Discogs API. Returns `Nil` if already on the last page.

  * next-page

Returns the next page of the information of this object, or `Nil` if already on the last page.

  * next-page-url

The URL to fetch the data of the **next** page of this object using the Discogs API. Returns `Nil` if already on the last page.

  * page

An integer indicating the page number of this object.

  * pages

An integer indicating the number of pages of information available for this object.

  * pagination

The [Discogs::API::Pagination](Discogs::API::Pagination) object associted with this object. Usually not needed, as its information is available in shortcut methods.

  * per-page

An integer representing the maximum number of items on a page.

  * previous-page

Returns the previous page of the information of this object, or `Nil` if already on the first page.

  * previous-page-url

The URL to fetch the data of the **previous** page of this object using the Discogs API. Returns `Nil` if already on the first page.

  * versions

A list of [Discogs::API::MasterReleaseVersion](Discogs::API::MasterReleaseVersion) objects.

Discogs::API::Member
--------------------

  * active

A Boolean indicating whether this member is still active with the [Discogs::API::Artist](Discogs::API::Artist) it is associated with.

  * id

The ID of this member as a separate [Discogs::API::Artist](Discogs::API::Artist).

  * name

The name of this member.

  * resource-url

The URL to fetch [Discogs::API::Artist](Discogs::API::Artist) object of this member using the Discogs API.

Discogs::API::Pagination
------------------------

This object is usually created as part of some kind of search result that allows for pagination.

  * items

An integer with the number of items in this page.

  * page

An integer with the page number of the information of this page.

  * pages

An integer with the total number of pages available with the current `per-page` value.

  * per-page

An integer with the maximum number of items per page.

  * urls

A hash of URLs for moving between pages. Usually accessed with shortcut methods of the object incorporating this `Pagination` object.

Discogs::API::Rating
--------------------

A rating, usually automatically created with a [Discogs::API::Community](Discogs::API::Community) object.

  * average

A rational value indicating the average rating of the object associated with the associated [Discogs::API::Community](Discogs::API::Community) object.

  * count

An integer value indicating the number of votes cast by community members.

Discogs::API::Release
---------------------

The `Discogs::API::Release` object represents a particular physical or digital object released by one or more [Discogs::API::Artist](Discogs::API::Artist)s.

  * artists

A list of [Discogs::API::ArtistSummary](Discogs::API::ArtistSummary) objects for this release.

  * average

A rational value indicating the average rating of this release by community members.

  * artists-sort

A string with the artists, sorted.

  * community

The [Discogs::API::Community](Discogs::API::Community) object with all of the Discogs community information associated with this release.

  * companies

A list of [Discogs::API::CatalogEntry](Discogs::API::CatalogEntry) objects of entities that had something to do with this release.

  * contributors

A list of [Discogs::API::User](Discogs::API::User) objects of contributors to the community information of this release.

  * count

An integer value indicating the number of votes cast by community members about this release.

  * country

A string with the country of origin of this release.

  * data-quality

String indicating the quality of the data.

  * date-added

A `Date` object of the date this release was added to the Discogs system.

  * date-changed

A `Date` object of the date this release was last changed in the Discogs system.

  * estimated-weight

An integer value to indicate the weight of this release compared to other release in the [Discogs::API::MasterRelease](Discogs::API::MasterRelease).

  * extraartists

A list of [Discogs::API::ArtistSummary](Discogs::API::ArtistSummary) objects for additional artists in this release.

  * fetch-master-release

Fetch the associated [Discogs::API::MasterRelease](Discogs::API::MasterRelease) object.

  * format-quantity

An integer value for the number of formats available for this release.

  * formats

A list of [Discogs::API::Format](Discogs::API::Format) objects that are available for this release.

  * genres

A list of strings describing the genres of this release.

  * have

An integer indicating how many community members have this release.

  * id

The integer value that identifies this release.

  * identifiers

A list of [Discogs::API::Identifier](Discogs::API::Identifier) objects for this release.

  * images

A list of [Discogs::API::Image](Discogs::API::Image) objects for this release.

  * labels

A list of [Discogs::API::CatalogEntry](Discogs::API::CatalogEntry) objects that serve as a "label" for this release.

  * lowest-price

A rational value indicating the lowest price if this release is available in the Discogs Marketplace in the currency that was (implicitely) specified when creating the [Discogs::API](Discogs::API) object.

  * master-id

The integer value of the [Discogs::API::MasterRelease](Discogs::API::MasterRelease) id of this release.

  * master-url

The URL to fetch the master release of this release using the Discogs API.

  * notes

A string with additional notes about this release.

  * num-for-sale

An integer value indicating the number of copies for sale for this release on the Discogs Marketplace.

  * rating

A rational number indicating the rating the members of the community have given this release.

  * released

A string with a machine readable for of the date this release was released.

  * released-formatted

A string with a human readable form of the date this release was released.

  * resource-url

The URL to fetch this [Discogs::API::Release](Discogs::API::Release) object using the Discogs API.

  * series

A list of [Discogs::API::CatalogEntry](Discogs::API::CatalogEntry) objects of which this release is a part of.

  * status

A string indicating the status of the information of this release.

  * styles

A list of strings indicating the styles of this release.

  * submitter

The [Discogs::API::User](Discogs::API::User) object for the submitter of this release.

  * thumb

A URL for a thumbnail image for this release.

  * title

A string with the title of this release.

  * tracklist

A list of [Discogs::API::Track](Discogs::API::Track) objects of this release.

  * uri

The URL to access this release on the Discogs image website.

  * videos

A list of [Discogs::API::Video](Discogs::API::Video) objects associated with this release.

  * want

An integer indicating how many community members want to have this release.

  * year

An integer value of the year this release was released.

Discogs::API::SearchResult
--------------------------

This object is usually created as part of a [Discogs::API::SearchResults](Discogs::API::SearchResults) object.

  * cover_image

A URL with an image describing this search result.

  * id

An unsigned integer for the ID associated with a release.

  * master-id

An unsigned integer for the ID associated with a master release.

  * master-url

A URL to fetch the information of the associated master release using the Discogs API.

  * resource-url

A URL to fetch the full information for this entry using the Discogs API.

  * thumb

A URL for a thumbail image for this entry.

  * title

A string with the title associated for this entry.

  * type

A string indicating the type of entry. Can be any of:

    release master artist label

  * uri

A URI to fetch the full information for this release on the Discogs website.

  * user-data

A [Discogs::API::StatsData](Discogs::API::StatsData) object with data. It's probably easier to use the `user-in-collection` and `user-in-wantlist` methods.

  * user-in-collection

An unsigned integer indicating whether the current user has this entry.

  * user-in-wantlist

An unsigned integer indicating whether the current user wants to have this entry.

Discogs::API::SearchResults
---------------------------

Retrieves a list of [Discogs::API::Searchresult](Discogs::API::Searchresult) objects that match the given query parameters, and pagination settings.

  * first-page

Returns the first page of the information of this object, or `Nil` if already on the first page.

  * first-page-url

The URL to fetch the data of the **first** page of this object using the Discogs API. Returns `Nil` if the there is only one page of information available.

  * items

An integer indicating the total number of [Discogs::API::SearchResult](Discogs::API::SearchResult) objects there are available.

  * last-page

Returns the last page of the information of this object, or `Nil` if already on the last page.

  * last-page-url

The URL to fetch the data of the **last** page of this object using the Discogs API. Returns `Nil` if already on the last page.

  * next-page

Returns the next page of the information of this object, or `Nil` if already on the last page.

  * next-page-url

The URL to fetch the data of the **next** page of this object using the Discogs API. Returns `Nil` if already on the last page.

  * page

An integer indicating the page number of this object.

  * pages

An integer indicating the number of pages of information available for this object.

  * pagination

The [Discogs::API::Pagination](Discogs::API::Pagination) object associted with this object. Usually not needed, as its information is available in shortcut methods.

  * per-page

An integer representing the maximum number of items on a page.

  * previous-page

Returns the previous page of the information of this object, or `Nil` if already on the first page.

  * previous-page-url

The URL to fetch the data of the **previous** page of this object using the Discogs API. Returns `Nil` if already on the first page.

  * results

A list of [Discogs::API::SearchResult](Discogs::API::SearchResult) objects.

Discogs::API::Stats
-------------------

This object is usually created as part of the [Discogs::API::MasterReleaseVersion](Discogs::API::MasterReleaseVersion) object.

  * user

The [Discogs::API::StatsData](Discogs::API::StatsData) object with statistics of the current user.

  * user

The [Discogs::API::StatsData](Discogs::API::StatsData) object with statistics of the Discogs community.

Discogs::API::StatsData
-----------------------

This object is usually created as part of the [Discogs::API::Stats](Discogs::API::Stats) object.

  * in-collection

An unsigned integer indicating how many people in the Discogs community have the associated [Discogs::API::MasterReleaseVersion](Discogs::API::MasterReleaseVersion) in their collection.

  * in-wantlist

An unsigned integer indicating how many people in the Discogs community have the associated [Discogs::API::MasterReleaseVersion](Discogs::API::MasterReleaseVersion) in their want list.

Discogs::API::SubLabel
----------------------

This object is usually created as part of the [Discogs::API::Label](Discogs::API::Label) object.

  * id

The ID of this sublabel.

  * name

A string with the name of this sublabel.

  * resource-url

The URL to get the full [Discogs::API::Label](Discogs::API::Label) information of this `SubLabel` using the Discogs API.

Discogs::API::Track
-------------------

The information about a track on a release, usually created automatically as part of a [Discogs::API::Release](Discogs::API::Release) object.

  * duration

A string indicating the duration of this track, usually as "mm:ss".

  * position

A string indication the position of this track, "A" side or "B" side.

  * title

A string containing the title of this track.

  * type

A string to indicate the type of track, usually "track".

Discogs::API::User
------------------

This object is usually created as part of other `Discogs::API` objects.

  * resource-url

The URL to get the full [Discogs::API::User](Discogs::API::User) information of this user using the Discogs API.

  * username

The string with which the Discogs user is identified.

Discogs::API::UserReleaseRating
-------------------------------

Provide the rating a user has given a release.

  * rating

An unsigned integerr with the rating by this user for a release.

  * release

An unsigned integer for the release ID.

  * username

A string for the username.

Discogs::API::Value
-------------------

An object usually created as part of the [Discogs::API::FilterFacet](Discogs::API::FilterFacet) object.

  * count

An integer indicating an amount.

  * title

A string for the title of this object.

  * value

A string indicating the value of this object.

Discogs::API::Video
-------------------

The information about a video, usually created automatically as part of a [Discogs::API::Release](Discogs::API::Release) object.

  * description

A string containing the description of this `Video` object.

  * duration

A string indicating the duration of the video, usually as "mm:ss".

  * embed

A Bool indicating whether this video can be embedded.

  * title

A string containing the title (usually "artist - title") of this `Video` object.

  * uri

The URL of the video, usually a link to a YouTube video.

AUTHOR
======

Elizabeth Mattijsen <liz@wenzperl.nl>

Source can be located at: https://github.com/lizmat/Discogs-API . Comments and Pull Requests are welcome.

COPYRIGHT AND LICENSE
=====================

Copyright 2020 Elizabeth Mattijsen

This library is free software; you can redistribute it and/or modify it under the Artistic License 2.0.

