package Object::Relation::Meta::Attribute::Schema;

# $Id: Schema.pm 3076 2006-07-28 17:20:08Z theory $

use strict;

use version;
our $VERSION = version->new('0.1.0');

use Object::Relation::Meta::Attribute ':with_dbstore_api';
use base 'Object::Relation::Meta::Attribute';

=head1 Name

Object::Relation::Meta::Attribute::Schema - Object::Relation database store builder

=head1 Synopsis

  # Assuming MyThingy was generated by Object::Relation::Meta and that we're building
  # a data store schema.
  my $class = MyThingy->my_class;

  print "\nAttributes:\n";
  for my $attr ($class->attributes) {
      print "  o ", $attr->name, $/;
      print "    Column: ", $attr->column, $/;
      if (my $idx = $attr->index) {
          print "    Index: $idx\n";
      }
      if (my $ref = $attr->references) {
          print "    References ", $ref->package, $/;
          print "       On Delete: ", $attr->on_delete, $/;
      }
  }

=head1 Description

This module is provides metadata for all Object::Relation class attributes while
building a storage schema. Loading
L<Object::Relation::Schema|Object::Relation::Schema> causes it to be used instead
of L<Object::Relation::Meta::Attribute|Object::Relation::Meta::Attribute>. This is so that extra
metadata methods are available that are useful in constructing the schema, but
are not otherwise useful when an application is actually in use.

=cut

##############################################################################
# Instance Methods.
##############################################################################

=head1 Instance Interface

=head2 Accessor Methods

=head3 on_delete

  my $on_delete = $attr->on_delete;

Returns a string describing what to do with an object that links to another
object when that other object is deleted. This is only relevant when the
attribute object represents that relationship (that is, when C<references()>
returns true). The possible values for this attributes are:

=over

=item CASCADE

=item RESTRICT

=item SET NULL

=item SET DEFAULT

=item NO ACTION

=back

The default is "RESTRICT".

=cut

sub on_delete { shift->{on_delete} }

##############################################################################

=head3 column

  my $column = $attr->column;

Returns the name of the database table column for this attribute. The table
column name will generally be the same as the attribute name, but for
contained objects, in which the column is a foreign key column, the name will
be the attribute name plus "_id".

=cut

sub column { shift->SUPER::_column(@_) }

##############################################################################

=head3 view_column

  my $view_column = $attr->view_column;

Returns the name of the database view column for this attribute. The view
column name will generally be the same as the column name, but for attributes
that reference other objects, the name will be the class key for the contained
object plus "__id". IOW, contained object foreign key columns have a
double-underscore in views and a single underscore in tables.

=cut

sub view_column { shift->SUPER::_view_column(@_) }

##############################################################################

=head3 foreign_key

  my $fk = $class->foreign_key;

Attributes that are references to another Object::Relation object will need to have a
foreign key constraint. This method returns the name of that constraint, which
starts with "fk_", then the key name the current class, then the name of the
current attribute, followed by '_id'.

For example, the foreign key for the "contact_type" attribute of the "contact"
class is named "fk_contact_contact_type_id" and applies to the "contact" table
and references, of course, the "id" column of the "contact_type" table.

If the C<parent()> method returns a false value, this method returns C<undef>.

=cut

sub foreign_key {
    my $self = shift;
    return unless $self->references;
    return 'fk_' . $self->class->key . '_' . $self->name . '_id';
}

##############################################################################

=head2 Instance Methods

=head3 index

  my $index = $attr->index;
  $idx = $attr->index($class);

Returns the name of an index for the attribute, if the attribute requires an
index. The name of the index is "idx_" plus the class key plus the name of the
attribute. Pass in a Object::Relation::Meta::Class object to use its key for the class
key part of the name. This is useful for creating indexes for classes that
inherit from the class for which the attribute was defined, and prevents
duplicately named indexes when more than one class inherits from a class with
an attribute requiring an index.

=cut

sub index {
    my ($self, $class) = @_;
    return unless $self->indexed || $self->references;
    my $key = $class ? $class->key : $self->class->key;
    my $name = $self->column;
    return "idx_$key\_$name";
}

##############################################################################

=head3 build

This private method overrides the parent C<build()> method in order to gather
extra metadata information necessary for building a data store schema.

=cut

sub build {
    my $self = shift;
    $self->SUPER::build(@_);
    if ($self->references) {
        unless ($self->{on_delete}) {
            my $rel = $self->relationship;
            $self->{on_delete} = $rel eq 'extends' || $rel eq 'mediates'
                ? 'CASCADE'
                : 'RESTRICT';
        }
    } else {
        delete $self->{on_delete};
    }
    return $self;
}

1;
__END__

##############################################################################

=head1 Copyright and License

Copyright (c) 2004-2006 Kineticode, Inc. <info@obj_relode.com>

This module is free software; you can redistribute it and/or modify it under the
same terms as Perl itself.

=cut
