#!/usr/bin/env perl
use strict;
use warnings;
use Test::More;
use Test::Exception;
use lib 'lib';
use Catan::Game::Bank;
use Catan::Game::Player;

BEGIN { use_ok 'Catan::Game::Trade' }

# setup data
my @players = ();
for (1..4)
{
  my $player = Catan::Game::Player->new({number => $_});
  $player->{resources} = {B=>9,G=>9,L=>9,O=>9,W=>9 };
  push @players, $player;
}
my $bank = Catan::Game::Bank->new;

subtest bank_trade => sub {
  my $details = { 1 => {B => 1, G => 1, L => -8 } };
  ok my $trade = Catan::Game::Trade->new($bank, \@players, $details), 'constructor';
  ok $trade->is_with_bank, 'is_with_bank returns true';

  my $expected_details = { 1 => {B => 1, G => 1, L => -8 }, bank => {B => -1, G => -1, L => 8} };
  is_deeply $trade->as_hashref, $expected_details, 'hashref matches expected data';
  cmp_ok @{$trade->resources('1')}, '==', 3, 'player side has 3 resources';
  cmp_ok @{$trade->resources('bank')}, '==', 3, 'bank side has 3 resources';
  cmp_ok @{$trade->players}, '==', 2, 'trade has 2 player';
};

subtest simple_trade_between_2 => sub {
  my $details = { 1 => {L => 1, O => -1}, 2 => {L => -1, O => 1} };
  ok my $trade = Catan::Game::Trade->new($bank, \@players, $details), 'constructor';
  is_deeply $trade->as_hashref, $details, 'hashref matches expected data';
  ok !$trade->is_with_bank, 'is_with_bank returns false';
  cmp_ok @{$trade->resources(1)}, '==', 2, 'trade has 2 resources';
  cmp_ok @{$trade->resources(2)}, '==', 2, 'trade has 2 resources';
  cmp_ok @{$trade->players}, '==', 2, 'trade has 2 players';
};

subtest trade_between_2 => sub {
  my $details = { 1 => {B => 1, G => 1, W => -1 }, 2 => {B => -1, G => -1, W => 1} };
  ok my $trade = Catan::Game::Trade->new($bank, \@players, $details), 'constructor';
  is_deeply $trade->as_hashref, $details, 'hashref matches expected data';
  ok !$trade->is_with_bank, 'is_with_bank returns false';
  cmp_ok @{$trade->resources(1)}, '==', 3, 'trade has 3 resources';
  cmp_ok @{$trade->resources(2)}, '==', 3, 'trade has 3 resources';
  cmp_ok @{$trade->players}, '==', 2, 'trade has 2 players';
};

subtest imbalanced_trade => sub {
  my $details = { 1 => {B => 1, G => 1, W => -2 }, 2 => {B => -2, G => -1, W => 1} };
  dies_ok { Catan::Game::Trade->new($bank, \@players, $details) } 'imbalanced trade dies';
};


subtest three_way_trade => sub {
  my $details = { 1 => {B => 1, G => -1 }, 2 => {B => -2, W => 1}, 3 => {B => 1, G => 1, W => -1} };
  dies_ok { Catan::Game::Trade->new($bank, \@players, $details) } 'three way trade dies';
};

done_testing;
