package Catan::Game::Trade;
$Catan::Game::Trade::VERSION = '0.02';
use warnings;
use strict;
use List::Util 'sum';


sub new
{
  my ($class, $bank, $players, $details, $resource_production) = @_;

  die __PACKAGE__ . ' new requires players and trade detail arguments'
    unless $bank && $bank->isa('Catan::Game::Bank')
      && $details && ref $details eq 'HASH' && keys %$details
      && $players && ref $players eq 'ARRAY' && scalar @$players;

  my $self = bless { }, $class;

  # validate trade details
  for my $player_number (keys %$details)
  {
    my @players = grep($player_number == $_->number, @$players);

    die __PACKAGE__ . ' is for invalid player number'
      unless scalar @players == 1;

    my $resources = $bank->resource_from_notation($details->{$player_number});
    $self->{$player_number} = { player => $players[0], resources => $resources };

    push @{$self->{players}}, $players[0];
  }

  my @players =  @{$self->players};

  if (@players == 1)
  {
    my $player = shift @players;
    my $ratios = $player->ratios;
    push @{$self->{players}}, $bank;

    # check trade ratios if its a bank trade
    my ($allowed, $requested) = (0,0);

    for my $r (@{$self->resources($player->number)})
    {
      if ($r->amount < 0)
      {
        $allowed += (-$r->amount) / $ratios->{$r->code};
      }
      elsif ($r->amount > 0)
      {
        $requested += $r->amount;
      }
      push @{$self->{bank}{resources}}, $r->invert;
    }
    die "$player requested $requested resources, but provided too few resources trade with the bank"
      unless $resource_production || $allowed == $requested;
  }
  # check that the trade contains only 2 players and the amounts balance
  else
  {
    die 'a trade must be between 2 players only!' unless @players == 2;
    my $resources1 = $self->resources($players[0]->number);
    my $resources2 = $self->resources($players[1]->number);

    for my $r (@$resources1)
    {
      my $total2 = sum map { $_->isa(ref $r) ? $_->amount : 0 } @$resources2;
      die 'a trade between 2 players must balance!'
        unless $r->amount + $total2 == 0;
    }
  }

  # check both players can afford the trade
  return $self if $self->can_afford;
}

sub is_with_bank { exists $_[0]->{bank} }
sub players      { $_[0]->{players} }

sub as_hashref
{
  my $self = shift;
  my $rv = {};
  for my $player (@{$self->players})
  {
    for my $r (@{$self->resources($player->number)})
    {
      $rv->{$player->number}{$r->code} = $r->amount;
    }
  }
  return $rv;
}

sub resources_all { [ map { @{$_[0]->resources($_->number)} } @{$_[0]->players} ]}

sub resources
{
  my ($self, $player_number) = @_;

  die 'resources requires a valid player number argument'
    unless $player_number && exists $self->{$player_number};

  return $self->{$player_number}{resources};
}

sub execute
{
  my $self = shift;
  if ($self->can_afford)
  {
    my %results;

    for my $player (@{$self->players})
    {
      for my $r (@{$self->resources($player->number)})
      {
        $player->resources->{$r->code} += $r->amount;
        $results{$player->number}{$r->code} = $r->amount;
      }
    }
    return \%results;
  }
}

sub can_afford
{
  my ($self) = @_;

  for my $player (@{$self->players})
  {
    for (@{$self->{$player->number}{resources}})
    {
      return die "$player does not have enough resources for that"
        unless $player->resources->{$_->code} + $_->quantity >= 0;
    }
  }
  return 1;
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

Catan::Game::Trade

=head1 VERSION

version 0.02

=head1 NAME

Catan::Game::Trade - process trades

=head1 AUTHOR

David Farrell <dfarrell@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is Copyright (c) 2015 by David Farrell.

This is free software, licensed under:

  The (two-clause) FreeBSD License

=cut
