package Archive::Har::Page;

use warnings;
use strict;
use Carp();
use Archive::Har::Page::PageTimings();

our $VERSION = 0.07;

sub new {
	my ($class, $params) = @_;
	my $self = {};
	bless $self, $class;
	if (defined $params) {
		$self->id($params->{id});
		if (defined $params->{title}) {
			$self->title($params->{title});
		}
		if( defined $params->{startedDateTime}) {
			$self->started_date_time($params->{startedDateTime});
		}
		$self->page_timings(Archive::Har::Page::PageTimings->new($params->{pageTimings} || {}));
		if (defined $params->{comment}) {
			$self->comment($params->{comment});
		}
		foreach my $key (sort { $a cmp $b } keys %{$params}) {
			if ($key =~ /^_[[:alnum:]]+$/smx) { # private fields
				if (defined $params->{$key}) {
					$self->$key($params->{$key});
				}
			}
		}
	}
	return $self;
}

sub id {
	my ($self, $new) = @_;
	my $old = $self->{id};
	if (@_ > 1) {
		$self->{id} = $new;
	}
	return $old;
}

sub title {
	my ($self, $new) = @_;
	my $old = $self->{title};
	if (@_ > 1) {
		$self->{title} = $new;
	}
	if (defined $old) {
		return $old;
	} else {
		return 'Unknown';
	}
	return $old;
}

sub started_date_time {
	my ($self, $new) = @_;
	my $old = $self->{startedDateTime};
	if (@_ > 1) {
		if (defined $new) {
			if ($new =~ /^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}\.\d+(?:\+\d{2}:\d{2}|Z)$/smx) {
				$self->{startedDateTime} = $new;
			} else {
				Carp::croak("startedDateTime is not formatted correctly:$new");
			}
		} else {
			$self->{startedDateTime} = undef;
		}
	}
	if (defined $old) {
		return $old;
	} else {
		return '0000-00-00T00:00:00.0+00:00';
	}
}

sub page_timings {
	my ($self, $new) = @_;
	my $old = $self->{pageTimings};
	if (@_ > 1) {
		$self->{pageTimings} = $new;
	}
	return $old;
}

sub comment {
	my ($self, $new) = @_;
	my $old = $self->{comment};
	if (@_ > 1) {
		$self->{comment} = $new;
	}
	return $old;
}

sub AUTOLOAD {
	my ($self, $new) = @_;
	my $type = ref $self or Carp::croak("$self is not an object");

	my $name = $Archive::Har::Page::AUTOLOAD;
	$name =~ s/.*://smx;   # strip fully-qualified portion

	if ($name =~ /^_[[:alnum:]]+$/smx) { # private fields
		my $old = $self->{$name};
		if (@_ > 1) {
			$self->{$name} = $new;
		}
		return $self->{$name};
	} else {
		Carp::croak("$name is not specified in the HAR 1.2 spec and does not start with an underscore");
	}
	return;
}

sub TO_JSON {
	my ($self) = @_;
	my $json = {
			title => $self->title(),
			startedDateTime => $self->started_date_time(),
			pageTimings => $self->page_timings(),
		};
	if (defined $self->id()) {
		$json->{id} = $self->id();
	}
	if (defined $self->comment()) {
		$json->{comment} = $self->comment();
	}
	foreach my $key (sort { $a cmp $b } keys %{$self}) {
		next if (!defined $self->{$key});
		if ($key =~ /^_[[:alnum:]]+$/smx) { # private fields
			$json->{$key} = $self->{$key};
		}
	}
	return $json;
}

1;
__END__
=head1 NAME

Archive::Har::Page - Represents a single page inside the HTTP Archive

=head1 VERSION

Version 0.07

=head1 SYNOPSIS

    use Archive::Har();

    my $http_archive_string = '"log": { "version": "1.1", .... ';
    my $har = Archive::Har->new();
    $har->string($http_archive_string);
    foreach my $page ($har->pages()) {
        print "DateTime: " . $page->started_date_time() . "\n";
        print "Id: " . $page->id() . "\n";
        print "Title: ". $page->title() . "\n";
        my $timing = $page->page_timings();
        print "Comment: " . $page->comment() . "\n";
    }

=head1 SUBROUTINES/METHODS

=head2 started_date_time

returns the date and time stamp for the beginning of the page load (ISO 8601 format)

=head2 id

returns the unique identifier of a page within the Archive. This is referenced by the Archive::Har::Entry objects

=head2 title

returns the page title

=head2 page_timings

returns the page timings object

=head2 comment

returns the comment about the Page

