package OpenInteract::Handler::Box;

# $Id: Box.pm,v 1.7 2001/07/13 14:41:11 lachoy Exp $

use strict;
use SPOPS::Secure qw( :level );
use Data::Dumper qw ( Dumper );

@OpenInteract::Handler::Box::ISA     = ();
$OpenInteract::Handler::Box::VERSION = sprintf("%d.%02d", q$Revision: 1.7 $ =~ /(\d+)\.(\d+)/);

$OpenInteract::Handler::Box::author            = 'chris@cwinters.com';
$OpenInteract::Handler::Box::default_method    = '';
@OpenInteract::Handler::Box::forbidden_methods = ();
%OpenInteract::Handler::Box::security = (
   process_boxes  => SEC_LEVEL_NONE );

use constant DEFAULT_METHOD       => 'handler';
use constant DEFAULT_SEPARATOR    => "<br>\n";
use constant DEFAULT_TEMPLATE     => 'base_box::main_box_shell';
use constant DEFAULT_BOX_WEIGHT   => 5;
use constant MAX_BOX_WEIGHT       => 100;
use constant BLANK_SHELL_TEMPLATE => '_blank_';

sub process_boxes {
    my ( $class ) = @_;
    my $R = OpenInteract::Request->instance;
    my $BOX_CONFIG = $R->CONFIG->{box};
    $R->DEBUG && $R->scrib( 2, "Box configuration: ", Dumper( $BOX_CONFIG ) );
    my $params = {};

    # First, do the system boxes -- this puts box information into the
    # box holding area

    my $system_box_class = $R->CONFIG->{box}->{system_box_handler};
    $system_box_class->handler();

    # If a website has boxes that it's adding on every page it can do so
    # in code rather than in a template

    my $custom_box_class = $R->CONFIG->{box}->{custom_box_handler};
    if ( $custom_box_class ) {
        eval { $custom_box_class->handler() };
        if ( $@ ) {
            $R->scrib( 0, "Cannot call custom box handler: $custom_box_class. Error: $@" );
        }
    }

    # Next, weed out boxes without at least a name and assign
    # class/method or template information as needed

    my @good_boxes = ();

BOX:
    foreach my $box_info ( @{ $R->{boxes} } ) {
   
        # Check to see if they are just a name, in which case they're
        # probably an old (deprecated) call

        unless ( ref $box_info ) {
            $box_info = { name => $box_info, template => $box_info };
            $R->DEBUG && $R->scrib( 1, "Found box with old scalar name-only call: $box_info->{name}" );
        }

        unless ( $box_info->{name} ) {
            $R->scrib( 0, "Box put into the holding area without a name! Not processed.\n",
                          "Info:\n", Dumper( $box_info ) );
            next BOX;
        }

        # If a template, class or coderef are not defined, first try to
        # lookup the box in the action table

        unless ( $box_info->{template} or 
                 $box_info->{class} or 
                 ref $box_info->{code} eq 'CODE' ) {

            # Modify this to lookup ALL information, which can include a
            # title, weight, etc.

            my $lookup_box_info = $R->lookup_action( $box_info->{name}, 
                                                     { skip_default => 1, 
                                                       return       => 'info' } );
            if ( $lookup_box_info->{class} and $lookup_box_info->{method} ) {
                $R->DEBUG && $R->scrib( 1, "Found class ($lookup_box_info->{class}) and ",
                                           "method ($lookup_box_info->{method}) for $box_info->{name}" );
                $box_info->{class}  = $lookup_box_info->{class};
                $box_info->{method} = $lookup_box_info->{method};
                $box_info->{title}  = $lookup_box_info->{title};
                $box_info->{weight} = $lookup_box_info->{weight};
                $box_info->{base_template} = $lookup_box_info->{base_template};        
            }

            # If the box isn't in the action table, then assume that the
            # box-name == box-template-name

            else {
                $box_info->{template} = $box_info->{name};
                $R->DEBUG && $R->scrib( 1, "Found box with old hash name-only call: $box_info->{name}" );
            }
        }

        # Assign default weight if not already there and ensure that the
        # weight isn't too large

        $box_info->{weight} ||= DEFAULT_BOX_WEIGHT;
        next BOX if ( $box_info->{weight} > MAX_BOX_WEIGHT );
        $R->DEBUG && $R->scrib( 1, "Putting box ($box_info->{name}) onto the",
                                   "stack with weight $box_info->{weight}" );
        push @good_boxes, $box_info;
    }

    # Now, sort the boxes by weight then name
  
    my @sorted_boxes = sort { $a->{weight} <=> $b->{weight} ||
                              $a->{name} cmp $b->{name} } 
                       @good_boxes;

    # Grab the template that we'll plug the box content into -- we
    # should probably cache this within the handler

    my $box_template_name = $R->{theme}->property_value( 'box_template' )
                            || $BOX_CONFIG->{default_template}
                            || DEFAULT_TEMPLATE;
    my ( $box_template_package );
    if ( $box_template_name =~ /^(.*)?::(.*)$/ ) {
        $box_template_package = $1;
        $box_template_name    = $2;
    }
    $box_template_package ||= 'base_box';

    my $shell_template = eval { $R->site_template->fetch_by_name( 
                                       $box_template_name,
                                       $box_template_package ) };
    if ( $@ ) {
        my $ei = OpenInteract::Error->set( SPOPS::Error->get );
        $R->throw({ code => 404 } );
        $R->scrib( 0, "Cannot fetch box template by name ($box_template_name): $ei->{system_msg}" );
        return "[[ Boxes failed ]]";
    }
    unless ( $shell_template ) {
        $R->scrib( 0, "No base box template to plug box info in" );
        return "[[ Boxes failed ]]";
    }
  
    # Generate content for each box
  
    my @boxes = ();
    $R->DEBUG && $R->scrib( 2, "Sorted boxes currently in the list:\n", Dumper( \@sorted_boxes ) );
    foreach my $box_info ( @sorted_boxes ) {    
        my $shell_params = {};
        $box_info->{params} ||= {};

        # Treat the box as a component and get the html back

        my ( $html, $info ) = $R->component->handler( $box_info, 'return info' );
        my ( $box_content );

        # If the user has requested to keep this box 'naked', don't
        # wrap it in the shell

        if ( $box_info->{base_template} eq BLANK_SHELL_TEMPLATE() ) {
            $box_content = $html;
            $R->DEBUG && $R->scrib( 1, "No wrapper template used by request, box is naked! (cover your eyes)" );
        }
        else {
            $shell_params->{content} = $html;
            $shell_params->{label} = $box_info->{title} || $box_info->{params}->{title} || $info->{title};
            $R->DEBUG && $R->scrib( 3, "Passing parameters to fill in the box of: ", 
                                       Dumper( $shell_params ) );
            $box_content = $R->template->handler( {}, $shell_params, 
                                                  { object => $shell_template } );
        }
        push @boxes, $box_content;
    }
    
    my $sep_string = $R->{theme}->property_value( 'box_separator' )
                     || $BOX_CONFIG->{default_separator} 
                     || DEFAULT_SEPARATOR;  
    return join( $sep_string, @boxes );
}

1;

__END__

=pod

=head1 NAME

OpenInteract::Handler::Box -- Handle input and output for independent "boxes" 

=head1 SYNOPSIS

 # Define box information in your server.perl

 box => {
    handler           => 'MyWebsite::Handler::Box',
    default_separator => '--',
    default_method    => 'run_box',
    default_template  => 'base_box::main_box_shell',
 },

 # Add a box

 my $zip = $R->{auth}->{user}->{zip_code};
 my $box = { name   => 'current_weather', 
             weight => 2,
             title  => "Weather in Zip Code $zip",
             params => { zip_code => $zip };
 push @{ $R->{boxes} }, $box;

=head1 DESCRIPTION

=head1 CONFIGURATION

This module allows you to define default information in two separate
locations for a number of parameters.

=head2 Server Configuration

In the C<conf/server.perl> file found in every OpenInteract website,
you can define certain information for your boxes. 

=over 4

=item *

B<handler> ($) (mandatory)

Define the class that will be used to process the boxes. Unless you
write your own class, this will B<always> be C<OpenInteract::Box> and
should not be changed.

=item *

B<separator> ($) (optional)

This is the string used to separate boxes. For instance, if you want
to put a short horizontal rule between each line, you could set this
to:

  separator => '<hr width="50%" noshade/>'

Or if you had a custom image you wanted to separate your boxes with:

  separator => '<div align="center"><img src="/images/box_sep.gif" height="2" width="25"/></div>',

This module defines the default separator as '<br>'. It will be used
only if there is no separator defined in the theme or in the server
configuration.

=item *

B<default_method> ($) (optional)

Define the method that will be used by boxes that do not specify
one. This module defines the default method as 'handler' and unless
you know what you are doing you should not change it.

=item *

B<default_template> ($) (optional)

This is the template into which the box content gets put. Normally
this is defined in the theme, but if for some reason someone blanked
the template out this will fill in. 

The default template is 'base_box::main_box_shell', which as the name
would indicate is installed with this package. (OpenInteract will
probably standardize to a 'package::template' naming convention in the
relatively near future.)

=back

=head2 Theme Properties

Two properties of the boxes can be defined on a per-theme basis.

=over 4

=item *

B<box_template> ($) (optional)

This is the template into which the box content gets put. OpenInteract
ships with one theme which has this property set to 'main_box_shell',
which is used if you do not specify anything. However, you can define
additional themes and change the look of a box entirely by modifying
this template.

=item *

B<box_separator> ($) (optional, but recommended)

See the discussion of B<separator> above in the L<Server
Configuration> section.

=back

=head2 Box Properties

An individual box also has a say as to how it will be rendered as well
as the content it will have.

The simplest case is a call:

 push @{ $R->{boxes} }, 'my_box_template';

Which simply uses the scalar passed in as the template name and the
box name, and uses all the defaults. However, you will likely get a
box with a title 'Generic Box', which is probably not what you want.

Another example:

 push @{ $R->{boxes} }, { name     => 'mybox', 
                          template => 'mypkg::mybox',
                          weight   => 1, 
                          title    => 'My First Box' };

Each box can define the following parameters:

=over 4

=item *

B<name> ($)

Just used to identify the box.

=item *

B<title> ($)

Display name of box used in the 'shell' wrapper, if you elect to use
that.

=item *

B<weight> ($)

Number between 1 (top) and 10 (bottom) indicating where you want the
box to be.

=item *

B<box_template> ($) (optional)

If you specify the keyword '_blank_' then your box content will be
'naked' and not wrapped by anything else. If you leave this empty you
will use either the box_template property in your theme, the
'box_template' defined in your server configuration, or the
DEFAULT_TEMPLATE defined in this class.

=item *

B<params> (\%) (optional)

Whatever you pass here will passed through to the template or method
that is implementing the box.

=back

=head1 BUGS 

None known.

=head1 TO DO

B<Cache base templates (wrappers)>

The base template wrapper should be cached in the handler so we do not
need to fetch it every time.

B<Flexible base_template handling>

Right now we allow you to use either the default base_template wrapper
(defined in either the theme or the server config) or none at all. We
need to allow each box to define its own wrapper.

=head1 SEE ALSO

L<OpenInteract::SiteTemplate>, L<OpenInteract::Theme>

=head1 COPYRIGHT

Copyright (c) 2001 Chris Winters. All rights reserved.

This library is free software; you can redistribute it and/or modify
it under the same terms as Perl itself.

=head1 AUTHORS

Chris Winters <chris@cwinters.com>

=cut
