package Mojo::Snoo;
use Moo;

extends 'Mojo::Snoo::Base';

use Mojo::Snoo::Multireddit;
use Mojo::Snoo::Subreddit;
use Mojo::Snoo::Link;
use Mojo::Snoo::Comment;
use Mojo::Snoo::User;

our $VERSION = '0.10';

has content => (is => 'rw');

sub multireddit {
    shift->_create_object('Mojo::Snoo::Multireddit', @_);
}

sub subreddit {
    shift->_create_object('Mojo::Snoo::Subreddit', @_);
}

sub link {
    shift->_create_object('Mojo::Snoo::Link', @_);
}

sub comment {
    shift->_create_object('Mojo::Snoo::Comment', @_);
}

sub user {
    shift->_create_object('Mojo::Snoo::User', @_);
}

1;


__END__

=head1 NAME

Mojo::Snoo - Mojo wrapper for the Reddit API

=head1 DESCRIPTION

L<Mojo::Snoo> is a Perl wrapper for the Reddit API which
relies heavily on the Mojo modules. L<Mojo::Collection>
was the initial inspiration for going the Mojo route.
Skip to L<synopsis|Mojo::Snoo/SYNOPSIS> to see how
L<Mojo::Snoo> can be great for one-liners, quick
scripts, and full-blown applications!

=head1 SYNOPSIS

    use Mojo::Snoo;

    # OAuth ONLY. Reddit is deprecating cookie auth soon.
    my $snoo = Mojo::Snoo->new(
        username      => 'foobar',
        password      => 'very_secret',
        client_id     => 'oauth_client_id',
        client_secret => 'very_secret_oauth',
    );

    # upvote each post in /r/Perl (casing does not matter)
    $snoo->subreddit('Perl')->things->each(
        sub { $_->upvote }
    );

    # Don't want to login? That's fine.
    # You can stick to methods which don't require login.
    # Omitting auth details is nice for one-liners:

    # print names of moderators from /r/Perl
    Mojo::Snoo->new->subreddit('Perl')->mods->each( sub { say $_->name } );

    # or do the same via Mojo::Snoo::Subreddit
    Mojo::Snoo::Subreddit->new('Perl')->mods->each( sub { say $_->name } );

    # print title and author of each post (or "thing") from /r/Perl
    # returns 25 "hot" posts by default
    Mojo::Snoo::Subreddit->new('Perl')->things->each( sub { say $_->title, ' posted by ', $_->author } );

    # get only self posts
    @selfies = Mojo::Snoo::Subreddit->new('Perl')->things->grep( sub { $_->is_self } );

    # get the top 3 controversial posts ("things") on /r/AskReddit
    @things = Mojo::Snoo::Subreddit->new('Perl')->things_contro_all(3);

    # print past week's top video URLs from /r/videos
    Mojo::Snoo::Subreddit->new('Perl')->things_top_week->each( sub { say $_->url } );

    # print the /r/Perl subreddit description
    say Mojo::Snoo->new->subreddit('Perl')->about->description;

    # even fetch a subreddit's header image!
    say Mojo::Snoo->new->subreddit('Perl')->about->header_img;

=head1 METHODS

=head2 multireddit

Returns a L<Mojo::Snoo::Multireddit> object.

=head2 subreddit

Returns a L<Mojo::Snoo::Subreddit> object.

=head2 link

Returns a L<Mojo::Snoo::Link> object.

=head2 comment

Returns a L<Mojo::Snoo::Comment> object.

=head2 user

Returns a L<Mojo::Snoo::User> object.

=head1 WHY SNOO?

Snoo is reddit's alien mascot. Not to be confused
with L<snu-snu|https://en.wikipedia.org/wiki/Amazon_Women_in_the_Mood>.

=head1 API DOCUMENTATION

Please see the official L<Reddit API documentation|http://www.reddit.com/dev/api>
for more details regarding the usage of endpoints. For a better idea of how
OAuth works, see the L<Quick Start|https://github.com/reddit/reddit/wiki/OAuth2-Quick-Start-Example>
and the L<full documentation|https://github.com/reddit/reddit/wiki/OAuth2>. There is
also a lot of useful information of the L<redditdev subreddit|http://www.reddit.com/r/redditdev>.

=head1 SEE ALSO
L<ojo::Snoo>
=cut

=head1 LICENSE

The (two-clause) FreeBSD License. See LICENSE for details.
