/* KinoSearch/Util/SortUtils.h - Sorting utilities.
 *
 * SortUtils includes a merge sort algorithm which allows access to its
 * internals, enabling specialized functions to jump in and only execute part
 * of the sort.
 */

#ifndef H_KINO_SORTUTILS
#define H_KINO_SORTUTILS 1

#include "charmony.h"

typedef int 
(*kino_Sort_compare_t)(void *context, const void *va, const void *vb);

/* Perform a mergesort.  In addition to providing a contiguous array of
 * elements to be sorted and their count, the caller must also provide a
 * scratch buffer with room for at least as many elements as are to be sorted.
 */
void
kino_Sort_mergesort(void *elems, void *scratch, 
                    chy_u32_t num_elems, chy_u32_t width,
                    kino_Sort_compare_t compare, void *context);

/* Standard mergesort function.
 */
void
kino_Sort_do_sort(void *elems, void *scratch, 
                  chy_u32_t left, chy_u32_t right,
                  kino_Sort_compare_t compare, void *context);

/* Merge two source arrays together using the classic mergesort merge
 * algorithm, storing the result in [dest].
 * 
 * Most merge functions operate on a single contiguous array and copy the
 * merged results results back into the source array before returning.  These
 * two differ in that it is possible to operate on two discontiguous source
 * arrays.  Copying the results back into the source array is the
 * responsibility of the caller.
 * 
 * KinoSearch's external sort takes advantage of this when it is reading back
 * pre-sorted runs from disk and merging the streams into a consolidated
 * buffer.
 * 
 * merge4 merges elements which are 4 bytes in size; merge8 merges 8-byte
 * elements.
 */
void
kino_Sort_merge4(void *left_ptr,  chy_u32_t left_num_elems,
                 void *right_ptr, chy_u32_t right_num_elems,
                 void *dest, kino_Sort_compare_t compare, void *context);
void
kino_Sort_merge8(void *left_ptr,  chy_u32_t left_num_elems,
                 void *right_ptr, chy_u32_t right_num_elems,
                 void *dest, kino_Sort_compare_t compare, void *context);


/** Quicksort. 
 */
void
kino_Sort_quicksort(void *elems, size_t num_elems, size_t width, 
                    kino_Sort_compare_t compare, void *context);

#ifdef KINO_USE_SHORT_NAMES
  #define Sort_compare_t            kino_Sort_compare_t
  #define Sort_mergesort            kino_Sort_mergesort
  #define Sort_merge4               kino_Sort_merge4
  #define Sort_merge8               kino_Sort_merge8
  #define Sort_quicksort            kino_Sort_quicksort
#endif

#endif /* H_KINO_SORTUTILS */

/* Copyright 2006-2009 Marvin Humphrey
 *
 * This program is free software; you can redistribute it and/or modify
 * under the same terms as Perl itself.
 */

