package Lingua::ZH::WordSegment;

use 5.008007;
use strict;
use warnings;

require Exporter;
#use AutoLoader qw(AUTOLOAD);

our @ISA = qw(Exporter);
our %EXPORT_TAGS = ( 'all' => [ qw(
	
) ] );
our @EXPORT_OK = ( @{ $EXPORT_TAGS{'all'} } );
our @EXPORT = qw(seg seg_STDIO set_dic);
our $VERSION = '0.02';

my %wordFreqList=();				#Chinese word as key of the hash table, frequency is the value
my %longestWordListStart=();		#Chinese character as key in the hash table, length of the longest word starting with
								# this character is the value in the table

#reading the dictionary
sub set_dic {
	my ($dictFileName)=@_;
	%wordFreqList=();
	%longestWordListStart=();
	if (defined $dictFileName) {
		local *dictFile;
		open dictFile, $dictFileName or die "Can't open file $dictFileName.";
		load_dic(*dictFile);
		close dictFile;
	} else {
		load_dic(\*DATA);
	}
}

sub load_dic {
	my ($dictFile)=@_;
	while(<$dictFile>){
		s/\x0A//;s/\x0D//;s/\x20/\t/g;
		my $line=$_;
		my @entries=split(/\t/,$line);
		my ($thisChnWord,$thisFreq)=@entries;

		#print "$thisChnWord,$thisFreq\n"; #for debug
		$wordFreqList{$thisChnWord}=$thisFreq;

		if($thisChnWord=~/[\x00-\x7F]/){
		}else{
			my $headChar=$1 if ($thisChnWord=~/^([\x80-\xFF].)/);
			my $thisLen=length($thisChnWord)/2;		#the length of the Chinese word in character
			$longestWordListStart{$headChar}=0 if (! defined $longestWordListStart{$headChar});
			$longestWordListStart{$headChar}=$thisLen if($longestWordListStart{$headChar}<$thisLen);
		}
	}	
}

sub seg_STDIO {
	while(<STDIN>){
		s/\x0A//g;s/\x0D//g;
		print seg($_); print "\n";
	}
}

sub seg {
	my ($thisSent)=@_;
	my $finalResult="";
	my $sentLen=length($thisSent);

	my $partialChnString="";
	my $index=0;
	while($index<$sentLen){
		my $thisChar=substr($thisSent, $index,1);
		
		if($thisChar ge "\x80")	{		#this is half of a Chinese character
			$thisChar=substr($thisSent,$index,2);
			$index+=2;
			$partialChnString=$partialChnString.$thisChar;
		}else{
			$index++;

			if($partialChnString ne ""){
				my $partialSegString=segmentZhStr($partialChnString);
				$finalResult=$finalResult.$partialSegString;
				
				$partialChnString="";
				$partialSegString="";
			}

			$finalResult=$finalResult.$thisChar;
		}
	}

	#in case of pure Chinese characters
	if($partialChnString ne ""){
		my $partialSegString=segmentZhStr($partialChnString);
		$finalResult=$finalResult.$partialSegString;
		
		$partialChnString="";
		$partialSegString="";
	}

	$finalResult=~s/^\x20+//;
	$finalResult=~s/\x20+\Z//;
	$finalResult=~s/\x20+/\x20/g;

	return $finalResult;
}

sub segmentZhStr{		#segmenting a string of Chinese characters, there should be no non-Chinese character in the string
	my $inputString=$_[0];
	my $result="";

	#for debug
	#print STDERR "Try to segment string $inputString\n";

	my $lenOfString=length($inputString)/2;
	my @arcTable=();

	#----------------------------------------------------------
	#step0, initialize the arcTable
	for(my $i=0;$i<$lenOfString;$i++){
		for(my $j=0;$j<$lenOfString;$j++){
			if($i==$j){
				$arcTable[$i][$j]=1;				
			}else{
				$arcTable[$i][$j]=-1;
			}
		}
	}


	#-----------------------------------------------------------
	#step1: search for all possible arcs in the input string
	#		and create an array for them

	for(my $currentPos=0;$currentPos<$lenOfString;$currentPos++){		 #currentPos is the index of Chinese character
		my $currentChar=substr($inputString,$currentPos*2,2);

		#from this position, try to find all possible words led by this character
		my $possibleLen=$longestWordListStart{$currentChar};
		$possibleLen=1 if (! defined $possibleLen);
		#for debug
		#print STDERR "\n$currentChar=$possibleLen\n";

		if(($possibleLen+$currentPos)> ($lenOfString-1)){
			$possibleLen=$lenOfString-$currentPos;
		}

		while($possibleLen>=2){		#all possible words with more than 2 characters
			my $subString=substr($inputString,$currentPos*2,$possibleLen*2);

			#for debug
			#print STDERR "s=$subString\n";

			if($wordFreqList{$subString}){
				#for debug
				#print STDERR "$subString found\n";

				$arcTable[$currentPos][$currentPos+$possibleLen-1]=$wordFreqList{$subString};
			}


			$possibleLen--;
		}

	}

	#for debug
	#for($i=0;$i<$lenOfString;$i++){
	#	for($j=0;$j<$lenOfString;$j++){
	#		print "  ",$arcTable[$i][$j];				
	#	}
	#	print "\n";
	#}


	#--------------------------------------------------------------------------
	#step2: from the arc table, try to find the best path as segmentation at
	#each point use the longest possible arc
	# Try from two directions for the search: left to right and right to left
	# using the one with higher product of frequency of the arcs

	my @leftRightSegLabel=();
	my @rightLeftSegLabel=();

	#initialize the segmentation label array
	for(my $k=0;$k<$lenOfString;$k++){
		$leftRightSegLabel[$k]=0;
		$rightLeftSegLabel[$k]=0;
	}
	
	#from left to right
	#-------------------------------
	my $leftToRightFreq=0;
	my $thisCharIndex=0;
	my $charIndexEnd=$lenOfString-1;

	while($thisCharIndex<$lenOfString){
		my $endCharIndex=$charIndexEnd;
		my $found=0;

		while((!$found)&&($endCharIndex>=$thisCharIndex)){
			if($arcTable[$thisCharIndex][$endCharIndex]!=-1){
				$leftToRightFreq+=log($arcTable[$thisCharIndex][$endCharIndex]);
				$found=1;
			}
			else{
				$endCharIndex--;
			}
		}

		$leftRightSegLabel[$endCharIndex]=1;
		$thisCharIndex=$endCharIndex+1;
	}

	#for debug
	#print STDERR @leftRightSegLabel,"\n $leftToRightFreq\n";

	#from right to left
	#---------------------------------
	my $rightToLeftFreq=0;
	$thisCharIndex=$lenOfString-1;

	while($thisCharIndex>=0){
		my $startCharIndex=0;
		my $found=0;
		while((!$found)&&($startCharIndex<=$thisCharIndex)){
			if($arcTable[$startCharIndex][$thisCharIndex]!=-1){
				$found=1;
				$rightToLeftFreq+=log($arcTable[$startCharIndex][$thisCharIndex]);
			}
			else{
				$startCharIndex++;
			}
		}

		$rightLeftSegLabel[$startCharIndex]=1;
		$thisCharIndex=$startCharIndex-1;
	}

	#for debug
	#print STDERR @rightLeftSegLabel,"\n $rightToLeftFreq\n";


	#---------------------------------------------------------------------------------
	# Step3: create result
	if($leftToRightFreq>$rightToLeftFreq){			#using left to right solution, prefer right to left
		for(my $p=0;$p<$lenOfString;$p++){
			$result=$result.substr($inputString, $p*2, 2);

			if($leftRightSegLabel[$p]==1){
				$result=$result." ";
			}
		}
	}
	else{
		for(my $p=0;$p<$lenOfString;$p++){
			if($rightLeftSegLabel[$p]==1){
				$result=$result." ";
			}
			$result=$result.substr($inputString, $p*2, 2);
		}
	}

	$result=~s/^\x20+//;
	$result=~s/\x20+\Z//;

	#for debug
	#print "result=$result\n";

	return " $result ";
}

set_dic();
1;


=head1 NAME

Lingua::ZH::WordSegment - Simple Simplified Chinese Word Segmentation

=head1 VERSION

This document describes version 0.01 of Lingua::ZH::WordSegment, released
September 21, 2006.

=head1 SYNOPSIS

	use Lingua::ZH::WordSegment;
	print seg($str_in);
	seg_STDIO();# Read from STDIN, and print the segmented result to STDOUT

	set_dic($dictionary_file_name); #load word from the file, this is not a must
	perl -MLingua::ZH::WordSegment -e 'seg_STDIO();' < input_file > output_file
	
=head1 DESCRIPTION

This code is mainly written by Joy, joy@cs.cmu.edu in July 4th, 2001.

This program is a perl version of left-right mandarin segmentor
As LDC segmenter takes a long time to build the DB files which makes the
the training process last too long time.

For ablation experiments, we do not need to create the DB files because the
specific frequency dictionary will be used only once for each slice.

The algorithm for this segmenter is to search the longest word at each point
from both left and right directions, and choose the one with higher frequency 
product.

The above is Joy's original declarations.



=head1 METHODS

	seg($str_in);# return the string of segmentation result.
	seg_STDIO();	# Read from STDIN, and print the segmented result to STDOUT
	
	set_dic($dictionary_file_name)
	#The format of the dictionary file for each line is: 
	# "chineseWord\tFrequency\n"
	# 
	#Notice that if you don't call set_dic, 
	#the default dictionary in GBK encoding will be loaded.
	#The default dictionary is extracted from corpus of the People's Daily, 
	#January, 1998. 
	#Thanks to Institute of Computational Linguistics, Peking University,China.

=head1 SEE ALSO


=head1 AUTHORS

Rewrited by Chen Yirong E<lt>cyr.master@gmail.com<gt> ,September 21, 2006
Original Author: 
	Joy, joy@cs.cmu.edu in July 4th, 2001
	
=head1 KUDOS

Many thanks to Joy who made the code available.
Thanks to the PKU Corpus (from Institute of Computational Linguistics, Peking University, China) to help to automatic generate the default dictionary.

=head1 COPYRIGHT

This program is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.

See L<http://www.perl.com/perl/misc/Artistic.html>

=cut

__DATA__
	2488
	398
һ	242
	92
ѧϰ	326
	4330
	66
	418
	26378
ʮ	146
	264
	30526
ͳȫ	12
	34
	70
ץ	26
ʵ	114
Ž	54
һ	36
ܶ	54
	46
ǰ	50
Ϊ	1848
ȡ	6
	228
ĸ	480
	146
	292
ִ	100
	620
	982
ʤ	68
	648
ܶ	56
	28786
	946
쵼	56
	220
	158
	7218
	422
	84
	48
ʯ	74
	62
F	14
	16
	310
	12
ξ	26
	26
	20
	1152
	1136
	130
	394
׶	78
	60
Ⱥ	448
Ͷģ	18
	184
һ	158
	24
	8674
	74
	16
	550
	122
ֻ	58
	18
Ŀ	46
	6366
׳	6
	4
	560
	20
	4
	560
	1524
	16
Ļ	10
չ	8
	12
	8
ʹ	646
	270
·	24
	186
	1182
Ǻ	6
찲	10
¥	6
	1890
ȽȽ	4
	16
	836
	68
ָ	14
	668
й	1304
	578
	652
	4
	14
Ų	8
	42
	88
ί	64
	56
	312
	236
֯	286
	34
ȫ	30
	8
	536
	892
	10
	54
	1388
Ա	32
	234
	348
	112
	14
	370
	12
	166
	12
	48
	62
ᶨ	28
	74
սʤ	30
ֺ	10
	54
	122
쵼	468
ɲ	382
Ҫ	1672
	936
ȫ	42
ս	152
߶	90
ʶ	172
Դ	14
ƶ	116
ʵ	50
ǿ	294
	1500
	222
	206
	80
Ӵ	78
	40
	78
ȫ	174
	488
	256
	952
ս	14
	1378
Ӣ	68
ģ	36
ͬ	242
	72
	140
	84
	26
	120
	260
ھҵ	12
	42
λ	52
	46
	26
	104
	6
	128
ϯ	194
	2924
	288
	82
	246
	2872
	168
Ҫ	326
ƶ	114
ȫ	182
	530
˹	4
	182
	52
	462
	764
	320
	72
	304
ͻ˹̹	18
˹	14
	56
	1116
л	4
	1116
ѡ	12
ԭ	96
߷Ժ	6
󷨹	6
	516
	30
	64
	50
Ժ	10
Լ	176
ʡ	12
ѡ	50
ѡ	42
ھ	26
	84
ͳ	164
	12
	124
	1040
	6090
Ϊ	240
	118
	48
	210
	798
	270
	56
	118
	172
	148
Ա	26
	114
	1668
Ѿ	418
	74
	1128
	126
¹	40
	44
	268
	52
ŷ	38
	1206
Ϊ	258
	748
	22
ɹ	10
	558
	44
	1502
Ŀ	250
	2962
ú	6
ˮվ	8
Ƥ	4
ӹ	6
ϴ·۳	4
·	32
೧	4
	4
ֽ	4
	52
᳧	4
ҩ	4
ţ	40
Ѫ	22
ȡ	6
ӣϣ	4
	24
	4
ﻯ	4
绯ø	4
	38
	1520
ѧ	40
	42
	290
	4272
һ	346
	516
	8
	82
һ	4210
	468
ٷ	2
Ծ	2
	856
ĸ	26
	404
ʱ	178
	4
Ĩ	4
΢Ц	10
佱	4
	6
͸	10
	2
ϣ	2
ڴ	8
	74
	586
	20
	40
	8
һ	2
	2
	6
	28
	250
Щ	454
̯λ	4
ǰ	360
ǻ	4
ʱ	642
	20
	112
ͷ	2
˵	1276
	658
ɫ	4
	2
	6320
ɽ	2
	2
	2
ٿ	98
߾	38
ΰ	118
	52
ܽ	52
ʷ	212
չ	12
δ	116
ƶ	128
	22
׶	116
	174
	14
	64
	180
չ	1226
	142
»	404
	570
	62
	8
	30
	12
	24
	104
	74
	1306
	146
ѧ	16
	36
	8
ѧ	132
ͬѧ	58
	484
質	12
	86
	8
ӭ	26
	38
ί	28
	300
	116
ǿ	4
	82
	6
	6
Ǹ	2
	358
չ	154
ȫ	4
	6
	36
	62
ͨ	354
	2930
	12
ȫ	566
μ	236
ʽ	128
	254
	26
	68
	16
	2
ս	46
	2
߸	4
	66
	174
	1124
	52
칫¥	4
	32
֮	16
д	196
ʹ	4
	2
ʽ	52
	6
	1214
	6
Ȼ	4
Ŀ	8
һ	6
ʻ	20
	794
	22
ʱ	28
	22
	376
̨	2
	10
	10
ķ	2
	142
л	2
ͬ	14
	16
	6
	4
	26
ڳ	10
	196
	36
ҪԱ	4
Ե	6
	2
Լͬ	2
ȫ	48
	4
	52
עĿ	2
ͬʱ	250
ֶ	6
	494
	6
	4
ʵ	12
	8
	2
	2
	4
	2
С	384
д	2
	4
	16
	2
	2
¯	2
	356
Ʒ	84
ں	14
	4
ʱ	58
	2
׹	2
	50
	156
ר	84
ϲ	16
Ϊ	188
ʳ	70
	158
	8
	32
ũѾ	2
	460
	226
ƽ	94
	180
	14
	76
ũ	4
	106
ũ	16
˫	2
Ӫ	266
	72
ʲ	150
	338
	22
˴	56
	478
	6
	38
	2
	10
Ȫӳ	2
	58
ɽ	4
ף	52
Ӣ̨	4
Ŀ	6
	60
質	18
֣	24
ӽ	2
	4
־	16
ݳ	10
˹	94
	16
	304
	86
	24
	22
ƶ	4
	42
ÿ	108
	10
Ԫ	526
Ǯ	380
	1250
	24
	14
Ŭ	280
	628
֧	36
ƺϴ	2
˾	10
ȴ	234
	6
Ծ	6
ﵽ	162
	2
	42
Ӳ	6
	4
	18
	24
	120
϶	54
	8
	40
ͳ	154
Ҳ	1386
	34
	6
	38
ƻ	16
θ	10
ʹ	4
̫	10
	30
	10
º	6
	70
	36
	174
	338
	154
	14
ͷ	4
	18
ô	46
	54
	56
û	12
	12
	154
Ԫ	10
	122
Ҫ	270
Ȼ	64
ʩ˹	2
	4
ֵ	40
	30
	4
	2
֮	360
	132
ɫ	18
觺	2
丸	2
´Ļ	2
	2
	20
	876
ʵ	4
ѧ	18
ַ	10
ı	4
	44
	2
չ	14
	46
˼	162
	384
	72
	104
	16
ʻ	6
	56
Ǹ	214
ض	8
	4
	6
	22
	36
ѡ	2
	16
	16
ѡ	68
	2
ϵ	4
	2
	494
	8
	80
	4
	4
	28
֪	6
	140
	68
	174
԰	8
	6
	64
	46
	4
	20
	12
һ֮	2
	4
	4
	366
	510
ֻ	52
	50
	558
	14
	2
	42
	160
	10
ͷ	2
	472
	2
	340
û	378
	1076
	818
ϰ	26
һ	1324
	6
	4
	14
	92
ȫ	4
	64
	38
ˮ	12
	6
ϵ	68
ׯ	2
	34
γ	184
ȫ	2
ֹ	4
	10
	4
֧	18
ҵ	264
ũҵ	172
	24
һ	8
	4
	18
ʵ	186
״	12
	8
	664
	538
׼ȷ	36
	224
	2
	26
	194
ɻ	162
ȷ	6
λ	40
ٶ	90
	2
	30
С	16
	6
	26
	38
	594
	26
	8
	164
	2
λ	34
	2
ƺ	32
	4
԰	6
ռ	6
Ķ	58
	14
̬	16
۹	6
	532
һ	8
	38
	144
Ͷ	26
	14
ȫ	98
	72
ÿ	72
	328
ṩ	216
	90
	2
	6
ũƷ	4
	78
ʢ	6
һ	26
ȥ	942
	22
	40
˾	438
ҵ	132
	52
ǿ	130
֮һ	108
ȡ	224
	2
ʩ	24
	376
֤	12
	36
ְ	4
ְ	246
	8
	1660
	18
	2
ƶ	2
	14
媽	2
	122
	42
	2
	4
ס	84
ò	2
·	102
	4
ǰ	2
	250
	4
	150
в	24
	96
	70
Ʋ	16
ȫ	160
Ӱ	234
ɶ	6
	1026
Դ	46
	4
Դ	6
	8
	198
	226
Դ	2
	70
۵	14
֮	96
	204
ѧ	2
Ŀ	24
	104
Ƭ	70
	28
ճ־	2
	2
һʱ	24
	3314
	708
ʲô	474
	6
	18
ϣ	4
Ȱο	2
Ҳ	54
ܶ	176
	136
Ȼ	44
άϣ	2
ʱ	90
ʷ	2
հ	12
һ	130
	6
̵	78
Ʒ	126
	4
Ӷ	4
ձ	172
ҵ	10
Ʒ	340
	10
װ	20
	8
ƤЬ	2
	6
ˮ	2
Ӣ	76
ױƷ	4
	14
Ʒ	2
Ŀ	6
Ӧо	2
	1208
Ȼ	32
	4
֪	324
	2
ħ	2
	10
	58
ʵ	62
	92
	938
װ	2
	468
	2
ʽ	118
	6
	90
ر	196
ϲ	4
	2
	10
	6
ְ	34
	34
	18
ֹ	2
	4
	2
	16
	796
ԭ	54
	102
Ͻ	24
ʵ	268
	4
	158
	54
	64
Щ	104
ҵ	1150
	292
ģ	28
	46
沢	104
	38
	280
û	822
	126
ط	302
	16
	122
	52
	10
	8
	252
	12
	6
	32
ֱ	52
	52
	410
	612
ҵ	156
	14
ҵ	130
	20
	26
	98
	36
Ԫ	62
	8
	102
֮	24
һ	218
	8
	88
	10
	262
	164
ص	164
ҵ	14
	212
	40
ģ	140
	16
ס	236
	168
г	628
	114
	16
	128
	74
ܲ	36
Ӧ	236
	6
Լ	510
Ч	66
ת	22
Ͷ	282
	154
о	336
Ӫ	16
	98
	52
	12
ӪȨ	4
ְ	40
·	82
	450
	64
	64
Ժ	186
ί	28
ʹ	24
	4
	134
ල	142
	134
Ѷ	248
	290
	6
	46
ҵ	16
	12
	102
ֵ	4
	18
	152
	40
	14
ν	12
ˮƽ	288
ƫ	30
	86
ʵ	180
ҵ	8
ṹ	204
	194
	46
Ч	24
ΰ	30
ۺ	14
	78
ϵͳ	200
	116
	6
	98
	38
	56
	26
	16
	134
	6
·	22
	22
	42
	6
ݳ	118
	68
	200
	30
	68
Ϻ	208
	50
	22
	106
	124
	4
	140
	172
ֳ	58
	86
	254
	4
˴	36
¼	18
	6
ٻ	6
	10
ص	50
	20
վ	140
	28
	78
	6
ͨ	6
˫	62
	18
	4
ҧ	28
	952
Ѫ	12
Ȼ֪	4
	134
	30
赹	4
ƽ	4
ί	66
	36
	30
ĳ	80
ڱ	4
	260
ϰ	22
	232
ȫԸ	6
	44
	1458
	136
	16
	18
޽	18
	462
	4
	176
	198
ˮ·	6
ȡ	4
һ	10
·	116
ֱͨ	8
Ӫ	6
	20
	16
	150
	26
	34
	164
	146
	64
տ	8
	44
ʾ	254
л	78
ҽ	6
տ	8
ӹ	6
	4
	4
	54
	4
	122
ҽԺ	148
ǽ	28
У	8
ʦ	116
Ƹ	4
ͬ	138
ѵ	70
ʦ	8
ʵ	30
ʵϰ	10
	32
ͼ	44
Դ	102
	8
	130
	28
ѧ	12
	204
ѧ	36
ƻ	228
	18
ѡ	42
γ	18
걨	18
	56
	40
о	20
ǿ	120
̴	8
Ϲ	92
	28
	4
	70
	32
ʯ	62
	38
ʳƷ	46
Э	60
	14
	68
	28
ҩƷ	20
Ӣ	26
	12
	8
	8
Υ	20
	6
	4
ʷ	10
׼	16
Ϸ	18
ʵ	40
Ӽ	16
	38
	26
人	32
	86
Ա	18
	76
	86
	8
̳	4
	46
	30
Ӽ	4
	10
ϲ	52
˳	20
л	8
	20
	434
	28
һ廯	18
ѧ	50
Ѹ	16
	272
ŷ	84
ͳһ	102
	506
ҵ	26
	28
	8
	38
	58
	486
	48
	108
	56
	80
ƶ	206
	156
߶	8
	6
	26
Զ	68
κ	146
޷	80
˽	160
	218
ȫò	4
ʼ	4
	74
Ʒ	28
޹˾	56
	10
һ	20
˶	72
ӵ	58
װ	4
Ьҵ	8
Ƥ	4
	4
	6
ľ	10
Ʒ	26
	4
ȥ	356
	16
ڶ	182
	148
	104
	400
չ	20
׻	4
	20
չ	6
̨	126
	10
	16
	32
ͶӰ	4
	134
	8
	24
	92
ʵ	54
	66
ڶ	20
	118
ͼ	86
˶Ŀ	4
˽	20
Ƿ	34
װ	14
	140
Ϸ	46
	6
	106
	114
	16
	18
	32
	230
	14
	48
	4
	12
ϸ֤	6
һ	14
	36
	102
ʧ	84
޴	114
Σ	152
ҹ	402
	28
δ	96
ܵ	148
񾭼	64
	188
	118
	146
	54
	284
	94
	146
ȶ	210
	100
	40
	78
	202
˰	44
	432
	4
ǿ	132
	20
	26
Ծ	18
ƴ	18
ȷ	28
ȫ	32
Ԥ	84
Բ	24
	52
	12
շ	128
绰	152
	376
	48
	118
Ծ	36
ҵ	18
	14
	106
Ȥ	10
	14
ѧ	132
֪ʶ	108
ֵ	68
Ϣ	152
̼	18
	422
Ļ	314
Ʒλ	10
	74
	66
	58
Ҫ	302
õ	202
	44
	6
	68
	100
Ӯ	42
Ե	6
	4
ãģͣ	32
ͬ	124
ÿ	8
ʹ	8
	18
վ	24
	80
	72
Ե	16
Զ	18
	84
	136
	8
ȷ	10
ֻ	20
뿪	138
	240
Ͽ	4
ϵ	110
ʵ	170
̽	52
ּ	6
	20
	42
	22
	34
	86
	4
	172
	140
ƪ	88
	6
ع	18
ϵ	100
	114
е	152
ο	8
ӳ	80
	66
ɫ	4
˼·	26
ʩ	180
	126
Ƶ	4
Ի	28
	98
Ů	8
	4
	44
	18
	184
	34
Ƿ	38
	106
	30
	22
	10
	34
͵	4
ҵ	34
	4
	20
ʮ	8
Ӫ	96
	64
չ	32
	336
Ӧ	166
۸	418
	84
涨	202
	6
	8
ע	8
Ʒ	4
	12
	12
ȼ	14
Ƽ	4
λ	276
	136
׼	62
й	328
	292
ʮһ	6
	162
	12
	16
쳣	12
״̬	58
Χ	98
ȡ	170
	100
Ȩ	34
	6
	42
	62
ͻ	64
ץ	90
Ч	32
	104
ץ	32
	568
	208
Ч	56
	4
ؼ	16
	90
	150
	86
п	6
Ա	20
	182
Ʒ	20
	6
ȫ	22
	6
ɼ	84
¼	12
	232
λ	432
	68
Ͷ	122
ٲ	4
	48
	96
ϸĥ	4
	68
Ϸ	30
	6
	64
	136
	16
Ŀ	22
	182
ִ	10
	14
	8
ն	12
ȡ	56
	54
	116
	60
й	36
ʡί	76
	146
	16
ٷ	6
	18
	40
Ѻ	88
	56
	28
	78
	640
˳	4
㶫	78
	76
һֱ	150
	42
ȫʡ	52
ۼ	22
	4
Ԫ	316
ռ	164
ͬ	44
ܶ	52
	210
ƽ	58
	12
	20
	102
ͬ־	444
	12
	54
֮	34
	4
	40
Ӱ	10
Э	50
а	12
	20
Ӱ	32
Ʒչ	6
	90
	8
չ	18
ͼƬ	176
	8
	36
	8
	4
ʮ	18
Χ	16
Ԫս	6
Ժ	4
ս	14
	46
Ԫ	6
	92
	50
	94
	24
̭	12
Ա	28
ѧ	22
Ƕ	4
	36
ۺ	52
	14
ղ	6
	74
	4
רҵ	58
	4
ȵ	46
֢	4
;	38
Ī	8
̸	42
	92
Ĵ	4
Ѷ	4
ɷ	52
ָ	170
	202
ļ	86
Ҫ	448
	6
Ȼ	24
໥	54
	8
ծ	52
	116
	250
ҹ	10
ͼ	8
	8
Ϯ	32
	4
	34
	34
	16
տķ	4
ׯ	22
	448
	4
ɱ	10
	40
	30
	16
	30
	12
鴦	4
	34
	102
	50
ֲ	46
	34
ի	16
ڼ	118
	42
	64
˷ָ	6
	8
֤	68
	14
ͽ	32
˹	4
	4
Ϊ	180
ֻ	378
ƻ	30
ɱ	6
Ŀ	48
ͣͣ	48
	12
	14
	40
ҵ	54
Ŵ	14
쳧	4
	20
ɷ	16
	66
֤	28
	12
̽	12
	34
Ũ	6
	30
	52
	20
	6
	134
ݿ	14
Ƚ	112
ж	16
߲	40
ˮ	52
Ƿ	140
Ҫ	12
	10
	198
	308
ƴ	14
	4
ֵ	42
	20
	134
ǰ	34
	36
	26
	26
	34
	1814
	204
	4
	4
	28
ͯ	6
Ͷ	4
ɳ	20
Ӣ	4
ʵ	40
Сѧ	22
	22
	18
ô	60
ƽ	18
ȫ	26
Խ	4
緢	4
	106
·	50
	32
˼	28
	870
ʵʵ	10
ΰ	30
Ŀ	156
Ծ	40
	440
ȥ	156
˿	108
ô	142
	40
	46
	40
л	36
	150
	12
	24
ת	64
	206
ʧʱ	6
ó	28
Ƴ	62
ش	142
	12
ʡ	28
ͷ	16
ʮ	18
ͨ	180
ִ	28
̨	6
	68
ͦ	4
	22
ָ	44
λ	14
	30
Υ	28
һ˿	8
	70
	128
羰	6
ů	32
ɳ	26
	38
	42
֧	74
Ƚ	190
	18
	10
¼	4
	6
	16
ǹ	16
ƾ	4
	4
Ͷ	100
	78
̶	60
	4
ؼ	4
	6
	166
	50
漰	46
Ȩ	6
	8
׫д	16
	16
	4
̱	6
˭	240
	658
	8
ʦ	26
־	46
ڣ	4
	38
	194
	10
	4
	22
ӡ	4
	46
ӡ	22
	4
	62
	20
	4
	28
칫	18
־	6
	16
	4
	82
	18
	36
	22
	82
	4
Щ	118
	8
	370
	6
	32
	62
㻨Ǯ	4
ҳ	30
	106
ʮ	42
	14
	32
԰	4
ɳ	22
ʮ	12
	20
	42
	10
	36
ת	54
־Ը	4
	8
	22
Ա	4
	4
	18
	30
	20
	6
ʡ	196
ӵ	10
Ӣ	22
	8
յ	64
	94
	8
	28
	14
	46
д	4
	6
	6
Ժ	38
ʦ	4
	16
	70
	34
	90
ܱ	20
	34
ֵ	62
	130
	120
	10
ѹ	34
	40
	54
	158
	20
Ҫ	54
ζ	22
̶	16
	32
	36
	26
仯	130
С	6
	52
	54
	46
Ӧ	202
	6
	12
	142
ж	104
	14
ƶ	4
	14
ũ	270
	74
	178
Ƽ	258
	36
	44
Э	58
	8
	38
	18
	70
˷֧Ԯ	4
	4
¥	24
ػ	10
ͣ	16
	4
	90
֧Ԯ	28
	30
ָ	88
Ŀ¼	48
޸	24
ܹ	8
	4
˰	10
Ż	26
Ŀ	14
	8
	6
	102
⿪	44
	218
	4
취	90
ȫ	28
	16
	274
	38
	16
֮	138
	14
ȷ	22
	20
	62
Ա	258
ʳ	4
֧	32
һ	20
޶	18
	58
Э	24
ｨ	6
	10
	170
ίԱ	124
	90
	44
ҵ	160
	238
׿Խ	6
ǰ	118
	88
ͨʽ	4
	10
	6
	30
ͨ	18
	24
	86
	4
	34
	42
	4
	4
	290
	24
	4
ѧ	154
ռ	28
Ͷ	30
	74
ḻ	74
	16
Ϊ	10
	28
	80
һ	26
ҽ	38
ͬ	6
	574
̻	8
㻯	4
	8
	14
ͨ	40
ɫ	32
	6
ͨ	14
ʵʩ	102
	4
ƾ	4
ܰ	10
	14
	56
ٰ	72
˻	12
ί	34
̨	50
	128
	58
̸	62
˫	98
̶	8
	108
ʱ	4
	40
	30
	14
	44
Ư	10
Խ	10
	98
ȫ	12
	12
ǿ	10
ʶ	88
ڼ	18
	22
	30
	100
	16
	6
Ŀǰ	288
ж	42
ƽ	208
	90
˵	88
ַ	4
ʮ	182
ؼ	76
ϰ	24
ų	10
	38
ά	62
ʱ	24
ְ	42
	22
	20
	10
ɫ	86
	4
	66
ռ	4
Ү·	10
Χ	42
˽	18
̫	8
ӵ	14
	8
ִ	106
˹̹	20
ҵ	8
	24
	64
ͣ	10
ʧ	50
	22
鳤	18
	4
	90
˼	32
	390
	34
Ա	24
	8
	8
	4
Э	38
	14
ǡ˹	4
	4
	6
װ	38
	4
ī	58
ǿ	52
	76
Ͷ	42
	24
ջ	20
ծȨ	4
׷	40
	16
	30
Ȼ	4
	12
	84
Ԫ	36
	10
	8
	218
һ	18
	4
	6
	28
	6
	166
ͦ	10
ͼ	4
ǣ	4
ֵ	18
ũ	252
	42
	70
	116
	66
ָʾ	32
ǧ	8
	28
	250
	34
	86
ӱ	20
	4
ּ	4
	40
	102
ľ	16
ǧ	6
	38
ʦ	14
	36
ǽ	10
ɳ	42
Ἲ	8
ΰ	50
	22
¾	4
ˮ	8
ͯ	44
	16
	12
	10
	8
	364
	96
޳	8
͹	14
	26
	26
·	24
	22
ֵͲ	10
	6
˿	4
	108
	4
	30
	22
	6
֫	4
	46
	8
м	12
ǰ	10
	8
	16
	4
	38
	6
	116
	4
ʻ֤	22
	32
ʻ֤	4
	12
	4
	4
	4
	62
	20
֤	8
	22
	18
	78
	22
ݲ	8
ˮ	4
	88
Ҫ	172
	4
ʮ	8
׿ɽ	4
ʱ	116
ú	24
	198
	36
	18
ϸ	22
	134
ƶ	198
ĸ	134
״	24
Ҹ	100
	260
	68
	102
ʵ	20
	134
Ӧ	24
	136
	36
	14
ɷ	38
ɽ	20
	14
ú	14
	38
	20
Ů	48
ı	2
տ	2
ɽ	4
ͷ	84
	30
С	8
	30
ǰ	74
ű	2
ȡ	48
ż	12
	96
ֽ	48
	20
űȺ	2
Ժ	112
	148
	72
	88
ʼ	24
׷	2
	58
̫	4
ݵ	20
	4
	10
	4
ս	44
	14
	2
Ա	18
	24
ҳ	24
սʿ	56
ֿ	6
	22
	88
	6
	12
ͳһս	18
	26
	92
ҵ	12
̫	2
·	8
	10
	50
ί	8
紦	2
ί	16
	2
ί	6
»	6
ձ	10
	2
	88
ǰ	14
عҵ	4
	92
	2
	12
촬	2
	10
齨	34
춨	24
عҵ	4
	22
Լ	94
ʲ	2
ʯͲ	2
	48
	2
˿	2
	4
	2
Ĺ	2
	8
	102
	12
ҵ	100
һ	6
	2
Ʋ	8
	32
	4
	62
	38
ͬ	12
һ	10
	50
	26
	272
	8
ս	34
	8
	6
	2
ϵ	4
ʳƷ	2
	32
֧	26
֧	2
	26
	18
ũƷ	16
ԭ	10
Ʒ	26
	52
	18
	32
	256
	74
	4
	30
ṹ	10
	2
ì	62
	10
	54
˵	8
ˮʯ	2
	36
	18
Ʒ	10
	22
	6
˸	8
	22
	46
ΰ	12
	4
	92
	8
	86
Ȳ	12
ź	4
	8
	4
	14
	40
ؿר	2
	6
	18
ʹ	2
	60
	24
	356
ͳƾ	8
	4
ͳ	58
	28
	60
	10
ָ	44
׼	8
ӿ	104
	138
	94
	24
Э	22
ս	2
	6
ɽ	2
	6
	2
	16
	2
Ѿ	4
ս	12
	68
߻	2
ƽ͹	2
ÿ	2
ս	6
	40
	18
	108
Ӧ	42
淶	28
Ӧ	52
	116
	54
䷶	4
	166
	32
	32
Ժ	6
	14
	8
淶	64
	8
	2
׷	32
	2
	62
߷	8
ʵ	6
	26
	22
༪	2
ί	4
óί	8
	2
ͨ	2
	6
ž	44
ܲı	6
ܺڲ	10
	16
ο	4
	6
ִ	6
	2
	128
ʵҵ	6
³	10
	66
	10
	102
	2
	12
	46
Э	2
	52
³Ѹ	10
ѧ	4
é	2
ͯѧ	2
	20
ר	20
	20
	86
ʼ	18
ɹ	26
	4
	56
С	44
	4
	70
	10
ͼ	14
	10
	10
	36
	44
Ǹ	32
	68
Դ	2
ת	12
	8
ʵ	26
	10
	200
	78
³	8
ְȨ	14
ֹ	20
	92
	56
ݰ	2
ʶ	12
	94
ʹ	14
	18
	16
	2
	2
	48
ֱ	6
	268
	32
ٱ	30
	6
	2
	4
	6
׷	2
֤	4
	208
ɭ	2
ڻ	8
	64
	2
	2
ĵ	2
Ӣ	14
	34
	2
	2
ӽ	28
ʱ	80
֮	2
	44
ý	4
ѭ	14
	38
ϼ	8
Ԭ	26
ΰ	6
˻	16
	96
	8
	26
	4
ɭ	2
ս	14
ǿ	40
ת	4
	20
ҵ	58
ǧټ	8
·	12
ֵ	60
ע	106
ת	8
־	54
ȱ	14
	4
ʣ	2
ԽԽ	106
	48
͸	10
ճ	16
	26
˼ά	16
	2
	6
	4
	2
	8
	22
	50
¼	20
	36
ҵ	14
ӿ	16
	122
	14
ȫ	6
л	2
ѧ	44
	46
	156
ֶ	56
ı	116
	46
	40
˷	44
	14
	36
Ȩ	38
	112
ƽ	10
	2
һЩ	496
޶	2
	74
	2
ʾ	12
	40
ֵ	6
	6
Ȥζ	2
΢	6
Ǩ	4
Фӡ	4
֮	2
Ф	12
	68
	64
	56
	2
	12
	2
Ӷ	58
	30
ƻֺ	2
	2
ǰδ	2
ѻ	2
ɹ	170
	12
Դ	2
ж	48
	22
ϸ	12
	122
Ƿ	4
˼	2
	2
	2
	2
	22
	6
ؾ	2
˹	4
	24
	10
ԭ	4
	6
	10
ʦ	34
ܱ	2
ȫ	14
˵վ	18
ƹ	58
ѧ	240
ʦ	170
	10
	94
ǰ	14
	6
	146
ͨ	44
	138
ʿ	16
Ϊֹ	22
ʮ	12
	2
	2
	2
	30
㲥	46
̨	26
	4
鳤	6
	226
	148
	28
	58
˶ʿ	16
	28
ԣ	2
һͷ	12
	12
	178
ԥ	2
ɽ	22
Ⱥ	84
	16
ƶ	12
ҹ	2
д	2
	2
׳	2
ഺ	6
	42
½	28
ɽ	2
ʯ	8
ܳ	20
ϩ	2
ˮ	14
	122
	6
λ	56
ˮ	8
	2
	72
һ	26
ŷ	8
	74
	60
ܹ	136
ִ	64
	14
Ӯ	20
ս	2
	6
ſ	36
	8
ع	2
	4
	8
	12
	2
	48
ƾ	16
	2
һ	2
̸	124
ϵ	390
	2
	144
	18
	196
	394
	10
	14
ֲ	2
ѧ	18
糾	2
	4
	4
ͷ	4
	114
ʵ	14
ɹ	112
	24
	14
ƶ¸	6
	26
	24
У	8
	30
	30
	138
	102
ѧ	8
	10
÷	6
ѹӢ	2
ٶ	8
	36
	104
׼	54
Ļ	10
ѹӢ	2
	4
ʽ	84
̨	18
֪	4
ơƳ	4
	6
ó	2
	4
	60
	2
԰	20
	4
	48
˶Ժ	2
	84
Ѫ	12
ͻ˹	12
	6
	34
촺	6
β	2
	2
ͽ֯	2
Ǵ	14
к	6
ˮ	8
	74
	2
	12
	10
	240
	18
	32
ƽ	8
	238
Ȼ	144
	2
ٷ֮ʮ	2
	2
	2
ȡ	2
	6
¼	82
	20
ʡ	2
	126
֪	16
	34
ƭ	10
Ʊ	6
	94
˽	10
	66
	2
	64
	22
	2
	2
	16
ֲ	44
С	12
	18
Ф	20
	10
	2
	48
ս	4
»	20
Ҷ	4
ѵ	4
	30
	40
	16
·	34
	6
	14
ҹҹ	4
	16
	36
	10
	60
	2
೤	2
	46
	2
	22
һǧ	8
Ʒ	2
һ	32
һ	2
ȡ֮	2
Դ	6
Ʒ	2
̵	2
	146
һ	28
	24
	28
Ƭ	2
	2
	2
	2
	280
ִ	4
ȭ	2
̫	2
	2
ӰƬ	2
	4
	18
ʢ	4
	2
	8
̲	12
	2
	6
ͷ	6
	16
ʮ	2
	60
	2
Ը	58
	4
ˮ	2
	20
ܾ	58
Ͷ	2
þ	6
Ӱ	2
	4
	98
˼	24
	36
	4
	8
	2
ѧ	2
ʼ	292
	2
	26
Ӱ	104
Ƭ	4
Ʒ	2
Ƭ	4
	46
	8
	10
	2
	258
	2
	50
С˵	52
	36
ģʽ	40
	58
	4
	30
·	2
	32
ͼ	4
	2
	20
ʹ	10
ѱ	2
ĥ	6
	24
ȵ	22
ע	50
	52
	2
ѡ	2
Ҫô	32
Ե	2
	22
	2
	106
ʾ	32
	24
۹	16
Ը	16
	46
	58
	16
ɧ	2
	12
Ŵ	6
	100
	2
	20
¹ԭ	2
	4
ź	12
	54
	2356
ֽ	2
ݻɫ	2
װ	4
ս	12
	2
ñ	2
	18
	2
̷	2
ȫ	116
Ұ	2
ĹԱ	2
ģ	8
	2
	12
ձ	2
	8
̫	2
ζ	2
ò	8
ǰ	2
	4
ĩ	8
	12
㿪	2
	90
Ϊ	212
	366
	4
Ӵ	30
	56
鷳	24
	4
෴	12
	56
ô	30
ִ	2
	40
	68
ʮ	2
	172
	22
	2
ְ	12
	148
ϼ	64
	26
ܾ	28
Ц	2
	52
	26
	2
	16
	30
ĩ	48
ܲ	6
С	32
±	22
ʵ	24
	8
	28
	14
ճ	14
	120
	26
ɫ	6
	174
	12
ͻ	32
	26
β	8
	44
ͨ	8
	2
	2
	2
೤	32
	42
	22
䷢	18
	12
֤	12
	4
	2
ع	14
	68
ũ	10
	4
	28
	4
ӭ	16
	6
ͬЭ	4
㽭	2
ʢ	6
̻	4
	2
ֽ	2
	20
	2
	166
̽	2
Ѳ	10
	122
͵	2
ٱ	6
	50
绰	38
׶	42
	4
	22
	6
	2
ϰ	10
	60
ֵ	2
ϯ	6
	80
	6
	24
	26
	2
е	4
	2
	14
	66
	16
	24
	40
	52
	28
	88
	6
ˤ	6
	188
	16
λ	8
ҵ	8
	128
	6
֮	14
	34
	26
	12
	26
	4
	14
ǰ	6
	6
	6
豸	120
	26
һͳ	4
	58
	2
	2
	30
ʯ	2
ģӣ	2
	10
	36
ȫԶ	4
	4
Ԥ	36
	10
	6
	20
	2
	46
	62
	10
δ	22
ɱ	2
	2
й	74
У	14
	22
	52
	20
	2
	10
	2
ѧ	12
±	2
	22
̽	18
ԭ	14
	4
̲	30
	4
С	6
	22
	52
ˮ	2
	20
ˮɻͷ	2
װ	4
Զ	2
Ϊ	222
ʹ	4
	10
	78
	42
Ů	4
ƽ	2
Ϸ	2
ˣ	6
	2
̳	2
	8
	4
	16
	24
칬	2
	2
	2
	8
	2
	106
	2
	2
	2
	2
̸ɫ	2
־	8
ο	6
	2
	2
	2
ͬ	8
¯	2
	2
	42
ʩ	16
	2
	6
ƶ	2
	2
	2
	4
	8
	2
Ů	58
	14
	20
	114
ǿ	16
	6
ʵ	6
	2
ճ	2
ǵ	44
	8
	26
Ϣ	2
	2
ҽ	6
	172
æ	4
ҽ	4
ҩ	2
Ч΢	2
	48
	8
	180
ɤ	6
	46
	2
	52
	32
	4
	52
	26
	20
	2
	2
	4
	2
	4
	8
С	2
	2
	2
	30
	2
ë	2
Ҳ	6
׳	2
	2
ƿ	12
ę́	4
	12
±	2
	12
	18
С	4
	8
	2
	2
ɶ	2
ι	70
	2
	8
	58
	48
	4
	10
	10
˥	4
	10
	28
	116
	80
	46
½	2
	50
	102
	26
̽	10
	18
	12
ѭ	26
ѹ	2
־	2
	26
	12
	2
Խ	28
	2
	14
	2
Խ	178
	24
	54
ɾ	68
	8
ݳ	12
ʻ	4
ٹ·	14
	76
	2
ǰ	46
˫ĵ	2
	6
ɲ	2
	2
ʻ	2
¹	114
	24
Ѥ	6
	6
ҵƻ	2
ƽ	2
滺	2
޺	6
	8
ż	6
	6
бϸ	2
	8
Ʈ	2
	2
ˮ	10
˼	40
	588
ƽ	4
ۼ	18
߼	2
ƽʱ	14
ͬ	16
ദ	4
Ŀ	2
Զ	2
	4
ıְ	2
ҵ	2
һ굽ͷ	2
չ	38
	66
	8
б	10
	2
˹	12
	2
ݳ	2
ɲ	2
	2
ָ	68
ԭ	22
	4
ӭ	76
	4
ҵ	2
Ϸ	42
	16
	12
齭	12
ذ	6
ڴ	6
	54
ģ	2
п	2
·	2
	2
һ	54
ζ	8
	32
	24
	2
	2
Ʊ	12
ߺ	2
	32
᳹	82
ʾ	6
	76
ֹͣ	38
ɷ	2
Ȼ	2
ٴ	24
	24
	16
۲	2
ֹ	34
¼	20
	4
	34
	4
ͻ	16
	2
	14
	8
	4
Ȼֺ	6
	158
൱	84
Ȼ	2
	4
һ	196
	32
ܷ	2
ʹ	114
	2
̺	2
˵	2
	2
	20
	4
	4
	20
	8
	168
	8
	2
	2
	6
̩	104
ο	70
	2
	4
	2
	2
	20
	8
˫ӵ	14
װ	2
	2
	14
ע	24
	10
	10
	8
	10
黳	4
Ϊ	2
¬	16
پ	2
ʡ	36
	4
̫	2
	4
ɽ	2
	2
	2
Դ	4
	2
	28
	28
	18
	60
	12
ο	46
С̷	4
ɢ	4
	6
	10
徻	2
	204
	272
	2
	28
	8
	16
ũ	20
⾭ó	4
ó	2
ӹ	14
Ҫ	10
˲	90
	10
˻	2
	26
	2
	4
	2
	4
	4
	6
ͭ	2
Ļ	4
	28
	208
	2
ð	4
	2
	54
	34
	2
ɽ·	6
н	4
ãã	6
ֺ	2
ͨʲ	2
	2
Ũ	2
	4
ĸ	2
ɽ	2
ж	2
´	2
	6
	4
ɳ	2
	2
	6
ϳ	2
ε	4
	6
ҽѧ	18
佱	4
	24
	2
	66
	34
ٴ	12
	46
ϲ	20
	44
ý	2
	14
	12
	22
	4
	42
	46
δ	6
˵	14
Է	4
	134
Լ	6
	12
	16
ͥ	2
	58
	50
	8
Ԥ	10
ֹ	14
	26
	32
ϵ	4
	106
	20
	20
	46
	20
⽻	88
	304
	22
	64
	6
	2
ͷ	20
Ȼ	26
ǿ	6
	66
	34
ɼ	10
	78
	136
	12
Ʊ	60
	36
	4
	4
	112
ʷɭ	2
о	50
	8
ѧ	4
Ү³	2
̫	22
Զ	12
	30
	4
	2
ϵ	2
	2
۲	6
	4
ЧӦ	8
	10
	224
	36
	18
м	4
Ѫ	2
Ծʽ	2
	2
	12
	20
	14
ؼ	8
ʦ	22
	2
ʿ	14
ɣ	2
	4
	2
	8
	16
ӡ	10
ϵ	2
ھ	44
ƺ	28
֣	14
	40
ʡ	8
	2
ȥ	78
	2
	32
	30
	6
ĳ	2
	18
	38
	16
	46
	26
	2
	16
ʧ	8
	2
֤	4
α	16
ɶ	20
˻	2
֤ȯ	60
Ӫҵ	6
	6
	2
	2
Ѱ	42
	62
	50
Ŷ	2
ǰ	54
״	84
Ҫ	64
	16
Ǳ	24
	166
	2
Ŵ	4
ʽ	158
	4
	2
ʹȨ	2
	42
	14
	2
ֲ	14
	10
	112
	10
Ϣ	16
α	2
	2
	8
ִ	44
ǳ	126
	122
	116
ר	38
ð	2
֤	6
	6
ǧ	2
	26
	10
	28
	34
׳	6
	12
׳	6
	4
	6
	6
	4
һ	22
Խ	2
һ	30
	42
	30
д	12
Ӹ߲	4
ȷ	76
	28
	38
	16
Ⱦ	10
ߵ	4
д	4
	50
	4
	2
	2
	86
	16
˻	8
	26
ƾ֤	10
֧	2
	18
۷	2
	4
	8
˰	2
	6
߰	12
ȡ	44
	18
ַ	8
	38
һ	2
	16
	38
ת	24
	116
	2
ͻƿ	28
	22
ƽ	8
ɽ	2
㽭ʡ	14
ĳ	2
̩˳	2
	2
	2
	108
ʡ	6
ϯ̨	4
	34
ƶ	38
ͭ	6
	2
޷֯	6
	36
ʺ	20
	94
ףԸ	20
	62
	10
	44
	8
	4
Ѹ	96
Ա	10
	4
ٻ	2
ˮ	6
	22
	2
ܿ	64
ҹ	2
	6
	60
Ϊ	50
	6
¶	10
ֻ	10
	28
	8
	6
ľ	2
԰	4
	14
ҵ	2
ֳ	8
е	14
ϣ	216
	6
	38
	28
	2
Ч	4
й	4
Ա	14
	48
	2
ҵ	2
	6
֪ʶ	2
ʷ	4
Ȥζ	6
񹲽	6
ε	4
ó	2
	2
	28
԰	50
	4
	18
	120
Բ	10
Ȩ	8
Ⱥ	10
	14
	24
	2
ϲ	2
ʯ	2
	26
	2
Ϊ	16
	30
̰	14
ܻ	10
Ų	8
ռ	2
޶	16
	46
	10
Υ	62
	80
Ż	10
Ա	80
	6
	2
ְ	4
	84
	14
	10
ǹٷ	2
	12
	10
	14
	2
	8
	60
ȫ	38
	32
з	2
¿ѭ	2
	46
	20
	8
Դ	12
Ӣʽ	2
Ժ	4
	6
ô	84
̸	2
	104
	1636
	2
Ӣ	6
Ű	2
	26
ͶƱ	8
	2
ͶƱվ	2
	2
˴	18
	2
Ϥ	18
	10
	14
ѧ	20
	28
	4
ί	4
˲	12
Ĵ	28
	4
	42
µϰŵ	2
	14
ӡ	14
չ	4
	72
	14
	10
	30
	10
Ӫ	2
˽Ӫ	24
	16
	10
	42
	18
	122
ɧ	2
	4
	12
ѡ	28
	2
Ů	8
	184
Ӿ	4
	6
	8
	4
	2
	10
¼	22
	34
	8
	26
	24
Ӿ	56
߶	8
	2
ƹ	16
̨	4
	8
ֱ	2
	40
	30
	4
վ	38
㳡	48
	6
	4
	8
	20
	42
	2
ʮ	2
	2
͹	10
۴α	2
Ԫ	64
ʡ	10
ʩ	2
	10
	10
	10
֣	16
ŵ	32
	10
ȫ	2
	8
	10
	2
ƶ	26
	22
é	2
·	8
	20
;	6
	12
Ľ	34
	42
	156
	28
	10
Ϊ	48
	14
ȡ	4
	312
Ļ	6
	66
	16
˦	4
ƶñ	4
¸	24
Ա	50
ȴ	4
ͬ	4
ʿ빲	4
	4
˰	12
	14
ͼ	24
	14
ٱ	26
	38
	6
	32
	44
	12
	12
	20
ǰ	22
	22
ϰ	14
	44
	18
	6
	8
ڲ	4
ɽʡ	18
	4
˰	46
	12
˰	4
Ż	42
	44
	10
	8
	8
	36
΢	12
˰	8
	4
	10
ܰ	6
ѷ	4
÷	8
	4
߽	4
	12
	4
	10
	136
	8
	26
	20
	12
	34
	60
η	8
	4
龫	4
ս	6
ʤ	6
	58
	50
˰	6
	48
ƭ˰	4
	10
ά	72
	6
	10
	70
֧	142
	36
	8
Ц	50
ʮ	12
ܻ	26
ʮֻ	16
»	8
ϻ	14
	28
	34
ű	14
ؽ	8
Ԯ	20
	2
ץ	10
	28
ڽ	10
ǰ	48
ѹ	12
	52
ͣ	2
ڶʮһ	8
½	22
ܿ	2
	70
·	36
	60
Խ	14
	8
ȼ	2
ױ	2
Ʒ	12
ֿ	40
һ	52
	14
	46
ʡ	34
ʡ	4
	14
	46
һ	2
ƾ	8
	26
	106
˷	4
	6
Ȼ	42
ഴԱ	2
	6
	2
Ż	4
˧	2
	2
Ԫѫ	2
	4
	6
·	4
ٽ	142
˫	24
ó	50
	52
ֱ	84
	46
פ	86
ƶ	2
߬	2
·	2
	8
	6
ʤ	2
ש	2
ƽ	2
	2
	26
ʩ	58
֧	148
פ	4
	122
䵭	6
	22
ʽ	10
Ҷ	30
ű	4
ǩ	40
Լ	12
	6
鳤	32
	6
д	38
Ȱ	18
	24
³ç	2
	14
	4
	10
ֲ	4
	40
	16
	2
᳤	14
	14
	120
	72
Ŀ	2
ʹ	56
	54
׼	6
·	8
	4
	18
	2
	6
	138
ָ	4
	62
	30
	22
	6
ȥ	2
	2
Դ	22
	2
	18
˹	2
	46
Ⱥ	4
ˮ	2
	6
	2
ͻ	2
ܰͽ	2
ܰ͹	2
̷	2
֢	2
	40
	32
δ	12
	18
	8
ν	32
г	8
	2
Ⱦ	14
Ũ	2
ζ	14
	16
ƫԶ	8
ũ	30
ǧʱ	38
	52
	2
	22
ȥ	26
ɷ	42
	62
	18
	24
¹	2
	16
	46
	6
׼	132
	38
ر	12
	12
ί	34
	24
	2
ǿ	2
	14
	18
	158
滮	34
ż	2
ָӲ	16
	68
	2
	2
	2
	48
	8
Ķ	16
	2
	2
	2
޻	16
	8
	2
	2
Ƥ	2
װ	6
Ǯ	4
	2
	48
	24
ֽ	2
	10
Ϊ	10
	10
	100
	6
ԭ	198
	12
	6
ůʪ	10
	50
ʪ	6
	6
ʱ	358
ˮ	2
	4
̨	2
	4
ͻ	2
ո¶ͷ	6
	58
	2
	12
ܼ	2
ҵ	4
ؿ	12
Ը	8
	12
	12
ͻ	66
е	92
	8
ͻ	6
չ	68
	10
	12
	6
	2
ͬ	16
	40
	28
Ͼ	10
ʱ	60
	22
	8
Ԧ	6
߳	2
֤	120
ǰ	34
	10
	4
	10
	6
	16
	14
	2
鴦	26
α	2
ֵ	2
ж	4
̨	2
ʳ	2
	30
	2
	2
ɱ	14
	282
	16
	46
ѩ	4
	18
	2
	32
	50
ѩ	4
	8
չʾ	28
	14
	16
	32
黭	6
Ž	12
ʿ	36
ʵ	20
ְ	32
	2
	64
ƽ	36
	2
ͥ	100
˾	42
ס	22
	72
	2
	2
ȿԺ	2
ũ	2
	10
ʡ	2
Ч	8
ƻ	2
	22
	20
	96
	18
	2
л	26
	2
	28
	148
̸	8
	32
Ⱦ	4
	6
Է	2
	104
ͷ	14
	2
	2
	14
	2
	2
	32
»	4
	10
Ԩ	4
	4
˵	4
	4
ɢ	4
	10
żҿ	26
֧	36
Ϣ	2
	12
ԶԶ	10
	28
н	4
	6
	26
	8
	28
ʳƷ	10
ͬ	12
˵	8
	18
ĳЩ	36
	124
»	8
Զ	14
Ƕ	18
	2
	30
ع	18
	12
	10
	6
	2
	30
	32
ƶ	12
ϳ	4
	2
	6
س	8
	94
Ӣ	2
	90
ذ	2
	4
ᳫ	16
	26
ս	20
	10
Χ	26
ɫ	88
	26
	10
ս	8
	26
ӵ	14
ë	80
	24
ȷ	90
	2
Ͷ	2
	32
	62
	2
	8
ܵĿ	2
	6
̬	20
ұ	4
	14
	2
ũҵ	12
	102
ѡ	6
ӹ	2
	2
ͳ	2
Ʒ	30
	18
˾	6
С	28
֪	6
	8
ѯ	24
	2
Ԫ	18
һ	40
Ԯ	8
	40
ظ	14
	34
Ҫ	10
	18
	12
	10
ָӲ	2
Ϥ	18
ů	2
׷	4
ůůͺ	2
ս	14
չ	26
	42
	22
	14
	24
ѧУ	192
	2
Сѧ	22
	8
Ľ	6
У	4
	2
	26
ÿ	2
У	2
У	2
У	2
	10
ÿ	18
У	26
	8
Ƹ	2
	2
	2
	4
̸	2
	38
	48
෽	14
ϲ	12
	2
³	4
ϲ	6
˹ʡ	2
	124
ʥ	2
С	10
ʯ	2
ʧ	6
	34
	30
	26
Լ	26
	2
ֶ	16
	18
ʢ	28
	6
ָ	20
	10
	4
	6
	44
ֵ	14
֮ǰ	82
	8
	160
	2
	10
	22
	28
	14
˫	2
	8
	14
	2
	2
	2
ǿ	2
	6
	16
һ	4
	2
Դ	2
	24
	18
	4
	16
̸	14
Ԯ	4
ƶ	6
ƽ	26
	8
	46
	42
Խ	16
	8
¹	16
	4
ƹ	26
	2
	2
	2
һ	10
Ʊ	6
Ԥ	6
Ʊ	60
Ԥ	2
Ա	2
ǰ	4
	2
ÿ	4
	2
Ӧ	16
ǰ	16
̨	38
ӭ	4
׳	2
˿	2
鳱	2
	2
һ	2
	4
	2
	2
	2
ֶ	2
Э	2
	56
	2
Ĭ	8
ʴ	2
ġ	2
	2
	88
	82
	2
ͯ	2
	2
һ	96
ͯδ	2
Ȥ	8
³	12
ֵ	2
ʷ	2
	2
	2
	2
	12
	30
	6
	2
	18
	16
ĵ	2
͵	2
ŵ	2
	16
	2
ӡ	34
ŵ	2
ʮ	22
ѧԺ	28
ҵ	40
ʮ	46
	10
	14
ѧԺ	2
	2
̰	2
	4
	10
	6
	2
	10
û	90
	346
	28
	2
	2
š	6
ʽ	2
ϼ	12
	6
¡	2
	4
	4
	4
	2
ү	2
	2
	6
߶	6
һ	8
ʮ	20
Ҽһ	2
쳹	2
žž	2
	2
	10
Ⱥɽ	2
	2
ع	4
	12
	8
	2
ɱ	2
	6
ٽ	2
ʮ	186
	2
Ʒ	2
Ȥ	46
ί	2
ί	2
	6
	34
ϴ	8
Ҫ	16
	32
	6
	52
ů	6
	12
	8
ѹ	4
	26
	148
ȥ	14
	50
	60
ɱ	116
	54
	28
	12
	6
	2
Ӫ	8
	6
̽	2
	2
	20
ӭ	4
ƻ	8
	34
վ	2
ڹ	6
פ	14
Ź	2
ϯ	50
	32
	8
	2
	8
	2
糤	2
ų	6
ձ	68
ȫ	10
Ѽ	4
	2
ֻҪ	66
	16
	24
	68
	38
	2
ǰ	4
	4
	22
	4
	10
	8
Ҫ	4
˵	54
	22
	6
ǿ	22
	2
ۺϹ	4
	8
	6
г	94
ѧ	2
ֲ	12
Ƭ	4
	6
	18
	2
Ԫ	2
ʶ	4
	2
	2
	2
	20
ƫִ	2
ȫ	6
	6
	8
Ϊ	20
	16
ת	2
ǰ	40
	36
ʳ	4
ҵ	4
	8
	8
	20
	2
	6
	8
	4
	6
Ʒ	2
䲼	18
ձ	60
	26
˴	54
ί	34
Ȩ	18
	2
鼯	2
㷺	54
ʱ	20
	4
Ԥ	2
ڸ	2
	56
	2
	4
	26
ƻ	52
̽	2
Ǻ	2
	2
Ͳ	2
	22
	4
ũ	2
	42
ѧ	4
ｱ	2
ѧ	50
	2
Ů	12
ɾͽ	2
	6
΢	4
	2
	2
	38
	30
	24
	10
	4
ǰ	2
Ȥ	28
ɳ	6
	48
Э	2
	2
	10
	4
ʦ	2
	2
	28
	2
	2
ֲ	22
	16
ָ	4
׶	8
ظ	8
	36
ߵѧУ	6
	24
˷	40
	10
У	14
	4
ߵ	8
ְҵ	32
	6
	8
	2
Я	14
ο	14
οƷ	2
	2
ֵ	50
ѻ	2
ҵ	4
	18
Ԥ	50
	18
	18
	10
	8
	22
С	4
	22
	2
Сѩ	2
	8
̨	58
Ͽ	10
Ϻ	4
	22
	12
ʦ	4
	6
	6
ˢ	10
	38
	6
	6
	20
ɫ	16
ѩ	60
ɽ	52
	4
	4
Ȼ	2
	20
ʫ	2
	28
Ƶ	20
ֽ	2
ָ	12
	4
	4
	8
	2
һ	4
հ	2
ʡ	4
	8
	14
·	8
ϴ	54
ƺ	6
	2
	56
	2
	28
ˮ	2
ʵ	6
׹	6
֦ͷ	2
ʱ	10
	2
ĸ	30
	6
	2
	2
ʿ	6
Ʊ	30
ë˹	2
¡	20
ٰ	2
Ʊ	4
շ	2
	4
	4
	18
ʷ	2
֤ʵ	20
	2
ʳ	4
Σ	14
	10
	2
ֹ	2
	2
ǿ	6
۸	28
	4
ֹ	32
թ	16
	6
¢	26
˳	2
ƺ	2
ׯ	2
կ	2
	6
	8
	6
	42
ɽկ	2
Խ	2
ɽ	2
	90
	52
ҹ	26
	70
	28
	6
ѹ	2
ȫ	8
ҵ	2
	32
	12
	4
	28
	34
ҫ	4
	2
	8
Ϻ	8
ѩ	16
	28
	6
϶	14
	2
	4
籩ѩ	2
ű	4
ͨ	6
Ψһ	32
	4
	54
	2
ͼ	4
	12
	2
С˵	2
	174
	12
ֲ	16
	60
	22
칫	18
ӡ	6
	10
	12
	56
ʡ	8
	2
ʼ	50
	2
	6
	40
	2
ɨ	2
	2
Ч	32
ר	10
ר	18
ʵ	20
	4
˼	32
ҽί	20
	50
	22
	4
	12
	8
	20
	4
	4
ͨ	2
	6
	64
°	10
޵ĺ	8
Լ	24
Ը	98
	36
ղ	2
޵	10
ͼ	2
	16
	2
˹	2
	14
	6
֮	2
ծ	6
	2
	4
	6
	6
ծ	36
	36
	2
	84
	2
	6
	12
	10
ר	8
	136
	4
	202
ŷ	4
ɫ	14
	10
	32
ȴ	24
	62
	52
Ӿ	2
	2
Ӿ	2
	2
	10
	30
	4
˹	2
ë	6
	2
˶Ա	40
ϲ	6
	4
Ů	4
Ů˫	2
Ǿ	2
е	4
Ǿ	2
	4
	4
	2
״	2
	2
ʮһ	2
	2
ί	2
	2
α	2
Ʒ	16
е	74
覴	2
	6
Ϻ	12
ҹ	2
	8
ӹ	12
	2
Բ	4
ײ	2
	4
	22
	14
	2
籨	12
	20
Ͼ	22
	6
	24
ؿ	2
	44
ʡ	10
	4
˫	2
	4
ʽ	4
ȫ	38
	16
ữ	2
ί	24
ۿ	36
	40
	4
ؼһ	2
	16
	6
ȸ	6
Щ	102
	2
	24
ˣ	4
	6
ຣ	6
	2
	6
ϲ	22
	8
	2
Ԥ	18
Ӧ	34
	8
	6
	18
	30
ͨ	10
ծ	10
	18
ʺ	24
	8
	4
¸	58
Ѻ	4
	12
Ƹ	16
	2
	18
ת	10
	10
	8
	4
	2
ȫѵ	2
ѵ	50
	8
ǹ	2
	2
	10
߷	22
һ	30
ʮ	4
ǿ	10
ίԱ	12
	48
	16
	22
	8
ƶ	2
̨	4
	4
	6
	40
	4
Ȩ	12
	10
	8
	4
	4
־ǿ	4
	20
	4
	2
	6
ũԺ	4
չƷ	6
	4
չ	6
	2
	6
Ա	8
ٷ	2
뺯	4
Ϊֹ	6
	4
ŦԼ	36
	2
	22
Ű	8
´	24
	4
ӱʡ	22
	30
	10
	4
	4
ӿ	6
ƶ	2
	4
ֹ	36
	10
鲼	16
ȫ	62
λ	32
	26
	24
͸¶	12
	10
	2
	2
	4
ذ	2
	2
ɼ	4
»	4
ͼ	4
Ϣ	120
	14
Ƴ	24
	2
	8
Ի	14
ʧҵ	10
	20
ش	68
ͬʱ	24
	12
	26
ʵ	10
	82
	8
	34
ڲ	90
	26
	2
	2
˹	2
ȱ	2
Ͻ	2
	2
֧	10
ȱ	54
	6
	10
	14
	4
԰	42
	8
	6
	4
	4
ý	38
Դ	10
	6
Ȱ	24
	24
ʮ	2
Ӣģ	2
İ	6
	8
	2
	126
ͬ	8
Ϲ	6
ϲ	4
Ф	6
Υ	2
	4
	38
	6
	4
	10
ũ	4
	72
	82
	8
ͻȻ	60
Ԥ	2
	12
ߵȽ	2
	36
ƾ	2
	12
Ŀ	2
	2
	14
ɻ	4
̬	16
Ƭ	6
Ϊ	32
	62
Ӧ	2
۳	2
	8
	4
ҽʷѧ	2
ѧ	30
ר	4
	24
Ҽ	12
ѧ	6
	10
	18
	8
	10
ޱ	12
	2
	6
	8
	4
	2
	10
⾭ó	4
ɽ	2
	4
	2
	2
ó	66
Ϣ	4
ʱ	26
	10
	234
չ	2
	8
չ	4
ǩ	28
	2
ë̺	4
زƷ	8
һ	2
Ծ	6
	6
	2
˰	6
Ծ	8
ͬҵ	8
ǰ	10
	2
	2
ժ	2
	12
	14
	2
	2
	6
	54
	4
ָ	4
ʵ	54
ȡ	16
	2
	58
Ŀ	2
ȡ	2
	2
	4
Ժ	18
Ժ	16
	4
ס	4
	2
ͬ	76
Ҫ	8
Ƶ	14
ʵ	6
	16
	2
Ļ㱨	2
	24
̸	8
Ժ	10
֮	4
	6
	28
	2
	32
ҩ	4
ҩ	8
Ч	6
	72
ģ	8
ɼ	2
	4
	2
Ҥ	2
	6
	6
	2
	2
Ҥ	2
	20
	2
Ҥ	2
붬	6
ʡ	18
ׯ	2
ӿ	2
	4
ȳ	6
ÿ	90
	16
ɽ	2
	2
	12
	38
鼮	14
Ϸ	4
ķ	2
	2
־	2
	4
³	2
Ը	12
շ	2
	6
	6
	8
Դ	2
	44
̬	10
ŷ	4
Ӵ	14
跨	10
¡	4
̥	4
	12
	6
	10
ϸ	8
Ȼ	40
ֲ	4
ӹ	2
ȫ	10
	2
	2
	4
	2
	16
	2
	2
	4
Ļ	2
	10
	82
컨	4
	80
Ƭ	72
	22
ٰ	2
	2
	8
	4
ѫ	10
ɭ	4
˺	8
޳	2
ȡ	2
	12
Ѻ	4
	2
	16
	2
Ǩ	4
ƽƺ	2
С˳	2
	4
ݳ	2
	2
	16
	2
	2
	2
	10
	4
е	4
ü	4
	2
	2
	6
	2
	2
ؼ	66
	20
	6
	4
ǿ	12
䶾	2
	2
Ʒ	2
䶾	2
	4
	18
ͷ	2
	18
Ȱ˵	6
	62
֤	2
Ͽ	2
ȡ	4
Ͷ	8
	6
	16
	2
	2
	4
ζ	8
	2
	14
ְ	6
֧	4
Ϫ	20
ٱ	2
	30
ɽ	4
Ƶ	4
	8
	12
ʹ	2
	18
ί	2
	2
ϵ	2
Ⱥ	2
	2
ӱ	2
	8
	2
	8
	16
	16
άΰ	8
ξ	28
Э	66
	18
	2
	16
	4
ָ	4
ת	2
׼	26
	4
	2
ɺ	6
	14
Ը	2
	12
	74
	10
绪	2
¼	24
Ƭʽ	4
մ	4
	2
	2
	10
ܵ˵	6
ʶ֮ʿ	6
˳	64
	26
	6
Ʒ	2
Ӫ	4
ڳ	2
	22
	184
	2
	34
θ	4
ԭ	84
	14
һ	16
ͳս	12
ί	4
ί	2
軰	6
	2
	16
	2
	4
Ѽ	4
ʡ	8
Ʒ	2
	2
	4
	24
	34
	2
Ա	30
ͬ	32
ѫ	2
ŷ	10
	16
½	18
	24
̲	2
	12
	4
ɱ	32
Ī˹	10
	2
	42
	32
	2
ʯ	4
׵	2
	2
	8
	32
	22
	6
쳼ö	2
	2
縺	8
	4
	2
	2
¶	2
Ħ	2
	2
	2
󲿷	36
	2
	4
	2
Լ	14
	20
	64
	4
	46
	18
	2
	10
ߵ	22
	12
Ա	32
ﳤܶ	2
ܺ	2
	52
˰	6
һ	8
	2
˰	2
·	6
Ŀ	6
ƽ	28
ȷ	12
	2
	16
	102
ͼ	4
	6
߶	2
	2
	2
	4
־Ը	4
	2
	20
ƫƧ	10
ˮԴ	2
ѷ	2
ƶˮ	2
	10
æ	98
ʮ	2
ƶ	56
	4
	10
󽧴	2
ѹ	28
ͷ	4
	8
	10
	80
̳	4
	26
Ͼ	18
ȫ	6
ϸ	60
	12
	4
	18
	4
	10
	28
ð	32
	2
侯	32
ʦӪ	2
	12
ϧ	2
	14
ֲи	6
	4
	2
	16
ת	4
ȶ	6
	10
תҵ	2
ͬ	2
	38
Ը	10
	2
֯	4
ֱ	124
	2
ͼվ	2
ҽ	66
	6
	2
Ӣɳ	2
ǡ	2
٤ʦ	2
ʾ	10
	16
ģ	2
	18
̫	6
	6
ǰ	18
ļ	2
	6
	6
ֱ	2
Ͻ	6
	2
ŷ	2
˾	6
żӴ	8
	4
	2
³	2
͸	6
άҲ	4
	20
	4
ͷ	2
֣	2
	2
	4
޲	2
ܾ	2
ֲ	4
Ѫ	4
	4
	28
Ԥ	10
Ժ	2
	14
ţ	12
	4
	18
ҽҩ	62
	18
֮	8
	12
	36
Խ	2
	4
	12
	2
	2
	2
	20
	2
ͨ	46
	50
	2
չ	2
	4
	2
	2
ɳ	16
ʿ	2
Э	2
	174
	2
	12
Ҳ	10
͸	2
һ֮	2
	2
	46
	2
ͨ	4
	2
	4
ȡ֮	4
鳪	2
ϳ	4
	14
ʮһ	18
Ӿ	6
Ůǿ	2
	2
	2
	2
ϲƬ	2
	2
װƬ	2
绪	2
ƪ	2
һȽ	4
Ƹ	8
д	6
	8
ʤ	6
	8
	22
Э	6
ͬ	8
	2
	2
ռ	2
	2
ڭ	2
Ÿ	2
	4
	16
ʲ	4
Ǯ	4
	6
	2
	40
	8
Ļ	14
	18
ӥ	2
߹	2
żҽ	2
	20
Ư	6
Ǯ	14
	26
	32
	2
Ƥ	2
Ӱ	6
	4
	2
	90
	2
	4
Ҫ	60
ʹ	2
ز	2
	4
	6
	2
	24
	8
ǧ	2
ǧ	4
	60
ʼ	8
֦Ҷ	2
	8
	2
Ǳ	6
	2
ͷ	6
	12
	30
	114
	2
	10
	4
į	4
;	2
	6
	4
	14
ٵ	4
Ϻ	10
ⳤ	60
з	8
	2
ڽ	8
	4
һ	4
ɹ	12
б	2
	2
Ӧ	2
ֽ	4
Ǧ	30
б	8
	2
	30
	4
	2
ͷ	2
ó	10
ź	28
	186
	2
	8
עĿ	12
	18
	8
	2
	2
	2
֯	60
	4
ֳ	30
Ҷ	14
С	2
	16
ڽ̽	4
	16
ֿ	8
ҵ	2
ط	6
	4
۶	6
׿	2
	14
ţɽ	2
ũó	2
Ź˾	38
Ҭ	2
Ҭ֭	2
ȴ	8
ӹҵ	2
Ũ	8
	4
֣ãĻ	2
	56
	2
	2
ϣ	4
Ӱ	12
	8
	28
	8
пԺ	6
	18
׵	2
	2
̳	20
	2
	32
	34
	14
	4
	8
̫	6
忪ʼ	2
	6
	2
	70
	8
	4
	2
Ƚ	2
С	4
վ	6
ƶ	2
	2
ֽ	12
	4
	4
ϸ	10
޺	4
	2
	8
	2
	2
	6
Ӳ	2
	2
	4
Ӣ	144
Ȼ	2
	12
	8
Ա	18
д	2
书	2
רע	4
	8
	24
	4
	28
ʵ	2
ʷ	2
д	8
׷	8
ս	8
	4
	20
	2
ʧ	6
սʷ	2
ο	6
ʮһ	24
Ʋ	2
	2
	4
ʧȥ	24
	32
	6
	14
ȱ	6
Ȼ	32
ע	12
	2
	6
	46
	8
Ϸ	4
Ϊ	4
Ƴ³	2
ʷϷ	2
ִϷ	4
ͳϷ	2
	20
	14
˶	6
ңԶ	8
Ӿ	6
ÿ	16
	4
	26
ģ	2
Υ	2
	6
·	8
Ӽ	8
	50
ۺ	4
	2
	4
	6
糡	8
	2
ڻ	2
˼	4
	2
˼	8
	2
й	2
Ķ	4
	2
任	4
л	2
	30
	20
	2
	2
˶	2
СƷ	2
	4
ͯ	10
Ȥ	2
	10
ʧ	12
Ω	2
	14
	4
龰	32
ʱ	6
	4
	2
	26
	2
̧ͷ	10
ͦ	2
	102
׼	36
ͷӶ	2
С	2
	20
	10
׶	2
	8
ƨ	4
	2
ݶ	4
ռ	6
	2
	8
	2
	6
	86
	6
	2
	16
	10
ò	4
	10
	2
־Ը	10
	18
ú	12
ˮ	4
	4
	18
	36
ˮ	2
ʧ	6
	2
ƽ	10
	4
ͷ	2
ת	18
λһ	2
	2
	14
	2
	16
	2
	18
֪ͨ	84
ί	8
差	2
Ѯ	12
	6
	4
	20
	6
־	36
ӵ	26
	2
ί	4
ϯ	2
ݽ	4
ѡ	2
	2
	62
	26
	4
	2
	44
	4
	104
	2
	4
	44
	64
͸	8
	2
ɼ	20
֮Ϊ	12
	2
	8
人	4
Ѵ	2
人	2
	32
겻	2
Ѵ	2
ˮλ	2
	22
	20
	18
	2
	2
	8
鷨	6
	18
	2
	102
	2
	16
	2
	2
Ӻ	2
	8
	40
ǧ	58
鷨	16
	2
	2
	2
	6
־Ը	2
	2
	2
	4
̯	8
̤	16
ũ	2
ũ	10
	14
	6
ְ	32
	2
	4
	6
ڶʮ	6
	2
	2
ͬ	2
	12
	12
ʮ	4
	2
	60
ͨ	40
º	10
黹	12
	8
	2
	2
	66
	4
õ	8
Ƕ	2
	4
ﺣ	4
	12
	8
	16
岻	2
	6
̫	2
	4
ʮ	26
	2
	26
Թ	2
	2
Ȼ	78
г	4
ʳƷ	4
ߵ	2
װ	12
	2
	4
ʤ	2
æ	8
ѡ	4
	2
Ь	36
	4
	2
	36
	2
	8
	8
	6
Сѧ	6
	28
߼	36
	24
פ	2
	8
	18
ѵ	2
˶	10
	2
	30
	2
	14
	2
	8
	6
	24
	24
ȡ	12
Ӫ	12
֮Ч	8
ⲿ	18
	26
	18
ֱ	16
ʡֱ	2
ֱ	6
ֱ	2
	4
	8
	6
	2
ֵ	2
ʵ	12
̬	12
ȥ	76
	2
ԭ	10
	22
Ч	22
ʱֵ	2
	2
	40
	4
	8
רҵ	2
	2
ׯ	2
կ	2
ͷ	2
ů	4
	4
ַǷ	4
Ů	2
	6
ϲ	14
	10
	10
Ǵ	2
Զ	10
㼣	6
ɽ	4
ͭ	2
	6
ɽɽ	2
	2
۲	10
ɼ	2
һ	6
ȫ	14
	2
ǻ	18
ֻ	4
С	20
֮	22
	54
տ	2
Ʊ	2
ӡ	6
ǩ	2
ƶ	26
	4
Ѱ	2
ȱ	2
	2
	2
ʱ	26
	60
	20
	2
	4
»	2
	4
ƶ	2
	12
ѿ	4
϶	4
	4
Բ	20
	2
	42
	22
	2
	16
	2
ίԱ	60
	10
	4
żҿ	4
鿴	8
	12
ǰ	8
	14
	2
꾮	6
	2
	2
	2
	2
Ʒ	10
ɫ	4
Ⱦ	32
	26
ɨ	2
	2
ÿ	106
·	10
	2
	2
	4
	4
Ԥ	2
	8
	4
ʱ	4
	10
	6
ͬ	2
	20
	2
ʦ	2
Ծ	4
ٴ	48
ʾ	4
Ȩ	26
Լ	22
ʴ	2
	6
̫ԭ	2
	20
	50
	6
	6
	10
ѽ	14
	22
ɳ	26
׳	18
	2
	18
	12
	28
	32
ݵ	4
	10
ʲ	2
Ը	2
	4
	144
	6
ѧ	6
ʿ	20
ѧλ	30
ν	4
ع	22
ֲ	22
ѧ	2
ʼ	4
ǣͷ	6
⽻	22
ί	24
ʪ	6
	4
	2
ʮ	10
Ա	4
ܽ	8
	2
յ	2
ˮ	2
	38
	4
	26
к	2
̶	4
	14
	10
	6
籩	12
	6
Ϊ	6
ˮ	4
ƶ	2
	12
	14
ĳ	18
	8
ȫ	4
м֤ȯ	2
ծȯ	20
	4
ز	2
	26
	4
Ͷ	18
Ұ	18
Ա	22
	2
	2
	6
	16
	10
	2
	2
̫̫	6
̫	2
ͥ	8
	2
	2
֮	14
	12
	4
	36
	12
	6
	6
ֶ	2
	14
紵ݶ	2
	14
һ	14
糱	2
	2
ƵƵ	10
	28
	146
	10
׼	2
ҩ	46
ڣ£	2
̱	6
ע֤	2
ȡ	2
	24
	2
	8
Ʒ	12
	10
ʿ	2
Ůʿ	28
װ	28
Է	4
	2
	16
Ů	46
Ʒ	2
Ʒ	12
ľ˹	4
׸	2
	4
½	46
г	14
ͼ	2
ͬ	2
ͳ	4
ȫϵ	4
	86
ȫλ	14
ȷ	48
	2
	8
	18
	2
ҵ	10
ɫ	18
	22
ز	46
н	46
Ʒ	2
ʵ	34
	12
	20
Ͻ	4
	12
һǧһ	6
ѹ¶	2
ز	2
ʯׯ	6
	24
	8
	4
	2
	4
Ƿ	2
	2
	88
ж	4
Ȩ	14
	12
	2
	14
	12
ż	6
ֱϵ	2
ֱϽ	20
	20
ץס	50
Ը	8
	6
͸	40
ɽ	2
	4
	28
	10
	2
	2
	44
	12
	14
ͷ	6
ռ	12
	12
ֵһ	6
	14
	12
	2
ר	4
ŵ	4
Ȼ	2
	2
	6
Ӧ	24
	10
ԭ	6
ո	50
	2
ɻ	2
˹	4
	2
̫շ	2
̫	292
פ	6
	4
	10
°	4
ȩ	2
	2
	6
ܲ	4
	2
	2
	20
ΰ	4
Ѫ	2
	14
Σ	28
	8
	6
콭	2
˰	2
鴨	2
	6
	2
ɭ	18
	6
ı	6
	112
	2
ɷݹ˾	4
	10
Сƽ	88
	2
ƽɽ	2
úҵ	2
Ѫ	6
ȼ	6
	4
ȼü֮	4
α	2
α	2
	4
	2
ƺ	2
̶	2
	2
Ʒ	2
쵽	4
ȼý	4
ȼ	2
	10
Բ	2
	10
	2
޶	2
ټ	2
	14
ͳ	2
Ǩ	4
	2
	6
	4
	2
	12
	2
˴	16
г	4
ʶ	14
	38
	8
ÿ	8
β	2
	2
	8
	12
	4
	58
ӡˢ	4
	2
ݻ	2
	2
	6
س	6
յ	22
	2
	6
	10
	28
ʶ	4
	6
ǰ	42
	6
	2
	8
Ͷ	14
	4
Ϊ	4
ͨ	6
ͨ	8
ͨ	4
	20
ԥ	6
԰	4
	24
	2
޲	14
ֺ격	2
	2
	6
	42
ˮ	2
	6
	10
	4
	20
	18
	2
ίԱ	16
	2
	112
	4
	12
ҵ	4
	2
ϯ	4
ִй	2
	2
Ȱ	6
	2
	18
	2
	6
ս	4
ͷ	2
	2
ƪ	6
绰	14
ؼ	2
	2
	4
ʹ	2
߾	18
ͼ	2
	4
Ϧ	2
Χ	10
ӻ	10
	4
ʯɽ	2
	4
㽭	24
	6
ƽ	2
粿	4
ӰƬ	22
	12
	2
	4
	2
ɳ	2
	2
Ʈ	4
	2
	6
	2
	2
	2
Ѳ	2
	4
	68
ҹĻ	10
	8
	2
˶	4
ɫ	2
	8
ʨ	2
赸	10
	4
л	22
	2
콱	2
ӥ	2
Į	2
ᱮ	2
	2
ⲿ	52
Ƭ	4
ػ	4
	2
	4
ϡ	2
	2
	6
ԭ	4
ڻ	2
	4
	2
ֻ	2
	2
	2
	2
С	2
	4
	10
	8
ʧ	2
	10
ʷʵ	6
ͳս	6
	6
α	2
	18
ķ	2
ݺ	6
֯	4
ش	10
	2
	18
ԭ	18
ų	8
	4
ϲ	2
	6
	20
	14
	2
	2
ط	2
	2
	8
	2
	2
	2
·	4
	2
ͣ	82
	24
	22
	2
ڿ	2
	2
	2
	2
	2
	12
ľ	2
Ϣ	2
	6
	2
	6
	16
	4
	2
	2
ⶩ	4
Կ	12
	6
ظɲ	14
	4
Ϊ	8
	2
ͼ	28
һһ̨	4
ͼı	4
ע	4
Ա	6
н	2
³	2
ѩ	2
	2
ŷԪ	6
	2
ͲΤ	4
ͨ	8
	6
רһ	2
	4
	2
ˮ	8
Ƭ	2
	2
ʡ	6
	106
Э	2
	26
	2
Э	16
ʵ	44
غ	16
	66
	2
	58
ͩ	2
ΰ	2
	6
	2
	2
	2
	2
	4
һһ	6
ի	2
䶯	34
ʡ	22
	2
	10
	2
	22
	24
һ	2
	8
ָ	48
	2
ȿֺ	2
	6
ʳ	2
ĸԸ	2
	2
	6
Ƥ	2
ڷ	16
	4
ٿ	2
	2
	20
	2
	4
ñ	4
Ƥ	2
ī	2
	8
Ե	26
ǿ	2
Ӣ	2
»	2
	52
	2
ͽ	4
ɢ	6
ʤ	4
С	14
	6
	2
	10
γ	4
	72
	88
ʮ	34
ʮһ	12
	2
	116
ͬ	18
ƽ	6
־	4
ʮ	12
ͨ	2
	48
粨	12
	16
߲	12
	8
	30
	8
	18
˳Ӧ	2
	44
༫	2
	4
Ȳǰ	6
ؽȡ	14
	2
	6
	2
ɷָ	4
	4
	2
	2
Ӧ	2
	2
	6
	4
ϱ	8
ƶ	4
ʱз	2
	42
	4
	18
һ˳	2
	8
	12
ƽ	2
Զ	38
һ	26
־	6
иŬ	4
Ǿӭ	8
	28
лл	6
״	2
ţ	2
	2
־	2
ӭ	14
	20
	14
	2
׳	2
ᶨ	12
	8
	6
ƪ	2
	4
	2
	14
	10
	8
д	6
ȫ	16
	42
	2
ֻ	96
	4
к	6
ʱ	6
͹	52
	14
̱	6
	22
۵	42
ܷ	26
ץ	6
	4
Ӳ	28
ǿ	4
	18
ʵ	2
䵽ʵ	6
״	4
˵ʵ	4
	12
ʵ	2
ʵЧ	2
	2
	2
ʽ	10
ž	4
	6
	6
	10
μ	12
ȫȫ	30
ּ	20
	6
	10
	6
Ŀ	6
	10
ȡ	6
ܷ	2
ǿ	16
	6
	4
Ӱ	14
	2
	2
ȹ	10
Ƿ	4
	4
	2
	2
ƻԻ	2
ϲ	8
	24
Я	4
	2
	8
	4
ָӼ	4
	2
	2
̷	6
	2
ּ	6
	6
	2
ˮ׼	12
ֶ	2
	14
	4
	8
	2
ı	4
	2
	4
㷢	4
ϴѩ	2
	2
	142
	4
ר	10
Ů	2
ݶ	2
	50
	2
Ұ	8
ɽ	8
	4
̳	2
и	4
Ī	14
	2
ͼ	2
ѡ	2
ҹ	4
˯	6
	4
	2
С	10
	6
	2
ʫ	2
	4
ϯ	2
	2
òϢ	2
	14
лĻ	2
	2
	4
Ӿ	4
ѡ	4
ëŮ	2
ϲѶ	6
կ	2
߳	16
ף	34
	10
ӳ	16
	24
	18
	8
	8
Ӣ	8
ʯɽ	10
	20
Ҫ	4
ʱ	2
ʮ	2
г	28
	2
ʼ	2
	30
	20
	10
װ	2
ǧ	6
귢	2
ʮ	2
˰	4
ܾ	4
	30
ȵ糧	6
糧	12
ѯ	14
	28
	34
ʵ	8
쿴	4
	2
һŻ	2
Ż	2
	4
һһ	6
	14
֮	4
ص	38
	60
	6
	24
һ	36
	16
	12
	142
բ	6
	10
	2
	4
ҵ	2
	28
	4
	6
	4
	32
	46
糧	4
	2
ϰ	12
	2
	66
Ƿ	4
	8
	2
һ	2
	4
簮	2
	2
	4
Ϊ	14
	4
	4
ƽ	6
һ	6
˽	6
	40
ʷ	6
	4
ʫ	2
	12
	2
ʡ	32
Ϸ	4
·	2
	2
	6
Ӫ	16
	4
	10
	2
	2
	4
	10
ȫ	4
	20
	6
˽׼	12
	6
ҵ	8
ܹ	6
׼	6
ϵ	8
Ž	4
м	2
Ͷ	6
ת	2
׼ײ	2
	14
	2
	18
	30
	10
ƶ	2
ӭѶ	2
	2
̬	2
	2
Ȩ	26
	2
ְ	2
м	2
þ	10
޲׼	14
	16
	10
ܳ	8
	22
	72
	98
ҽ	10
Ч	8
ʱ	8
	4
	4
	6
Ը	4
	16
ɥ	8
Ӽ	8
ǻ	10
ѩ	14
ȫ	24
	20
ɣ	8
ش	16
ŵ	6
ƹɽ	2
һ	4
ѩ	18
·	4
֧	2
ϵ	2
	14
	6
տ	4
ɽ	4
	4
һ	32
	2
	2
	6
Сʱ	122
	4
	10
	10
	4
һ	2
ȼ	12
	2
	10
	12
	2
Ʒ	2
ʵʵ	2
׿гЧ	6
	6
ܶԱ	2
	10
	14
	4
Խ	8
Ԥ	16
Ԫ	12
ʱ	16
	100
С	6
	12
ǰ	2
	4
Сѩ	6
Ȱ	2
ش	16
	2
	2
	14
칫	82
	2
ý	8
	10
Ⱦ	10
ׯ	8
	6
	2
	4
	4
ͼ	6
	2
	10
ͬ	2
	6
Ա	16
	14
֤	8
ϸ	28
ʸ	14
֤	4
½	6
½	24
	2
ί	2
	2
	40
ͨ	10
֤	6
ƻ	4
ʥ	6
һһ	2
	2
	2
ո	4
ò	4
	52
	34
ұ	20
	4
Ӹ	2
Ǵ	10
Զ	2
	2
ǭ	2
	6
	2
ǭ	2
	2
ʡ	2
Է	16
ũˮ	6
	4
	14
ֲ	4
ˮ	2
ǭ	2
ʦ	8
̸	20
ò	36
	6
	2
ϥ̸	2
ϸ	16
	2
ഴҵ	6
˫	14
˼	2
ٹ	2
	10
	2
	2
滯	6
Ӧ	2
	14
	20
	2
	2
	4
	2
ҽ	4
ѧԱ	10
	24
	2
	2
Ҿ	36
	4
	4
	14
	2
ܷ	2
	18
	2
	2
	2
ط	2
	12
ȾԴ	4
	14
	6
	4
	2
	4
	2
	6
	6
	14
	22
	12
	2
	8
	4
ׯ	4
	10
	2
	4
	2
ͣ	6
	2
Ʋ	24
	24
	4
	24
ͣ	2
	16
վ	2
	2
֧	6
	4
⴬	2
	2
ˮ	2
	2
´	10
ͣ	2
	30
	2
Ƴ	2
Ŀ	8
	22
	2
ֳƿ	2
	16
	8
̤ʵ	8
	6
	32
	12
	6
Ļ	2
Ļʽ	8
	4
ҹ	2
ɽ	2
ײ	14
	36
ڣ	2
	2
	2
	6
	28
Ļ	16
ξ	4
ǧ	16
	2
	4
	18
´	2
·	2
	54
뾳	2
	8
	4
ò	4
ޱ	2
ͯ	6
	2
	14
	2
	2
Թ	2
	6
	6
	4
	10
	46
	4
Щ	28
Ѻ	2
	6
	2
	28
	2
	4
жŷ	2
	10
ӽ	6
	14
û	12
	6
ŷ	4
ؽ	10
	4
ά	8
þ	6
	2
	10
	20
¸	2
ԭ	4
	4
ʮ	4
	2
ǳ	6
	4
ս	8
	2
	2
	2
ʮ	34
	12
ʮ	4
	6
	2
	2
Ӱ	28
ˮ	14
ͨˮ	4
	20
ʮ	4
ʥ	2
ȱˮ	14
ʮ	12
ˮ	2
	2
ˮ	2
ȱˮ	2
	6
ˮ	2
ʡ	6
	6
ʮ	2
ˮ	2
;	6
	2
ˮ	6
г	10
ˮ	2
	2
	6
	4
ǧ	4
	18
	2
ʮʱ	2
	2
	2
̨	6
ʴ	2
	44
ͻ	2
װ	2
	18
	4
װ	6
	4
	52
Ӽ	2
	2
ӻ	2
ٽ	16
	38
̯	14
	2
	10
	14
	32
	14
ϵ	4
	6
	10
µ	8
	4
	34
	20
Ѷ	2
׸	2
	2
άͷ	2
	2
ܸ	2
ҪЮ	2
	2
⵽	16
Ƿ	6
	4
	2
Ƿ	6
ҹ	12
ǰ	8
Σ	2
	2
ƴ	2
	4
¹	4
	14
	4
ݶ	12
	8
ϯλ	12
Ǯ	12
	8
ʵ	18
	6
˵	8
˽	4
	4
˹	2
	18
	28
	20
	8
ְ	6
	2
ѡƱ	6
	2
	14
ʤ	10
	26
ѧԺ	2
ְ	4
϶	4
Ժ	62
ͬ	12
ִ	2
	50
	6
	40
	6
Ҳ	4
	4
ǫ	4
⽻	16
	58
Ϸ	142
	32
ѧ	8
	12
ϵʷ	2
ƪ	4
˻	10
л	6
͹	2
Զ	2
	18
Ԯ	2
	12
	16
	2
	2
ó׶	12
	6
	8
	4
	4
ҵ	2
	2
	42
	14
׼	8
	34
ͬ	2
Ȩ	24
޵	2
ٳ	2
	10
	2
	10
	10
	2
Ҫ	16
	16
ι	8
	4
ֳ	4
	2
ϳ	4
	4
	6
ݲ	4
ʺ	8
	4
	26
ٷϴ	2
ֳ	2
ͳ	2
	2
˽	10
ɽ	10
	16
	10
	12
	6
	6
	24
	10
м	16
	2
	6
	16
ǿ	2
	4
	4
	2
Ϲ	8
ͬ	4
ʹ	2
չʷ	6
	34
¹	6
	2
ն	2
°	10
	4
ʹ	36
	12
	28
Զ	2
ʼ	6
	4
	8
	8
	2
½	4
ٷ	12
	2
	14
	2
	48
ͨ	2
洦	2
	2
	8
	2
̬	54
Զ	2
	4
	4
	2
	6
ͨѶ	12
ɹ	4
	4
	26
Ī˹	34
	2
	4
ͽ	8
	6
	8
	22
Բ	2
׶˹	2
ڿ	16
̸	54
ᵽ	18
	4
ʧ	4
	2
	4
˶	6
ı	10
Ȼһ	4
ƽ	2
蹵	2
	10
	6
	2
ڶĿ	8
	4
	6
ں	8
	2
	2
	10
Դ	2
	4
	2
	2
	2
	2
	2
һ	2
	12
	2
	6
	2
	2
ɾ	4
	4
Ǹ	2
	6
Դ	4
	2
	48
ŵ˹	2
	8
	38
	2
	2
Ԥ	4
	6
˿	8
	20
һ	18
ۿ	12
Ű	6
䡪ʡ	14
β	4
	2
ǹ	2
	2
ͬ	4
°	4
	14
	32
ƾ	8
³	2
	6
ϰ	36
	10
ʡ	2
	6
´	2
Ϊ	12
	14
³	8
	4
Ͳû	2
	6
ɳ	2
	2
ɽˮ	4
ڰ	4
̫	4
	20
Ϊ	14
	4
Ǵı	4
չ	4
ͨ	34
	4
	12
	24
ҹ	4
	2
	4
	2
Ȩ	2
	14
Ŀ	6
ھ	4
нڼ	2
ƽ	24
Խͨ	4
С	2
	4
	28
	66
֮	6
־	16
	6
յ	18
	20
ϣ	6
װ	2
	6
Ը	2
	20
	12
	8
	22
	10
	4
	2
谭	10
	2
޳	6
¼	4
	10
	2
ר	2
е	8
ǲ	2
ҽƶ	10
	2
	4
	4
ﰣ	2
ƽ	10
ɷ	2
μ	2
ǩ	12
;	8
Һ	16
ʽ	2
̹	6
	4
·	4
	10
·	4
	12
	12
չ	16
ͼ	4
Ͽ	20
	6
	8
·	2
	4
	2
	2
	80
޶	2
ɷ	2
	2
	2
ľ	2
	2
	2
	20
ά	6
	22
	4
¼	2
Ƽ	2
	2
	8
ŤΪӯ	4
	4
ƽ	2
ʡ	14
ܲ	2
	4
	8
	2
	2
	4
	42
һ	2
	2
Ĵ	2
	20
	4
±	12
ĳ	6
	6
Ľ	4
	10
һ	6
	8
ȹ	10
	2
	2
	10
	14
	4
ʡ	12
	2
Ͼ	2
Ŵ֤	2
ʮǼ	2
	2
	16
	24
	2
	8
	6
ǹ	6
ѧ	22
㲥	2
	2
ֱ	2
С	2
	24
ӯ	12
	8
	2
ƽ	10
̸	8
г	10
	8
켣	2
	12
ֵܲ	8
	2
	4
	4
	34
ش	6
	4
	6
ֲ	16
ҵ	2
ɫ	4
ұ	2
	2
	2
	4
Ӫ	4
ʲ	2
	46
	8
	4
	2
ʵ	2
	2
	2
	2
ת	18
	2
	8
ɽ	2
ռ	6
	20
	10
ٷֵ	16
оԺ	6
	14
ָ	4
	10
̰	4
ᷨ	4
	6
	4
ƾ	10
	58
	10
	28
	2
	2
	34
缶	4
һ	4
ȷʵ	28
۶	20
ʵ	4
Ȼ	16
Ȼ	20
Ʃ˵	2
а	4
	2
	26
Ͽ	6
	82
	4
Ŀ	6
Ŧ	8
߱	26
	2
	52
ͬ	6
ʵ	2
	8
	10
	4
С	6
	4
Ϊ	14
	12
	4
	4
	2
	4
ͶƷ	2
	8
	12
רҵ	10
̬	4
ϸ	4
˼	2
ʱ	2
	14
	4
ϩ	14
	22
	4
޶	14
	2
߼	2
ݼ	2
֮	18
	10
Ѱ	8
	20
ǩԼ	8
	10
ڲ	2
Լ	14
㼶	2
	2
	4
ź	18
ʧ	2
ʹ	14
ҵ	2
	2
	4
ƫ	4
¢	2
	6
	2
׬ȡ	4
ռ	4
Ϊ	12
¢Ϸ	2
	8
Сҵ	2
	6
	4
ٻ	6
	14
ֹ	4
	16
ɹ	2
	12
ͨ	4
	2
˦	8
	28
	10
	10
	2
	2
	8
ƶ	6
Ӫ	4
	16
Ӳ	10
ҵ	2
	34
ں	4
	12
	20
ɢ	16
	8
	2
	4
	2
	2
չ	2
	2
޹	10
	4
	2
	4
ɴ˿ɼ	4
	24
	14
ƺ	30
Ϥ	24
ۼ	4
³	2
	10
	30
֧	20
	2
ǡ	8
ɹ	2
	28
	10
	2
	2
ã	2
	2
	2
	22
ã	2
	2
	2
	2
ã	2
Ӧ	20
ò	56
	8
	18
̯	2
	2
׬	14
	8
ծ	2
̲	2
Ч	2
	8
ת	6
˳	6
	4
ȷ	12
	4
ر	8
	8
	6
ǰհ	2
	4
	2
	28
	2
뷨	32
Դ	2
ʼ	2
	2
ǧ	2
˵	2
ο	16
	4
С	8
	8
	10
Сҵ	10
	6
	2
	16
	2
	6
	14
ծʽ	2
	2
	2
ͣϢ	4
	2
	16
Ƿծ	4
	18
ä	2
ͬ	40
÷	2
ɳ	6
	2
	2
	2
פ	10
	44
ע	2
ά	22
ʱ	4
	10
	6
֧	10
	2
	6
	2
	4
	8
߰	2
	6
	6
ҵ	6
ױ	2
	2
	2
	16
ʱ	4
֮Σ	2
	10
	4
ز	4
ܲ	16
	2
ȴ	2
	2
ν	14
	2
	4
ռ	6
Ե	14
	4
	4
ʱ	2
ʵҵ	4
۽	6
	2
äĿ	2
	8
ּ	14
	4
	26
ɴ	16
ֻ	48
	16
	8
һҵ	2
	6
ʵ	12
֪֮	2
ﻮ	4
	4
	2
ʢ	8
Ӧ	4
	2
ǽ	2
Ԩ	2
	42
˶	2
ʵ	2
˾	2
˾	2
ƺ	2
ĸӹ˾	6
	2
	2
	6
ĸ˾	2
	16
	2
ó	2
о	2
ƻ	6
	2
	2
	16
	4
	6
	10
	2
Ⱥ	4
䵱	2
ƫ	4
	6
Ա	6
	50
ݺύ	4
Ϊ	12
	12
ͬ	28
	8
	2
	20
ɶ	20
ò	6
	2
ͷ	2
	12
	2
ػ	2
ҵҵ	6
ͷ	6
	4
	32
Ī	2
ȻԴ	4
ŵ	20
ĳ	6
ռ	14
ȥ	18
ʧ	8
֤	8
ҡǮ	2
Ǯ	2
޵׶	2
	4
	6
͸	2
	4
	14
Ȫ	4
ĩ	4
ʡ	10
ϢϢ	6
г	2
	2
ʮĴ	24
Ǽ	4
	4
ͨ	4
ͨ绰	2
	2
ɨ	8
׳	6
ä	6
ҵ	2
	2
ʵ	12
	2
	4
ʱ	22
ӹ	4
	2
	2
ȶ	4
ͨ	2
	4
پҵ	2
	10
	4
	2
С	2
	2
̿	8
	2
	6
	2
	6
δ	2
ȼ	2
	18
	2
ҽ	8
Ѳ	4
	20
ɽ	2
	2
	2
	6
Ŭ	2
м	16
ר	4
	16
Ѳݳ	2
	12
	4
	2
	4
֫	2
	4
	10
	2
	2
˵	2
վ	6
	14
ʱ	22
	4
Ȩ	4
	4
	6
	6
ƽ	2
չ	4
³	2
	18
ɱȼ۸	2
	2
	2
	28
·	6
	4
Ӵ	12
	2
ʱ	4
	6
	2
	2
	2
	46
ֽ	4
	4
	2
	2
	2
	2
޷	6
ֹ	2
ˮȾ	4
֣	6
	4
Э	2
	4
	4
֣	2
ˮ	2
³	2
	88
ɳĮ	14
˶	4
	8
	2
̻	2
	4
	18
	4
	16
	4
	2
	12
	4
	2
	18
	4
	2
̴	4
	28
ϸ	22
	4
־	4
	8
̷	10
û	4
ʹ	2
ȱ	4
Ӻ	4
˿	14
	20
	2
	2
	2
	12
ͻ	12
	2
	4
	2
	2
	10
ҵ	6
	2
ŵ	8
	2
	8
	14
ǰ	16
ϰ	16
	18
	16
廨	2
	26
	2
̽	2
봨	2
ں	2
	2
Ķ	42
	8
æ	4
Ȫ·	2
	2
	6
	2
ʼվ	2
ƽ	2
	4
ϸ	24
	2
	2
ù	2
	2
	2
	4
	12
㲻	2
ŵ	2
	2
ð	6
	2
ֻ	6
	50
	42
	16
ȫ˳	2
ۻ	4
	4
Υ	8
	2
Ǧ	2
ĥ	12
	4
	12
	20
	8
ҵ	2
	4
׼	2
˵	12
Цֹ	2
	10
ղ	8
Ȼ	2
	10
һ·	12
Դγ	2
	4
	16
ܿ	38
	8
	6
	6
	14
ר	14
ڰ	14
	20
	6
޼	6
	6
	6
	52
	2
	12
ɷ	2
ò	2
	8
ÿһ	4
Ϣ	6
ɶ	4
	6
	34
	2
֤ȯί	4
֤	4
	12
˰	26
ܾ	10
	4
ѵ	12
Ը	26
һƵ	2
	2
벻	4
ȳ	2
	8
	8
	14
	14
	8
ֿ	12
ԱЧ	8
	12
Ťӯ	6
	12
־	8
	2
Ť	4
Ż	2
ſ	6
Ըӯ	4
	14
	4
	4
	2
	34
	2
ܹ˾	24
Ա	8
	8
	4
	4
	4
	8
	4
ӻԾ	10
	8
ũ	2
ʹ	2
Ӫҵ	22
	4
	2
	14
	2
	6
̹	8
	2
	2
	20
	6
	2
	2
	2
	2
	2
	2
	2
	2
	2
	2
޿	4
Ѯ	4
г	8
	2
ǩ	4
ж	6
Ʊ	4
	2
	2
	2
̳	4
	2
	2
	2
	16
ҵ	10
	8
̯	8
	30
	2
ϯ	10
	6
	6
ֲ	2
סլ	26
Ƽ	16
	4
ά	18
Ͷ	12
	10
	18
	2
	8
	2
ֲ	2
	16
	22
	4
ľ	12
	4
ʹƷ	2
ֻ	2
	2
	2
	18
̨	8
ǧ	10
	22
	12
ǹ	8
ʱ	2
̨	4
Ա	8
ƶ	2
	10
	2
֯	18
ѡ	22
ɫ	18
¡	2
ǡ	6
	2
뵳	12
ϻ	2
	18
	8
	2
	2
ֵ	10
	4
հ׵	2
	4
	2
	2
ڵ	4
Ƹ	4
רְ	4
Ǩ	6
»	10
»	2
	2
ʯ	2
	2
	12
	2
	2
Ӣ	2
	26
ҵ	14
޳	26
½	8
	2
	2
ͷ	6
	2
	4
	6
֧ӳ	10
ˮ	34
ױ	2
д	8
	2
	10
㡾ְ	4
ע	6
Ӳ	4
	12
	14
·	2
г	8
	18
̯	4
ϴ	2
	16
ֻ	6
	2
·	8
	8
̯	4
ҵ	2
ũóг	6
	24
Ȱ	2
ǿ	4
	4
Ȱ	20
	10
	2
ʽ	4
	18
	4
һ	10
	12
	2
	2
ͨ	30
	4
	2
	16
	110
һ	2
	6
	16
	2
	36
ʱ	8
ʱ	6
ѵ	4
ǰͷ	2
	4
	6
̹	2
	2
	24
첲	10
	4
	34
Ѫ	2
	2
ֹ	12
	8
	4
վ	2
Ϊ	18
Ȫ	2
	30
ò	8
	4
	2
־	8
	28
г	80
С	4
ǰ	8
	4
Ȱ	2
Ի	2
	2
	2
ʧְ	2
	4
	8
	30
	12
Ǹ	8
	8
ˮ	10
	8
ն	4
ں	12
	48
	26
	2
	16
	4
	60
Υ	2
	2
	2
	102
	12
	4
	2
	10
	16
	8
	24
賿	4
ʱ	8
	14
	6
	4
	2
ӳ	4
	38
Σ	4
Ϸ	2
	584
жӳ	4
	24
ְͣ	2
ְ	4
ֳ	4
ί	10
ί	2
	8
˵	8
	2
	14
ĳʡ	2
»	2
	2
	2
	6
Լ	2
	2
	6
	10
	16
С	10
׮	10
ֵһ	2
ͷ	2
	8
	2
	2
׼	2
Ժ	16
	8
Ч	2
	2
	2
Թص	2
Ϊʲô	114
Žֹ	4
Ƿȱ	2
쵼	6
	6
	34
	4
	2
Ⱥϵ	2
	2
	2
֮	14
Բ˵	2
Ⱥ	4
Ⱥ	2
ʢ	4
	2
	12
ɽ亣ζ	2
ͺ	2
ｻ	2
ϯ	2
ͨ	10
к	4
˷	6
ò	2
֢	4
޽	2
	2
	6
	28
Ҫ	4
ͷ	12
	4
	4
ձɽ	2
	6
	30
	20
	2
	2
	10
	6
	2
	4
	2
	2
	4
	4
׼	2
	32
	2
	4
	16
	10
	4
	2
	2
ȫ	6
	2
	6
ϵ	2
·	4
õ绰	2
	10
ش	8
ˮ	14
	2
	2
£л	22
	2
	92
	6
ɾ	18
	46
	8
	2
	2
	4
	4
ְ	2
ȺȺ	2
	2
	4
Ź	2
	2
	4
Ԭׯ	2
ĳ	10
ƺ	4
	8
	64
	2
վ	4
	18
˻	2
Ѧׯ	2
	2
	66
	4
ץ	4
鰸	4
ê	12
İ	2
	14
	4
	2
	4
ܲ	4
̱	6
	10
ԤӦ	4
ʷ	2
˶	4
	12
ӹ	8
	8
ȫ	10
	4
Ӳͷ	2
	4
	2
	8
紦	4
ʺ	2
̽	2
и	2
	2
ѻ	2
	16
	16
	2
	4
	4
	4
	4
	2
	2
Ҳ	2
ê	2
	2
Ѷ	8
	2
С˵	2
ɽ	2
	22
¥	52
	2
	2
ػ	6
	2
̮	2
Σ	8
	2
Ͷʷ	2
ܹ	2
Ԯ	4
ʤ	4
ȫ	2
ê	4
ԭλ	2
һ	4
˲	8
ѵ	10
貿	4
֧	2
ް	2
ת	2
	2
	4
	72
·	16
	2
ʫ	18
	4
	4
ͼƬչ	4
ﲻ	6
ѵ	22
	4
	2
	8
վ	2
	10
	2
	6
	2
	4
	4
	6
ͼ	2
Ϻ	2
	2
	4
þ	14
	4
Ͻ	4
Ű	2
	8
	4
	4
	2
	2
	2
	10
Ǯ	14
	4
֮	2
û	6
	2
	4
	8
֮	6
	30
ѿ	8
	4
ְλ	10
	36
	2
һ	8
ʵ	4
֮	2
л·	2
ʦ	16
δ	2
	8
Ƴ	12
æ	4
	2
¼	2
	78
Ե	2
	30
ֱ	2
	8
	2
	2
	2
ں	4
	8
Ĵʡ	10
ɽ	2
	2
	8
ɽ	4
	10
	2
	56
	2
	4
	6
Ǣ	4
־	2
	2
ɺ	2
	2
ᾧ	2
	6
	2
	10
˳	4
	34
ɽʡ	4
ί	10
֯	12
ֱ	6
	2
һ	12
ѡ	2
ʵ	12
	2
֧	6
	32
	2
	6
	2
ƶ	10
	12
	2
ְ	6
ϲ	2
	2
	2
׵	4
	2
	2
	4
	20
ܹ	2
	2
	2
	4
ȥ	2
ʮ	8
ÿ	14
	2
	2
	2
	6
	2
	2
	4
	2
	2
¹	8
Ժ	4
	58
Ÿ	2
ʡ	24
	2
ɽ	6
	2
	2
	2
԰ʽ	2
ί	12
	2
ϳ	6
·	6
ϵо	2
	2
	2
ɱ	2
ǰϦ	14
	2
ײ	2
	2
Ƭ	4
Ƭ	2
м	6
ѧ	2
ϵ	2
̨	2
Է	2
	4
	4
	10
	2
	2
	18
վ	8
	4
	46
ȫ	4
ȥ	10
	4
ⳡ	2
Ȼ	4
װ	12
	2
ְ	2
	2
	16
	2
ƽ	2
ũ	4
ò	4
	2
	2
	10
־	2
	26
ʡ˴	8
ͮ	2
	14
	2
	2
	2
	4
	4
ս	4
	2
	2
ǰ	2
ʧΪ	2
	2
ܷ	10
ѡٷ	6
	30
ת	2
	2
ʶ	4
Ȩλ	2
ʼ	2
	90
	6
	2
ɹ	4
˼	4
	2
	14
	4
λ	6
Ը	2
	6
	8
	12
˼	6
ر	2
	22
	2
ְ	2
	2
	20
Ϊ	6
ɢ	2
	2
ʺ	4
ɥʧ	4
ѭؾ	2
˵	2
Ļ	12
и	2
ɭ	4
	10
	2
ˮ	2
̩ɽ	2
	2
ػ̹	4
	2
ʡ	2
	2
Ӱ	4
	12
Ļ	2
	4
	2
	2
	2
	2
Ժ	4
µ	2
Ժ	2
	2
	4
ɽ	2
Ů	2
	4
ҹ	2
	4
	2
	4
ʢ	2
	4
	2
	12
ռ	24
ḻ	6
Դ	2
	2
ʾ	4
	6
	10
	12
ֱ	16
	4
	14
⻯	2
ί	10
ί	12
	4
д	4
޼	2
	4
һ	6
ıȡ	10
	18
	8
л	12
Ψ	2
	2
	48
Ū	6
	2
ʮ	4
ʵ	10
	10
	16
	2
	2
ȡ	2
	2
	4
	4
	2
ҿ	2
ϰ	2
	2
Ļ	6
	2
	4
	2
	2
	2
Сվ	2
	4
	8
	2
⳵	12
˾	64
㲻	2
˳	2
	2
	12
	4
	2
	2
ϻ	2
ɨ	18
	8
	10
	14
	6
	2
ݶ	2
ͷ	8
¼	6
	10
ʡ	6
	6
㶫ʡ	14
ͷ	6
ֲ	2
չ	2
α	4
ڳ	2
	2
	12
ΰ	2
絽	4
Ĵ	26
еѩ	2
	8
	4
Ľ	18
	20
	4
Ӣ	2
һ	26
	12
Ȼ	2
	6
	2
ٻ	2
	2
	2
ǧ	2
ʦ	10
	2
ҵ	2
	2
	2
Ů	2
	2
	2
	12
	4
ó	8
	4
	2
	2
ɽ	2
ͼ	2
	2
	4
	2
	2
	26
ӱ	4
	4
ƽս	2
	2
	4
	60
ͽ	2
	2
ĵ	4
	20
	2
Э	4
ʷ	2
	2
Ա	2
	2
Ա	16
	10
	14
	2
ǹ	2
ŻѪ	2
	4
	22
	2
ͷ	24
	4
	4
	6
벻	8
	2
	8
	4
˻	2
ʵ	4
	6
	14
	4
	12
	4
ʼ	4
	16
ѡ	12
ѡ	8
ӽ	2
	2
	2
	2
	6
ɫ	8
	4
Ʒ	2
ٻ	14
	20
	8
	10
	6
	2
	8
	16
	6
	2
	6
	2
	12
	2
콭	2
С۴	2
	2
	2
	2
	2
	2
ҩ	2
ڷ	2
ʩ	6
	2
׻˵	4
	2
Ͱ	8
ѧԺ	16
	2
ձ	4
	10
	28
	30
	10
	6
΢	2
	2
Ʒ	2
	2
	4
	6
	2
н	4
л	2
д	10
	2
ֹ	2
⻨	2
	8
	2
	10
	10
	2
־Զ	2
	4
	2
Ů	8
ھʮ	2
	2
	2
˼	4
	4
	20
	2
	2
	62
һ	2
	4
	6
	8
	2
ʻ	10
ر	6
	12
	2
ڵ	10
	2
ѩ	20
ߺ	2
ȱ	2
	10
ѩ	8
	12
ѩ	20
ȱ	2
ũ	6
	6
	4
ץи	2
ϲ	8
	2
ӵ	6
ӵ	2
ɽ	2
ͬ	2
	2
	4
פ	4
˫	2
	6
	2
ʡ	10
	4
	6
	30
	2
ʾ	4
	2
ʡ	4
Ԯ	4
ʧѧ	6
ط	4
У԰	14
	6
	8
Կ	2
	10
	2
	8
ԣã	4
ɶ	12
ǰ	2
ɻ	8
	6
	6
	2
	6
ŵʽ	2
	34
׷	14
	6
	2
	12
	2
	2
	2
	12
	4
	8
͵	2
߼	2
	14
	2
һͬ	6
	8
	2
	6
عչ	2
;	2
׻	2
	2
滭	2
	2
й	4
ѧ	2
ݱ	4
Э	2
Ժ	2
	2
ɣ	6
	4
ʵҵ	2
ʿ	2
Ͼ	2
	4
佱	2
	2
	12
	4
ֻ	2
Ƚ	2
֯	2
Ԫ	2
׿	2
ൺ	4
Ƶ	4
ڰ	4
ػ	2
ȰƤ	6
ֲ	6
߾	2
	56
ǧ	6
	2
	2
	12
Ϸ	2
ɢ	6
	18
Ѭ	2
	2
Ͻ	12
	4
Σ	4
溦	4
ѧ	2
	2
	24
	8
	10
	2
	8
ſ	4
ҵ	22
Ѹ	2
	18
	6
	8
	2
չ	8
ﱸ	8
ά	2
	12
	14
Ժʿ	12
	2
	6
¡	6
	6
	10
½	8
ԭ	36
	10
	14
ת	8
ˮ	20
	4
䶯	6
	2
Ƽ	2
	2
һ	10
ը	2
	2
ŨŨ	4
	4
侲	6
	14
ӡ	2
Ժ	10
	4
Ժ	12
ѡ	10
ֱ	6
	8
	20
ӽ	8
	2
Ŵ	4
	2
ũ	10
	4
	2
ݼ	2
	2
	2
ʵû	2
	2
	4
ˮ	2
	4
֮	2
Э	2
	6
	2
ѡ	2
Ұ	4
ѡ	4
ȡ	12
	16
˻	2
ɽ	2
Ա	16
̹	4
ѧ	2
	4
ƶ	2
	2
	18
ѧ	4
	2
ܶ	4
洢	4
	2
ֱ	10
	2
	4
洢	2
	2
ӡ	2
	2
	10
ͻ	2
	2
ˮ	6
	24
̫ƽ	8
	2
¡	4
	4
	4
־	2
ֲ	2
	2
	6
	2
	2
	8
	10
о	4
Ʒ	6
Ӣ	12
ֱ	2
赥	4
	6
	2
տ	4
ϼ	4
	12
	8
	2
ְҵ	14
˫ʮ	2
	10
ҽ	12
ͨ	14
ѹ	8
	10
Ʒ	8
	2
	2
֢	8
	6
	2
	2
	2
ҽ	2
	12
	2
ʡ	4
Ͳ	2
	2
ҽ	2
	2
	4
ҽ	24
ͼ	10
ǰ	12
ռ	14
Ů	14
ҩ	10
	4
	6
	2
	12
¼	18
	10
	2
̨	2
Ľ	4
Я	8
	20
	2
	2
Ů	6
	2
	2
ѹ	2
ϲ	10
ƶ	14
	8
	8
ʣ	8
	2
	8
	4
	2
ʳ	20
	2
	2
³	4
շ	2
	2
	2
	2
ƾ	4
	6
	10
ҽѧ	6
ҽҩ	2
Ƽ	2
д	2
	20
	18
Ƹ	2
	2
	2
	2
	2
ϸ	2
鶯	4
̽·	2
ɴ	8
ʡ	2
ѧԺ	2
	2
	2
	4
ѧ	2
	6
Ⱦɫ	4
	10
ͼ	2
	6
Ⱦɫ	2
²	4
	16
ڶ	4
ִ	2
	6
⿪	2
ѧ	2
	4
	12
ԭ	2
ѧ	4
	2
	2
	4
̽	6
Ҹ	6
	4
	6
	2
ҩ	4
	6
	6
	4
֢	2
	2
ҽԺ	4
	42
	6
ָͷ	2
ͷ	4
ҩ	2
Ĺ	6
	28
ҩ	4
ҽҩ	10
ֻ	8
ѡ	12
	2
ʨ	6
	6
	6
ţ	2
󲿷	12
	10
з	2
Ȳ	2
ر	2
	2
֢	6
Э	4
Ч	2
Ч	16
ر	2
	2
ίԱ	2
	2
ڿ	6
	6
츣	2
ȵ	4
׶	12
	2
绰	10
	8
绰	6
ý	10
	10
ʽ	4
	6
Ʒ	8
л	2
绰	2
ͨ	6
	2
	2
	8
ʹ	2
	16
	4
	8
	6
ע	14
	8
	2
ռ	4
ӵ	2
	16
½	2
˵	2
	2
	4
	8
γ	2
	2
ϣ	8
ҵ	10
	2
	2
ƾ	8
۲	2
	18
ִ	2
	82
	10
Ÿ	6
ƽ	4
	6
»	2
ר	4
	2
Υ	4
Ϸ	4
н	2
	2
	2
	8
û	2
	44
Ĳ	6
ϻ	6
ʼ	2
ǹ	10
ΰ	30
Ϸ	16
	10
س	2
ɺ	2
ʱ	6
	10
ɰǹ	2
	6
˰	6
ר	2
Ļ	4
̾	6
	2
ս	2
Υ	4
	2
԰	2
	2
ۺ	4
	10
ع	10
Ϸ	2
Ӫҵִ	4
һ	2
ɰ	18
ʷչ	2
	4
Ӣ	4
Ϫ	2
	2
ɣ	4
רƬ	6
	4
	8
ż	4
	2
	10
	6
	4
걾	4
	6
	16
֤ȯҵ	10
	8
Ȩ	6
	2
	4
	4
ѧԺ	2
	2
ֱ	4
	2
ԺУ	12
	4
óٻ	2
¼	6
	2
ѡ	2
ҩ	4
	8
	2
Ƴ	8
ҳ	8
	2
	6
ĩ	12
ķ	4
Ƹ	4
	12
Ϊ	12
	4
	6
	4
	12
	18
ͬ	4
ӿ	2
ű	2
ҵ	4
ܵ	12
뵺	20
	14
	2
	12
	2
	2
Խ	4
	2
	2
ƴ	2
	4
	2
Ѯ	8
֪	12
	4
	12
߹	2
	2
ծȨ	4
	2
	12
	2
̫	2
̳	4
	4
	20
Ч	12
δ	2
»	22
	6
ó	8
׼	2
	4
ǧ	8
ȫ	6
ɻ	10
г	2
۽	2
ȡ	22
Σ	4
³	6
	8
	4
ĪϿ	12
λ	2
	2
ֿ	6
ֲ	2
	4
	2
	2
	2
	14
	2
͹	6
	12
Թ	24
߮	2
ֻ	6
ñ	2
Ԯ	2
Ī	2
	6
	2
ֱ	4
ַ	4
	8
	20
о	2
⽻	6
糿	40
	4
ǹ	6
Ȼ	4
ʹ	6
	6
	2
ºĬ	4
Ԣ	2
	4
	2
ӵ	8
	2
漴	8
	14
	8
	4
³	12
	10
	12
ڱ˹	2
	8
ͨ	4
˹	4
	2
	6
	2
	6
̸	8
ž	2
ⶨ	4
	4
Ļ	2
θ֮	2
	2
	4
	6
ãȻ	2
	2
׶	6
	8
оԱ	8
	2
Ų	2
	34
	2
	12
ȫ	2
	2
	2
	4
ʧҵ	6
	14
	2
ǰ;	14
ã	2
ܷ	2
	2
	10
	2
	2
Ǯ	2
ž	4
	4
ȫ	12
ҵ	4
ѧҵ	2
ס	2
	4
о	28
	6
ס	2
Ͷ	2
	16
	2
ֹ	12
˳	2
	2
ʹ	2
	2
̾	10
Գ	6
˫ְ	4
˫ҹ	2
	8
˵	2
ҵг	2
	12
	8
	6
С	2
	6
̸	14
	2
ȱ	10
	16
	2
	4
	4
	6
	2
ͳ	8
	6
ó	2
׹	12
	20
̳	4
λ	4
Ů	2
	2
	6
˾	2
	6
	6
	4
	2
	4
	2
Ϥ	10
ѧ	2
ʥ	2
	2
	46
	14
ȥ	6
˽	2
	14
	26
	6
Ȳ	2
	2
һ	8
	2
ȶ	4
	50
ԭ	4
	2
ô	84
	6
̽	6
	2
߽	18
	2
Ũ	6
Χ	16
Ⱦ	4
ö	2
	4
ۻ	2
	4
ʨ	4
Ŀһ	2
	4
	8
	2
ֻ	12
	2
Ů	10
	2
֫	4
	6
	2
	10
	2
	4
Ʊ	4
ת	2
	4
ת	8
ۼ	2
ˮ	4
	2
	10
	2
о	2
	16
	2
	2
	2
ȼ	2
ʮ	2
	8
	2
	2
ײ	2
Բ	2
	6
	12
	2
ȫ	8
ѻȸ	4
	12
ݼ	2
ַ	2
	6
ѹϷ	2
	30
ʵ	6
б	8
	8
	6
Ե	2
α	2
	2
Σ	2
Ϣ	2
	2
	8
	2
Ϣ	2
	6
ƮȻ	2
	6
	60
̾	2
Ŀ	2
ü	18
ϲ	268
	24
	4
֮	18
	10
	6
	32
֮	2
	14
˺	12
	2
ϰ	2
	10
	2
	12
	2
	2
	4
	2
	2
	24
	4
	6
ӡ֤	2
ף	12
	36
ʮ	2
	2
Ա	2
ʮ	2
	2
Ѫ	2
δ֪	6
	2
ʧѪ	4
ò	2
	12
	26
µƽ	8
֮	8
䷨	4
	4
³	4
	12
½	34
	2
ʡ	2
ι	2
	2
	8
Ժ	8
ר	2
	2
Ȩ	2
ױһ	2
ǳ	2
ѡ	6
	2
	8
¹Ұ	4
ַ	2
	6
	6
	6
	6
ά	6
	16
ݽ	2
ְ	2
ָ	6
	18
	30
ײ	10
	6
л˶	4
Զ	4
	14
	8
	16
	12
	10
¬	2
	2
	14
ǹʵ	2
Է	6
ٶ	6
ֵ	2
	2
Ҧ	4
ΰ	2
	2
	10
ʮ	6
ʮ	6
	4
	6
ʮ	2
	10
	4
	4
Ӧ	14
ݿ	6
Τ	2
	8
	2
ʮ	10
	2
˹	2
ʮһ	10
ʮ	4
	20
	2
ְ	2
Ĳ	2
н	6
Ͻ	4
	2
ٽ	4
	8
ܲı	4
	2
	4
ʢ	2
	2
	2
У	2
ھ	2
ϯ	6
ϣ	6
һ	12
	2
ʮ	2
˹	2
ս	10
Ԫ	4
ְ	18
ʹ˼ʹ	4
׸	2
Ӧ	2
ת	4
ɰ	8
һ	2
Ժ	2
	16
µ	30
ĭ	20
	14
͵	6
	14
	2
	2
	2
ڸ	30
פ	6
ʹ	2
	6
	4
	2
Դ	2
	4
ϸ	16
ر	14
	4
	4
ͣҵ	10
ֹ	2
ͬ	2
	2
	12
	2
	2
	2
	2
Ѻ	4
	12
÷	2
ҵ	8
	2
ֵ	4
м	36
	2
	4
	4
޶	6
	2
ʱ	8
ȡ	20
Ƴ	2
ҵ	2
λ	2
ȫ	6
	2
ʹ	12
½	2
	46
	4
	2
	2
	4
	2
	2
ü	2
	4
	2
͹	2
ѹ	6
м	4
	6
ͬһ	4
©	2
ä	2
	12
	2
	2
	4
	2
	2
ˮ	8
	4
ˮ	132
	14
	6
ð	4
ѷ	8
	2
	2
	2
	6
	8
	2
ܺ	8
⳥	34
	6
	30
ܺ	6
ά	6
ȷ	6
	2
	2
	6
	6
°	2
ͥ	10
þ	2
	2
⳥	4
	6
ʧ	4
	2
о	6
	6
ɽɽ	2
	4
ǰ	14
ȥ	18
Ů	10
˾	2
	40
֧Ʊ	12
	22
ͷ	2
	6
	4
Ⱥ	36
	2
®	2
ն	4
ۧ	2
˪	2
Ⱥ	2
ҰѼ	2
г	6
	6
Ⱥ	16
Ͷ	24
ι	22
	4
	6
	2
ȹ	4
	2
	2
Ǩ	4
	2
סլ	2
	26
Ұ	2
	16
	2
ס	2
ͥԺ	6
	2
	2
ɽè	2
Ұ	6
û	2
	2
	2
	12
Ϊ	4
ܿ	6
Ҫ	4
	2
ʳ	36
Ϣ	2
	2
ͼ	2
Ұ	2
ů	24
	18
	4
	8
ң	4
	26
	54
	2
ͼ	10
	4
ؽ	6
	2
	2
Ư	4
Ƚ	8
	2
ؽ	2
ǰ֮	2
	6
¬ɭ	14
	6
Ⱥ	8
ӹ˾	2
ɱ	2
	2
ڣ	2
	2
	6
ϴǮ	2
ְԱ	8
	4
Ӣ	20
	2
ӹ	10
ͣ	16
	16
	2
	2
	2
ע	6
	2
	2
	6
ʲծ	2
	2
֪	2
	4
Ϣ	2
ת	28
	8
	26
	6
	16
	2
	2
ʧ	2
	2
	2
	6
	2
	2
һ	10
	4
ɿ	2
	2
ӱ	12
	8
ð	2
	10
ݬ	2
ʧ	2
	8
ɳ֮	2
ע	2
	8
	2
ڸ	2
	4
ʦ	2
	10
ȱ	2
̵	6
	2
֮	2
ʾ	10
ϼ	4
	2
ҵ	8
	8
	8
	6
˴	4
·	2
ֹ	2
֧	10
	2
	22
	2
µ	2
	4
	12
߽	8
ѵ	4
Ѽ	4
ƹ	2
ë	2
ٶ	2
ض	2
	2
	8
	2
˻	2
	4
˷	16
Ƽ	2
ǩ	10
	36
	6
	4
	10
	30
	4
Ѧ	4
	4
ף	8
Ԥ	2
	6
	4
	14
	2
ͳ	2
ͳ	6
	12
ճ	2
	2
Ťת	8
	8
	4
վ	22
	6
	10
	4
亹	4
Ϻ	2
	8
ȡʤ	4
һ	14
	4
	8
ȷ	14
ʤ	24
Զ	2
	6
ض	2
׸ֶ	4
	2
ɽ	60
Ԫ	6
ϸֶ	4
ʮһ	8
	2
	2
ʤ	4
޳	4
	4
ϡ	4
	6
̨	16
	4
	10
	8
	2
ã	2
ʧ	12
ã	6
	10
С	4
誺	2
	2
ս	2
ƣ	8
	34
ѵ	2
س	4
	12
	16
	2
ã	2
ƹ	2
	4
㾢	2
	2
	2
	4
	4
	4
	12
׼	2
	6
߸˿	6
	2
˹	2
	16
ƥ	2
˿	4
	4
	2
	2
Ӿ	2
	10
̳	4
	12
	4
Ů	6
	4
	2
ӡ	2
°ʱ	2
	2
	2
Զ	2
	2
౭	4
ս	2
	6
	6
	12
ִ	2
	2
ս	6
	4
	24
	4
	2
ջ	22
	12
תΪ	8
и֮ս	2
	2
ս	4
	2
ʤ	2
ʮ	4
ʮ	4
	12
	2
	10
ʮ	4
	2
	4
ܼ	2
Ͻ	6
ȫ˻	6
˫	2
	18
	12
	8
	12
ã	4
Ұ	4
ն	12
	2
	4
ʤ	2
	4
	6
	2
˷ܵ	2
	2
ɱ	2
	6
쾡	4
ã	10
ȡ	4
Ĵ	2
	2
ǿ	2
	2
	4
ɾ	40
	2
ؾ	2
	2
	2
ϼ	2
	18
Ů	20
ã	2
ã	4
	2
	2
	2
ĺ	2
Ʒ	52
	20
Ӣ	28
ĸ	4
ö	36
ʺ	2
	2
ռ	2
	2
	2
ӭ	6
չ	4
	22
	4
	4
Ⱥ	2
ר	8
	2
۵	2
ѵ	12
	4
ʽ	4
ѷɫ	2
ʦ	8
	4
	2
	6
ܻ	2
	2
	2
Ϳ	2
ϴ˹	2
	2
֪	10
īسɹ	4
ǳ	2
֪	4
	2
ר	6
ҵʿ	2
	2
չλ	2
Ĳ	2
Ϊ	2
ȫʽ	2
չ	2
	4
	6
	10
	2
չ̨	2
	4
	2
ɧ	2
	2
ֲ	2
	2
	12
ǻ	2
	2
Ӥ	32
	2
	2
Ϊ	4
	4
	32
	6
ǿʢ	4
	4
	2
	2
	2
	2
	2
һϯ֮	2
־	2
ռͷ	2
	2
Ͷ	14
	6
	14
	2
ԴԴ	2
֪	2
	2
Ȳ	2
	2
Ű	2
	20
û	2
	2
	2
ʶ	4
	2
	2
ʵ	8
ֹ	2
	2
	2
ɱ	2
ϻ	34
	2
	24
Ϣ	2
	4
	2
	16
è	4
	6
ģ	20
ģ֣	2
Ӱ	4
ס	4
ͷ	4
	4
͸	2
θ	2
߳һ	2
л	6
ϸ΢	2
	6
ȫ	2
ֻ	16
	4
ģ	4
ɵ·	4
ãУ	2
	6
	2
	2
ǳ	2
	2
̸	10
	10
ڰ	8
¼	2
ݲ	2
ת	4
ջ	4
	4
	4
һ	6
	6
տ	2
ģʽ	2
Ƶ	2
Ƶ	2
	2
	2
	2
Դ	6
	4
	10
Ǳ	10
	32
˳	4
ã	2
ã	2
Ƶ	2
	10
ɨ	4
	2
ʽ	2
	2
ͷ	4
	18
ҵ	2
֣ã	2
	2
оƬ	2
Ӧ	4
	22
	4
	46
Ӥ׶	10
䷽	8
	10
	2
Ʒ	2
	24
	14
	2
ʯׯ	16
ҵ	8
¹	2
	16
̷	18
ʡ	12
Ʒ	8
ʵ鳧	2
	2
	4
ɭ	4
ζȫ	2
ζȫ	2
԰	2
ʳƷ	2
	4
	2
	4
Ʒ	2
	2
	2
	2
ȸ	2
ʾ	2
	8
Խ	2
	2
	2
	6
	4
ʽ	6
	2
	8
	4
ˣ	6
´	2
ѩӡ	4
Ȩ	10
	2
	8
	28
	2
	2
	22
	16
γ	10
	4
β	2
Ƕ	4
	4
	4
	6
Сγ	6
	10
	2
ŷ	2
ҡ	2
	4
	4
д	6
ͣ	12
	2
ص	2
	4
	6
Ѿ	4
	2
ж	2
װ	10
Դ	2
¿	12
г	2
	6
	2
ר	8
麣	2
ԭװ	2
͵	2
ɢ	4
	4
	2
䳧	2
αװ	2
Ϊ֪	2
͵	2
˰	16
ֵ˰	6
˰	2
	2
װ	8
Ħг	22
Υ	4
	8
	4
м	4
ڳ	2
	4
	12
	4
	2
	4
ֱ	4
	54
	2
	2
	2
ȫ	2
ҵ	2
ʶ	4
	6
ͷ	2
ͷ	2
רó	2
˳	2
ȳ	2
β	2
©	6
	2
͹	2
	4
	2
ʡ	8
һ	134
	4
	4
ˮȪ	2
ؽ	4
ֵ	2
	20
׷	12
	2
ݺ	4
	2
ѹ	8
	12
	4
	10
Ǧ	12
	12
	8
Ʒ	6
	4
վ	24
	4
ףɣ	4
ۺ	2
⸶	2
	4
ʵ	2
֤	4
۾	2
ҵ	8
ָ	44
	8
ѧ	6
	2
	4
	4
	2
	2
	4
	2
·	2
ˮ	8
г	6
ַ	2
	22
	8
	2
ͨ	2
ʵ	8
	2
ٳ	6
ȱ	4
ϵ	4
н	2
õ	10
ս	4
ɽ	4
	4
	6
	2
ǰ	2
λ	6
Ӭ	12
	8
	10
	2
֮	4
̩	4
Ӧ	2
	2
	4
̩	12
	16
	10
	10
	4
	2
	4
ǰ֮	2
֮	2
漣	6
	10
	4
õ	2
	10
ѹ	4
Σķ	2
¶ͷ	2
	4
Ͷ	2
֮	2
˳	14
Ż	2
	2
	4
߶	6
ж	2
Ͷ	2
ծ	2
	14
	4
ʶȴӽ	8
	2
ѻս	2
ʾͼ	2
	8
ÿ	2
ÿ	16
ͼ	6
	2
̼	16
̼	16
߼	8
ͼ	8
λ	2
	6
Ϳ	8
ƽ	2
ʮ	2
	10
Ӱ	4
Ӱ	4
ͼ	2
	2
ɷݹ	2
ͼ	4
	2
	2
	22
	2
ֽ	2
	2
ɣͣ	2
ס	6
	8
	6
ƽ	6
	12
̶	6
	6
	6
	2
	2
	2
ֵ	2
ȡ	4
֤	2
	2
˰	6
	6
Ｏ	4
	4
ʹ	18
ѧ	8
ƶȻ	2
˰	2
˰	12
	10
	14
ָ	2
Ӧ	4
ƽ	4
	2
֧	2
	2
	10
	4
	4
Ͷ	2
ҵ	6
Ԥף	6
	2
½	2
Ƿ	4
Χ	2
͹	4
	2
˰	2
	2
Ӫҵ˰	2
˰	4
ӡ˰	2
	2
	2
	4
	20
	2
ծ	2
	2
ȯ	2
	4
	2
	4
·	6
ӳ	10
ʽ	4
ҵ	2
	4
	2
	2
ܲ	2
Ŵʽ	8
¶	2
	6
ܹ	4
ʵ粿	14
	18
	4
֧	26
	2
	4
	4
	6
	6
	10
	6
֤	2
֤	2
֤	2
֤¹	2
֤	2
ɷ	10
֤	2
֤¹	2
ǵ	2
	2
	2
	6
ϴ	2
	4
	8
ͬ	2
	2
	2
	2
	70
	2
	4
ϴ»	4
	2
С	2
յ	8
ӵ	2
Ŀ	10
	2
ֲ	2
	10
	2
	2
ΰΰ	6
ھ	18
	12
п	2
	8
	2
ѯ	10
С	2
	4
	4
	4
ջ	10
Ȼ	2
	2
ѯ̨	2
	4
Ѷ	2
̨	4
Ӱ	2
ͷ	2
̳	2
׷	6
Ȥ	4
	2
ʷ׳	2
ջ	2
	10
̹	4
ר	10
	2
Ӫ	2
	2
	2
²	2
	4
һ	14
	2
	2
һ	4
δ	2
	2
һ	2
	2
	2
С	8
ڼ	4
	34
и	2
֪	4
	10
	80
ⲻ֪	6
㹻	26
볡ȯ	4
֪	4
	40
֪	22
ͨ	6
ԩ	4
	6
걨	6
Է	4
	4
ԩ	2
	2
	4
ô	24
	4
۾	30
δ	8
֪	26
	2
ɶ	2
	2
ƾ	2
ȡ	8
ܽ	2
	8
	2
ӽ	4
	8
ϰ	40
׬Ǯ	8
Ӯ	2
ܲ	14
	48
	6
	4
	2
ἰ	2
©	2
ǲ	4
	8
	10
־	4
Ц	8
ÿ	72
	2
	4
	20
ʱ	16
	10
	8
	2
ͻ	10
	2
ǻ	8
	4
	10
	4
Ա	4
һﾪ	2
	2
ը	16
	2
㱨	18
Ϻ	4
	2
	4
	2
ǰ	20
Ӧ	6
	8
	26
»	2
	2
ж	6
	6
	2
Ц	6
	2
	22
ʱ̴	2
	6
	4
ը	2
	2
	6
Ͳ	4
˯	20
	6
֤	10
	4
	2
ΣƷ	2
	2
	6
С	46
ǻ	2
С	2
	12
ģ	8
	2
	4
Ե	2
ʱװ	2
	4
	24
	10
ʱװ	12
ģ	6
פ	2
˼	2
	4
	2
ŵ	2
	2
	2
ֻ	2
¼	2
Ѱ	6
δ	14
	14
װ	4
	2
	2
	4
	4
	2
Դ	4
װ	2
	6
羰	10
ս	2
Żݳ	2
	2
н	2
̨	4
	2
Ͼ·	2
	6
	4
	2
	2
ʨ	2
̳	2
	2
	2
껭	2
	2
ɽ	6
̨	2
ʥ	14
׺	2
ʥ	2
	2
¥	2
	8
ͼӰ	2
ѧ	2
	2
Ⱦ	2
	6
	4
	2
¥	2
չ	2
	4
޿κ	2
ֻ	24
ڶ	2
	12
ť	4
	128
	2
	42
	24
̾	2
	2
չʾ	2
ϸ΢	2
ʶ	4
׹ί	2
Զ	2
ƽ	2
ͨ	2
	6
	2
һ仰	2
	2
	6
	4
	4
	6
	12
	2
	6
	6
	6
	2
	2
Ҷ	2
͸	2
	2
	6
д¥	2
	10
	4
걸	2
	2
	4
¥	2
	2
	2
	2
ÿ	2
ס	6
	12
	2
	4
Ԥ	4
̨	4
̨	2
ϴ̨	2
	2
	2
	6
΢¯	2
緹	2
̻	2
	6
	2
֧	2
ӵ绰	4
ӿ	2
	4
	4
ͯ	2
	8
	2
ɹ	2
	4
	4
ײ	6
	8
	2
ݴ洦	2
	6
ﶫ	2
ְ	2
СС	8
ٶ	2
ʡ	10
	8
	2
	6
	4
ƽ	4
Ҳ˵	6
Ŀ	12
	4
ͼ	2
	6
	2
	6
	2
	2
	4
	4
	2
	4
ë	2
	2
ƶ	2
	2
	2
Ծ	2
	8
	4
	4
	2
	10
	2
ڸ	6
ͳ	2
	2
ά	2
	6
	18
	6
ƶ	4
	4
	8
	6
	6
	2
	2
	2
ٲ	2
ڹ	2
	2
	40
һ	2
ʮ	2
	28
	2
	2
	2
	4
	4
	2
	2
Ϣ	2
վ	18
ͨ	2
	22
𳤳	6
	2
У	4
	4
һ	2
	18
	44
	2
	8
	2
ͣԣ	4
	2
	8
Զ	12
ƥ	4
ʱ	6
ʹ	20
ʽ	2
	2
	2
	2
ʵʱ	2
Ŀ	12
Ա	2
籨	6
ѧϰ	4
	12
	4
	2
Դ	2
	12
䱸	2
Ӳ	10
	2
	2
	4
	108
	2
	2
ף	2
	2
ӷ	2
ʱ	2
ַ	10
װ	18
	6
	2
	6
	2
ٵ	2
Ʒ	2
	2
	2
	4
	12
ת	2
֤	2
˵	24
	16
Ƭ	4
ͨ	8
	8
	8
ǣӣ	8
	2
	4
	16
л	8
	8
	2
ɿ	2
ӵ	8
	2
Ԯ	2
	2
	2
ֶַ	2
	6
Ƶ	2
΢С	8
ʱ	2
ֻ	2
ʽ	6
	2
	2
ȡ	2
	2
	4
	4
㼸	2
	4
û	2
	2
ǰ	2
	18
ģ	2
ʽ	2
ģ	2
ԭ	2
̿	2
	6
	2
Ѱ̨	20
й¶	10
˽	6
	10
Ѱ	14
	18
ڴ	2
	2
©	6
̽	2
	24
̽	2
ɫ	10
	10
Ļ	2
Բ	14
ƭ	12
	2
	2
Ķ	2
ð	4
һ	4
	2
	2
Ÿ	2
Ը	12
	2
綨	6
	2
Ѱ	12
ط	4
	2
	2
	2
ͨ	2
̨	2
ѡַ	2
̨	2
	4
	8
	2
	2
	2
η	2
շ	2
һζ	4
ɽ	2
	2
ԣ	2
ͣ	2
ר	16
	2
Ļ	6
	2
	2
	2
	2
	4
û	2
	2
Ӳ	2
Ƶ	2
ߵ	2
	2
ŵ	2
	2
	2
ٱͨ	2
¼	4
Ի	2
绰	2
	4
	2
	2
	2
	6
	2
	2
С	2
Ȩ	2
ɴ	4
Զ	4
	6
	4
ӡ	14
ī	2
ŷ	12
ǩ	12
ͼ	6
ױ	2
༭	16
	2
֮	2
	2
	20
	6
Ȩ	6
	4
·	6
Ǩ	2
Ȫ	2
´	30
	4
ס	2
	2
Ǽǿ	2
	4
	6
ȹ	4
ʧʱ	2
	4
Ľ	4
԰	4
	2
	2
ػ	2
˰	2
Դ	12
	2
ֵ	4
	2
ί	2
	12
ص	2
	2
Ƭ	2
	2
	2
	2
֮	2
	8
	32
	18
	4
ѧ	2
ʩ	12
ƽսʱ	2
	2
	6
շ	2
	6
ƽ	34
	2
ս	6
ܲ	2
޶	12
	24
	2
	2
	14
	6
ˮ	6
	2
Ѽ	10
	2
Ѽ	2
	2
	2
¹	2
ʵ	16
¹Ⱥ	4
¹	2
һ	6
	2
B	4
	2
߹	14
	14
	4
	6
	2
	4
	2
ˮƷ	2
	2
	2
	16
ռ	2
	6
˹ҵ	18
	12
	4
	2
ɽ	8
˵	8
ȼ	4
̨	10
	4
ҵϵ	2
ľ	2
˹	4
ף	2
Ť	2
̨	6
	2
	2
	2
	4
	4
ƹί	4
ȫ	2
	2
	6
ǿ	4
	4
	10
Ǭ	2
	6
	8
	10
ʱ	6
	4
аͳ	8
	6
ҽ	6
	6
Ժ	8
ž	6
	2
Ժ	6
¶	8
	2
	10
ѫ	2
	2
	4
	6
Ժ	6
	6
ɫ	2
	4
Χ	8
Σ¥	6
Ե	2
	4
	36
ŭ	2
¥	2
谵	2
ľ	2
ʽ	2
¥	4
	4
	2
	4
᳡	10
Ⱥ	8
	8
	2
λ	2
	2
ƶ	2
	2
¥	8
ھ	4
ͷ	2
	14
	20
	10
	54
Y	2
ϼ	2
	2
д	16
żȻ	14
㱨	4
ҳ	2
Ư	36
վ	2
Ȼ	12
	2
ͣ	2
̶̵	2
İ	2
	8
	2
Ķ	6
һ	2
	4
	2
	4
ر	2
߿Ƽ	2
	10
Դ	2
	14
Ű	2
ʮ	2
	2
г	2
	28
ʱ	24
ժ	2
	6
ѧ	4
	2
	2
ѧ	6
װ	4
	4
ҵ	6
	4
˾	2
	2
Ű	2
ȫ	2
	4
ѡ	6
	4
	6
¼	2
Ͽ	2
ʷ	4
	2
ȣ	2
ʱ	2
ʱ	10
	4
ͨ	8
	14
	2
	2
Ȼ	28
ʱЧ	4
Ӣİ	4
	2
	50
ο	2
	2
	2
	2
	4
	12
	12
	2
	10
ռ	4
	10
֤	12
	2
֤	2
	6
ִҵʸ	2
׼	2
	2
	2
ȼ	2
	2
ɱ	2
΢	2
߲	12
ά	6
ø	8
	2
߻	4
	2
	2
	2
	2
	2
߲	2
	2
	2
ʾ	2
	2
ڰ	14
ڶʮ	8
ϯ	2
ھʮ	2
	6
	34
	24
	2
ֵ	2
ڼ	8
׼	2
ʮ	4
ʵ	22
˶	6
	8
	12
	2
	2
	4
ظ	8
Ӽ	6
ʮ	10
ͨ	2
	2
	4
ż	2
	2
	2
	2
ɢ	4
Ǽ	6
̧	2
	20
ƭ	2
ͬ	6
̧	4
ѹ	4
	6
Ĳȡ	4
ʮ	4
Ʒ	8
ʮ	2
ϡȱ	2
ʮ	2
	8
ڶʮ	10
	2
	6
	2
	2
	22
ȡ	18
	6
˲	6
ڶʮ	4
	8
Ҫ	4
ڶʮ	2
ڶʮ	2
ʱ	4
ڶʮ	2
ڶʮ	2
ڶʮ	2
ʮ	8
	2
	2
޼	2
	2
	2
	6
ʮ	2
ʮ	2
ʮ	2
	6
˶	4
	10
ʧ	4
Ǽ	12
	6
ʮ	2
ʮ	2
ʮ	2
	4
ʮ	2
	26
	16
ʮ	2
	10
϶	20
ʮһ	4
ۿ	2
ʮ	2
ʮ	2
ʮ	2
	2
	4
ʮ	2
Ա	2
	2
ʮ	2
	2
˽	2
ְ	2
	2
ʮ	2
	4
շ	2
ʮ	2
άϵ	2
	2
Ӫ	4
Լ	18
	4
	2
	2
Э	2
Ũ	2
Σ	2
	4
	2
	2
	12
̶	8
	6
	8
	6
	4
	4
	2
	2
״	2
	8
û	2
	2
	4
ȺȺ	2
Ԥ	6
	2
	2
	2
Ĥ	2
С	20
	2
ϲ	2
	20
	2
	2
	2
	2
Ҫ	2
ص	2
ð	6
˷	2
永	4
ʹ	4
Ȼ	2
˻	6
	2
	2
	4
	2
	2
պ	6
	2
	4
ҵ	4
ú̿	8
۲	4
ʮ	2
̲	4
İ	2
b	2
	2
˰	2
ʮ·	2
ӡ	18
ɽ	4
ù	2
ֹ	4
Ա	2
ȫԱ	2
һ	2
	2
	2
	2
	2
	8
	2
ķ	2
	2
	2
	2
ú	4
	2
װ	2
ŵ	2
	2
ʢ	2
׳	6
	6
	4
	2
	4
쵼	18
ץ	8
Ƚ	2
	2
	2
	12
̰۸	4
ݳ˷	10
	2
Ҫ	2
̫	2
̫ԭ	4
	28
־ֳ	2
쵼	4
巿	2
Բ	2
ϯ	2
칫	2
ͨ	4
ְ	2
˾Ӷ	2
	2
	2
	2
	2
ʿ	4
	4
־	2
ķ	12
	2
Ӣ	2
ű´	2
̫	78
ǿ	2
	2
	6
	2
Ͷ	8
ŷ	10
ҵ	2
	4
˳	4
¥	30
Ȼ	2
	2
α	10
ͷ	4
	2
	2
	4
	8
	2
ҿ	2
	2
¾	14
	4
ҵ	4
ȼ	4
	8
ʱ	2
ŵ	4
ūʽ	2
ͳν׼	2
	10
	6
ֺ	2
	2
ӵ	6
ʳ	2
	2
	2
ϵ	2
	10
	2
	4
ƺ	2
	2
	2
	26
ִ	2
˳	2
ʮ	4
ҵ	8
	2
õ	2
	6
	14
ȡ	6
	6
	2
	4
¼ȡ	6
˫˫	6
Ϊѻ	4
	2
	2
ɽ	2
	6
	8
߿	8
	4
	80
	12
	6
	14
ɻ	22
	4
	4
鱾	6
ѧ	10
Ƹ	4
Ա	6
	6
Ա	2
ƴ	4
ϧ	6
̿	12
	2
	4
	2
	2
ʱ	2
	6
	2
	2
	4
Ѳ	2
	2
	2
ɳ	2
˾	2
	2
˾	2
˾Ա	2
	2
ִ	4
	2
ȭͷƷ	2
	6
	2
찲	2
ŵ	2
˱	2
ֵ	4
Ӵ	2
趨	2
	2
ʧ	2
ɽ	2
ֽ	2
ɽ	2
	2
	2
	2
	2
	2
	10
	4
ɽ	4
ƻ	10
ǹ	4
ɽ	4
	8
	10
	2
ͬ	2
	4
Ͷ·	4
	4
	4
	4
ǹ	6
ɻ	6
	2
ս	2
	4
	2
ǹ	2
	14
ǹ֧	2
Ǳ	2
ʯ	2
Ӫ	2
ͨ	2
	6
	2
Χ	2
ʱ	4
	4
Ȧ	2
غ	2
ǹ	2
	6
	4
	4
	2
ǹ	2
	2
ֻ	2
Ӵ	2
	4
	2
Ӣ	2
־	2
ϵ	4
	2
	2
Ͻ	2
ͣ	2
ɢ	2
н	2
ȥ	2
м	2
֯	2
յ	2
	18
ǳ	6
	8
ȱ	2
	2
	6
	2
	2
	4
ڿ˼	2
	2
	2
	8
Ŵ	8
ɩ	18
Ѽ̽	4
ֽ	8
	10
	4
	6
	2
	12
ڱ	2
	2
	2
	2
	2
Ա	2
ó	6
·	6
豸	2
	10
äͯ	4
̳	2
ѧУ	2
	10
ä	2
ֽ	32
	2
ä	2
	2
ĸ	2
˶	2
ɯ	2
ί	6
	4
	6
ʮ	2
ʮ	2
ʵ	2
	2
Ʒ	2
Ų	14
ľ	2
	2
һ	2
ɽ	6
Ƽ	4
	6
	10
	2
	4
Ϸ	2
	4
Ⱥ	4
˸	6
	2
ή	2
ٲά	2
	2
Ӣ»	2
Э	2
	2
Ų	4
	2
ĥ	2
ݲ˥	2
Ŀ	2
	8
籾	6
	4
ϳ	4
ؽ	4
ʵ	4
	2
	10
ʷ	2
	10
־	4
Ժ	4
	2
	2
	2
	4
	8
	4
	2
ӱ	6
	2
ܼ	2
г	2
	2
켫һʱ	2
ʧ	2
Ӵ	6
Ⱥ	2
ס	6
	50
Ժ	4
	4
ֻ	34
ղ	2
	6
	2
	4
ҵ	2
÷	2
໥֮	4
־	8
	62
	4
ѩ	4
ˮ	2
ƻ	4
ƺ	4
Ϸ	2
Ƽ	2
	6
	6
	2
	4
·	2
	8
϶	8
ȱ	22
	2
	2
	2
	2
ɽ	2
ѩ	14
	2
	2
	6
ʮ	2
	2
Χ	2
·	96
	2
	2
	4
	8
ѩ	4
	6
	6
԰	4
	2
	18
	6
˾	8
۰İ	4
	4
	4
	6
	8
	20
	2
᳤	2
ֵ	4
	2
	2
	2
ϻ	2
	2
Ӣ	2
С	2
	4
	4
	2
	6
	2
޲	18
ˮ	18
	4
³ľ	4
ȹ	2
֥	2
	2
ɼ	2
	6
	2
	2
ʶ	4
󻯶	4
Զ	4
Ȧ	2
ó	12
	6
ˮ	2
	2
	2
	2
	2
֤	4
׼ȷ	2
̽	10
ÿ֧	2
	4
ר	4
ˮ	12
ϡ	8
	26
Σ	10
	2
ʩ	2
	4
	2
	4
	2
	2
	6
	4
	6
ˮ	6
ϵ	20
	2
˥	4
ˮϵ	4
	2
Ⱥ	4
Ŀ	16
	4
	2
ǰ	12
β	6
	2
̬ѧ	2
ֳ	2
	8
״	4
	2
	2
Ӷ	2
	2
	2
¸	2
ó	6
	2
	4
	10
ǿϢ	2
ֱ	20
ʻ	4
ǰδ	4
Ļ	2
	4
ǰ	2
	4
Ե	4
	2
	4
	2
Ϊ	4
Сֵ	2
	2
ͯ	4
־	2
пǧ	2
ǧ	2
	2
	6
һ	2
	2
	2
	6
	6
ĥ	2
	6
	2
ؼ	4
	2
	2
	2
	12
	2
˳	44
	4
	4
ǿ	4
Ͼ	2
޷	2
˪	4
	2
תӡ	2
	4
£	2
ţԣ	2
	2
	2
ֵ	4
֮·	4
ҵ	2
	2
	8
װ	4
	2
	2
ץС	4
	4
	2
	2
	6
þ	2
۰	4
	4
	4
	2
	2
⾭óί	2
־	2
	4
˹	2
ˮ	6
ϵ	2
	2
ʵ	2
	2
	14
Щ	4
廯	2
	6
	4
	20
	2
һ	2
εض	2
	4
ݴ	2
ƽ̹	4
	10
	2
ļ	4
	4
ľ	8
ʶ	2
	2
	2
	6
	4
羰	2
´	8
	2
	2
	6
	4
	2
	2
Т	2
	10
׶԰	8
	48
	2
	2
Է	2
	2
	2
	2
ѧ	2
걾	2
һӦȫ	6
˭	2
Ƹѡ	2
	8
	2
Խ	2
	4
Ľ	10
е	2
	4
̶	2
ר	2
ϸ	4
	2
	4
	4
	2
	2
Ӧ	6
ῼ	2
У	2
	2
º	2
²	2
	8
	2
Ծ	2
ͯ	4
ϳ	6
ְҵѧ	2
	2
	2
	10
㽱	4
ʮ	2
	2
ͭ	6
	6
	6
	6
	4
	2
	8
ϼ	46
Ժ	6
	4
	8
ҹ	14
	2
ƿ	4
	2
	2
	4
ɫ	2
	14
¡ʽ	2
촰	2
	2
ʱ	6
ɫ	2
ִ	8
ɳ	4
	2
	2
	12
	54
	4
У	2
	2
	18
վ	10
ɽվ	8
	2
Խ	10
	8
	16
ϫ	2
	2
	6
	2
	2
	4
ϼ	2
ǵ	2
ϼ	2
˹	2
	2
	10
۲վ	2
ɿ	8
һ	4
۲	8
	2
	4
ľ	10
	14
ʱ	46
ֵ	4
	8
	2
	16
ס¥	2
ʱ	4
	4
	6
	2
ܼ	2
	8
չ	4
ƹ	4
ƫƫ	2
	2
	24
վ	8
ҽ	2
С	4
߲	6
ʧ	8
	4
	16
ּ	2
ȡˮ	2
	2
	18
	10
	8
	2
	2
	8
¶	2
	4
䰮	4
	10
	2
	10
ٽ	2
	2
	10
ɼ	2
	2
	16
	20
°	18
	16
	4
	20
ħ	4
	2
ٰ	2
Ա	2
	6
	4
̲	2
	8
	4
	10
	16
	2
վ	8
	4
ȥ	2
	2
Ը	8
ֽǮ	2
	2
	2
׷˼	2
	18
ʹ	2
Ĭ	8
Ƭ	4
	2
峱	2
˼	4
	6
	20
	2
ٱ	2
޷	8
¶	8
֮	2
	2
޷	4
ף	4
	4
ȫ	2
˾	2
Ƭ	2
ƽ˹	2
	2
	2
	8
	2
	2
	2
	4
廪ѧ	4
	2
Ӽ	10
ѧ	4
ʡ	2
	4
ǿ	2
չ	2
	4
	18
Ҽ	2
ּ	10
ֵ	26
ġ	2
	2
⽻	4
	2
κ	2
	2
ϰ	22
Բ	2
پ	2
	2
	2
	2
	2
ζ	2
	2
	2
	2
Ͽ	12
	20
ǩ	14
Ϊ	6
	2
˾	4
	6
	4
ܱ༭	6
ʳ	2
ѧ	4
Զ	2
	2
	4
ҵ	8
	2
ͻ	2
ٻ	2
ѿ	6
	8
	10
	2
Ӫҵ	2
	10
ݽ	2
ϸ	4
	4
	6
ȱһ	4
Ǣ̸	2
	4
н	4
Ϸ	2
ʿ	2
	8
ܷͼǿ	2
	6
	4
	4
ڿһ	4
ͼ	2
	2
	2
	2
ܷ	8
	2
ư	2
	2
λ	4
	2
	8
޹	2
ͷͷ	2
ĺ	2
й	2
ɤ	2
һ	36
	4
	2
	10
һ	2
	2
	16
滻	4
	2
Ԯ	2
	2
	6
ְ	6
㶫	6
	4
	2
֮	4
	2
⵹	2
ղķ˹	2
	2
ƴ	2
	2
	4
־	2
ɽ	4
ԶͶ	2
ŵ	2
	4
	6
׽	2
ͷ	4
	2
	8
Ѱζ	2
Ա	2
ʲô	4
ϳ	6
ǰ	6
͵	4
ս	4
ɽ	4
	2
	8
ȫ	2
ǡǡ	4
ʾ	2
	2
	4
¹	4
¾	2
	2
ϯ	2
Ĳ	4
	2
	2
	2
ͷ	8
	8
	2
	2
з	2
ս	8
	10
	2
	2
	4
	2
ƴ	6
ס	4
	16
ȭͷ	6
	4
	6
ǰ	2
	2
	2
С	2
	2
	8
ͷ	2
	2
	2
ϴ	2
ҫ	2
	2
λ	2
ս	2
Ԫ	4
	4
	4
ս	2
	2
	2
	2
³	14
̩ɽ	14
	8
	4
ҵ	10
ҵ	2
	2
	12
⼮	2
	2
ׯ	2
	2
	2
	2
	2
	2
Ƶ	6
	2
	2
С	2
Ԩ	2
	2
	2
	2
	2
٩Ө	2
	2
	2
Ż	2
	2
	2
	2
Ů	2
ָ	2
	4
ٻ	6
׹˾	4
	2
ʰ	4
ȴ	2
	2
ʱҴ	2
Э	4
	4
	2
ۼ	2
	2
	4
	2
	8
	42
	2
Ӣ	2
ļ	4
	2
	2
Թ	4
г׼	2
	2
֮	4
	6
	2
	2
¹	6
	2
	2
Ӿ	8
Ħ	8
ܴ	2
ծ	2
ֻ	2
	6
˹	2
	4
	92
	2
Ϊע	2
Ի	2
Ъ	2
	12
ͷ	2
	4
	2
ɴ˿	2
Ī	14
	2
	4
	4
ѡ	20
	8
ޱ	6
	2
ѹ	2
ᰲʡ	2
	2
½	2
貨	2
ͨѶ	4
ͨ	2
	18
Ȩ	4
ľͼ	6
ʲ	2
Ү	8
˹	2
	2
	2
ԨԴ	2
շ	2
ڹ	2
	6
ʲ	4
峥	2
	2
־ƽ	4
	2
	2
ǰé	2
	2
	4
	2
	2
	2
	2
ο	4
	8
Ȼ	2
	2
	2
̨	2
	12
	2
Ӱ	6
ЭԱ	2
˹	6
̳	4
	2
	4
	18
	2
	2
	2
Ԥ㰸	8
·	2
Ƴ	2
ŹŲ	4
	6
׷	2
	2
	8
ƶ	2
ŵ	6
ϯ	8
	2
	4
	2
̯	2
	14
	2
	2
	2
	4
	4
ע	2
ѺƷ	2
۲	2
	2
ͻ	2
辶	2
	2
	16
Ȧ	2
	2
ڶ	2
	6
	2
	2
ȥ	6
Ϊ	6
	8
ͷ	2
	22
	2
	2
¡	2
Լ	4
	4
ֲ	12
ͽ	6
	2
	2
	16
	2
Ұ	2
ʬ	4
	16
	6
̬ѧ	2
΢	4
	4
˫	4
	8
	4
	4
ɽ	2
ɽ	2
˹	4
	2
㽶	14
޼	2
	4
ḻ	4
	2
в	4
ʥ	8
ɽ	2
	8
	6
װ	2
ǳɫ	2
	2
㽶	2
԰	2
ī	2
ݺ	2
԰	18
	2
	4
ɳ̲	6
	2
	8
һ	6
ձȵ	2
	2
	2
	6
	8
	6
½	6
֮	2
	4
	2
	6
Ϊ	2
	4
	6
	2
Ǯ	2
	6
̻	4
	6
	2
	2
	2
غ	4
	10
ѵ	2
ù	14
	4
	6
	2
	6
	10
˾	10
	2
	2
˾	4
˹	4
	6
ɱ	12
޹	6
	6
	2
	4
	2
	2
	2
	2
ɱ	12
	2
	2
άŵ	6
	10
¡	2
ʹ	4
һžŶ	2
	10
͢	6
Ź	2
޿	2
ƽ̸	2
	2
	4
	2
ɴ	10
Ų	4
	2
ر	2
ɿ	2
	2
ƿ	2
	4
	2
ɱ	8
	2
ı	6
ʮ	2
ı	2
	4
	2
۲Ա	2
	2
Ӫ	6
	2
û	8
¸绪	2
	14
	2
	4
	8
	4
С	2
	4
	10
	2
	6
	2
	2
	4
	2
ǰ	4
۱ͤ	2
к	98
	4
	2
ۻԱ	4
þ	2
	2
߿	6
СС	6
	2
洦	4
޲	2
	2
¬	10
ۼ	6
ѡ	4
ϧ	6
	6
ֵ	2
ʮФ	4
	2
	2
	2
Ͼ	2
Ƶ	4
	6
л	4
	2
ֱ	2
	2
׷Ī	2
ϯ	4
թ	2
ǧ	2
	2
	2
ҡ	8
޼	4
	4
	2
	4
ʥ˵ñ	2
	4
	4
	2
ݽ	2
	2
ÿ	2
	2
	2
¥	2
	2
	2
ŵ	4
˰	2
©˰	2
Ž	2
	4
ֲ	2
	2
	2
ǰһ	2
ʽ	4
տ	4
Ϊ	4
ʱ	6
ನ	2
ʽ	2
	2
	4
ι	2
һ	2
ѭ	2
	4
ɽ	4
ѭ	2
Ҹ	4
ֺ	2
	2
	8
Ѫ޹	2
Ʒ	2
Ʋ	2
	2
	4
	2
	2
Ӱ	4
	2
	6
	2
Ϊʱ	2
ʲ	14
Ͷʼ	2
ֻ	2
Լ	6
ø	2
Ѫ	6
ĸŮ	2
	2
	48
	2
	2
	10
ȼ	2
	14
ɢ	2
	2
	2
ȼ	2
	2
	14
̻	2
ҳ	4
ƽ	2
	4
ͷ	2
	2
	2
	2
	8
Ԫ	4
	2
ʼ	2
Ƶҵ	2
ʳҵ	2
Ṥҵ	18
չ	6
	10
	6
ռ	4
	4
	2
	4
	4
	4
	4
÷	2
¾	2
	2
ؿ̩	2
	2
Ȥ	2
	2
ȼ	12
	4
ƭ	2
	6
ǩ	6
ع	2
˹	2
	2
	8
ͷ	4
	10
	2
	16
Ū	46
	8
	6
ɫͷ	2
	2
߽ű	2
չ	4
ˮ	2
罦	2
ˮ	4
̬	2
ѧ	8
	6
ÿ	18
ݻ	2
	6
Ѥ	2
	2
ְ	2
໥	4
	4
	6
ת	6
	4
	4
ͷ	2
	2
ʤ	2
	2
Ƥ	4
	4
ù	2
	4
Ŵ	2
Ȼ	10
	4
	4
ʼ	2
	2
˹	2
ѷ	2
У	2
	16
	4
	2
	6
ʶ	2
ճ	2
ɫ	6
	30
	4
	6
¶	26
	4
	2
Ʒ	2
	2
	2
	2
һ̼	2
	2
	12
ů	6
	2
̽ռ	2
Ͽ	10
ȡů	2
	2
	12
̽ն	2
	2
ȼ	4
װ	10
ʧ	4
Ŵ	4
տ	2
	4
	2
Ӳ	2
սս	2
	2
	2
½	4
ʻ	20
	2
	10
	8
	8
Ʈ	12
ذ	2
	2
ʳ	10
	2
	4
¼	4
Ҷ	2
Ҷ	2
к	4
	2
Ҷ	2
˿	4
̰	2
׻	2
ҽԺ	2
è	2
	2
	2
	2
	2
	2
	4
	2
ȸ	2
	12
е	42
ֱ	2
ڼ	8
ս	2
Ĵ	4
Ѳ	4
Խ	2
ë	12
	4
	4
	4
	8
鲻Խ	2
뵽	34
ϰ	2
Ժ	6
	2
	4
ʳ	2
к	2
һ	6
	4
ס	14
	6
ȫ	2
ҳ	16
	2
ѧ	2
̲ݻ	2
	2
Զ׼	4
	2
	2
	2
Ҫ	8
׶	2
ɲ	4
	2
ʦ	2
ר	6
޿	2
ʱ	2
ʡ	2
ο	2
Щ΢	2
	40
	2
	12
ע	2
	2
	2
ֱ	2
ǰ	2
˿	4
	6
	2
	10
	2
س	2
߰˵	2
Ϣ	44
	2
	20
˵	20
е	6
	2
ׯ	2
Ӫ	2
Ъ	2
п	2
	4
	6
	2
Ĺĵ	2
	10
ʱ	2
鱨	2
	6
	2
ұ	2
	12
Ҵ	4
	2
	22
֪	2
	8
	2
¶	2
	2
ˮ	8
·	4
޺	2
·	2
¥	2
Ŀ	2
	2
	4
һ	42
ֵ	2
ҵ	2
	2
	2
	2
	2
	4
ɨ	4
ɨ	22
	2
ͷ	4
	6
	2
	10
ڼ	16
һ	8
գ	2
	4
ݳ	4
պ	8
	2
	2
	4
ʱ	2
ֽ	4
	2
	2
֤	2
	8
	8
ŷ	2
	14
η	8
	6
	4
;	2
˴	10
	4
ʦ	2
̬	2
	2
	2
ĳ	2
ĳ	2
ȫ	4
ҵ	2
	6
˵	18
ѭ򽥽	4
ǧ	2
ʽ	4
ʳ	2
ͬ	2
ʵ	2
	4
ԸΥ	4
	6
	4
˳	4
ί	4
	4
	6
	6
ѽ	44
	12
к	2
ˮ	2
	6
	2
	2
˹	4
	8
쳵	2
ٷ֮һ	2
ٷ֮	4
˿	2
ҹ	14
н	4
	8
Ծ	2
	2
	4
ҹ	8
	8
פ	2
	2
	6
	2
	10
	4
ʱ	14
	2
	2
	22
	10
	28
ʱ	4
ӵ	4
	2
	10
Ԣ	2
	2
	4
	16
Ϊһ	4
	2
	4
	8
С	2
ȻԵ	2
	6
	2
β	6
	2
Ѽ	2
	2
	16
Ǻ«	2
ʱ	6
ֻϦ	6
	2
ի	4
ѩ	4
	2
а	22
ʱ	2
	4
	18
	2
	2
˼һ	2
	2
	4
	2
ֳ	2
а	2
	2
ּ	2
	10
	6
Ʊ	2
Ͻ	2
	2
	2
ֹ	2
	6
	2
ԭ	10
ͣ	2
˼һ	2
ռɽ	2
ֵǮ	4
ȫ	14
	2
ʷׯ	2
ӥ	14
	4
	2
	2
׵	2
ҵ	12
޽	2
	2
	6
	2
ҽ	4
˾ְ	2
	2
ԭ	4
»	2
Խ	2
	2
ͩ	2
̸Ļ	2
Ա	2
ԭ״	2
	2
Ч	2
Ժ	2
Ժò	2
Ȩ	4
	4
	32
	6
	2
	2
	2
޻	2
ŷ	6
ʽ	4
	100
	4
	4
	4
˲	4
	4
	2
Ů	4
֢	4
	2
	4
ǣ	2
	4
ٸǿ	2
ؿ	2
ӯ	2
	2
	4
ƣ	6
Ц	8
Է	4
ӿ	6
̽	4
л	4
	2
	2
	2
زʶ	6
	2
	34
ֻ	22
	8
ʵ	4
Թ޻	4
⻷	4
	2
˵	4
·	14
ϴ	2
֯	2
ë	6
廨	2
	4
	2
ͷ	2
ͷ	2
ȫ	2
ʹ	2
	2
	2
	4
ಡ	6
β	6
ծ	8
	2
	2
֫	2
ĬĬ	4
ֺǺ	2
Ӧ	2
	4
ϸ	4
ҧҧ	2
	12
һ	2
	4
	4
Ũ	2
	2
	2
	4
Ľ	2
㴫	2
Ե	2
	2
֪	2
	4
	8
ζ	2
α	4
ͼ	2
֧	2
	2
ۿ	4
ʪ	4
ο	2
̨	2
	2
	2
	4
Ӵ	2
	4
§	4
ұ	2
	2
	4
	2
ƿ	2
	54
	2
̽	2
ʵ	2
	4
	2
ÿ	2
ˮ	6
	2
	6
˼	6
֮	2
	2
	6
ͣЪ	2
	2
	2
칫	2
	2
	2
ƽ	4
¾	4
	2
	2
̳	2
仧	4
ת	2
ݽ	2
赲	8
Ȼ	2
ͻͽ	2
֮	2
ȡ	2
¸	20
	2
	6
ǧ	10
Ψ	4
	8
д	2
Ҷ	6
	2
	8
״	2
մմϲ	2
ͣͲǰ	2
Ψ	4
	2
ǡǡ෴	2
	2
	2
ѧʷ	6
żҽ	4
	2
֧	6
	4
ѧ	2
ͼ	2
	24
	4
λ	2
ۺ	4
	2
	2
	4
	2
ų	6
Գ	2
ع	4
	2
	14
	2
	2
	8
̸	6
ǳֹ	2
ѧ	10
	2
ϴ´	2
֪һ	2
Ϊ	2
¿๦	4
ѧ	2
岨	2
	2
۵	2
	2
ϵͳ	2
л	14
	2
	2
з	2
ӱ	4
	2
Ȩ֮	2
Ȩ	2
	2
	2
Ѫ	8
	28
	16
	2
	2
	6
ˮ	2
ֹ	4
ʮ	26
һĻĻ	4
ѪӰ	2
Ӣ	6
׳	2
	4
	2
ǰ	12
¬	2
	4
ǹ	2
	24
Ž	2
ѵͷ	2
Ӧ	4
ի	2
ƽԭ	4
λս	2
ս	2
	2
	6
	6
	12
޲	2
	2
	2
תս	2
	4
ԥ	2
	12
־	2
̫	2
¸	4
ս	6
ǰͺ	2
׳	2
	12
ż	2
	10
ǧ	4
ʿ	12
Ӣ	4
ӳ	4
ӳ	4
ʷ	2
	2
ϡ֮	2
	2
	2
ϵ׳	2
	4
	12
һƪƪ	2
	2
¼	2
	2
Ҵ	6
Ϧദ	2
	2
δ	2
ί	18
Ѱ	24
ҽ	2
ͷ	2
	2
	2
ͷ	4
	6
·һ	2
	2
	4
	6
ʱ	6
Ӱ	2
Ļ	4
²ż汸	4
	4
	10
	10
ʮ	18
½	2
λ	2
	4
	2
۳	8
	2
	2
ȫ	8
ศ	2
	2
мί	2
걾	2
׶	2
	2
᰸	2
	6
ش	2
	2
	4
ʷ	6
	6
Ʊ	4
ͼ	4
	2
ĳ	2
	2
Ҫ	14
	4
	2
	2
	2
ȫ	6
ԣ	4
	2
Զ׿ʶ	2
	2
	4
	2
	2
ï	2
ģ	20
	2
	2
	4
	2
	2
	22
	4
̵	6
ϸ	2
ѧũ	6
	4
	4
ǧͷ	2
һ	2
·	4
	2
ʾ	4
͸	4
ͼԱ	4
	4
	2
	18
ͣ	12
	2
	14
·	2
ֳ	14
	4
	2
ͣ	32
ɵ	2
ɽ·	2
	2
	10
	2
	4
ǻ	2
	6
ײ	2
	2
	2
	4
	4
޳	2
б	2
ɳ	4
	2
ĳ	4
ӭ	6
ʻ	2
ˮ	2
	10
	12
;	2
ʻԱ	8
	2
·	2
ʡ	2
	4
Ӱ	4
	90
	2
	8
Ϣ	2
	4
ũ	6
	2
	2
	14
಻	2
ʹ	4
	4
ðα	18
	2
	4
	2
	4
ȷ	6
	14
	2
	2
	2
	2
ڱ	2
	6
⳥	4
ǣ	2
ģ	2
ѡ	2
	4
ǣ	6
ɰǧ	4
	4
Ǹ	8
	4
	4
ӹ˾	4
˺	4
	2
Ǳ	2
	8
ֺ	6
	6
·	2
	2
	8
	2
	2
	2
	2
	8
	2
	2
	2
Ϸ	4
	2
	2
	4
֮	4
	2
	2
	4
	2
	12
ĳ	2
֤	8
	2
ߺ	2
ö֮	2
	2
ѿ	2
	4
	6
ס	2
Դͷ	4
ҷ	4
׫	2
ԭϯ	8
	2
	2
Ͳ	10
ܴ	10
״	2
	4
Ȩ	2
ͷ	6
³	2
	4
ԭ	8
Ȼ	2
Ȩ	2
ָ	6
ŵ	2
ŵ	2
	2
ĸ	2
Ǹ	6
ǽ	24
	2
	22
̳	2
	2
	6
	4
	2
ǽ	2
Ͷ	2
	2
ٳ	6
ͬѰ	2
	2
Ȼ	2
̨	4
˵	2
̷	2
̨	2
	2
У	6
仯	2
	2
Żﻯ	2
߿	2
	2
	32
ص	8
	2
ﷸ	6
ʹ	2
	2
	4
	2
껪	2
	2
	6
ĳ	2
	2
Ī	2
ɳ	2
	6
վ	14
Ʊ	2
	2
ض	6
ʾ	4
Ʊ	2
թ	4
ȫ	2
	2
	2
ʤ	10
	2
	16
̺	4
	12
⳥	4
	2
	22
	4
	4
	2
	6
	2
	4
	2
	2
ͥ	2
	2
԰	2
ͥ	2
ͥ	2
	2
	2
	2
ȡ֤	2
ʤ	2
о	2
ʵ	8
	8
Ѻ	2
	2
ԩ	2
	8
	18
鲨	2
	4
	2
޷	2
ξ	8
	12
ĳ	2
	28
Ҵ	4
̨֤	2
	18
Ŀ	4
	28
˿	4
ۼ	8
ٿȻ	2
׽	2
	2
̾	2
	2
˳	2
ܶ	8
Ա	6
	20
ʵ	2
һ	18
͵	10
	6
͵	6
Թ	2
۽	2
	2
װ	2
	2
	8
ǻ	2
Ȼ	6
æ	2
	12
ϳ	18
ðƻ	2
ҡͷ	4
δ	28
β	4
¶	4
	2
	2
ǩ֤	16
	4
ĵ	2
ʼ	2
	2
ʼǱ	4
п	2
һ	4
ǧ	2
ĥ	2
ͷ	6
ľս	2
	14
	2
	2
	2
	8
	2
	12
	2
ǿ	2
׷ʽ	2
	2
־	8
޶Ŀ	2
ٻ̵	6
Ƶ	6
ϻ	6
ɽ	2
һ	2
Ӿ	2
	2
	2
ƺ	2
˵	2
	2
л	2
һƽ	2
繤	26
Ƭ	4
	2
	18
ƶ	2
Ѳ	2
Ȩ	2
	2
ͤ	2
ִڵ	2
	2
	2
	2
ս	10
	8
	2
	4
ٷ	4
	2
ĳ	6
ר	2
	2
̾	4
	2
ѧ	2
η	2
Ӧ	2
˼	4
ӳɲ	2
	4
Ļ	2
ס	2
	10
μ	2
ȱ	18
Ӷ	4
Ϣ	4
	2
	2
	8
	6
	2
	8
˲	4
	4
Ī	2
Ͷ	4
	4
·	2
ѧ	4
	2
	4
	12
а	2
	2
СС	2
	2
	2
	2
ʮ	2
ʮ	2
Σ	4
һҶĿ	2
	2
Υ	2
	2
	6
	2
	2
	12
	14
е	2
	2
־ε	2
˾	8
ѧ	2
༭	4
	2
̷ѧ	2
	2
̷	10
̿	4
	6
	4
ѩ÷	2
Ʊ	4
Ժ	6
	2
÷	2
	2
ų	2
	8
ݶ	22
ï	2
ͨѶԱ	2
	2
	2
	2
	2
	38
	12
ڣ	2
ؼ	10
	4
ߣ	4
Ӣ	2
ݻ	2
	4
ߣ	2
	2
	2
	4
	2
	2
	2
	2
	4
	2
ʷ	2
Ļ	2
ͼ	2
Ѫ	4
ж	2
ս	2
ӳ	4
	8
ʢ	2
»	2
ʵװ	2
	2
ȻȻ	2
Ա	10
	10
	4
	2
Ϳ	2
	4
	2
Ŀ	4
	2
	2
Բ	4
	2
Ӳ	8
Ƥ	6
к	4
	2
̺	4
	4
	2
	6
Ϳ	2
	2
Ⱥ	2
ܼ	4
	4
Խ	4
	2
	2
	4
	2
ק	2
	2
β	6
	6
Ʒ	4
	18
ʷ	2
	2
	2
Է	2
֮	2
	6
Ż	2
	6
һ	2
	2
	4
	4
	2
ͷ	4
³	2
	2
	4
ͻ	2
	2
£	2
Ӵ	2
ɫ	2
	2
	8
ǰ	2
β	10
ɫ	2
֯	2
	8
Ѳ	2
ֿ	2
ޱ	2
	2
ÿ	2
˼	4
Ӫ	14
	2
ߦ	4
	14
	2
	4
ʮ	4
	2
ë	2
	32
	6
ָԱ	2
͸	2
ͷ	2
°	2
	2
	2
һ	6
	4
	2
ͷ	4
	4
	10
	2
ʽ	2
	4
	18
	2
ߦ	6
Ƥ	36
Эͬ	2
˵	2
	16
	2
	2
	4
	22
ϱ	10
ְ	2
	4
	10
	6
	2
׳	2
ﻰ	4
	2
ӡ	2
	2
	8
	2
黢	20
	2
	2
ͷ	6
ƽͷ	2
ŵ	2
	2
ͷ	10
	2
ŨŨ	4
ζ	2
޳	6
վ	8
	20
쵱	2
Ӳ	4
	4
޳	4
	2
	8
ֹ˾	6
	2
	2
	4
Ը	4
	2
	154
	2
޺	4
	8
	2
뱸	4
	12
Ѫ	2
ɹ	20
	16
	2
԰	2
һ	2
	2
ල̨	2
	2
ӪҵԱ	2
װ	2
	2
	2
ũҵ	6
һ	2
	2
ӵܱ	6
	4
	2
¾	2
	2
	12
	2
Ӿ	2
ɲ	4
ң	2
	2
	2
	2
ר	2
󱸾	2
	2
Ȩ	24
Ծ	4
	8
	2
	8
	2
	8
ٷ	4
̾	4
Ƹ	2
һȹ	4
	4
	2
ϸ	2
	2
ɻƺ	4
	4
ס¥	2
廧	4
Ӣ	4
Ǯ	2
	2
ͷ	2
²	2
ؿ	2
ͱ	12
˲	2
ũ	2
	4
	2
ֽ	14
ϼ	6
	2
	4
	6
	14
ǿ	2
	12
·	4
˵	2
ڴ	2
ס	2
ȥ·	2
	2
վס	2
	20
	2
	4
⵶	4
	2
	4
ס	2
	14
˿	20
ʹ	2
ƽ	2
Ţ	2
׷	4
	2
	2
ȫ	8
ϥ	2
	2
	4
	4
ɳ	2
Ť	2
ܻ	2
	2
	2
ⲿ	2
	8
ҡ	4
	8
	2
߰	2
Ѫ	2
Ѫ	2
׳	6
	2
ɲǼ	2
	2
	2
Ѳ	2
	2
ϵ	32
	2
κ	2
ס	6
	2
	2
˺	2
	4
	2
	2
	4
	2
	2
ȹ	2
ɺ̲	4
	2
ȱ	2
˵	4
	2
	16
Ӧ	2
	2
ͷ	4
	8
	6
촽	4
	4
	8
ά	2
д	34
	2
	6
γ	2
ʮ	2
̸	2
ɺ	2
	6
ˮ	2
	6
ⲻ	4
ι	2
ʵɼ	4
Ĳ	4
	6
Сײ	4
Ҷ	2
	10
з	2
ţ	6
֦	16
ʯͷ	2
ǽ	2
ֵ	6
ƶ	4
	2
	2
̨	2
Ӫ	2
װ	4
	4
	2
	2
ϻţ	2
	6
ٳ	4
	8
ͦ	16
ɴ	14
	2
	4
᲻	4
æ	2
	4
	4
Ԥ	2
	2
ͷ	4
ֵ	6
̺	4
	4
Ƥ	4
Ц	2
æ	10
	2
	4
	2
	4
	2
װ	2
ɵ	6
	10
	2
	2
Ѯ	2
Ӣ	10
ϲ	2
	2
	4
ϡ	4
ϲԵ	2
ͬһ	4
ǣߴ	2
ȷ	4
	2
ҹ	6
	2
	2
ɫ	4
	12
	6
	2
	6
	2
	8
һ	2
Ͷ	2
Ȣ	4
	2
	12
Ч	6
	2
ٱ	2
¡	6
	2
	2
濢	4
ǧ	2
߷	2
	2
ɽ	6
׸	4
仧	2
	2
	2
շ	4
ȷ	2
й	2
	2
Զ	2
	4
䲿	2
	4
ν	2
	2
ʧܺ	2
	2
ʤ	2
	2
	4
ȳ	2
	2
	2
	18
	2
Ȩ	2
	2
	2
	2
ƳƷ	4
	2
ͬ	2
ص	2
֯Ʒ	8
	4
ó	2
	6
	4
ƣ	4
	2
ʧ	6
	2
	4
ת	4
ù	4
	4
ٽ	2
ȫ	2
	2
밶	4
	2
ڰ	2
	2
	6
	2
	2
	6
Ǽ۸	4
	2
	2
ȡ	4
	2
	6
	4
	10
	4
	30
סԺ	8
	4
	4
ͱ	8
	10
˱ɽ	2
Ĺ	4
	2
	2
	2
	2
	4
	18
һ	2
	2
һ	16
	2
Тͨ	2
	2
	2
	2
	2
	2
ѡƽ	2
ѧǫ	2
	2
׹	4
ѧ	2
	2
ӳ	2
ΰ	2
Ӣ	2
	2
ѵ	2
	2
	4
ӽ	2
	2
	2
	2
	2
	2
	2
	2
	2
ϰ	2
ѫ	2
	4
	2
ï	2
	2
	2
	2
֮	2
	4
	4
	4
	4
ص	14
Ψ	4
	10
	2
Э	2
Ժ	2
	2
ũ	2
	2
ί	8
һ	2
ʮ	2
¼	2
ʮ	4
һʮ	2
	2
ʮ	4
ʮ	2
ǧ˰	2
Űһʮ	2
	2
ǧ	2
һ	2
	4
	2
֮Ժ	4
ʮ	2
	2
Ʒ	2
	2
	2
	2
Ŷ	20
հ	4
	2
һ	6
	4
	4
	10
ͷ	2
	2
	8
	2
Ϲ	2
һ	4
	6
հ	2
ˮ	2
	2
	8
	2
	4
	2
ע	6
	2
ڴ	2
̲ס	4
	2
հ	8
¹	2
ʢ	6
Ͱ	4
	2
	2
ԸԳ	2
	2
꼶	4
Ů	54
	8
	2
	2
Ӱչ	2
˲	2
Цò	2
ǧ	2
	2
ʷ	2
	2
	2
ý	2
	2
Ф	2
Ԫ	2
ѡ	6
	6
֪ʿ	4
	6
й	6
	2
	2
	2
	2
	12
ɽ	14
	2
	2
	2
ɽ	2
ĬȻ	2
	2
	2
	2
	2
	4
¼	2
	4
Ԫ˧	4
	8
Ϊ֪	2
	2
	2
	2
	2
	2
	4
һ	2
Ц	8
ְԱ	2
	2
ȫ	2
	2
ռ	2
װ	4
	2
ά	2
Ż	2
û	2
	4
	6
	12
ά	2
	4
	2
	2
	2
ҵƷ	4
ɴ	2
ж	2
	4
Ա	2
ᱣ	4
ת	6
ƽң	2
һ	2
֯	2
	4
	2
Ȫ	2
֯	2
	2
	2
֤	2
ſ	4
	2
ܴ	2
ʯ	4
籣	2
ת	4
	6
֯	8
ĸ	4
	2
¾	10
ɽ	2
	6
	8
	2
	2
̫	2
ƽң	2
ܴ	8
³	4
ĳ	2
	2
ʯ	2
ʯ	2
֧	2
	2
	4
	2
	2
	2
	6
ʽ	2
	2
ӡˢ	2
컨	4
	2
	2
	2
	4
̾	2
ůů	2
	2
԰	2
ˮ	6
	2
	2
	2
	4
ֻ	2
Ҽ	18
Ľ̾	2
	2
	8
	4
屣	2
¥	2
ô	24
ˮ	4
	4
	8
	2
	2
ʣ	18
ö	6
	2
̵	2
	2
ũί	2
	2
	6
	4
ɹ̫	14
	2
	2
˫	4
	6
	2
	2
û˵	2
ɲ	22
ͷ	2
ѪǮ	2
̣	6
ܵ	6
Ʃ	2
	2
	2
	2
	2
Ϊ	2
	4
	2
	6
խС	4
ֳ	4
	4
	2
ظ	2
	4
	14
	2
	4
	4
ԭ	2
	2
	12
	2
	2
	2
	2
޷Ķ	2
Ϣ	2
Ϣ	2
	2
	2
޷֯	2
ѹ	18
˰	2
	2
ñ	2
Ӫ	6
½	2
	2
˰	2
޷	4
ϸɴ	2
	2
˰	4
	4
Ķ	2
ͷ	4
	4
Թ	2
޲	2
	8
	2
Ṥ	16
	2
ֳ	2
߷	2
ͨ	2
	4
	2
ƽ	4
	2
	2
	2
	4
	4
	2
˳	2
ѧ	2
	6
	2
Ǧɽ	2
ƽ	2
ֳ	4
	4
	4
	2
	2
۶	2
	2
	2
ֵ	4
Դ	4
	2
	2
	4
ͳͳ֧	2
	2
	2
	2
	2
	4
	4
	2
	4
	4
	2
	2
ʧ	2
ٷ֮	2
ԭ	2
	2
ʮ	12
	2
	4
	2
	6
˺	10
	2
	2
	2
ɱڿ	2
	2
	4
ѼӰ	2
ȫ	6
ʮ	2
Ȼ	4
ֱ	2
л	4
ũ	2
	2
̾	4
	2
³ཱ	2
ʮ	8
ʮ	2
ҫ	2
֣ݣ	2
	2
ǣ	2
	18
ǧ	2
	2
	2
ʮ	2
һһ	4
	16
ó	2
ҽ	2
ï	2
	4
ΰ	2
	2
ƽ	2
Ϫ	2
ڽ	2
ʺů	2
	2
	6
	2
οʽ	6
	2
ӯ	2
	2
߷	10
	4
ݹ	2
ʥ	2
	2
ָ	2
	2
	8
	2
ϸ	2
ͨ	2
	2
	2
Я	2
ҵ	2
	8
Ƽ	2
	4
Ʒ	2
	2
־	2
	2
	2
	2
	2
Ȩı˽	2
	6
פ	2
	2
ְ	2
	2
ִ	2
Ϊ	2
Ը	2
ĥ	2
·	2
	2
	2
	2
	2
нԼ	2
	2
	12
Ӵ	12
׼	2
֧	2
Ŀ	4
ֵ	4
	2
	2
ƽϢ	2
	2
	2
뾳	4
	2
	2
ѧ	2
ι	6
רҵ	2
ߵԺУ	2
Ӧ	2
ѡƸ	2
ֱ	4
	2
	2
	2
ɲ	2
	4
Υ	4
в	2
Ƽ	2
Т	2
	4
	2
	2
һ	2
ƽ	4
֬	4
ƶ	2
ئ	2
ȹ	2
	2
	2
ί	4
֬	2
	2
Ұ	2
	4
	2
Ⱥ˶	2
	2
Ա	2
±	6
־	2
ӳ	2
	6
	2
	6
	6
	2
ʿ	4
	4
ͽ	2
	4
ƽ	4
	2
	2
	2
»	2
ǹ	2
	2
	4
ݾ	2
ʳ	2
ò	2
	2
	4
	2
	2
	2
	2
Ե	2
	2
Ͳ	4
	4
綫	2
	4
	8
ͨ	2
	4
	2
	2
켽	26
	4
	4
ί	2
᲻ɴ	2
	4
	6
к	4
־ս	2
	8
վ	2
ɨ	2
	2
ǧ	2
	2
ߴ	2
ίԱ	2
տ	6
Ͷ	14
	10
	4
ǰί	2
	2
˾	4
ȡ	2
	2
	2
ظĸ	2
³ԥ	2
	10
Զ	2
	4
ȫ	2
	4
	2
	8
	4
Ժ	2
	2
	2
	4
	4
ίԱ	4
	2
Ա	4
ŷ	2
	2
	2
	2
	4
	4
	2
ʩ	2
п	4
Ⱥ	4
	2
ԩ	4
ƽ	2
	4
	2
ʮ	12
ʮ	2
йί	2
	8
Ʒ	6
	2
˽η	2
а	2
ֱ	2
ȫ	4
	2
ǫ	2
ƽ׽	2
һ	2
	2
˽	8
ڿҿ	2
	2
߷	2
	2
	2
׸	8
	4
ǰ	4
	2
	2
	2
ũ	2
	2
һ	2
	10
	4
ȺӢ	4
	2
	2
	2
絽	2
	8
	2
	2
С	20
ӽ	4
	10
	8
	2
	2
СȺ	2
ʵ	2
	2
Կ	2
רѧУ	4
ר	4
ũ	2
ݺ	2
	2
ר	2
	2
	2
Ƭ	2
	2
۵	2
ס	4
۵	2
	4
ʱ	4
ʮķ	2
	6
	4
ʱ	2
ʮ	2
	2
¥	2
	4
	2
	6
ܽ	2
ȼƷ	2
	4
	4
ˮ	2
	2
	2
Ȼ	2
	2
	16
	4
Ŧ	2
	2
	2
Ƕ	2
վ	8
	2
	2
æ	4
	2
	2
	2
ݻ	4
ﳬ	2
	2
	2
Ӣ	2
	2
ȫȨ	2
	2
	4
ž	2
	2
Զ	2
Զ	2
	2
	2
	2
	2
	2
	2
	2
γ	8
ɵ	2
̲Ϳʽ	2
	2
	2
̲Ϳ	2
ƽ	2
	2
Ⱥ	4
ƽ	2
	2
Ǯ	2
ο	2
	2
	6
Բ	2
	4
ж	4
	2
ʮ	2
կ	4
	2
	2
	2
	2
ռ	2
Ƹߵ	2
	16
	4
	10
ַžӪ	2
ԼӪ	2
	6
	4
	2
ҵ	2
	2
װ	4
	2
˥	2
ǿ	2
ᴩ	4
	2
	4
	2
ѽ	6
רԺУ	2
˰	2
ѧ	2
	8
	32
	6
	2
	2
ɽ	2
	2
ԴԶ	2
Ѯ	2
	2
˾	2
ѧͯ	2
	2
׫	2
	14
һָ	2
	4
	8
	12
ϸ	4
½粺	6
ڶг	2
	2
	2
Ѳչ	2
	4
ǿ	6
ĸУ	6
¥	4
ȫУ	2
	2
	2
	2
	2
估	2
ޱ	4
ѧ	2
	2
	2
ϲ	2
	2
Ѫ	2
	2
	2
	2
	2
ϸϵ	2
Ŵ	4
ί	2
Ȼѧ	2
	10
	4
	2
	2
	2
ѪѲ	2
ɽҰ	4
֤	2
	2
ũʦ	2
˼	6
	2
	28
	2
Ȧ	2
	2
	2
˼	2
Ұ	6
ҹ	2
׷	2
һ	14
ĺ	2
	2
ũѧԺ	4
	2
	2
	4
	2
	26
Ứ	4
	2
	18
	4
	4
	2
ͭ	10
೵	4
	2
Ұ	2
ֳ	4
ҵ	4
߰	4
	2
	20
	4
ɽ	6
	2
ֳ	4
ʹ	6
	2
	2
	4
Ӧ	2
	4
ﻨ	2
ཬ	2
	2
Ⱦ	2
ⲡ	2
߿	2
	2
	2
İ	2
һ	6
	2
˭֪	4
	4
ʵ	2
ϵ	2
	4
ѡ	2
	2
Ʒϵ	2
	2
֧	2
ո	8
	2
ζ	4
	2
	2
	4
ΰ	2
	4
鱨	2
Ȼ	2
޼	2
ʱ	2
	2
ߴ	6
	8
ͬ	2
ת	2
	2
Ժ	4
ǧ	2
ã	2
Ź	2
Ѫ	2
͸	2
ƽ	2
	8
Ӳ	2
Ժ	2
ҽʦ	2
	2
	2
Ժ	2
	2
Σ	4
廯	2
	2
	2
	2
	2
	8
ʿ	8
ʦ	6
	4
	4
У	2
ʡ	6
	2
־	2
	2
ٻ	2
ĵ	4
	8
ĵ	4
	2
	2
Э	2
	2
ѩ	2
ɽ	4
߷	6
Ѷ	2
ʱ	2
	2
	2
ɷ	2
	8
	2
	2
	2
ǰ	2
¹	2
	6
	2
	2
趫	2
	2
ľ߹ģ	2
δ	2
ữ	4
	6
	2
	2
ͺ	4
֤	2
	6
	2
	2
С	2
Խ	2
ʰ	8
о	2
תȨ	14
	2
ƿ	2
ί	2
Σ£	2
ɣӣ	2
	2
Ʊ	2
ɱ	2
ż	2
	6
	2
С	2
ƹЭ	2
	2
Ӱ	4
ٰ	2
Ӱ	2
Ͷ	2
̨	2
	2
	2
ɫ	4
רҵ	2
ҵ	2
Ƚ	4
	2
Ʒ	4
	2
ɭ	2
	2
⽻ѧ	8
	4
	8
ϵͳ	2
ѧ	4
Ĭũ	2
ȴ	2
Ѭ	2
о	2
	2
Կ֪	2
Ⱦ	2
	2
	2
	2
װ	4
䱾	2
	6
	2
	6
	4
Ϣ	4
ڻ	24
ѩˮ	2
	2
ů	4
	2
	4
	2
	4
ɫ	2
	2
	4
	2
	6
	4
	18
	4
	2
	2
ϸ	2
	2
	4
	2
	2
綫	2
	2
ç	2
	2
	6
	4
	4
	4
ֶŮ	2
ת	2
ǰ	8
	4
鷿	4
ֻ	4
	2
ר	2
ܸ	2
ƻ	2
	2
л״	2
Э	2
˹	6
ɱ	6
ף	2
׸	2
	2
	28
	4
	6
	4
Ļ	2
	6
ټ	2
߰	2
Դ	2
	4
ɽ	2
տ	12
	4
Ī֮	2
ִ	4
	8
	8
	2
	2
	2
	2
̹ɣ	6
ķ	8
ʦү	2
	2
	2
ҵ	4
	4
	2
ܵ	2
Ρ	2
	2
	4
Ȩ	2
Ʊ	4
	2
⽻	2
	6
	2
	2
ɱ	2
ص	2
ί	2
Ͱ	2
ͬ	2
񲿳	2
׿	2
ԭ	2
	2
	2
ʲ	2
ʱ	2
ܱ	2
	2
	20
	4
ฮ	2
ͳһ	2
ֲ	2
	2
	6
	10
	4
Īķ	4
Ʈ	2
	10
	2
	6
׵	2
	2
Ҽ	6
ӳ	2
	2
ï	2
Ŀ	2
	8
ǽ	2
ľ	2
ɣ	2
	2
	10
	2
	4
ȶ	8
	2
ι	2
ɽ	2
	2
	2
ζ	2
	2
	2
ƹվ	2
	4
֦	2
ũѧ	2
㽶Ҷ	2
ɹ	2
С	4
	2
ժ	10
ժ	8
ֲ԰	2
Ƚ	4
ף	4
źǹ	2
	32
ٿ	2
ʵ	2
ת	2
С糡	2
ܸ	2
	6
	2
԰	2
	2
濾	4
	6
ǳ	2
ɫ	2
Ʒ	4
ζ	8
	2
	4
ҵ	2
ǧ	2
	8
ũҵ	2
ϣ˹	2
ʤ	4
ò	2
	4
ѹ	2
ش	2
	8
ƽ	2
	8
	2
Ƕ	2
	4
	2
˱	2
Ȫ	2
׹	2
	2
ע	4
	16
	4
ж	2
ͣ	2
	28
	2
Ұ	8
	10
	2
ص	4
Ϣ	2
	2
	2
	2
μ	4
	2
޼	6
ʶ	2
	2
絳	4
Կ	2
ѡ	6
ӱ	2
	4
	4
	4
	2
ϸ	4
¯	2
£	2
ɢ	2
С	4
ɢ	2
	2
ֺ	2
Сһ	2
ɵ	4
ƽµ	4
Ѱµ	2
С	2
Ժ	2
ϯ	4
	2
	2
	2
Ժ	2
ɻ	4
	2
	2
	4
´	2
	2
	2
ָ	2
ύ	4
۲	2
	6
	2
ж	2
	2
	2
	4
٤	2
Τ	2
˹	2
޵	6
	4
ӿ	4
ǰ	2
	2
	4
Ҿ	2
̩	4
	4
̩	8
ȼ	8
	2
	8
	4
	2
	2
վ	2
	2
ӡ	8
	2
	2
	2
	2
ծȨ	4
	12
	2
µ	2
	2
	2
	2
	2
	2
	2
	2
	4
Ұ	2
Ϯ	4
Ȧ	2
Ѷ	2
	2
	4
	2
	2
ۼ	2
	2
	2
	2
ݽ	4
	2
	4
ī	4
Ϣ	2
ڹ	2
Ʒ	2
ũ	2
	4
	2
Ҫ	2
޲	4
޼	2
ڽ	2
	8
	4
	2
Ӧ	2
	2
ǿ	4
	6
	2
·	2
	4
·	2
	2
֮	2
	2
ɹ	10
	2
	2
	2
իµ	2
ɫ	4
	2
涷	2
	4
̨	2
	8
	2
Ƿ	2
֬	2
Ҭ	8
	4
ز	4
	2
	2
ѷ	2
½	2
˷	2
	2
ּ	2
ڶ	2
	2
	2
С	4
׶	2
	2
	6
	2
Ҭ	4
	2
	2
˷	2
	2
ϰ	4
˿	2
	2
	38
̽	8
ӻ	2
ʹ	2
󸹬	2
	4
	12
	4
	2
	12
	2
Ӣ	8
	4
Ǽ	2
תվ	2
	4
͡	2
	2
	2
	2
׹	2
ϵ	4
	18
ͨ˴	4
	4
ϰ	2
	4
	6
	2
˿	8
ķ	4
ķ	4
	2
ξ	2
	6
	14
ͼ	2
	2
ϲ	2
ɨ	2
˯	4
	2
˯	16
	2
ԭʼ	4
ǰ	2
Ƥ	2
	2
	2
ϡŹ	2
˯	2
	4
ů	2
	6
	4
	2
	4
	8
	2
ŷ	2
	4
	6
	4
	32
	2
⻰	14
	2
˵	2
	4
Ͱ	2
	2
˽	2
̰	2
	2
Ⱥ	2
²	2
	2
Ǯ	2
	6
ߺ	2
ļ	2
Ǯ	10
	6
	4
һ	10
	16
Ϻ	2
	4
	2
˫Ŀ	2
ʧ	4
֢	2
ҽƷ	2
°	2
֫	2
û	6
	2
	2
ͼ	2
Ե	2
	2
ʮ	2
	2
	2
	2
ũ	4
ɩ	4
	14
ʶ	2
԰	2
	2
·	2
ӭ	4
	4
Ц	2
	4
ǰЩ	2
	2
	14
ú͵	2
õ	28
	26
ѹ	26
	2
վ	2
ũ	2
	16
	20
	2
޳	2
	2
ǧ	2
ѹ	8
ʮ	6
ªͼ	2
첻	2
ϻ	2
	4
	6
С	8
	2
	2
ѡ	6
ǧ	4
	2
ô	8
	2
뾶	8
Բ	2
	2
	6
©	2
	2
	2
	2
	2
	4
ȨƵ	4
	4
ũվ	4
	20
	2
ũ繤	8
Ȧ	4
ȡů	2
	6
	10
	2
ת	2
õ继	2
͵	4
	2
	2
ܺ	2
	4
͵	18
	2
̨	2
Ǯ	2
Ϊ	4
ҫ	2
	4
Ǯ	2
	2
ѳ	2
	2
	2
ʡ	2
	2
Ľ	2
ԭԭ	2
	12
	2
̤̤ʵʵ	4
ظ	2
	2
	4
	6
	2
ѧ	2
	2
ָ	6
	4
	4
	2
ͨ	8
	2
ǿ	2
ȡ	4
Ϫ	2
֧	10
С	2
㻪	2
	2
	6
л	2
Χ	6
˳	8
ǣ	4
	2
	2
	2
̨ɽ	4
ٲ	4
	2
¡¡	2
	2
״	2
ˮ̶	12
	2
	4
ѧ	4
	2
̶	2
æ	12
ת	4
ˮ	4
̱	2
ʧ	2
ʹ	2
˲	2
	10
	8
ݸ	14
ʶ	2
	2
þ	10
	4
̶ˮ	2
Ǻ	2
ð	2
̶	2
ͷ	2
ס	2
	2
	2
	2
ͷ	2
Σ	2
	4
	12
	4
ʪ	4
ʯͷ	12
ɹ	4
ο	4
ɽ	8
	2
	4
羰	2
	2
æ	2
Ӱ	2
	2
	6
֮	2
	2
סַ	6
˻	2
	4
	2
ס	4
С	2
ȫ	2
	6
զ	8
	4
	2
	4
ͷ	6
ʧ	16
	2
	4
	2
	4
ĸ	16
	2
	2
	4
չ	2
	4
	6
	2
	2
	2
	2
׻	8
	4
׳	2
	2
	10
ȥ	6
	2
	2
ʱʱ	4
	4
Х	2
	2
Ա	2
	2
Ƹ	2
	2
ˬ	2
	2
´	2
ڶ	2
תۼ	2
	8
	10
׼	2
Ű	10
һ	2
ύ	2
	2
˵	2
	2
۽	2
	2
ල	2
Ŀ	2
	2
ʵʵ˵	2
	2
̬	2
	2
д	2
	14
Ÿ	2
ʾ	2
	2
	2
С	14
Ļ	6
˹	32
	2
	2
ϲŭ	2
	6
	2
	4
Яֲ	2
ɲ	4
æµ	2
ϵ	2
	6
ȥ	2
̾	2
ȥ	6
С	2
ϴ	2
	2
	2
	2
֮Ժ	2
	4
ӻ	6
ʼ	2
	4
ӹ	4
	4
һ	2
	4
	4
ԳԺȺ	2
	2
	2
	2
ʱ	2
β	2
ר	6
	2
	2
	4
	2
ٹ˽	2
	2
˽	6
ߺ	4
Ͷұ	2
	6
ûʧ	2
ͽ	2
Ѱ	4
ɡ	2
	4
	2
	2
	2
֪	2
ϱ	4
	2
ɲס	8
	2
Թ	2
¹	2
	2
	2
ݷ	2
첿	4
й	2
˻	2
»	2
ծ	10
	6
	16
	2
Ŀ	2
ϲ	2
	2
	2
ӭ	2
	2
ﴨ	2
ȫ	2
	16
Ժ	2
	10
	4
˽	2
	2
	2
	6
ª	2
	2
	2
ӵ	2
Ͷǿ	4
ҵ	2
ɢ	2
Ϫ	2
˹	2
ٰ	2
²	6
	2
	2
	2
	4
ѭ	2
ְ	4
	2
֧	2
	2
١	2
	2
	6
	2
	2
	2
	2
̸	4
ֶ	4
ɻ	2
	6
	2
ѧϰ	2
ũ	2
޴	4
	4
	2
Ĳ	4
ǲ	4
;	2
	6
и	2
һ	2
	2
֤	2
	2
	2
	2
	8
	2
н	2
	2
	8
	4
	2
	2
㺺	2
	2
	6
	2
Ĺ涨	2
	4
ͨ	2
	2
Ϣ	2
	2
ɽ	12
	2
	6
	2
ʤ	2
	2
ľ	2
溦	2
	2
ʴ	4
	2
ɽͷ	2
԰	2
ʯ׵	2
ľ	2
ëձ	2
С	2
ɽ	2
	2
˹	2
ɽ	2
	2
	2
	2
ж	2
	4
	2
޿ɷ	8
Сѧ	16
ͣ	2
ʹ	8
	4
	2
	6
	16
	2
΢˽	2
	2
	2
׵	4
ҵ	2
	4
	2
߷	2
	2
	6
Ƕ	2
޸	4
	12
	2
	12
ɳѼ	2
С	2
һ	2
	2
ͨ	2
Ԫ	2
	2
	16
	6
	2
س	2
	2
	4
	10
ٽ	4
ɫ	2
	2
ټ	2
	2
	12
ù	2
ۼ	2
Ѫ	2
	2
ʹļ	2
˲	2
	2
ѫ	2
	2
	2
	6
	12
	4
Ц	2
к	6
ˮ	2
	2
	2
	10
ũ	4
ͷʹ	2
Ӳ	2
	14
	2
	48
Ȳ	2
	4
иҵ	4
̵	2
ҽ	2
ʦ	2
⽨	2
³	2
汸	2
	2
	2
	6
	4
Ϊ	2
Ա	4
ԣ»	2
	2
ʹ	2
	2
ʳĸ	2
	58
ϡ	2
	4
	18
ĳ	4
	2
·	2
	4
	8
Ƿ	2
ж	4
ׯ	2
	4
ߵ	2
;	2
ֳ	2
ĳ	4
	6
ĳ	10
	2
	12
	8
Ȼ	2
	2
	2
	2
ʱ	4
ɾ	12
ũó	4
·	2
	2
϶	2
ί	2
	2
ѩ	4
	8
޻	2
ȡ	2
	2
	2
	2
Ӣ	2
¸·	2
ƽ	4
ɽ	4
ƽ	2
ׯ	18
	2
	10
	8
	4
	10
	2
	2
	2
ţ	2
Ģ	2
絳ί	6
	2
ҵ	8
	10
а	2
	2
Ӣ	4
	4
	6
	2
ϵ	2
С	4
Ϣ	2
	2
һ	2
ʹ	2
	2
׸	2
Ҿ	2
	2
Ŀ	4
ż	8
	4
	6
	118
լԺ	2
ж	4
	4
˽	2
	2
ͷ	2
	2
	2
	86
	2
ճ	6
û	16
Ӣ	2
	2
ϲ	2
	24
	34
	2
̥	2
	2
	2
ʡ	2
	2
	2
û	2
Ǧʺ	4
Ʒѧ	2
	2
	4
	18
ս	2
	2
	4
ޥ	2
	2
	2
Ω	2
	2
	8
Ϫ	2
	10
Ϫ	2
	2
	8
ʵ	2
	2
շת	2
	2
һƽ	2
	2
	2
нײ	2
ԣ	26
а	2
ϸɲ	2
	2
ĸ׽	4
	8
	2
	4
ʫ	2
	2
	2
	8
	2
	2
	2
ֱ	2
ʮ	2
	2
ڴ	8
ӱ	2
	4
	2
ȱ	2
	2
޴	2
	2
	2
±	6
	2
ܶ	2
ܶ	4
	2
	2
	2
ĳ	2
	2
ϵͳ	2
	2
	2
	2
ä	2
	2
̬	8
֤	2
	2
	10
̬	6
Ȼ	4
˷	2
	2
	2
	8
	2
ֱ	4
ͷ	2
Ž	2
	2
	4
	2
	2
	2
ʧ	4
	2
ϲ	2
	14
	2
۸	4
һ	2
	4
Ǳ	2
	6
	2
ʢ	2
	2
ԴȪ	2
֪ʶ	10
Դ	4
	2
ط	2
֪	2
ֶ	2
ָ	6
ֹ	4
	2
	4
	2
㷺	2
Ȩ	2
¶	2
̢	6
а	2
	4
	4
ɳ	2
H	2
	4
Ψʷ	2
	10
涨	2
	2
	2
	2
չ	4
⽨	2
ֳ	2
Խ	2
ʱ	14
ʲԷ	2
޴	4
εɫ	4
	2
	4
	2
Ϊ	2
	2
׿	2
и	2
	2
	2
	2
	2
	4
ʯ	2
Į	2
	2
ٷ֮	2
ͷ	2
ͳ	2
ϲ㽨	4
	2
Խ	4
ƶ	2
Ĳ	2
Ѫ	2
	2
	2
	2
ˮ	2
ɽ	2
Ա	2
Ȳ	2
	2
ز	8
ֳҵ	2
徭	6
ɽ	2
	2
ס	2
	2
	4
޲ϯ	2
ʳζ	2
	2
	2
æ	2
	2
ֳ	2
	4
Ϸ	2
	2
ר	2
С	16
ˮ	2
Կͷ	2
ʴ	2
ʴ	8
ܻ	4
˽	2
	2
Ϊ	4
	8
ܺ	2
ݾ	2
ı	2
	4
Ȩ	4
ٳ	2
	2
	2
ٱ	4
ϳͲ	2
ΰ	2
	12
	8
귶	2
	4
	4
ϲӭ	2
ʱ	4
ճ̱	2
	2
ٲ	4
Ӧ	2
	2
֮	2
	14
Ѿ	2
ӹ	2
˥	4
	4
	2
ԱŸ	2
	2
ۺ	4
Դ	2
Ŀ	4
ķ	4
	2
	6
ɫ˵	2
˲	2
	2
Į	2
ס	2
	2
	2
Ԩ	2
д	2
ͽ	2
	2
	2
	6
ÿ	2
ǧ	2
	2
	4
	2
ݷ	4
أ	2
ƴ	2
Ϥ	2
	2
	8
Ǯ	2
	4
ӻ	2
	2
ױ	2
	2
¸	6
ð	2
	4
	2
ʹ	2
	2
	2
ĳ	2
ڷ	2
	2
ز	2
	2
	6
	2
Ժ	2
龿	2
ع	2
ץ	2
	2
	14
	2
	8
	6
ӿ	6
ٱ	4
	2
ලԱ	2
	16
ؿ	2
ʳ	4
ɫɫ	2
Ի	2
	2
Ϣ	2
ؿ۷	2
	2
н	2
	2
һ	2
	2
	12
Ŀŵ	2
ӭ	2
Ҫ	2
	2
Ȼ	2
	2
ĳĳ	2
˼	2
ʹ	2
ؽ	2
ͨ	4
	2
̶ʲ	2
참	4
ͷ	4
	2
ͷ	2
	2
ַ	2
֧	2
ƭ	2
	2
	4
Ч	2
	2
˽	2
	2
	2
֤	2
ȫ	2
ܻ	2
ȫ	2
ʴ	2
ִ˲ƣ	2
ͷ	2
¸	2
	2
ɷ	4
۴	8
һ	4
˴	2
ʾ	2
	2
	2
ʱ	2
	4
	2
ǽ	2
Ϸ	14
	2
Ͻ	2
	2
Ƕ	6
״	4
ֵձ	2
ü	6
	2
緹	18
	6
	2
Զ	2
֧	2
ĸ	8
	2
	2
֧	2
ܷ	2
֧	2
н	2
˯	46
	2
	2
	4
Ħ	2
Ҫ	2
	2
	2
͸	12
	2
	2
	2
и¥	2
߽	2
	2
ǽ	2
	6
ÿ	2
	2
	6
۵	2
	2
	2
б	2
ǽ	4
쳯	8
	2
ִ	10
	4
	4
ʤ	6
ͷ	2
	2
Ϊ	2
ƿ	2
ϲЦտ	2
	4
	2
	2
ղ	4
и	2
װ	6
ɲ	2
	2
	6
	2
خ	2
	6
	2
˳	2
ͷ	2
ؼ	2
ӵ	2
	2
	2
ݸ	2
ѹ	2
	2
	2
	2
	2
	2
	2
	4
	2
	2
	4
ȫ	4
ũ	10
	4
	2
ͽ	2
	58
Ŵ	4
	2
	2
ׯ	2
ַ	2
ɽƺ	2
ƶ	2
ƶʿ	2
屣	2
	2
˳	2
	2
е	2
	2
Ͻ	2
ʵ	2
	6
鷳	2
	2
	2
Դ	2
	2
	18
	4
	2
ǰ	6
	2
Ϊ	2
ʱ	2
δ	4
ƺ	2
л	2
ɿ	4
ɶ	6
	2
	2
	2
	2
ճǼ	2
Ϣ	2
ͳ	4
	2
˵	6
	2
־	2
ռ	2
	2
	16
˰	2
	2
ɽ	2
ִɽ	2
˵	2
޼ƿʩ	2
	2
	2
	2
Ȩ	2
ѡ	2
	2
	24
	2
԰	10
	16
Ǩ	2
ƶ	2
һ	12
	4
	2
	4
	10
ſ	2
	2
	8
	2
ͦ	2
󸹱	2
	4
	14
Ѽ	12
	2
	4
Ϊ	2
޶ż	2
Ұ	2
Ȼ	20
ȫȻ	4
	4
	2
ͬ	2
紺	2
ĺ	4
	2
ƻ	2
	2
	2
ݺݺ	2
ͤ	2
	2
ľ	10
ʫ	8
Ȼ	2
	2
	6
ſ	2
Ȼ	2
ɷ羰	2
¾	2
	8
Ұ	6
	2
ı·	2
ū	4
	2
	16
³	2
ɳ	2
	6
Գ	2
ס	2
֬	2
ٻ	2
	4
	2
	4
	2
	2
	4
	2
ҩ	2
	4
	4
é	2
Ь	4
	2
	2
	2
	4
	2
Ī	2
	4
	4
	10
˹	2
ԥ	2
Լ	4
޹غּ	2
	8
ȷ	4
	2
	4
	4
׷	2
岻	2
Ȳ	2
	2
	4
	2
й	2
ڷ	18
	4
ʥ	4
	2
ƴ	2
ڹ	4
滯	2
	2
Ӫ	4
̽	2
ʵ	2
Ż	2
ϵ	2
	2
	2
	4
	2
	2
	2
	6
	2
	2
	2
һ	2
Ϳһʱ	2
ʢŭ	2
ˮ	2
˾	6
ɱ	2
ȴ	2
ĸ֮	2
	2
ں	2
	2
ùͷ	2
	2
	4
	10
	2
ĳ	2
	2
	2
	6
Խ	2
	2
಻ס	2
ŭĿԲ	2
ǽ	2
	2
	2
ҧг	2
	2
	8
潬	4
	2
	2
	2
	2
	2
	2
ȥ	2
Ӱ	2
ͨ	2
	2
	2
ڽ	2
	2
	2
˼	2
	2
	6
齵	2
Ȼ	2
Ȼ	2
Ի	2
¶	14
Ӵ	2
̨	2
	2
	4
÷	2
	2
	6
	2
Ը	2
	2
	2
ư	2
	2
ʱ	4
˴	2
	2
	14
	8
	2
Ϸ	4
	4
	2
ȥȪ	2
	6
·	2
	2
ɽ	2
ʫ	2
Ŷ	2
	2
ɳɳ	4
	4
ů	8
ҹ	4
β	2
	10
գ	2
ƺӿ	2
	2
ɽ	2
ɽ	2
ɽ	2
ɽ	2
ȶ	2
	2
	2
	2
	4
	4
	2
	12
ҷ	2
˷	2
˰	2
	2
װ	2
	4
	4
СԺ	6
	2
ĺ	2
ƽ	2
	46
֨֨	2
	18
Ƥñ	2
	2
	4
	2
˵	28
۽	2
Ө	4
	2
ĵ	2
Ͱ	2
ûû	2
	2
ص㻧	2
	2
С	2
ؿ	2
	2
	2
	16
ʪ	26
ũʱ	2
	4
æ	2
	12
	4
	2
˼	2
	6
	2
	40
ɽ	2
֧֧	2
	2
޻	2
̺	2
Զ	2
ɫ	4
ݳ	2
	2
	2
Ϲ	2
	2
	2
Ժ	28
	2
	10
԰	6
֪	2
ÿÿ	4
չ	4
	2
	4
ͺ	2
	2
	2
	6
Ȼ	2
Ľ	2
	2
	4
	2
Ⱦ	2
Ե	2
Ϸ˵	2
ŭ	2
ʢ	2
	2
	2
ʵȻ	2
ƻ	8
ʯ	2
	2
ʯ	2
С·	6
м	2
ټ	2
˼	2
ϡ	4
ĺɫ	2
ϡ	6
̨ͤ¥	2
ˮ	6
С	4
ӹ	2
̨ͤ	2
¥	2
ӳ	2
	2
԰	2
ˮ	4
	4
һԴ	2
	2
	2
ż	2
ƽ	2
	4
ƻ	4
	2
ƿ	4
	2
	2
	2
С	2
Χǽ	2
԰	2
ջ	4
ӳ	4
	2
ɽ	2
Ȼͤ	2
ú԰	6
	2
ũ	2
԰	6
	2
ǽ	2
͸	2
¶	4
	2
ľ	2
	6
	2
	4
	2
ս	2
ɫ	2
	2
º	4
	2
˽	2
	4
	4
ͷ	6
	2
	2
	2
	2
	2
	2
	2
д	2
ڰ	10
̹	2
	2
	2
	2
	2
Ͽɽ	2
	2
	8
Ϊ	2
̨	2
Ĳƽ	2
	26
	2
	2
	2
	2
Գ	2
שʯ	2
ľṹ	2
	2
ʯǽ	2
ʽ	2
	2
	4
ʢ	2
ʮ	2
	2
ַ	4
յ	2
ͻأ	2
ש	4
ʯ	2
	2
߲	2
	2
ɼ	2
	2
	2
ҩ	2
½ս	4
Ȳ	2
Ĳƽ	6
αȨ	4
	2
Ҵ	2
Ĳƽ	6
Ƭ	4
ռ	2
һʱ	2
ʾ	2
	2
	2
	2
ʱ˿	2
츣ɽ	2
ȹ	2
˾Ա	6
	8
	4
	2
	2
о	2
	4
	2
	2
䲻	2
׽	2
αس	2
ʮʱ	2
²	2
սƷ	2
;	2
	6
λ	2
̵	2
ſ	4
	12
	2
	2
	18
	2
	2
һδ	2
ʮ	4
Ӧս	2
ĺԺ	2
	2
	2
	38
ݶ	4
˾ŵ	2
	2
߳ŭ	2
Ż	4
ȼ	2
	2
	2
	8
Сɽ	6
ʧ	2
ָսԱ	2
̧	16
	2
ͻΧ	2
	2
	2
	2
ǧ	2
ջ	4
	2
	2
	8
	6
	2
Ƥ	2
	18
ʵ	2
	2
	2
Ʒ	10
	54
֥	2
	4
С	2
ʱ	2
绯	2
ͼ	2
һ	4
ʳ	2
	2
	4
	4
	2
	2
ҩ	2
ֹ	2
Ƚ	2
	2
	2
	2
	2
	2
	2
ǿ׳	2
ʳ	8
	18
ũʳƷ	2
ʾ	14
	2
	6
ۼ	2
	2
	6
	2
	4
	2
	4
Ȫˮ	2
	2
˵ʵ	2
̻	2
һһ	2
	2
	12
	2
ζ	4
ζ	2
Ʒζ	2
Ũ	2
Ĵָ	2
˿	12
Զ	2
У	4
	2
ն	6
	2
װ	2
	4
ҹ	8
һ	2
ҹ	4
	4
һ	2
ե	8
˿	4
	2
˱	2
ζ	2
	2
	2
ʦ	2
	2
	2
	2
	2
ƽ	2
	10
ҹ	4
İ	2
ҹ	2
	2
	2
	2
	4
Ⱥ	2
	2
ľ	2
	2
	2
	2
	6
	2
ԲԲ	2
Դ	6
	2
ţ	4
ʳ	2
˿	2
ƥ	2
	4
	6
ֽ	2
	2
	2
	2
	2
Ʒ	2
	2
Ψ	2
	2
׷	2
ڼ󽨹	2
ϸ	2
̯	2
ų	2
	2
	2
ͼ	2
ܷΪ	2
̰ͼ	2
Ȱ	4
	2
Ƶ	2
	2
ǰ	2
	2
	2
Ž	2
û	2
·	2
	2
Զ	2
ഺ	2
ʮ	4
	2
	2
	2
⿬	2
Ľ	2
ı	2
	4
ᵳ	4
絳	4
	2
	2
ͬ	2
	2
	4
	2
ʮһ	4
	2
ï	4
ѽ	2
ҹ	2
	4
ѫ	2
	2
	2
	2
	2
	2
ï	2
	2
	2
	2
	2
	2
	4
	2
ܶ	2
س	6
Ա	2
	2
	2
з	6
Υ	6
	4
ְ	4
	2
	2
	2
	4
Σ	2
	2
˼	2
	2
	2
幤ҵ	2
	2
	2
ѧά	2
	2
ϳά	2
	2
ɫ	2
ֽ	2
ԭҩ	2
	2
	2
Ͷ	2
	2
ȼ	2
	34
	2
С	2
ػ	4
	2
	4
̾	4
峿	2
׷	2
	4
ԭ	2
	2
	4
ѩ	2
	2
ǣ	2
ž	4
	2
	4
ů	6
	4
	6
	4
¯	4
ů	2
	2
ʮһ	2
	2
ͼ	2
ĳ	2
	2
˰	2
	4
	2
Խ	2
¶	2
Ӫ	2
Ӵ	2
	2
µ	4
	2
ÿ	2
	2
	2
	2
˺	6
	4
	2
	2
ƽ	2
	4
	4
	2
	2
	2
	2
	2
ս	2
	2
ͻ	4
	4
	2
ٽ	2
	2
޼ҿɹ	2
Ȼ	2
˲	2
	2
	2
Х	4
Ӣ	2
	2
˷	4
ð	6
Ծ	2
	2
ѩϼ˪	2
	2
Ǩ	2
	2
ɴñ	2
	2
Ĺ	2
	2
ֿ	6
	2
	2
	2
	2
	2
ػʵ	2
	2
	2
Ӫ	2
ɽ	2
ļ	2
ҵ	2
¾	2
Ӧ	2
ʩ	2
ʩ	2
	2
ʩ	2
ѫ	2
	2
	2
Ա	2
1992	4
ϲ	4
	80
ʮ	4
̨	2
	4
ȫ	2
˵	12
	6
ʯ	2
	10
	2
	8
1919	2
1922	2
ѧ	4
˵	36
־	4
	2
͵	24
	6
־	2
	4
˵	4
ɫ	2
ɽ	4
ϵͳ	2
	6
	8
	24
ҩ	4
	2
ҪȻ	4
	122
Ա	4
	4
	6
	30
	2
	10
ܻ	4
ʵ	32
	2
	14
ͯ	2
С	14
	2
	8
	8
Һ	2
	2
	6
	4
	4
ɫ	8
	2
	2
	6
Ϲ	6
	198
	2
	4
Դ	2
	20
ԭ	4
	2
	10
	20
	16
	4
һ	6
Ʒ	2
ʵ	2
㼯	4
ը	10
ҵ	8
	60
С	16
ֲ	12
ж	2
ɿ	6
ѧ	2
	34
	12
	6
	2
	2
""""	12
	2
	4
ʯ	2
ˮ	8
	6
	2
԰	22
	4
	2
ѻ	6
	6
	4
	6
	6
ģ	2
	2
Զ	4
	4
	2
	14
	8
㵽	2
	2
ˮ	2
	8
	4
Ԥ	4
	4
	2
ۻ	8
෽	6
ӦƸ	2
	2
	16
ˮ	2
	2
Ϯ	2
	2
ɳ	6
ӵ	4
ϸɲ	2
ƽ	8
	4
	6
߰	2
ײ	4
	2
	6
	4
	2
	2
	10
ͷ	4
ʱ	4
ϧ	2
֦	6
ѧ	4
è	28
	2
̫ȭ	2
ĩ೵	6
Ӱ	2
ʽ	2
	8
	4
	6
һ	6
ר	2
	28
˵	14
	6
ô	22
ֱ	40
īˮ	12
ܵ	26
	6
ʮ	2
	6
һ	24
ĺ	2
Ȳ	6
а	4
ɵ	2
δ	4
	4
Ի	6
	12
ַ	2
	2
д	4
Ů	8
Ů	14
	8
	6
	6
ͨ·	2
ɳ	2
ͼ	26
ϴµ	4
˴	12
	2
ǡ	8
̽	4
	8
	4
	20
	24
λ	2
κ	22
ƾ	6
	4
	2
ɽ	4
	2
¶	2
ɽ	6
Ա	2
	2
ҵ	2
	2
Ͽ	18
	20
ж	4
ҩ	2
	4
	4
	128
	2
	18
	2
	2
У	10
Ҷ	14
ѧ	20
ɽ	2
鱨	8
͵	10
	4
	18
	2
ͷ	6
ĥ	2
԰	4
Ц	4
	2
֪ͨ	2
	10
	6
ȱϯ	4
ҿ	4
	2
Ů	2
	8
	10
	2
нˮ	4
	2
ӹ	2
	4
	20
ӱ	16
	2
	32
ë	4
	6
ɡ	24
д	10
	6
	4
	10
	12
	2
	2
ͻ	6
˹	4
	2
ñ	18
ʾ	22
	2
У	2
Ͻ	6
	6
ļ	2
ʮ	2
	6
	6
	14
	6
	20
	2
ǿ	2
	10
	6
ӷ	2
	2
У	2
	2
	6
ܵ	4
	20
ذ	12
Ƭ	2
Ϻ	4
װ	2
̫	6
羰	2
ѧ	2
ָ	2
Һ	4
	26
·	2
컨	4
	2
ֱ	10
	2
ŵ	18
ʷ	2
΢	4
;	4
Ǯ	10
	28
	2
	24
	12
ҳ	2
ղ	2
	8
	4
ֵ	2
ĸ	8
ѧ	6
	2
	2
	2
	10
	10
	2
	4
ɽ	2
ѹ	8
Ƶ	2
	8
ټ	2
ɢ	20
ӰԺ	4
ʩ	2
ֹ	2
	2
ѡȡ	2
	14
ﳵ	8
	8
	4
ɽҰ	4
ҹ	8
ʡǮ	2
	8
	2
ְ	14
Сʱ	6
С	34
˲	2
̤ʵ	2
հ	4
	2
ȫ	2
	10
	2
	16
	2
˶	2
	2
ĵ	6
ð	4
	6
һ	2
¿	8
ͨ	2
	2
Уѻ	2
У	2
˼	2
	2
һ	28
	22
	2
ţ	4
	4
	6
	4
	2
ջ	4
ѵ	2
	2
	4
ˮ	8
ʮһ	2
	4
ʮ	4
	2
̹	2
˿	2
	10
ɢ	2
	32
һת	2
ͷ	4
׷	4
	10
̹	4
ӣ	2
	4
ƽȻ	6
ң	2
ͼ	2
	4
ԭ	2
	2
С	4
	6
ֲ	4
Ŀ	2
ů	2
	2
	8
	16
	6
С	4
ָ	2
Ĺ	2
	8
	2
	6
	26
	2
	2
	2
	2
	2
	2
Ϫ	2
	2
һ	2
	2
	4
	2
	2
	2
	28
ľ	2
	4
	2
ó	2
	26
	2
	10
ϲּ	2
	8
Ǳ	2
	12
	8
ר	2
	2
	4
	2
ƽ	4
	2
	2
	18
	4
ʽ	2
	2
ű	4
Ϳ	6
μ	2
ɳ	2
	2
	2
ɽ	4
	6
չ	2
	2
	4
ͷ	8
	2
	2
	10
	4
	14
ҵ	2
1903	2
	2
ʶ	2
1916	2
1940	2
˹̹	4
1949	4
1967	2
Ұ	10
1977	4
1979	4
1991	2
	4
3026	2
440	2
1997	2
Ϊ	4
	6
ʻ	2
	2
	2
	2
³	4
	2
С	2
	2
ķ	146
˵ʱ	2
緹	12
	4
	2
	6
Ѿ	6
	8
	4
߰	2
ƿ	14
	10
	14
	24
	2
	28
	2
ͬ	2
	4
ɢ	8
	4
	4
	2
	12
	2
	6
׼ʱ	8
ҹ	6
⾰	4
ˮ	2
	2
	4
Ʊ	4
Ա	2
˿	2
Ա	6
ȥ	22
ؾ	4
	2
Ǹ	10
ô	2
ը	2
	8
	2
	4
У	2
	2
绤	4
	2
	4
	2
	2
Ҷ	2
	2
ı	4
Ƭ	2
ӳ	6
	2
˩	6
ȶ	38
ƥ	12
	2
	6
ƽ	8
	2
	2
	4
ص	2
һ	4
	10
	2
	2
ⵣ	4
	2
	4
û	6
§	2
	2
ϡ	4
Ū	2
ǹ˨	2
	12
	2
	4
	2
	2
	2
½	4
	2
	6
	2
ʹ	4
	2
	2
	2
	4
Ķ	2
񾭹	2
ˣ	2
д	2
Բ	2
	2
	2
ô	6
	4
	2
	2
	2
߶	4
׵	2
	4
˥	4
	2
	4
ʿ	6
	2
	8
	4
	16
	10
ˤ	14
	6
	2
	2
	6
Ը	14
Ƕ	56
	2
ûʲô	4
	6
	4
ɢ	2
ϸ	2
ײ	6
	4
	2
	6
	2
Ϊ	2
֦	2
ֹ	2
	2
ȫ	2
	8
ֱ	10
ɶ	2
	2
	4
£	2
	8
	2
	4
	8
ԷƱ	2
	2
	2
Ծ	2
īˮ	2
ԥ	4
	6
ë	2
̵	8
	2
	2
	2
˵	2
	4
ζ	2
;	2
ƿ	8
	16
Ա	2
	2
	4
	8
ɱ	2
	4
	2
	2
	2
ˮ	6
С	2
û	6
	4
	6
	4
	2
	2
	2
	2
	2
	2
̸̸	4
	16
	2
ǽ	2
ҽ	2
	2
ŷ	2
	2
	2
;	2
Ʊ	4
	2
	8
ٵ	20
	2
	2
	2
Ψ	2
	2
	8
С	16
ʦ	2
	8
	2
	4
ͨ	2
	6
	8
ͬ	4
	2
Ⱦ	4
ֹ	2
©	8
	6
Ư	8
	4
ͷ	4
խ	8
	2
ɸɾ	10
ƽƽ	2
ҵ	2
ų	2
	4
	4
	2
1996	2
;绰	2
ʮ	2
Ӷ	2
	2
齱	2
˳	2
	2
	2
	2
߿	2
ϣ	6
	4
	2
ӽԺ	2
	4
ԭҰ	2
ե	2
Ѫ	4
	2
	2
	4
պ	2
	4
ϧ	6
	2
	4
ڸ	4
	2
۹	2
Ҷ	2
˿	2
û	2
	2
	6
	2
ͭǽ	2
ס	2
ˮ	10
	10
	10
	4
	2
һЦ֮	2
	2
Ӱ	2
	2
	2
	2
	2
С	4
	2
ҡ	4
ŷ	4
û	2
Ʋ	8
ϥ	4
	4
	4
	8
	4
Ͽ	4
	2
	34
	34
ξ	2
	2
	4
	2
о	2
1957	2
	2
С㲥	2
	4
	2
ʹ	8
	2
	2
Ծ	4
	4
	4
	4
ĵ	2
þ	2
ƣ	2
	4
	4
	8
	42
שͷ	4
	2
ٶ	2
	2
ʳ	2
̥	4
ʹ	4
ǧ	4
	4
	2
	16
	4
Ž	2
Ʒ	6
	4
ˮţ	2
	2
	2
ɽ	2
	2
	4
	14
	2
л	2
о	10
	8
ڽڰ	2
ס	4
	2
	2
ݵ	2
	2
ȫû	2
Ӫ	2
ķ	2
Ȼ	2
	2
ͷ	4
	2
¡	2
ķ	6
	2
ͷ	6
ǹ	2
	2
	2
	2
6.5	2
ս	4
ʮ	2
ʡ	2
籭	2
ʢǰ	2
	2
٤	2
	2
	2
ƽ	2
糵	4
ѹ	2
ú	2
ͨ	2
	6
	8
ѧ	6
	8
	4
Ӿ	2
¹	8
ž	2
	2
	4
	2
ѵ	2
ʮ	8
	12
	6
鷳	2
Բס	2
	4
	4
ǳ	4
	4
	2
ǿ	2
	4
	2
	2
	4
л	4
	2
	2
	4
	2
	8
ʮ	6
	2
ƽ	2
Ȼ	2
ֿ	2
	2
ö	2
	4
η	2
˹	2
	2
ض	4
ϴ	6
	2
ɾ	4
˴	4
	6
	2
ˮ	2
	2
ɹ	2
	2
	2
ɻ	4
	4
	4
	6
	2
׹	6
ײ	2
ʽ	2
̼	6
	12
	2
Ҷ	2
	2
	2
	2
	2
蹷	2
	4
һ	4
	4
	2
	2
	2
	2
ɲ	2
ȼ	6
	6
	2
	2
ι	2
ʢ	2
	2
	2
	2
	2
Ӳ	2
ǿ	2
	4
	2
ת	10
	2
СȦ	2
к	16
	2
ѧ˵	2
ѧʷ	2
	2
	2
	2
	4
	2
гԱ	6
	4
Ѵ	2
	2
	2
λ	2
	2
Ц	2
	4
	4
ȹ	10
	2
	4
һ	4
㷹	2
ͷ	2
	6
ҵ	2
ػ	2
С	2
ê	6
	2
	6
Ա	2
	2
	8
ҡ	4
	2
	2
	2
	14
	2
	2
	2
Ҳ	8
ѩ	4
	2
	2
	4
ܼ	2
	2
	6
	6
	2
Ⲩ	2
ͷ	2
ȿ	2
	2
ת˲	2
	2
ɽ˺	2
	2
	2
	2
ƥ	2
	2
	2
ʮ	4
ձ	2
	2
	2
	2
	2
үү	4
	4
ߵ	16
	2
Ǳ	10
	8
ʯ	2
η	2
	2
ˮ	2
	2
	4
ˮ齻	2
ö˶	2
ú	6
	2
û	2
	4
üһ	2
˾	2
	2
ˮ	4
	4
״	2
	2
ˤ	4
ν	2
	2
	4
	4
ţ	6
ˮ	4
ұ	4
϶	2
	2
Ӣ	8
	2
ڰ	8
ƽ	2
ı	2
	6
	6
	2
	2
	2
	2
ǰ	4
λ	12
ɽ	2
	4
Ӧ	4
	2
	4
	2
չ	2
	2
	2
貽	2
ѩ	4
	2
	10
ӵ	2
̰	2
	2
仰˵	2
	8
be	2
̲	2
	2
Ʒ	4
ָ	8
	2
ຣʡ	2
	4
	2
ҳ	2
	2
	2
	2
	2
	4
1975	2
ڵ	2
ɳ	4
ʯ	2
	2
ѧ	2
	2
	2
	2
ë	6
	2
ͷ	2
	2
	2
	4
	2
ҹ	10
	4
	4
	2
Ӱ	2
Ʈ	2
ŵ	2
Ҫ	2
	2
ˬ	2
	6
Ը	2
ͷ	2
	2
	2
	8
ַ	2
	10
	4
	2
սǰ	2
	2
	2
	2
ɫ	2
ί	2
̴ɽ	2
ɡ	2
Զ	2
Ȧ	2
ʮ	4
ѧ	2
	2
Ҫ	4
	2
	2
	4
տ	2
	2
ǰ	4
	2
ծ	2
ծ	2
	2
	16
״	6
	6
Ϻ̲	2
	2
	4
ˮ	12
̵	2
	2
ճ	4
Ь	8
峺	2
	2
	6
	4
	10
	6
	2
ǧ	2
	2
	2
ѹ·	2
Լ	2
ԤԼ	2
ˬ	4
΢	2
	2
	2
ݵ	4
Ѳ	6
ҹ	6
ˮ	2
	4
	6
ͷ	4
	4
΢	2
	2
	4
д	2
	2
ͧ	2
	2
׽	2
ֻ	2
ư	2
С͵	12
ƽ	2
	18
߸	4
ٹ	2
ʼ	2
	2
̻	2
	2
	14
	2
ʻ	6
ľ	2
ˮ	2
	2
	2
κ	2
տ	2
	14
	2
˵	6
ѩƬ	2
	2
ù	4
Ż	2
	2
Ȧ	2
̨	2
ʵʵ	2
	2
	2
	2
Ͷ	4
ֱ	2
	2
ͷ	2
	8
Ԫ	16
ˮ	2
	2
ƽӢ	2
	2
ȸ	2
ë	2
ⷿ	2
	4
	2
˵˵	2
	2
	2
񻶽	2
ź	2
	4
	2
	6
ϴү	2
	6
ɽ	2
ײ	2
Ů	2
	2
ʦ	2
ͽ	6
ɢɢ	6
	4
	2
׹	2
	2
	2
	2
ʦ	2
ǧ	2
	2
	4
ĵ	2
	6
ֹ	4
ܽ	4
ţ	2
	2
ƽ	2
	2
	2
	2
	2
Ϧҹ	2
	2
ˤ	2
Ť	2
	2
ײ	4
	2
Ტ	2
	2
Ϣ	2
Ч	4
Ժ	2
1870	2
ʴ	2
	2
ɱ	2
	4
ͷ	4
	2
	2
	6
ҹ	2
	2
ڶ	4
·	2
ɫ	6
	2
ʱ	2
ѧ	2
˼	2
¶	2
ƽ	6
·	4
ˮվ	2
ʾ	2
дԱ	2
	4
	2
ʪ	6
ϳ	4
Ȼ	2
̥	4
	2
ɿ	2
	2
	2
	4
	2
	2
	10
	2
	2
ѱ	4
ľ	4
	2
Ҷ	2
Ź	2
Ʒ	2
	2
ԡ	2
ë	2
1976	2
	2
õ	12
֦	2
	4
쫷	2
·	4
	4
ǳ	4
ɨ	2
	4
Ϣ	2
ϰ	2
	2
̴	2
	2
ע	2
ͼ	2
	4
ֽ	2
	2
ָͷ	4
	4
	2
Ͳ	2
Ͳ	6
	2
ʱ	2
	2
ˮ	2
	2
ѱ	2
	2
Ŀ	2
Ī	2
	10
	2
Į	2
ĸ	8
	2
ľͷ	16
¯	6
	2
ص	2
ҩ	2
÷	6
	2
	4
	2
۹	4
	2
ŭ	8
	2
׷	2
׽	4
ƴд	4
ȭ	4
	2
	4
	2
·	2
	6
¯	2
ը	10
˳	2
	6
üĿ	2
	4
	6
ĥ	4
	2
	2
ȥ	6
ο	2
ĵ	4
	6
	2
	2
	4
	4
	2
	2
	2
ʱ	2
¶ȼ	8
ձ	2
	4
	4
	14
	2
	4
ȫ	2
Щ	8
ͬ	4
	2
-	10
ʿ	6
	6
˵	2
	2
	4
ǹ	4
	2
	2
ѩ	4
ز	2
	2
¹	4
ҫ	4
	2
	4
ζ	4
	2
	2
ս	2
Ӫ	2
	6
ţ	2
ʿ	2
	4
˳	12
ͬͬ	2
	4
	4
	2
ͨ	2
ɳ	2
	2
	2
	2
	6
۾	8
ʷ	2
ɡ	4
ص	2
	2
Ŵ	2
	4
һʶ	2
Ū	4
	6
Ū	2
軨	2
	4
	4
	2
	2
ϥ	2
ָͷ	2
ϣ	4
ȹ	2
	2
	4
	2
	4
	2
	2
	2
	10
˵	4
ǿ	8
¼	6
	8
ε	2
	2
ͬ	10
꾡	2
ר	2
	2
	6
ǣ	2
С	2
	2
	2
綯	2
ֻ	4
	6
񾭲	2
մ	4
	4
˼	6
ֱ˵	6
	2
ľ	2
	10
˦	2
	2
	2
	2
	2
Ӫ	4
	6
ͷ	2
ټ	2
ֻ	2
Ļ	4
κ	2
Ȥ	16
ˮ	2
ɫ	2
ƶ	2
	4
ǰ	2
	2
˰	2
ûϵ	2
Ӱ	2
	4
ʹ	2
ư	2
	6
	2
	10
	2
	2
ž	2
	2
	2
	2
	2
޷	2
â	2
ʲô	10
	8
Ŀ	2
	2
ѧ	2
	2
	2
	2
	2
	2
ȭ	4
	2
Χ	2
	4
	2
	2
п	2
ϷŪ	2
ϷԺ	4
ִ	2
	2
Һ	4
ֹ	2
͸	4
	2
	2
˶	2
	2
	2
	2
	6
ɫ	4
ǳ	4
ָ	6
	2
ʳ	2
ת	2
ƽ	2
	4
	4
	2
ţ	2
	4
θ	2
ũ	2
	2
	2
ŭ	2
	2
к	2
	2
շ	2
Ա	2
	12
Թ	2
ڵ	2
	2
ӵ	2
	4
ơ	8
ݼ	8
	8
ƿ	2
	2
	2
	2
	2
	2
	2
	2
ͻ	2
	2
	2
	2
	2
	2
	6
	2
	4
	6
ɳ	2
	2
ͻ	2
	2
ԭ	2
ʰ	2
ͻ	2
ǰ	2
·	2
	2
ǰ	2
	4
	4
Ѭ	2
ں	6
ǿʶ	2
۷	2
	4
	4
1725	2
	2
ۡ	2
	2
	2
˵	6
	2
Ծ	2
ڼԼ	2
ⵯ	2
ԭӵ	2
	2
	2
ɢ	4
ɫ	2
	2
ʩ	2
	4
	2
굶	2
ת	2
	2
	12
ʾ	8
ɢɢ	2
ָ	2
ʺ	4
ÿ	2
ѩ	2
	2
۱	6
	2
	2
Ų	2
ճ	2
Ҿ	6
ػ	2
һ	2
Ժ	4
	8
п	4
ʾ	4
	2
ָ	2
	8
	2
	2
ľ	2
д	2
ӡˢ	2
Ԥ	4
ԭ	8
	6
ʧ	2
ŵ	2
	4
	12
	2
	2
Ѿ	8
	2
Ȫˮ	2
	4
׶	2
ྲ	2
	6
ȫ	6
Ӽ	2
Ž	2
	4
	2
Ա	2
	4
޾	2
ѩ	2
	4
	2
	2
	4
	4
Ѱ˼	4
	6
	2
	4
	4
	2
ۻ	2
	4
	2
	4
ʥ	2
	6
	2
²	2
װ	2
Ⱥ	2
	6
ɢ	2
	2
õ	2
Թ	2
	2
	2
	8
ĳ	4
֤ͨ	6
	2
ʤ	2
	2
ϩ	2
ˮ	2
ʮ	4
	2
¯	2
׵	6
	4
	2
ʤ	4
շ	2
Ҫ	2
ֱ	2
ܲ	2
	6
	6
İ	2
	2
ײ	2
	2
ɱ	4
	8
û	2
Ȼ	2
	6
ϰ	2
	2
	2
淢	2
	6
	4
Ͷ	2
	2
	2
	4
	2
	4
С	2
ͧ	2
˳	2
ʻ	2
鲢	2
ѩ	2
˳	2
	6
ͤ	4
	2
	2
ƽ	2
	2
մȾ	2
ս	10
ɫ	2
ϰ볡	4
Ա	2
	2
	2
Ի	2
ɼ	2
־	8
Ұ	6
	2
	4
һ	2
	2
ʵ	2
	6
ˬ	4
Բ	2
	2
Ǻ«	2
	4
ʮ֮˾	2
	4
	2
	4
͹	2
ͳ	2
ʱ	2
	6
	2
֪	2
	2
һɲ	4
ֱ·	2
ʮȫʮ	2
	2
	10
۱	2
ҥ	2
Сƽ	2
	2
	2
Ҫ	2
	4
	8
	2
һ֪	2
	2
鹦	2
	2
Ȼ	2
ӷ	2
Ӧ	2
β	2
X-	2
	2
ױ̨	2
	2
	8
	2
	2
Ҷ	2
	8
	2
	2
	2
΢΢	2
ҡ	2
	2
	2
ѧ	2
ѧ	2
ִһ	2
Ĭ	2
դ	4
	6
Կ	14
ۻ	2
	2
õ	4
	2
ˮ	2
ˮ	8
	6
ˮ	2
	2
	2
˵	2
³	2
˵	2
	2
	2
Ĩ	2
˵	2
	2
ˬ	4
	2
ɿ	2
	2
	2
η	2
	2
Χ	2
Ұ	2
	2
Ƣ	8
ĳ	4
	6
ǫ	6
ٶ	6
	2
ˮͷ	2
Ͻ	2
	2
ִ	8
ˮ	4
	2
Ѹ	2
	2
°	2
Ҵ	2
	8
	4
ʯ	4
	6
Ͽ	2
	2
	8
һ	4
	2
	2
ĵʮ	2
	2
	2
	2
˺	6
ͻ	2
	2
	2
	2
	4
	2
ͽ	2
	2
ϲ	2
Զ	2
	2
ƾ	8
Ȼ	2
	6
	2
	6
	4
	4
	2
Ը	2
	16
ӽ	2
ο	2
鿯	2
Ϸ	2
ص	6
Ұ	2
ݸ	2
	2
1948	2
	2
	2
Ϊ	2
ϻ	2
	2
	2
	2
	4
	2
Ĵָ	2
	2
	2
ζ	4
˵˵	2
ɥ	2
	2
«	2
ζ	2
	18
	2
	2
	20
	2
	2
ƽ	2
ָ	2
ǿ׳	8
ʳ	2
	6
	2
Զ	2
Ц	2
ϳ	2
ճ	2
	2
Ӷ	2
һ	2
	2
һһ	2
	4
¶	2
и	2
	2
Ԥ	2
Ȼ	4
Ӧ	2
	2
	2
	2
	2
˯	4
˯	2
	4
仯	2
	2
	2
	4
	2
ص	2
ϧ	2
ţ	2
ܲ	6
Ӳ	4
	2
ܱ	4
	2
ȱ۶	2
	6
Զ	2
ͬ	2
ʯ	2
嶯	2
罻	2
ۻ	4
˵	2
	2
ͼ	2
	2
	2
	2
ö	2
	2
װ	2
ֱɻ	6
	8
	12
	2
	2
ݸ	2
ס	4
	4
ˤ	2
	4
	2
	2
ˮ	4
ͮͮ	2
	2
ߴ	2
	4
ѹ	6
	12
С	2
	2
ó	2
	2
ͨͨ	2
Գ	2
	2
ʮ	2
	2
	2
	2
Ʊ	2
߼	4
Ⱥ	4
ʾ	2
	4
	2
ͨ	2
滰	2
	4
˺	2
ָ	2
׼	2
ķ	2
	2
	2
	2
Խ	2
	2
׶	2
	2
	2
ûû	2
	4
	2
	2
	2
	2
	2
	2
˺	4
ҹУ	2
	2
	8
ϵ	2
	2
ѹ	2
	6
	4
ɷ	2
ټ	4
	2
	2
	2
·	6
Ա	2
	2
ǹ	2
	2
	2
	2
ĩ	2
	2
ש	8
1988	2
V	2
	2
ж	2
ϵ	4
ɽ	2
ͱ	2
ի	2
	2
ը	4
	6
Ĥ	4
м	2
	2
ͻ	2
	2
䲻	2
ؾ	2
λ	2
	2
ŭ	2
ͷ	2
	2
	2
Ǳ	2
	2
ʤ	2
	2
ɨ	2
ɽˮ	4
򽻵	2
	2
1973	2
ʧ	2
ʰ	4
̻Ҹ	2
	2
	2
ڷ	2
ӡ	2
޹ؽҪ	8
ٶ	2
	4
	2
	4
Ĭ	2
С	2
	2
ε	2
	2
թ	2
	2
ѧ	8
ݳ	2
Ƿ	2
	2
ʸ	2
	2
ò	2
	2
	8
ʶ	4
༭	2
³	6
	2
	2
	2
߸	2
ʷѧ	2
	2
ͷ	2
	2
	2
ó	8
բ	2
ǰ	2
	4
	2
ͷͷǵ	2
а	2
	2
ͷ	4
˲	2
	4
	2
	4
̸	2
ŵ	2
	2
ߵ	2
ϵ	2
͵	4
	2
ʵ	2
ϵ	6
ס	2
	2
ë	2
¶	2
	2
̿	2
	4
Щ	4
	2
з	2
	2
ʩ	2
ҹɫ	2
Ƭ	4
ز	4
	2
ջ	2
	2
˯	2
	2
ܸ	4
	2
	2
	2
	18
Ա	2
	2
	2
	6
	2
ǲ	2
	2
	4
ϴ	2
	4
	4
ݫ	2
	2
	2
	2
	2
	2
ӢĶ	2
	2
	2
	4
	6
	2
	2
߷	2
ȥ	4
	2
	4
ܾ	2
	2
	4
	6
Ϣ	2
п	2
ĩ	2
	2
ũ	2
´	2
	2
ѩ	2
ưɼ	2
þ	4
ֱ	2
	2
ȱ	2
	2
	2
	2
	2
˿֮·	2
;	6
	2
ŭ	2
չ	4
	2
	6
	2
ר	4
ѧ	2
	2
Ա	2
	2
	2
	2
һ	2
ӡ	2
游	4
	4
ʧ	2
Ԥϰ	2
	4
	2
	4
൱	4
＾	4
ʹ	2
	2
	8
	4
	2
	2
	2
	2
	2
	2
	2
	4
ڲ	2
ΪЦ	2
Ϸ	2
۲	2
ŵ	4
	2
	2
Ƥ	2
¹	2
	2
	2
˼	6
	2
	2
	2
԰	4
ס	2
Ⱥ	2
Ȧ	2
ͨ	2
	2
	2
	2
	2
	2
²	2
ζ	4
	2
θ	2
ѵ	2
	2
	2
ͻͻ	2
	2
˵ùȥ	2
߻	2
ͷ	2
	2
	2
	2
Խ	4
鱦	2
ָ	2
	2
	2
滭	2
ô	2
Ƭ	2
Ƭ	2
	2
	4
ʵ	2
΢	2
豭	2
ڿ	2
	2
	2
͸	2
	2
ˢ	6
ȹ	2
	2
к	2
	2
Ĺ	4
ʯ	8
Ī	2
	2
ǽͷ	2
	2
	2
	2
	2
	2
ȻȻ	2
	2
	2
	2
ֶն	2
	2
	2
	2
ʧ	2
	2
Ű	2
˯	2
	2
	4
	2
	2
	4
ɽ	2
ͷ	2
˥	2
̸	2
Ω	2
ų	2
ѹ	2
	2
	2
һ	2
	4
	4
ɵ	2
	2
	2
˫	2
ˡ	2
	2
	2
ɤ	4
Χ	4
ڳ	2
1978	2
	4
	2
	2
	4
	2
	2
	2
	2
Ƿ	2
	2
ǧİ	2
ƫ	2
װ	2
ص	2
ƽ	2
ץ	2
Է	2
ͣ	8
	2
	2
̨	2
	2
	2
ɫ	2
ס	2
	2
Ӱ	2
	2
	2
	2
̻	2
Ż	2
ϴˢ	2
ȱ	2
һ	2
͸	2
ʹ	2
	2
᰸	8
	4
ѹ	2
	2
	4
ʱ	6
է	4
Ƶ	2
ɫ	6
	2
	2
ɢʧ	2
ɺ	2
ͬ	4
	2
ͨ	2
	2
	2
˿	2
ʵ	2
	2
	2
ϲ	2
	2
	2
ͨ	2
	4
ͬ	2
	6
	2
Բ	2
	2
	2
	2
	2
߹	2
	2
	2
	2
	4
ѩ׷	2
	2
㶹	6
	4
˪	2
һ	10
	4
Ұ	2
	2
	4
ֵ	2
μ	2
˿	2
	2
	2
Ѷ	2
Ψ	2
	2
	2
ָ	2
ݻ	4
	6
ӳ	2
ʩ	2
ҹԼ	2
	2
	4
	2
	8
	2
д	2
ɰͰ	2
	2
嵭	4
Ͳ	2
ϴ	2
	2
	2
	8
	6
ͷβ	2
Ż	2
	2
ҹ	2
ٳ	2
	2
Ħ	2
	4
	2
	2
	2
ñ	2
	2
	2
	2
	4
Ӱ	6
	6
Ƥ	2
Ѷ	2
	2
游ĸ	2
ά	2
Ƭ	2
	6
	2
	2
	2
	2
	2
ĳ	4
绰	2
	4
۸	2
	2
	2
˵˵	2
ѩ	2
ѩ	2
ǰ	2
	2
	2
ͷ	2
	2
ۺɫ	4
	2
	2
̴	2
ҹ	2
	2
	2
շ	2
	2
ƽ	2
	2
ü	2
	2
	4
	2
;߹	2
	2
ʽ	2
	2
	2
΢	2
	4
	2
Ϻ	2
	2
	2
Ѱ	2
	2
	4
	2
	2
	2
Ӫ	2
Ʋ	2
	2
1989	2
ٳ	2
ķ	2
ţ	2
	2
	2
1961	2
ͨ	4
ñ	2
	2
·	4
ѧ	12
	2
	2
ռ	2
Ȩ	2
	2
	2
ī	2
	2
֪	2
	2
չ	6
δ	2
	2
а	2
׻׻̵	2
	2
ͷ	2
	2
	2
Ƭ	2
	2
ø	2
	2
Ǯ	2
	2
ڻ	2
	2
	2
Ͷ	2
ȷ	4
ж	2
綯	2
ˮ	4
޷	2
伢	2
ʮŵ	2
ʮһ	2
ѧ	2
	4
	2
Զ	14
׿	2
Ͷ	2
Ӱ	2
	4
1966	2
	4
	2
	2
	2
	8
ʤ	2
	2
ʿɾ	2
	2
Ӿ	2
	2
	2
	2
1987	4
ʮ	2
	2
ɽ	2
˯	2
̲	4
	4
˵	4
	2
Ӣ	2
ǵ	2
	2
ս	2
Ȼ	2
׽	2
֯	2
	4
ľż	2
֩	6
ܶ	2
ҹ	2
ǡ	2
	2
	2
	2
л	2
	2
1963	2
	2
	2
	2
ǵ	2
Ϊ	2
̫̫	2
ݻ	2
ͣ	4
ŵ	2
©	2
˫ɹ	2
	2
޿	2
	2
	2
	2
	2
	4
	2
	2
Ա	2
¼	2
	4
	2
	2
	6
	2
ǰ	2
	4
װ	2
߽	4
	2
1964	2
	2
1939	2
	2
Ӣ	2
	2
չ	4
	2
һ	2
Ƭ	2
	2
¯	2
	2
	4
ϵ	2
߸	2
129	2
ɽ	2
	2
	2
	2
绰	2
11	2
	2
	2
	2
	2
Ⱥ	2
һͿ	2
̸	2
	2
	4
η	4
֪	2
	2
ڿն	2
	2
û	2
	2
ϡ	2
	2
ص	2
	2
	6
Ҷ	2
Ϫˮ	2
ϯ	2
ٱ	2
ϵͳ	2
	4
	2
̵	2
ĩ	2
	2
ڶ	6
ʮ	2
ϰ	2
Ӫ	2
ʮ	2
ˮ	2
۵	2
	2
	2
	2
	2
	2
	2
	2
	2
Ϊʱ	2
	2
	2
Ӱ	4
	6
ѹ	2
	2
	4
͸©	2
	2
ε	2
С	2
	2
Сֶ	2
	2
ɿ	2
202635	2
յ	2
	2
׷	2
	2
	2
һ	2
ˣ	4
	2
	2
Ƥ	2
	2
	2
۶	2
Ť	2
	2
ʡ	2
β	2
ס	2
ǰ	2
	2
СϪ	2
	2
	2
	2
	2
ľ	2
Сׯ	2
	4
д̨	2
Ͳ	2
̨	2
Ŀ	2
¾	2
¥	2
	2
	2
	2
ʱ	2
ļ	2
Ԥ	2
	2
	2
20:	2
	2
һ	2
	2
	2
Ϊ	2
޶	2
	2
	2
	2
ҩ	2
ᷳ	2
Ȼŭ	2
Ϸ	2
һѷ	2
	2
	2
εȻɷ	2
ʶ	2
	2
ż	2
ٲ	2
մ	4
Ѫծ	2
Ѫ	2
Υ	2
ҡҡλ	2
Ѽ	2
	2
Խ	2
	2
Ŀ	2
	2
ۿ	2
	2
	2
	2
	2
	2
	2
Ҷ	2
ˢˢ	2
	2
	2
ʱ	2
	4
ƽװ	2
	4
Ч	4
޲	2
ѩ	2
ײ	2
	2
ȫ	2
	2
ʧ	2
¿๦	2
	2
	2
	2
ҹ˾	2
	2
һ˵	2
ͷ	2
к	2
	2
	2
	2
ֽ	2
	2
	4
ʶ	4
	2
	2
	4
С	4
	2
ȫע	2
һ	4
ָ	2
Ȼ	2
ʱ	2
ʡ	2
ҡҷ	2
Ϸ	2
	2
1789	2
302	2
	2
	2
ը	2
ػ	2
	2
̲	2
	2
	2
鹫	2
ͣ	2
	2
	2
	2
	2
һ	2
	4
	2
µ	2
ΰ	2
ب	2
׹	2
	2
	2
	4
	2
	2
	2
	2
	2
	2
һ	2
	2
	2
ũׯ	4
	2
Ӧ	10
Ʒ	2
꼾	4
ܳ	2
	2
ˮ·	2
	2
κ	2
ʵԱ	2
	2
ɾ	2
	2
	2
	4
ĥ	2
	2
	2
첹	2
	2
ѧ	2
	2
	2
洬	2
ƺ	2
ô	2
	2
	2
	2
	2
	2
Ĵ	2
	2
	4
	2
	2
	2
	2
ԭӺ	2
	2
ѹѹ	2
	2
¡	4
ã	2
Զɽ	2
	2
յ	2
ˢ	2
ȫʳ	2
ɽ	2
	2
ڻ	2
1990	4
԰	6
	4
ƹ	6
ӳ	2
	2
	2
	2
O	2
H	2
	2
ֻ	2
͵	2
ʡ	2
	2
	4
	2
	4
	2
Ѻ	2
ʷ	2
δй	2
༶	2
	2
Ϣ	2
ɿ	2
ˮ	2
ѡȨ	2
ѡȨ	2
ݺ	2
ӡ	2
Ѫ	4
ѪҺ	4
	4
	2
ʱ	2
	2
Ӳֽ	2
ͼ	2
Ķ	2
	2
	2
	6
Ƥ	2
ȱ	2
	2
ë֮	2
ըҩ	2
¡	2
ɽʯ	2
	2
ճ	2
	2
	8
о	2
չ	2
ʮߵ	2
	2
	2
	2
ϥ	2
һ	2
˯	2
	2
	2
	4
	2
ī	2
	4
	2
	2
	2
«έ	2
	2
ʼ	2
	2
󲻹	2
	2
	4
	2
Ǧ	2
¼	2
С˵	2
һǧ	2
׾	2
ϣ	2
	2
˫	2
ͷ	2
ٶ	4
	2
˳	2
Ӳͦ	2
˼	2
ê	2
ٰ	2
һһʮ	2
깤	2
ˬ	2
ô	4
	2
	2
¶	2
	2
	2
	2
˫̥	2
洬	2
ν	4
	2
	4
	2
㺯	2
	2
	2
	2
	2
ߴ	2
ʵ	2
	2
	2
ë	2
ľ	2
ͩ	2
	2
	2
	2
Ѫ	2
	2
ͬ	2
вͨ	6
ĸ	2
ʡȥ	2
	2
¿	2
	2
	4
׳	2
	2
ûϢ	2
	2
С	2
	2
	2
	2
ʧ	2
һ	2
崫	2
	2
Ӿ	6
ǹ	2
	2
	2
	2
ˮ	2
	2
ʮþ	2
Ƥ	4
ǻ	2
ˮ	2
۰	2
׷	2
ת	2
	2
	2
	2
	2
첻	2
	2
	2
	2
	2
	2
	2
	2
Ʋ	2
	2
	2
	4
ϸ	2
Ҳ	2
	2
Ӳ	2
й	2
	2
ɽˮ	2
Ȼ	2
	2
©	2
	2
	2
	2
	2
Ѱ	2
	2
	2
	2
	4
ư	2
	2
	2
	2
ϲ	2
	2
չվ	2
	2
װ	2
	2
ζ	2
	2
ɫ	2
Ƭ	2
	2
	2
ƻ	2
	2
	2
	2
	2
	2
	2
	2
ʱ	2
	2
ǧ	4
ͷ	2
	2
	2
ۿ	2
Ƭ	4
	2
ͼ	2
	2
	2
	2
˿֯	2
ıɱ	2
	2
	2
ʧ	2
	2
	2
	2
	4
ˢ	4
ˮ	2
	2
	2
	2
в׼	2
	2
	2
򵥾	2
ͷ	2
ҩζ	2
10000	2
֮	2
ԭ	2
汾	2
	2
˰	2
	2
	2
	2
	2
	4
ª	2
8:	2
	2
ˮ	2
ʢ	2
Ϻ	2
ͷ	2
ͨ	2
С	2
	2
	2
	2
	2
һ	2
ע	2
ͷ	2
	2
	2
	2
ס	2
ҩ	2
ű	2
	2
	2
мһ	2
Ʒ	2
Ǯ	2
巳	2
Ķ	2
ֳ	2
	2
ҩ	2
	2
ٸƶ	2
	2
ŵù	2
еͨ	2
ʧ	2
Ҫ	2
Ч	2
	2
ƽ	4
ҩ	2
ʧЧ	4
	2
ȶ	2
Ҫ	2
PARSIFAL	2
﷨	2
	2
	2
һ	2
	2
Ц	2
	2
	2
ʮ	2
Ʊ	2
ƽ	2
	2
˫	2
д	2
	2
˿	2
	2
	2
ػ	2
	2
ɫ	2
ڴ	2
	2
֭	2
ͷ	2
˷	2
	2
	2
	2
ײ	2
ɽ	2
ɨ	2
չ	2
	2
й	2
	2
д	2
Ҫ	2
ָ	2
	2
ͷ	2
	2
	2
Ȼ	2
û	2
ֽϻ	2
֤	2
Ǻ	2
	2
	2
ֹ	2
ǽ	2
ڷҩ	2
	2
޻˵	2
챦ʯ	2
԰ټ	2
	2
1970	2
	2
ӵ	2
õ廨	2
	2
	2
ũҩ	2
ѹ	2
΢	2
	2
	2
	2
ɾȥ	2
	2
˾	2
ѡ	2
ٲʢ	4
ų	2
Ȭ	2
һ·ƽ	2
	2
	2
	2
ǰ	2
ǹ	2
ˮƿ	2
	2
ʱ̱	2
	2
Ȼ	2
	2
ܵ˵	2
Ȼ	2
ں	2
ս	2
ɽ	2
	2
	2
¼	2
	2
	2
ʧӭ	2
	2
	2
˳	2
ף	2
˿	2
	2
	2
	2
	2
	2
ϰ	2
	2
