package Linux::Info::DiskUsage;
$Linux::Info::DiskUsage::VERSION = '0.5';
use strict;
use warnings;
use Carp qw(croak);

=head1 NAME

Linux::Info::DiskUsage - Collect linux disk usage.

=head1 SYNOPSIS

    use Linux::Info::DiskUsage;

    my $lxs  = new Linux::Info::DiskUsage;
    my $stat = $lxs->get;

=head1 DESCRIPTION

Linux::Info::DiskUsage gathers the disk usage with the command C<df>.

For more information read the documentation of the front-end module L<Linux::Info>.

=head1 DISK USAGE INFORMATIONS

Generated by F</bin/df -kP>.

    total       -  The total size of the disk.
    usage       -  The used disk space in kilobytes.
    free        -  The free disk space in kilobytes.
    usageper    -  The used disk space in percent.
    mountpoint  -  The moint point of the disk.

=head2 GLOBAL VARS

If you want to change the path or arguments for C<df> you can use the following
variables...

    $Linux::Info::DiskUsage::DF_PATH = '/bin';
    $Linux::Info::DiskUsage::DF_CMD  = 'df -akP';

Example:

    use Linux::Info;
    use Linux::Info::DiskUsage;
    $Linux::Info::DiskUsage::DF_CMD = 'df -akP';

    my $sys  = Linux::Info->new(diskusage => 1);
    my $disk = $sys->get;

=head1 METHODS

=head2 new()

Call C<new()> to create a new object.

    my $lxs = Linux::Info::DiskUsage->new;

It's possible to set the path to df.

     Linux::Info::DiskUsage->new(
        cmd => {
            # This is the default
            path => '/bin',
            df   => 'df -kP 2>/dev/null',
        }
    );

=head2 get()

Call C<get()> to get the statistics. C<get()> returns the statistics as a hash reference.

    my $stat = $lxs->get;

=head1 EXPORTS

Nothing.

=head1 SEE ALSO

=over

=item *

B<proc(5)>

=item *

L<Linux::Info>

=back

=head1 AUTHOR

Alceu Rodrigues de Freitas Junior, E<lt>arfreitas@cpan.orgE<gt>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2015 of Alceu Rodrigues de Freitas Junior, E<lt>arfreitas@cpan.orgE<gt>

This file is part of Linux Info project.

Linux-Info is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Linux-Info is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Linux Info.  If not, see <http://www.gnu.org/licenses/>.

=cut

our $DF_PATH = undef;
our $DF_CMD  = undef;

sub new {
    my $class = shift;
    my $opts = ref( $_[0] ) ? shift : {@_};

    my %self = (
        cmd => {
            path => '/bin',
            df   => 'df -kP 2>/dev/null',
        }
    );

    foreach my $p ( keys %{ $opts->{cmd} } ) {
        $self{cmd}{$p} = $opts->{cmd}->{$p};
    }

    return bless \%self, $class;
}

sub get {
    my $self   = shift;
    my $class  = ref $self;
    my $cmd    = $self->{cmd};
    my $df_cmd = $DF_CMD || $cmd->{df};
    my (%disk_usage);

    local $ENV{PATH} = $DF_PATH || $cmd->{path};
    open my $fh, "$df_cmd|" or croak "$class: unable to execute '$df_cmd' ($!)";

    # filter the header
    { my $null = <$fh>; }

    while ( my $line = <$fh> ) {
        next unless $line =~ /^(.+?)\s+(.+)$/;

        @{ $disk_usage{$1} }{
            qw(
              total
              usage
              free
              usageper
              mountpoint
              )
        } = ( split /\s+/, $2 )[ 0 .. 4 ];

        $disk_usage{$1}{usageper} =~ s/%//;
    }

    close($fh);
    return \%disk_usage;
}

1;
