#!/usr/bin/perl -w                                         # -*- perl -*-
# $Id: 10-features.t 4093 2009-02-24 22:07:12Z andrew $

use strict;
use Cwd qw(abs_path);
use FindBin qw($Bin);

use lib ($Bin, "$Bin/../lib");
use Pod::POM;
use Pod::POM::View::DocBook;
use XML::LibXML;

use Data::Dumper;

use Test::More;
use Test::Differences;

use TestUtils;


my @tests = get_tests(\*DATA);

plan tests => 2 * int @tests;

my $pod_parser = Pod::POM->new();
my $xml_parser = XML::LibXML->new();

foreach my $test (@tests) {
    my $view = 'Pod::POM::View::DocBook';
    if (keys %{$test->params}) {
        $view = $view->new(%{$test->params});
    }

    my $pom    = $pod_parser->parse_text($test->input);
    my $result = $pom->present($view);
    $result =~ s{(<!-- Generated by Pod::POM::View::DocBook) \d+\.\d+ (using Pod:::POM) \d+\.\d+ (-->)}
    {$1 X.XX $2 Y.YY $3};


    eq_or_diff("$result", $test->expect, $test->description);

    eval { 
        my $doc = $xml_parser->parse_string($result);
    };
    is($@, "", "parsed output of " . $test->description);
}


__DATA__

== TEST ==

desc: basic test

=head1 NAME

Basic Test

=head1 DESCRIPTION

This is just a basic test.

=cut

-- expect --

<?xml version='1.0'?>
<!DOCTYPE article PUBLIC
    "-//OASIS//DTD DocBook XML V4.5//EN"
    "http://www.oasis-open.org/docbook/xml/4.5/docbookx.dtd">
<!-- Generated by Pod::POM::View::DocBook X.XX using Pod:::POM Y.YY -->
<article>
<title>Basic Test</title>

<sect1>
<title>Description</title>

<para>This is just a basic test.</para>

</sect1>

</article>

== TEST ==

desc: handling of entities

=head1 NAME

Entity & Test

=head1 DESCRIPTION

This is a test of entity handling.

=cut
-- expect --
<?xml version='1.0'?>
<!DOCTYPE article PUBLIC
    "-//OASIS//DTD DocBook XML V4.5//EN"
    "http://www.oasis-open.org/docbook/xml/4.5/docbookx.dtd">
<!-- Generated by Pod::POM::View::DocBook X.XX using Pod:::POM Y.YY -->
<article>
<title>Entity &amp; Test</title>

<sect1>
<title>Description</title>

<para>This is a test of entity handling.</para>

</sect1>

</article>

== TEST ==

desc: itemized lists

=head1 NAME

Itemized List Test

=head1 DESCRIPTION

This is a test of itemized list handling.

=over 4

=item *

item 1

=item *

item 2

=back

=cut
-- expect --
<?xml version='1.0'?>
<!DOCTYPE article PUBLIC
    "-//OASIS//DTD DocBook XML V4.5//EN"
    "http://www.oasis-open.org/docbook/xml/4.5/docbookx.dtd">
<!-- Generated by Pod::POM::View::DocBook X.XX using Pod:::POM Y.YY -->
<article>
<title>Itemized List Test</title>

<sect1>
<title>Description</title>

<para>This is a test of itemized list handling.</para>

<itemizedlist>
<listitem>
<para>item 1</para>
</listitem>
<listitem>
<para>item 2</para>
</listitem>
</itemizedlist>

</sect1>

</article>

== TEST ==

desc: ordered lists

=head1 NAME

Ordered List Test

=head1 DESCRIPTION

This is a test of ordered list handling.

=over 4

=item 1

item 1

=item 2

item 2

=back

=cut
-- expect --
<?xml version='1.0'?>
<!DOCTYPE article PUBLIC
    "-//OASIS//DTD DocBook XML V4.5//EN"
    "http://www.oasis-open.org/docbook/xml/4.5/docbookx.dtd">
<!-- Generated by Pod::POM::View::DocBook X.XX using Pod:::POM Y.YY -->
<article>
<title>Ordered List Test</title>

<sect1>
<title>Description</title>

<para>This is a test of ordered list handling.</para>

<orderedlist>
<listitem>
<para>item 1</para>
</listitem>
<listitem>
<para>item 2</para>
</listitem>
</orderedlist>

</sect1>

</article>

== TEST ==

desc: bold and italic

=head1 NAME

Bold/Italic Test

=head1 DESCRIPTION

This is a test of B<bold> and I<italic> handling.

=cut
-- expect --
<?xml version='1.0'?>
<!DOCTYPE article PUBLIC
    "-//OASIS//DTD DocBook XML V4.5//EN"
    "http://www.oasis-open.org/docbook/xml/4.5/docbookx.dtd">
<!-- Generated by Pod::POM::View::DocBook X.XX using Pod:::POM Y.YY -->
<article>
<title>Bold/Italic Test</title>

<sect1>
<title>Description</title>

<para>This is a test of <emphasis role='strong'>bold</emphasis> and <emphasis>italic</emphasis> handling.</para>

</sect1>

</article>

== TEST ==

desc: verbatim text

=head1 NAME

Verbatim Test

=head1 DESCRIPTION

This is a test of verbatim text handling.

    This is verbatim text.
    This is more verbatim text.

That was it.

=cut
-- expect --
<?xml version='1.0'?>
<!DOCTYPE article PUBLIC
    "-//OASIS//DTD DocBook XML V4.5//EN"
    "http://www.oasis-open.org/docbook/xml/4.5/docbookx.dtd">
<!-- Generated by Pod::POM::View::DocBook X.XX using Pod:::POM Y.YY -->
<article>
<title>Verbatim Test</title>

<sect1>
<title>Description</title>

<para>This is a test of verbatim text handling.</para>

<verbatim><[CDATA[    This is verbatim text.
    This is more verbatim text.]]></verbatim>

<para>That was it.</para>

</sect1>

</article>

== TEST ==

desc: casing of section titles

=head1 NAME

Basic Test

=head1 EXAMPLE OF A CASED TITLE

This is just a basic test.

=cut

-- params --
root = book

-- expect --

<?xml version='1.0'?>
<!DOCTYPE book PUBLIC
    "-//OASIS//DTD DocBook XML V4.5//EN"
    "http://www.oasis-open.org/docbook/xml/4.5/docbookx.dtd">
<!-- Generated by Pod::POM::View::DocBook X.XX using Pod:::POM Y.YY -->
<book>
<title>Basic Test</title>

<chapter>
<title>Example of a Cased Title</title>

<para>This is just a basic test.</para>

</chapter>

</book>

== TEST ==

desc: parameter test (root='book')

=head1 NAME

Basic Test

=head1 DESCRIPTION

This is just a basic test.

=cut

-- params --
root = book

-- expect --

<?xml version='1.0'?>
<!DOCTYPE book PUBLIC
    "-//OASIS//DTD DocBook XML V4.5//EN"
    "http://www.oasis-open.org/docbook/xml/4.5/docbookx.dtd">
<!-- Generated by Pod::POM::View::DocBook X.XX using Pod:::POM Y.YY -->
<book>
<title>Basic Test</title>

<chapter>
<title>Description</title>

<para>This is just a basic test.</para>

</chapter>

</book>

-- stop --
