#!/usr/bin/perl -I/home/phil/perl/cpan/DataTableText/lib/Data/Table/Text.pm
#-------------------------------------------------------------------------------
# Tree operations
# Philip R Brenan at gmail dot com, Appa Apps Ltd Inc., 2020
#-------------------------------------------------------------------------------
# podDocumentation
package Tree::Ops;
our $VERSION = 20200701;
require v5.26;
use warnings FATAL => qw(all);
use strict;
use Carp qw(confess cluck);
use Data::Dump qw(dump);
use Data::Table::Text qw(genHash);
use feature qw(say current_sub);

#D1 Build                                                                       # Create a tree.

sub new($)                                                                      #S Create a new child recording the specified user data.
 {my ($user) = @_;                                                              # User data to be recorded in the child
  genHash(__PACKAGE__,                                                          # Child in the tree.
    children   => [],                                                           # Children of this child.
    user       => $user,                                                        # User data for this child.
    parent     => undef,                                                        # Parent for this child.
    lastChild  => undef,                                                        # Last active child chain - enables us to find the currently open scope from the start if the tree.
   );
 }

sub activeScope($)                                                              #P Locate the active scope in a tree.
 {my ($tree) = @_;                                                              # Tree
  my $active;                                                                   # Latest active child
  for(my $l = $tree; $l; $l = $l->lastChild) {$active = $l}                     # Skip down edge of parse tree to deepest active child.
  $active
 }

sub setParentOfChild($$)                                                        #P Set the parent of a child and return the child.
 {my ($child, $parent) = @_;                                                    # Child, parent
  $child->parent = $parent;                                                     # Parent child
  $child
 }

sub open($$)                                                                    # Add a child and make it the currently active scope into which new children will be added.
 {my ($tree, $user) = @_;                                                       # Tree, user data to be recorded in the interior child being opened
  my $parent = activeScope $tree;                                               # Active parent
  my $child  = new $user;                                                       # New child
  push $parent->children->@*, $child;                                           # Place new child last under parent
  $parent->lastChild = $child;                                                  # Make child active
  setParentOfChild $child, $parent                                              # Parent child
 }

sub close($)                                                                    # Close the current scope returning to the previous scope.
 {my ($tree) = @_;                                                              # Tree
  my $parent = activeScope $tree;                                               # Locate active scope
  delete $parent->parent->{lastChild};                                          # Close scope
  $parent
 }

sub single($$)                                                                  # Add one child in the current scope.
 {my ($tree, $user) = @_;                                                       # Tree, user data to be recorded in the child being created
  $tree->open($user);                                                           # Open scope
  $tree->close;                                                                 # Close scope immediately
 }

#D1 Navigation                                                                  # Navigate through the tree.

sub first($)                                                                    # Get the first child under the specified parent.
 {my ($parent) = @_;                                                            # Parent
  $parent->children->[0]
 }

sub last($)                                                                     # Get the last child under the specified parent.
 {my ($parent) = @_;                                                            # Parent
  $parent->children->[-1]
 }

sub indexOfChildInParent($)                                                     #P Get the index of a child within the specified parent.
 {my ($child) = @_;                                                             # Child
  return undef unless my $parent = $child->parent;                              # Parent
  my $c = $parent->children;                                                    # Siblings
  for(0..$#$c) {return $_ if $$c[$_] == $child}                                 # Locate child and return index
  confess 'Child not found in parent'
 }

sub next($)                                                                     # Get the next sibling following the specified child.
 {my ($child) = @_;                                                             # Child
  return undef unless my $parent = $child->parent;                              # Parent
  my $c = $parent->children;                                                    # Siblings
  return undef if @$c == 0 or $$c[-1] == $child;                                # No next child
  $$c[+1 + indexOfChildInParent $child]                                         # Next child
 }

sub prev($)                                                                     # Get the previous sibling of the specified child.
 {my ($child) = @_;                                                             # Child
  return undef unless my $parent = $child->parent;                              # Parent
  my $c = $parent->children;                                                    # Siblings
  return undef if @$c == 0 or $$c[0] == $child;                                 # No previous child
  $$c[-1 + indexOfChildInParent $child]                                         # Previous child
 }

sub firstMost($)                                                                # Return the first most descendant child in the tree starting at this parent or else return B<undef> if this parent has no children.
 {my ($parent) = @_;                                                            # Child
  my $f;
  for(my $p = $parent; $p; $p = $p->first) {$f = $p}                            # Go first most
  $f
 }

sub lastMost($)                                                                 # Return the last most descendant child in the tree starting at this parent or else return B<undef> if this parent has no children.
 {my ($parent) = @_;                                                            # Child
  my $f;
  for(my $p = $parent; $p; $p = $p->last) {$f = $p}                             # Go last most
  $f
 }

#D1 Location                                                                    # Verify the current location.

sub context($)                                                                  # Get the context of the current child.
 {my ($child) = @_;                                                             # Child
  my @c;                                                                        # Context
  for(my $c = $child; $c; $c = $c->parent) {push @c, $c}                        # Walk up
  @c
 }

sub isFirst($)                                                                  # Return the specified child if that child is first under its parent, else return B<undef>.
 {my ($child) = @_;                                                             # Child
  return undef unless my $parent = $child->parent;                              # Parent
  $parent->children->[0] == $child ? $child : undef                             # There will be at least one child
 }

sub isLast($)                                                                   # Return the specified child if that child is last under its parent, else return B<undef>.
 {my ($child) = @_;                                                             # Child
  return undef unless my $parent = $child->parent;                              # Parent
  my $c = $parent->children;
  $parent->children->[-1] == $child ? $child : undef                            # There will be at least one child
 }

sub singleChildOfParent($)                                                      # Return the only child of this parent if the parent has an only child, else B<undef>
 {my ($parent) = @_;                                                            # Parent
  $parent->children->@* == 1 ? $parent->children->[0] : undef                   # Return only child if it exists
 }

#D1 Put                                                                         # Insert children into a tree.

sub putFirst($$)                                                                # Place a new child first under the specified parent and return the child.
 {my ($parent, $child) = @_;                                                    # Parent, child
  unshift $parent->children->@*, $child;                                        # Place child
  setParentOfChild $child, $parent                                              # Parent child
 }

sub putLast($$)                                                                 # Place a new child last under the specified parent and return the child.
 {my ($parent, $child) = @_;                                                    # Parent, child
  push $parent->children->@*, $child;                                           # Place child
  setParentOfChild $child, $parent                                              # Parent child
 }

sub putNext($$)                                                                 # Place a new child after the specified child.
 {my ($child, $new) = @_;                                                       # Existing child, new child
  return undef unless defined(my $i = indexOfChildInParent $child);             # Locate child within parent
  splice $child->parent->children->@*, $i, 1, $child, $new;                     # Place new child
  setParentOfChild $new, $child->parent                                         # Parent child
 }

sub putPrev($$)                                                                 # Place a new child before the specified child.
 {my ($child, $new) = @_;                                                       # Child, new child
  return undef unless defined(my $i = indexOfChildInParent($child));            # Locate child within parent
  splice $child->parent->children->@*, $i, 1, $new, $child;                     # Place new child
  setParentOfChild $new, $child->parent                                         # Parent child
 }

#D1 Steps                                                                       # Move the start or end of a scope forwards or backwards as suggested by Alex Monroe.

sub step($)                                                                     # Make the first child of the specified parent the parents previous sibling and return the parent. In effect this moves the start of the parent one step forwards.
 {my ($parent) = @_;                                                            # Parent
  return undef unless my $f = $parent->first;                                   # First child
  putPrev $parent, cut $f;                                                      # Place first child
  $parent
 }

sub stepEnd($)                                                                  # Make the next sibling of the specified parent the parents last child and return the parent. In effect this moves the end of the parent one step forwards.
 {my ($parent) = @_;                                                            # Parent
  return undef unless my $n = $parent->next;                                    # Next sibling
  putLast $parent, cut $n;                                                      # Place next sibling as first child
  $parent
 }

sub stepBack                                                                    # Make the previous sibling of the specified parent the parents first child and return the parent. In effect this moves the start of the parent one step backwards.
 {my ($parent) = @_;                                                            # Parent
  return undef unless my $p = $parent->prev;                                    # Previous sibling
  putFirst $parent, cut $p;                                                     # Place previous sibling as first child
  $parent
 }

sub stepEndBack                                                                 # Make the last child of the specified parent the parents next sibling and return the parent. In effect this moves the end of the parent one step backwards.
 {my ($parent) = @_;                                                            # Parent
  return undef unless my $l = $parent->last;                                    # Last child sibling
  putNext $parent, cut $l;                                                      # Place last child as first sibling
  $parent
 }

#D1 Edit                                                                        # Edit a tree in situ.

sub cut($)                                                                      # Cut out a child and all its content and children, return it ready for reinsertion else where.
 {my ($child) = @_;                                                             # Child
  splice $child->parent->children->@*, indexOfChildInParent($child), 1;          # Remove child
  $child
 }

sub dup($)                                                                      # Duplicate a parent and all its descendants.
 {my ($parent) = @_;                                                            # Parent

  sub                                                                           # Duplicate a child
   {my ($old)  = @_;                                                            # Existing child
    my $new    = new $old->user;                                                # New child
    push $new->children->@*, __SUB__->($_) for $old->children->@*;              # Duplicate children of child
    $new
   }->($parent)                                                                 # Start duplication at parent
 }

sub unwrap($)                                                                   # Unwrap the specified child and return that child.
 {my ($child) = @_;                                                             # Child
  return undef unless defined(my $i = indexOfChildInParent $child);             # Locate child within parent
  my $parent = $child->parent;                                                  # Parent
  $_->parent = $parent for $child->children->@*;                                # Reparent unwrapped children of child
  delete $child ->{parent};                                                     # Remove parent of unwrapped child
  splice $parent->children->@*, $i, 1, $child->children->@*;                    # Remove child
  $parent
 }

sub wrap($$)                                                                    # Wrap the specified child with a new parent and return the new parent.
 {my ($child, $new) = @_;                                                       # Child to wrap, new wrapping parent
  return undef unless defined(my $i = indexOfChildInParent $child);             # Locate child within existing parent
  my $parent     = $child->parent;                                              # Existing parent
  $new->parent   = $parent;                                                     # Parent new parent
  $new->children = [$child];                                                    # Set children for new parent
  splice $parent->children->@*, $i, 1, $new;                                    # Place new parent in existing parent
  $child->parent = $new                                                         # Reparent child to new parent
 }

#D1 Traverse                                                                    # Traverse the tree.

sub by($$)                                                                      # Traverse a tree in order to process each child and return an array of the results of processing each child.
 {my ($tree, $sub) = @_;                                                        # Tree, method to process a child
  my @r;                                                                        # Results

  sub                                                                           # Traverse
   {my ($child) = @_;                                                           # Child
    __SUB__->($_) for $child->children->@*;                                     # Children of child
    push @r, &$sub($child);                                                     # Process child saving result
   }->($tree);                                                                  # Start at root of tree

  @r
 }

sub select($$)                                                                  # Select matching children in a tree. A child can be selected via named value, array of values, a hash of values, a regular expression or a sub reference.
 {my ($tree, $select) = @_;                                                     # Tree, method to select a child
  my $ref = ref $select;                                                        # Selector type
  my $sel =                                                                     # Selection method
             $ref =~ m(array)i ? sub{grep{$_[0] eq $_} @$select} :              # Array
             $ref =~ m(hash)i  ? sub{$$select{$_[0]}}            :              # Hash
             $ref =~ m(exp)i   ? sub{$_[0] =~ m($select)}        :              # Regular expression
             $ref =~ m(code)i  ? sub{&$select($_[0])}            :              # Sub
                                 sub{$_[0] eq $select};                         # Scalar
  my @s;                                                                        # Selection

  sub                                                                           # Traverse
   {my ($child) = @_;                                                           # Child
    push @s, $child if &$sel($child->user);                                     # Select child if it matches
    __SUB__->($_) for $child->children->@*;                                     # Each child
   }->($tree);                                                                  # Start at root

  @s
 }

#D1 Print                                                                       # Print the tree.

sub print($;$)                                                                  # String representation as a horizontal tree.
 {my ($tree, $print) = @_;                                                      # Tree, optional print method
  my @s;                                                                        # String representation

  sub                                                                           # Print a child
   {my ($child, $depth) = @_;                                                   # Child, depth
    my $user = $child->user;                                                    # User data
    push @s, join '', '  ' x $depth, $print ? &$print($user) : $user;           # Print child
    __SUB__->($_, $depth+1) for $child->children->@*;                           # Print children of child
   }->($tree, 0);                                                               # Print root

  join "\n", @s, ''                                                             # String result
 }

sub brackets($$;$)                                                              # Bracketed string representation of a tree.
 {my ($tree, $print, $separator) = @_;                                          # Tree, print method, child separator
  my @s;                                                                        # String representation
  my $t = $separator // '';                                                     # Default child separator
  sub                                                                           # Print a child
   {my ($child) = @_;                                                           # Child
    my $user = $child->user;                                                    # User data
    my ($p) = ($print ? &$print($user) : $user);                                # Printed child
    my  $c  = $child->children;                                                 # Children of child
    return $p unless @$c;                                                       # Return child immediately if no children to format
    join '', $p, '(', join($t, map {__SUB__->($_)} @$c), ')'                    # String representation
   }->($tree)                                                                   # Print root
 }

#D1 Data Structures                                                             # Data structures use by this package.

#D0
#-------------------------------------------------------------------------------
# Export
#-------------------------------------------------------------------------------

use Exporter qw(import);

use vars qw(@ISA @EXPORT @EXPORT_OK %EXPORT_TAGS);

@ISA          = qw(Exporter);
@EXPORT_OK    = qw(
);
%EXPORT_TAGS  = (all=>[@EXPORT, @EXPORT_OK]);

# podDocumentation

=pod

=encoding utf-8

=head1 Name

Tree::Ops - Tree operations.

=head1 Synopsis

Create a tree:

  my $t = Tree::Ops::new 'a';
  for(1..2)
   {$t->open  ('b');
    $t->single('c');
    $t->close;
   }
  $t->single  ('d');

Print the tree:

  is_deeply $t->print(sub{@_}), <<END;
  a
    b
      c
    b
      c
    d
  END

Locate a specific child in the tree and print it:

  is_deeply [map {$_->user} $t->select('b')], ['b', 'b'];

=head1 Description

Tree operations.


Version 20200701.


The following sections describe the methods in each functional area of this
module.  For an alphabetic listing of all methods by name see L<Index|/Index>.



=head1 Build

Create a tree.

=head2 new($user)

Create a new child recording the specified user data.

     Parameter  Description
  1  $user      User data to be recorded in the child

B<Example:>


  if (1)                                                                               
   {my $t = Tree::Ops::𝗻𝗲𝘄 'a';
    for(1..2)
     {$t->open  ('b');
      $t->single('c');
      $t->close;
     }
    $t->single  ('d');
    is_deeply $t->print, <<END;
  a
    b
      c
    b
      c
    d
  END
  
    is_deeply [map {$_->user} $t->select('b')], ['b', 'b'];
   }
  

This is a static method and so should either be imported or invoked as:

  Tree::Ops::new


=head2 open($tree, $user)

Add a child and make it the currently active scope into which new children will be added.

     Parameter  Description
  1  $tree      Tree
  2  $user      User data to be recorded in the interior child being opened

B<Example:>


  if (1)                                                                               
   {my $t = Tree::Ops::new 'a';
    for(1..2)
     {$t->𝗼𝗽𝗲𝗻  ('b');
      $t->single('c');
      $t->close;
     }
    $t->single  ('d');
    is_deeply $t->print, <<END;
  a
    b
      c
    b
      c
    d
  END
  
    is_deeply [map {$_->user} $t->select('b')], ['b', 'b'];
   }
  

=head2 close($tree)

Close the current scope returning to the previous scope.

     Parameter  Description
  1  $tree      Tree

B<Example:>


  if (1)                                                                               
   {my $t = Tree::Ops::new 'a';
    for(1..2)
     {$t->open  ('b');
      $t->single('c');
      $t->𝗰𝗹𝗼𝘀𝗲;
     }
    $t->single  ('d');
    is_deeply $t->print, <<END;
  a
    b
      c
    b
      c
    d
  END
  
    is_deeply [map {$_->user} $t->select('b')], ['b', 'b'];
   }
  

=head2 single($tree, $user)

Add one child in the current scope.

     Parameter  Description
  1  $tree      Tree
  2  $user      User data to be recorded in the child being created

B<Example:>


  if (1)                                                                               
   {my $t = Tree::Ops::new 'a';
    for(1..2)
     {$t->open  ('b');
      $t->𝘀𝗶𝗻𝗴𝗹𝗲('c');
      $t->close;
     }
    $t->𝘀𝗶𝗻𝗴𝗹𝗲  ('d');
    is_deeply $t->print, <<END;
  a
    b
      c
    b
      c
    d
  END
  
    is_deeply [map {$_->user} $t->select('b')], ['b', 'b'];
   }
  

=head1 Navigation

Navigate through the tree.

=head2 first($parent)

Get the first child under the specified parent.

     Parameter  Description
  1  $parent    Parent

B<Example:>


      is_deeply $a->brackets, 'a(b(c)d(efgh(i(j))))';
      is_deeply $c->parent,   $b;
      is_deeply $a->𝗳𝗶𝗿𝘀𝘁,    $b;
      is_deeply $a->last,     $d;
      is_deeply $e->next,     $f;
      is_deeply $f->prev,     $e;
  

=head2 last($parent)

Get the last child under the specified parent.

     Parameter  Description
  1  $parent    Parent

B<Example:>


      is_deeply $a->brackets, 'a(b(c)d(efgh(i(j))))';
      is_deeply $c->parent,   $b;
      is_deeply $a->first,    $b;
      is_deeply $a->𝗹𝗮𝘀𝘁,     $d;
      is_deeply $e->next,     $f;
      is_deeply $f->prev,     $e;
  

=head2 next($child)

Get the next sibling following the specified child.

     Parameter  Description
  1  $child     Child

B<Example:>


      is_deeply $a->brackets, 'a(b(c)d(efgh(i(j))))';
      is_deeply $c->parent,   $b;
      is_deeply $a->first,    $b;
      is_deeply $a->last,     $d;
      is_deeply $e->𝗻𝗲𝘅𝘁,     $f;
      is_deeply $f->prev,     $e;
  

=head2 prev($child)

Get the previous sibling of the specified child.

     Parameter  Description
  1  $child     Child

B<Example:>


      is_deeply $a->brackets, 'a(b(c)d(efgh(i(j))))';
      is_deeply $c->parent,   $b;
      is_deeply $a->first,    $b;
      is_deeply $a->last,     $d;
      is_deeply $e->next,     $f;
      is_deeply $f->𝗽𝗿𝗲𝘃,     $e;
  

=head2 firstMost($parent)

Return the first most descendant child in the tree starting at this parent or else return B<undef> if this parent has no children.

     Parameter  Description
  1  $parent    Child

B<Example:>


      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';
      is_deeply $a->print(sub{@_}), <<END;
  a
    b
      c
    y
      x
    y
      x
    d
      e
      f
      g
      h
        i
          j
  END
  
      is_deeply $a->𝗳𝗶𝗿𝘀𝘁𝗠𝗼𝘀𝘁->brackets, 'c';
      is_deeply $a-> lastMost->brackets, 'j';
  

=head2 lastMost($parent)

Return the last most descendant child in the tree starting at this parent or else return B<undef> if this parent has no children.

     Parameter  Description
  1  $parent    Child

B<Example:>


      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';
      is_deeply $a->print(sub{@_}), <<END;
  a
    b
      c
    y
      x
    y
      x
    d
      e
      f
      g
      h
        i
          j
  END
  
      is_deeply $a->firstMost->brackets, 'c';
      is_deeply $a-> 𝗹𝗮𝘀𝘁𝗠𝗼𝘀𝘁->brackets, 'j';
  

=head1 Location

Verify the current location.

=head2 context($child)

Get the context of the current child.

     Parameter  Description
  1  $child     Child

B<Example:>


      is_deeply $a->brackets, 'a(b(c)y(x)z(st)d(efgh(i(j))))';
  
      is_deeply [map {$_->user} $x->𝗰𝗼𝗻𝘁𝗲𝘅𝘁], [qw(x y a)];
  
      is_deeply join(' ', $a->by(sub{$_[0]->user})), "c b x y s t z e f g j i h d a";
  
      $z->cut;
      is_deeply $a->brackets, 'a(b(c)y(x)d(efgh(i(j))))';
  
      $y->unwrap;
      is_deeply $a->brackets, 'a(b(c)xd(efgh(i(j))))';
  
      $y = $x->wrap(new 'y');
      is_deeply $y->brackets, 'y(x)';
      is_deeply $a->brackets, 'a(b(c)y(x)d(efgh(i(j))))';
  
      $y->putNext($y->dup);
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';
  

=head2 isFirst($child)

Return the specified child if that child is first under its parent, else return B<undef>.

     Parameter  Description
  1  $child     Child

B<Example:>


    if (1)                                                                          
     {is_deeply $a->brackets, 'a(b(c)d(efgh(i(j))))';
      is_deeply $b->singleChildOfParent, $c;
      is_deeply $e->𝗶𝘀𝗙𝗶𝗿𝘀𝘁, $e;
      ok !$f->𝗶𝘀𝗙𝗶𝗿𝘀𝘁;
      ok !$g->isLast;
      is_deeply $h->isLast, $h;
     }
  

=head2 isLast($child)

Return the specified child if that child is last under its parent, else return B<undef>.

     Parameter  Description
  1  $child     Child

B<Example:>


    if (1)                                                                          
     {is_deeply $a->brackets, 'a(b(c)d(efgh(i(j))))';
      is_deeply $b->singleChildOfParent, $c;
      is_deeply $e->isFirst, $e;
      ok !$f->isFirst;
      ok !$g->𝗶𝘀𝗟𝗮𝘀𝘁;
      is_deeply $h->𝗶𝘀𝗟𝗮𝘀𝘁, $h;
     }
  

=head2 singleChildOfParent($parent)

Return the only child of this parent if the parent has an only child, else B<undef>

     Parameter  Description
  1  $parent    Parent

B<Example:>


    if (1)                                                                          
     {is_deeply $a->brackets, 'a(b(c)d(efgh(i(j))))';
      is_deeply $b->𝘀𝗶𝗻𝗴𝗹𝗲𝗖𝗵𝗶𝗹𝗱𝗢𝗳𝗣𝗮𝗿𝗲𝗻𝘁, $c;
      is_deeply $e->isFirst, $e;
      ok !$f->isFirst;
      ok !$g->isLast;
      is_deeply $h->isLast, $h;
     }
  

=head1 Put

Insert children into a tree.

=head2 putFirst($parent, $child)

Place a new child first under the specified parent and return the child.

     Parameter  Description
  1  $parent    Parent
  2  $child     Child

B<Example:>


      is_deeply $a->brackets, 'a(b(c)d(efgh(i(j))))';
  
      my $z = $b->putNext(new 'z');
      is_deeply $z->brackets, 'z';
      is_deeply $a->brackets, 'a(b(c)zd(efgh(i(j))))';
  
      my $y = $d->putPrev(new 'y');
      is_deeply $y->brackets, 'y';
      is_deeply $a->brackets, 'a(b(c)zyd(efgh(i(j))))';
  
      $z->putLast(new 't');
      is_deeply $z->brackets, 'z(t)';
      is_deeply $a->brackets, 'a(b(c)z(t)yd(efgh(i(j))))';
  
      $z->𝗽𝘂𝘁𝗙𝗶𝗿𝘀𝘁(new 's');
      is_deeply $a->brackets, 'a(b(c)z(st)yd(efgh(i(j))))';
  

=head2 putLast($parent, $child)

Place a new child last under the specified parent and return the child.

     Parameter  Description
  1  $parent    Parent
  2  $child     Child

B<Example:>


      is_deeply $a->brackets, 'a(b(c)d(efgh(i(j))))';
  
      my $z = $b->putNext(new 'z');
      is_deeply $z->brackets, 'z';
      is_deeply $a->brackets, 'a(b(c)zd(efgh(i(j))))';
  
      my $y = $d->putPrev(new 'y');
      is_deeply $y->brackets, 'y';
      is_deeply $a->brackets, 'a(b(c)zyd(efgh(i(j))))';
  
      $z->𝗽𝘂𝘁𝗟𝗮𝘀𝘁(new 't');
      is_deeply $z->brackets, 'z(t)';
      is_deeply $a->brackets, 'a(b(c)z(t)yd(efgh(i(j))))';
  
      $z->putFirst(new 's');
      is_deeply $a->brackets, 'a(b(c)z(st)yd(efgh(i(j))))';
  

=head2 putNext($child, $new)

Place a new child after the specified child.

     Parameter  Description
  1  $child     Existing child
  2  $new       New child

B<Example:>


      is_deeply $a->brackets, 'a(b(c)d(efgh(i(j))))';
  
      my $z = $b->𝗽𝘂𝘁𝗡𝗲𝘅𝘁(new 'z');
      is_deeply $z->brackets, 'z';
      is_deeply $a->brackets, 'a(b(c)zd(efgh(i(j))))';
  
      my $y = $d->putPrev(new 'y');
      is_deeply $y->brackets, 'y';
      is_deeply $a->brackets, 'a(b(c)zyd(efgh(i(j))))';
  
      $z->putLast(new 't');
      is_deeply $z->brackets, 'z(t)';
      is_deeply $a->brackets, 'a(b(c)z(t)yd(efgh(i(j))))';
  
      $z->putFirst(new 's');
      is_deeply $a->brackets, 'a(b(c)z(st)yd(efgh(i(j))))';
  

=head2 putPrev($child, $new)

Place a new child before the specified child.

     Parameter  Description
  1  $child     Child
  2  $new       New child

B<Example:>


      is_deeply $a->brackets, 'a(b(c)d(efgh(i(j))))';
  
      my $z = $b->putNext(new 'z');
      is_deeply $z->brackets, 'z';
      is_deeply $a->brackets, 'a(b(c)zd(efgh(i(j))))';
  
      my $y = $d->𝗽𝘂𝘁𝗣𝗿𝗲𝘃(new 'y');
      is_deeply $y->brackets, 'y';
      is_deeply $a->brackets, 'a(b(c)zyd(efgh(i(j))))';
  
      $z->putLast(new 't');
      is_deeply $z->brackets, 'z(t)';
      is_deeply $a->brackets, 'a(b(c)z(t)yd(efgh(i(j))))';
  
      $z->putFirst(new 's');
      is_deeply $a->brackets, 'a(b(c)z(st)yd(efgh(i(j))))';
  

=head1 Steps

Move the start or end of a scope forwards or backwards as suggested by Alex Monroe.

=head2 step($parent)

Make the first child of the specified parent the parents previous sibling and return the parent. In effect this moves the start of the parent one step forwards.

     Parameter  Description
  1  $parent    Parent

B<Example:>


      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';
  
      $d->𝘀𝘁𝗲𝗽;
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)ed(fgh(i(j))))';
  
      $d->stepBack;
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';
  
      $b->stepEnd;
      is_deeply $a->brackets, 'a(b(cy(x))y(x)d(efgh(i(j))))';
  
      $b->stepEndBack;
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';
  

=head2 stepEnd($parent)

Make the next sibling of the specified parent the parents last child and return the parent. In effect this moves the end of the parent one step forwards.

     Parameter  Description
  1  $parent    Parent

B<Example:>


      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';
  
      $d->step;
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)ed(fgh(i(j))))';
  
      $d->stepBack;
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';
  
      $b->𝘀𝘁𝗲𝗽𝗘𝗻𝗱;
      is_deeply $a->brackets, 'a(b(cy(x))y(x)d(efgh(i(j))))';
  
      $b->stepEndBack;
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';
  

=head2 stepBack()

Make the previous sibling of the specified parent the parents first child and return the parent. In effect this moves the start of the parent one step backwards.


B<Example:>


      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';
  
      $d->step;
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)ed(fgh(i(j))))';
  
      $d->𝘀𝘁𝗲𝗽𝗕𝗮𝗰𝗸;
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';
  
      $b->stepEnd;
      is_deeply $a->brackets, 'a(b(cy(x))y(x)d(efgh(i(j))))';
  
      $b->stepEndBack;
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';
  

=head2 stepEndBack()

Make the last child of the specified parent the parents next sibling and return the parent. In effect this moves the end of the parent one step backwards.


B<Example:>


      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';
  
      $d->step;
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)ed(fgh(i(j))))';
  
      $d->stepBack;
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';
  
      $b->stepEnd;
      is_deeply $a->brackets, 'a(b(cy(x))y(x)d(efgh(i(j))))';
  
      $b->𝘀𝘁𝗲𝗽𝗘𝗻𝗱𝗕𝗮𝗰𝗸;
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';
  

=head1 Edit

Edit a tree in situ.

=head2 cut($child)

Cut out a child and all its content and children, return it ready for reinsertion else where.

     Parameter  Description
  1  $child     Child

B<Example:>


      is_deeply $a->brackets, 'a(b(c)y(x)z(st)d(efgh(i(j))))';
  
      is_deeply [map {$_->user} $x->context], [qw(x y a)];
  
      is_deeply join(' ', $a->by(sub{$_[0]->user})), "c b x y s t z e f g j i h d a";
  
      $z->𝗰𝘂𝘁;
      is_deeply $a->brackets, 'a(b(c)y(x)d(efgh(i(j))))';
  
      $y->unwrap;
      is_deeply $a->brackets, 'a(b(c)xd(efgh(i(j))))';
  
      $y = $x->wrap(new 'y');
      is_deeply $y->brackets, 'y(x)';
      is_deeply $a->brackets, 'a(b(c)y(x)d(efgh(i(j))))';
  
      $y->putNext($y->dup);
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';
  

=head2 dup($parent)

Duplicate a parent and all its descendants.

     Parameter  Description
  1  $parent    Parent

B<Example:>


      is_deeply $a->brackets, 'a(b(c)y(x)z(st)d(efgh(i(j))))';
  
      is_deeply [map {$_->user} $x->context], [qw(x y a)];
  
      is_deeply join(' ', $a->by(sub{$_[0]->user})), "c b x y s t z e f g j i h d a";
  
      $z->cut;
      is_deeply $a->brackets, 'a(b(c)y(x)d(efgh(i(j))))';
  
      $y->unwrap;
      is_deeply $a->brackets, 'a(b(c)xd(efgh(i(j))))';
  
      $y = $x->wrap(new 'y');
      is_deeply $y->brackets, 'y(x)';
      is_deeply $a->brackets, 'a(b(c)y(x)d(efgh(i(j))))';
  
      $y->putNext($y->𝗱𝘂𝗽);
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';
  

=head2 unwrap($child)

Unwrap the specified child and return that child.

     Parameter  Description
  1  $child     Child

B<Example:>


      is_deeply $a->brackets, 'a(b(c)y(x)z(st)d(efgh(i(j))))';
  
      is_deeply [map {$_->user} $x->context], [qw(x y a)];
  
      is_deeply join(' ', $a->by(sub{$_[0]->user})), "c b x y s t z e f g j i h d a";
  
      $z->cut;
      is_deeply $a->brackets, 'a(b(c)y(x)d(efgh(i(j))))';
  
      $y->𝘂𝗻𝘄𝗿𝗮𝗽;
      is_deeply $a->brackets, 'a(b(c)xd(efgh(i(j))))';
  
      $y = $x->wrap(new 'y');
      is_deeply $y->brackets, 'y(x)';
      is_deeply $a->brackets, 'a(b(c)y(x)d(efgh(i(j))))';
  
      $y->putNext($y->dup);
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';
  

=head2 wrap($child, $new)

Wrap the specified child with a new parent and return the new parent.

     Parameter  Description
  1  $child     Child to wrap
  2  $new       New wrapping parent

B<Example:>


      is_deeply $a->brackets, 'a(b(c)y(x)z(st)d(efgh(i(j))))';
  
      is_deeply [map {$_->user} $x->context], [qw(x y a)];
  
      is_deeply join(' ', $a->by(sub{$_[0]->user})), "c b x y s t z e f g j i h d a";
  
      $z->cut;
      is_deeply $a->brackets, 'a(b(c)y(x)d(efgh(i(j))))';
  
      $y->unwrap;
      is_deeply $a->brackets, 'a(b(c)xd(efgh(i(j))))';
  
      $y = $x->𝘄𝗿𝗮𝗽(new 'y');
      is_deeply $y->brackets, 'y(x)';
      is_deeply $a->brackets, 'a(b(c)y(x)d(efgh(i(j))))';
  
      $y->putNext($y->dup);
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';
  

=head1 Traverse

Traverse the tree.

=head2 by($tree, $sub)

Traverse a tree in order to process each child and return an array of the results of processing each child.

     Parameter  Description
  1  $tree      Tree
  2  $sub       Method to process a child

B<Example:>


      is_deeply $a->brackets, 'a(b(c)y(x)z(st)d(efgh(i(j))))';
  
      is_deeply [map {$_->user} $x->context], [qw(x y a)];
  
      is_deeply join(' ', $a->𝗯𝘆(sub{$_[0]->user})), "c b x y s t z e f g j i h d a";
  
      $z->cut;
      is_deeply $a->brackets, 'a(b(c)y(x)d(efgh(i(j))))';
  
      $y->unwrap;
      is_deeply $a->brackets, 'a(b(c)xd(efgh(i(j))))';
  
      $y = $x->wrap(new 'y');
      is_deeply $y->brackets, 'y(x)';
      is_deeply $a->brackets, 'a(b(c)y(x)d(efgh(i(j))))';
  
      $y->putNext($y->dup);
      is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';
  

=head2 select($tree, $select)

Select matching children in a tree. A child can be selected via named value, array of values, a hash of values, a regular expression or a sub reference.

     Parameter  Description
  1  $tree      Tree
  2  $select    Method to select a child

B<Example:>


  if (1)                                                                               
   {my $t = Tree::Ops::new 'a';
    for(1..2)
     {$t->open  ('b');
      $t->single('c');
      $t->close;
     }
    $t->single  ('d');
    is_deeply $t->print, <<END;
  a
    b
      c
    b
      c
    d
  END
  
    is_deeply [map {$_->user} $t->𝘀𝗲𝗹𝗲𝗰𝘁('b')], ['b', 'b'];
   }
  

=head1 Print

Print the tree.

=head2 print($tree, $print)

String representation as a horizontal tree.

     Parameter  Description
  1  $tree      Tree
  2  $print     Optional print method

B<Example:>


  if (1)                                                                               
   {my $t = Tree::Ops::new 'a';
    for(1..2)
     {$t->open  ('b');
      $t->single('c');
      $t->close;
     }
    $t->single  ('d');
    is_deeply $t->𝗽𝗿𝗶𝗻𝘁, <<END;
  a
    b
      c
    b
      c
    d
  END
  
    is_deeply [map {$_->user} $t->select('b')], ['b', 'b'];
   }
  

=head2 brackets($tree, $print, $separator)

Bracketed string representation of a tree.

     Parameter   Description
  1  $tree       Tree
  2  $print      Print method
  3  $separator  Child separator

B<Example:>


      is_deeply $a->𝗯𝗿𝗮𝗰𝗸𝗲𝘁𝘀, 'a(b(c)y(x)y(x)d(efgh(i(j))))';
      is_deeply $a->print(sub{@_}), <<END;
  a
    b
      c
    y
      x
    y
      x
    d
      e
      f
      g
      h
        i
          j
  END
  
      is_deeply $a->firstMost->𝗯𝗿𝗮𝗰𝗸𝗲𝘁𝘀, 'c';
      is_deeply $a-> lastMost->𝗯𝗿𝗮𝗰𝗸𝗲𝘁𝘀, 'j';
  

=head1 Data Structures

Data structures use by this package.


=head2 Tree::Ops Definition


Child in the tree.




=head3 Output fields


B<children> - Children of this child.

B<lastChild> - Last active child chain - enables us to find the currently open scope from the start if the tree.

B<parent> - Parent for this child.

B<user> - User data for this child.



=head1 Private Methods

=head2 activeScope($tree)

Locate the active scope in a tree.

     Parameter  Description
  1  $tree      Tree

=head2 setParentOfChild($child, $parent)

Set the parent of a child and return the child.

     Parameter  Description
  1  $child     Child
  2  $parent    Parent

=head2 indexOfChildInParent($child)

Get the index of a child within the specified parent.

     Parameter  Description
  1  $child     Child


=head1 Index


1 L<activeScope|/activeScope> - Locate the active scope in a tree.

2 L<brackets|/brackets> - Bracketed string representation of a tree.

3 L<by|/by> - Traverse a tree in order to process each child and return an array of the results of processing each child.

4 L<close|/close> - Close the current scope returning to the previous scope.

5 L<context|/context> - Get the context of the current child.

6 L<cut|/cut> - Cut out a child and all its content and children, return it ready for reinsertion else where.

7 L<dup|/dup> - Duplicate a parent and all its descendants.

8 L<first|/first> - Get the first child under the specified parent.

9 L<firstMost|/firstMost> - Return the first most descendant child in the tree starting at this parent or else return B<undef> if this parent has no children.

10 L<indexOfChildInParent|/indexOfChildInParent> - Get the index of a child within the specified parent.

11 L<isFirst|/isFirst> - Return the specified child if that child is first under its parent, else return B<undef>.

12 L<isLast|/isLast> - Return the specified child if that child is last under its parent, else return B<undef>.

13 L<last|/last> - Get the last child under the specified parent.

14 L<lastMost|/lastMost> - Return the last most descendant child in the tree starting at this parent or else return B<undef> if this parent has no children.

15 L<new|/new> - Create a new child recording the specified user data.

16 L<next|/next> - Get the next sibling following the specified child.

17 L<open|/open> - Add a child and make it the currently active scope into which new children will be added.

18 L<prev|/prev> - Get the previous sibling of the specified child.

19 L<print|/print> - String representation as a horizontal tree.

20 L<putFirst|/putFirst> - Place a new child first under the specified parent and return the child.

21 L<putLast|/putLast> - Place a new child last under the specified parent and return the child.

22 L<putNext|/putNext> - Place a new child after the specified child.

23 L<putPrev|/putPrev> - Place a new child before the specified child.

24 L<select|/select> - Select matching children in a tree.

25 L<setParentOfChild|/setParentOfChild> - Set the parent of a child and return the child.

26 L<single|/single> - Add one child in the current scope.

27 L<singleChildOfParent|/singleChildOfParent> - Return the only child of this parent if the parent has an only child, else B<undef>

28 L<step|/step> - Make the first child of the specified parent the parents previous sibling and return the parent.

29 L<stepBack|/stepBack> - Make the previous sibling of the specified parent the parents first child and return the parent.

30 L<stepEnd|/stepEnd> - Make the next sibling of the specified parent the parents last child and return the parent.

31 L<stepEndBack|/stepEndBack> - Make the last child of the specified parent the parents next sibling and return the parent.

32 L<unwrap|/unwrap> - Unwrap the specified child and return that child.

33 L<wrap|/wrap> - Wrap the specified child with a new parent and return the new parent.

=head1 Installation

This module is written in 100% Pure Perl and, thus, it is easy to read,
comprehend, use, modify and install via B<cpan>:

  sudo cpan install Tree::Ops

=head1 Author

L<philiprbrenan@gmail.com|mailto:philiprbrenan@gmail.com>

L<http://www.appaapps.com|http://www.appaapps.com>

=head1 Copyright

Copyright (c) 2016-2019 Philip R Brenan.

This module is free software. It may be used, redistributed and/or modified
under the same terms as Perl itself.

=cut



# Tests and documentation

sub test
 {my $p = __PACKAGE__;
  binmode($_, ":utf8") for *STDOUT, *STDERR;
  return if eval "eof(${p}::DATA)";
  my $s = eval "join('', <${p}::DATA>)";
  $@ and die $@;
  eval $s;
  $@ and die $@;
  1
 }

test unless caller;

1;
# podDocumentation
__DATA__
use warnings FATAL=>qw(all);
use strict;
require v5.26;
use Test::More tests=>70;

#goto latestTest;

if (1)                                                                          #Tnew #Topen #Tsingle #Tclose #Tprint #Tselect
 {my $t = Tree::Ops::new 'a';
  for(1..2)
   {$t->open  ('b');
    $t->single('c');
    $t->close;
   }
  $t->single  ('d');
  is_deeply $t->print, <<END;
a
  b
    c
  b
    c
  d
END

  is_deeply [map {$_->user} $t->select('b')], ['b', 'b'];
 }

if (1)
 {my $a = Tree::Ops::new('a');  is_deeply $a->brackets, 'a';
  my $b = $a->open      ('b');  is_deeply $b->brackets, 'b';
  my $c = $a->single    ('c');  is_deeply $c->brackets, 'c';
  my $B = $a->close;            is_deeply $B->brackets, 'b(c)';
  my $d = $a->open      ('d');  is_deeply $d->brackets, 'd';
  my $e = $a->single    ('e');  is_deeply $e->brackets, 'e';
  my $f = $a->single    ('f');  is_deeply $f->brackets, 'f';
  my $g = $a->single    ('g');  is_deeply $g->brackets, 'g';
  my $h = $a->open      ('h');  is_deeply $h->brackets, 'h';
  my $i = $a->open      ('i');  is_deeply $i->brackets, 'i';
  my $j = $a->single    ('j');  is_deeply $j->brackets, 'j';

  is_deeply [map {$_->user} $a->select(['b', 'c'])],        ['b', 'c'];
  is_deeply [map {$_->user} $a->select({e=>1})],            ['e'];
  is_deeply [map {$_->user} $a->select(qr(b|d))],            ['b', 'd'];
  is_deeply [map {$_->user} $a->select(sub{$_[0] eq 'c'})], ['c'];

  if (1) {                                                                      #Tparent #Tfirst #Tlast #Tnext #Tprev
    is_deeply $a->brackets, 'a(b(c)d(efgh(i(j))))';
    is_deeply $c->parent,   $b;
    is_deeply $a->first,    $b;
    is_deeply $a->last,     $d;
    is_deeply $e->next,     $f;
    is_deeply $f->prev,     $e;
   }

  is_deeply $a->brackets, 'a(b(c)d(efgh(i(j))))';
  is_deeply $b->parent,  $a;
  is_deeply $c->parent,  $b;
  is_deeply $d->parent,  $a;
  is_deeply $d->first,   $e;
  is_deeply $d->last,    $h;
  is_deeply $e->next,    $f;
  is_deeply $f->prev,    $e;

  ok !$c->first;
  ok !$e->last;
  ok !$h->next;
  ok !$e->prev;

  if (1)                                                                        #TsingleChildOfParent #TisFirst #TisLast
   {is_deeply $a->brackets, 'a(b(c)d(efgh(i(j))))';
    is_deeply $b->singleChildOfParent, $c;
    is_deeply $e->isFirst, $e;
    ok !$f->isFirst;
    ok !$g->isLast;
    is_deeply $h->isLast, $h;
   }

  if (1) {                                                                      #TputFirst #TputLast #TputNext #TputPrev
    is_deeply $a->brackets, 'a(b(c)d(efgh(i(j))))';

    my $z = $b->putNext(new 'z');
    is_deeply $z->brackets, 'z';
    is_deeply $a->brackets, 'a(b(c)zd(efgh(i(j))))';

    my $y = $d->putPrev(new 'y');
    is_deeply $y->brackets, 'y';
    is_deeply $a->brackets, 'a(b(c)zyd(efgh(i(j))))';

    $z->putLast(new 't');
    is_deeply $z->brackets, 'z(t)';
    is_deeply $a->brackets, 'a(b(c)z(t)yd(efgh(i(j))))';

    $z->putFirst(new 's');
    is_deeply $a->brackets, 'a(b(c)z(st)yd(efgh(i(j))))';
   }

  my ($y, $z) = map {$a->select($_)} 'y', 'z';
  is_deeply $y->brackets, 'y';
  is_deeply $z->brackets, 'z(st)';

  $y->putNext($z->cut);
  is_deeply $a->brackets, 'a(b(c)yz(st)d(efgh(i(j))))';

  my $x = $y->putFirst(new "x");
  is_deeply $a->brackets, 'a(b(c)y(x)z(st)d(efgh(i(j))))';

  if (1) {                                                                      #Tcut #Tunwrap #Twrap #Tcontext #Tby #Tdup
    is_deeply $a->brackets, 'a(b(c)y(x)z(st)d(efgh(i(j))))';

    is_deeply [map {$_->user} $x->context], [qw(x y a)];

    is_deeply join(' ', $a->by(sub{$_[0]->user})), "c b x y s t z e f g j i h d a";

    $z->cut;
    is_deeply $a->brackets, 'a(b(c)y(x)d(efgh(i(j))))';

    $y->unwrap;
    is_deeply $a->brackets, 'a(b(c)xd(efgh(i(j))))';

    $y = $x->wrap(new 'y');
    is_deeply $y->brackets, 'y(x)';
    is_deeply $a->brackets, 'a(b(c)y(x)d(efgh(i(j))))';

    $y->putNext($y->dup);
    is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';
   }

  if (1) {                                                                      #Tbrackets #TfirstMost #TlastMost
    is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';
    is_deeply $a->print(sub{@_}), <<END;
a
  b
    c
  y
    x
  y
    x
  d
    e
    f
    g
    h
      i
        j
END

    is_deeply $a->firstMost->brackets, 'c';
    is_deeply $a-> lastMost->brackets, 'j';
   }

  if (1) {                                                                      #Tstep #TstepBack #TstepEnd #TstepEndBack
    is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';

    $d->step;
    is_deeply $a->brackets, 'a(b(c)y(x)y(x)ed(fgh(i(j))))';

    $d->stepBack;
    is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';

    $b->stepEnd;
    is_deeply $a->brackets, 'a(b(cy(x))y(x)d(efgh(i(j))))';

    $b->stepEndBack;
    is_deeply $a->brackets, 'a(b(c)y(x)y(x)d(efgh(i(j))))';
   }
 }

done_testing;
#   owf(q(/home/phil/z/z/z/zzz.txt), $dfa->dumpAsJson);
