use Test2::V0;
use Bitcoin::Secp256k1;
use Bitcoin::Crypto qw(btc_prv btc_transaction btc_utxo btc_block);
use Bitcoin::Crypto::Util qw(to_format);

# Data from:
# https://github.com/bitcoin/bips/blob/master/bip-0341.mediawiki#test-vectors

my %base_data = (
	auxiliary => {
		fully_signed_tx =>
			'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'
	},
	given => {
		raw_unsigned_tx =>
			'02000000097de20cbff686da83a54981d2b9bab3586f4ca7e48f57f5b55963115f3b334e9c010000000000000000d7b7cab57b1393ace2d064f4d4a2cb8af6def61273e127517d44759b6dafdd990000000000fffffffff8e1f583384333689228c5d28eac13366be082dc57441760d957275419a418420000000000fffffffff0689180aa63b30cb162a73c6d2a38b7eeda2a83ece74310fda0843ad604853b0100000000feffffffaa5202bdf6d8ccd2ee0f0202afbbb7461d9264a25e5bfd3c5a52ee1239e0ba6c0000000000feffffff956149bdc66faa968eb2be2d2faa29718acbfe3941215893a2a3446d32acd050000000000000000000e664b9773b88c09c32cb70a2a3e4da0ced63b7ba3b22f848531bbb1d5d5f4c94010000000000000000e9aa6b8e6c9de67619e6a3924ae25696bb7b694bb677a632a74ef7eadfd4eabf0000000000ffffffffa778eb6a263dc090464cd125c466b5a99667720b1c110468831d058aa1b82af10100000000ffffffff0200ca9a3b000000001976a91406afd46bcdfd22ef94ac122aa11f241244a37ecc88ac807840cb0000000020ac9a87f5594be208f8532db38cff670c450ed2fea8fcdefcc9a663f78bab962b0065cd1d',
		utxos_spent => [
			{
				amount_sats => 420000000,
				script_pub_key => '512053a1f6e454df1aa2776a2814a721372d6258050de330b3c6d10ee8f4e0dda343'
			},
			{
				amount_sats => 462000000,
				script_pub_key => '5120147c9c57132f6e7ecddba9800bb0c4449251c92a1e60371ee77557b6620f3ea3'
			},
			{
				amount_sats => 294000000,
				script_pub_key => '76a914751e76e8199196d454941c45d1b3a323f1433bd688ac'
			},
			{
				amount_sats => 504000000,
				script_pub_key => '5120e4d810fd50586274face62b8a807eb9719cef49c04177cc6b76a9a4251d5450e'
			},
			{
				amount_sats => 630000000,
				script_pub_key => '512091b64d5324723a985170e4dc5a0f84c041804f2cd12660fa5dec09fc21783605'
			},
			{
				amount_sats => 378000000,
				script_pub_key => '00147dd65592d0ab2fe0d0257d571abf032cd9db93dc'
			},
			{
				amount_sats => 672000000,
				script_pub_key => '512075169f4001aa68f15bbed28b218df1d0a62cbbcf1188c6665110c293c907b831'
			},
			{
				amount_sats => 546000000,
				script_pub_key => '5120712447206d7a5238acc7ff53fbe94a3b64539ad291c7cdbc490b7577e4b17df5'
			},
			{
				amount_sats => 588000000,
				script_pub_key => '512077e30a5522dd9f894c3f8b8bd4c4b2cf82ca7da8a3ea6a239655c39c050ab220'
			}
		]
	},
);

my @cases = (
	{
		expected => {
			witness => [
				'ed7c1647cb97379e76892be0cacff57ec4a7102aa24296ca39af7541246d8ff14d38958d4cc1e2e478e4d4a764bbfd835b16d4e314b72937b29833060b87276c03'
			]
		},
		given => {
			hash_type => 3,
			internal_privkey => '6b973d88838f27366ed61c9ad6367663045cb456e28335c109e30717ae0c6baa',
			merkle_root => undef,
			txin_index => 0
		},
		intermediary => {
			internal_pubkey => 'd6889cb081036e0faefa3a35157ad71086b123b2b144b649798b494c300a961d',
			precomputed_used => [
				'hash_amounts',
				'hash_prevouts',
				'hash_script_pubkeys',
				'hash_sequences'
			],
			sig_hash => '2514a6272f85cfa0f45eb907fcb0d121b808ed37c6ea160a5a9046ed5526d555',
			sig_msg =>
				'0003020000000065cd1de3b33bb4ef3a52ad1fffb555c0d82828eb22737036eaeb02a235d82b909c4c3f58a6964a4f5f8f0b642ded0a8a553be7622a719da71d1f5befcefcdee8e0fde623ad0f61ad2bca5ba6a7693f50fce988e17c3780bf2b1e720cfbb38fbdd52e2118959c7221ab5ce9e26c3cd67b22c24f8baa54bac281d8e6b05e400e6c3a957e0000000000d0418f0e9a36245b9a50ec87f8bf5be5bcae434337b87139c3a5b1f56e33cba0',
			tweak => 'b86e7be8f39bab32a6f2c0443abbc210f0edac0e2c53d501b36b64437d9c6c70',
			tweaked_privkey => '2405b971772ad26915c8dcdf10f238753a9b837e5f8e6a86fd7c0cce5b7296d9'
		}
	},
	{
		expected => {
			witness => [
				'052aedffc554b41f52b521071793a6b88d6dbca9dba94cf34c83696de0c1ec35ca9c5ed4ab28059bd606a4f3a657eec0bb96661d42921b5f50a95ad33675b54f83'
			]
		},
		given => {
			hash_type => 131,
			internal_privkey => '1e4da49f6aaf4e5cd175fe08a32bb5cb4863d963921255f33d3bc31e1343907f',
			merkle_root => '5b75adecf53548f3ec6ad7d78383bf84cc57b55a3127c72b9a2481752dd88b21',
			txin_index => 1
		},
		intermediary => {
			internal_pubkey => '187791b6f712a8ea41c8ecdd0ee77fab3e85263b37e1ec18a3651926b3a6cf27',
			precomputed_used => [],
			sig_hash => '325a644af47e8a5a2591cda0ab0723978537318f10e6a63d4eed783b96a71a4d',
			sig_msg =>
				'0083020000000065cd1d00d7b7cab57b1393ace2d064f4d4a2cb8af6def61273e127517d44759b6dafdd9900000000808f891b00000000225120147c9c57132f6e7ecddba9800bb0c4449251c92a1e60371ee77557b6620f3ea3ffffffffffcef8fb4ca7efc5433f591ecfc57391811ce1e186a3793024def5c884cba51d',
			tweak => 'cbd8679ba636c1110ea247542cfbd964131a6be84f873f7f3b62a777528ed001',
			tweaked_privkey => 'ea260c3b10e60f6de018455cd0278f2f5b7e454be1999572789e6a9565d26080'
		}
	},
	{
		expected => {
			witness => [
				'ff45f742a876139946a149ab4d9185574b98dc919d2eb6754f8abaa59d18b025637a3aa043b91817739554f4ed2026cf8022dbd83e351ce1fabc272841d2510a01'
			]
		},
		given => {
			hash_type => 1,
			internal_privkey => 'd3c7af07da2d54f7a7735d3d0fc4f0a73164db638b2f2f7c43f711f6d4aa7e64',
			merkle_root => 'c525714a7f49c28aedbbba78c005931a81c234b2f6c99a73e4d06082adc8bf2b',
			txin_index => 3
		},
		intermediary => {
			internal_pubkey => '93478e9488f956df2396be2ce6c5cced75f900dfa18e7dabd2428aae78451820',
			precomputed_used => [
				'hash_amounts',
				'hash_outputs',
				'hash_prevouts',
				'hash_script_pubkeys',
				'hash_sequences'
			],
			sig_hash => 'bf013ea93474aa67815b1b6cc441d23b64fa310911d991e713cd34c7f5d46669',
			sig_msg =>
				'0001020000000065cd1de3b33bb4ef3a52ad1fffb555c0d82828eb22737036eaeb02a235d82b909c4c3f58a6964a4f5f8f0b642ded0a8a553be7622a719da71d1f5befcefcdee8e0fde623ad0f61ad2bca5ba6a7693f50fce988e17c3780bf2b1e720cfbb38fbdd52e2118959c7221ab5ce9e26c3cd67b22c24f8baa54bac281d8e6b05e400e6c3a957ea2e6dab7c1f0dcd297c8d61647fd17d821541ea69c3cc37dcbad7f90d4eb4bc50003000000',
			tweak => '6af9e28dbf9d6aaf027696e2598a5b3d056f5fd2355a7fd5a37a0e5008132d30',
			tweaked_privkey => '97323385e57015b75b0339a549c56a948eb961555973f0951f555ae6039ef00d'
		}
	},
	{
		expected => {
			witness => [
				'b4010dd48a617db09926f729e79c33ae0b4e94b79f04a1ae93ede6315eb3669de185a17d2b0ac9ee09fd4c64b678a0b61a0a86fa888a273c8511be83bfd6810f'
			]
		},
		given => {
			hash_type => 0,
			internal_privkey => 'f36bb07a11e469ce941d16b63b11b9b9120a84d9d87cff2c84a8d4affb438f4e',
			merkle_root => 'ccbd66c6f7e8fdab47b3a486f59d28262be857f30d4773f2d5ea47f7761ce0e2',
			txin_index => 4
		},
		intermediary => {
			internal_pubkey => 'e0dfe2300b0dd746a3f8674dfd4525623639042569d829c7f0eed9602d263e6f',
			precomputed_used => [
				'hash_amounts',
				'hash_outputs',
				'hash_prevouts',
				'hash_script_pubkeys',
				'hash_sequences'
			],
			sig_hash => '4f900a0bae3f1446fd48490c2958b5a023228f01661cda3496a11da502a7f7ef',
			sig_msg =>
				'0000020000000065cd1de3b33bb4ef3a52ad1fffb555c0d82828eb22737036eaeb02a235d82b909c4c3f58a6964a4f5f8f0b642ded0a8a553be7622a719da71d1f5befcefcdee8e0fde623ad0f61ad2bca5ba6a7693f50fce988e17c3780bf2b1e720cfbb38fbdd52e2118959c7221ab5ce9e26c3cd67b22c24f8baa54bac281d8e6b05e400e6c3a957ea2e6dab7c1f0dcd297c8d61647fd17d821541ea69c3cc37dcbad7f90d4eb4bc50004000000',
			tweak => 'b57bfa183d28eeb6ad688ddaabb265b4a41fbf68e5fed2c72c74de70d5a786f4',
			tweaked_privkey => 'a8e7aa924f0d58854185a490e6c41f6efb7b675c0f3331b7f14b549400b4d501'
		}
	},
	{
		expected => {
			witness => [
				'a3785919a2ce3c4ce26f298c3d51619bc474ae24014bcdd31328cd8cfbab2eff3395fa0a16fe5f486d12f22a9cedded5ae74feb4bbe5351346508c5405bcfee002'
			]
		},
		given => {
			hash_type => 2,
			internal_privkey => '415cfe9c15d9cea27d8104d5517c06e9de48e2f986b695e4f5ffebf230e725d8',
			merkle_root => '2f6b2c5397b6d68ca18e09a3f05161668ffe93a988582d55c6f07bd5b3329def',
			txin_index => 6
		},
		intermediary => {
			internal_pubkey => '55adf4e8967fbd2e29f20ac896e60c3b0f1d5b0efa9d34941b5958c7b0a0312d',
			precomputed_used => [
				'hash_amounts',
				'hash_prevouts',
				'hash_script_pubkeys',
				'hash_sequences'
			],
			sig_hash => '15f25c298eb5cdc7eb1d638dd2d45c97c4c59dcaec6679cfc16ad84f30876b85',
			sig_msg =>
				'0002020000000065cd1de3b33bb4ef3a52ad1fffb555c0d82828eb22737036eaeb02a235d82b909c4c3f58a6964a4f5f8f0b642ded0a8a553be7622a719da71d1f5befcefcdee8e0fde623ad0f61ad2bca5ba6a7693f50fce988e17c3780bf2b1e720cfbb38fbdd52e2118959c7221ab5ce9e26c3cd67b22c24f8baa54bac281d8e6b05e400e6c3a957e0006000000',
			tweak => '6579138e7976dc13b6a92f7bfd5a2fc7684f5ea42419d43368301470f3b74ed9',
			tweaked_privkey => '241c14f2639d0d7139282aa6abde28dd8a067baa9d633e4e7230287ec2d02901'
		}
	},
	{
		expected => {
			witness => [
				'ea0c6ba90763c2d3a296ad82ba45881abb4f426b3f87af162dd24d5109edc1cdd11915095ba47c3a9963dc1e6c432939872bc49212fe34c632cd3ab9fed429c482'
			]
		},
		given => {
			hash_type => 130,
			internal_privkey => 'c7b0e81f0a9a0b0499e112279d718cca98e79a12e2f137c72ae5b213aad0d103',
			merkle_root => '6c2dc106ab816b73f9d07e3cd1ef2c8c1256f519748e0813e4edd2405d277bef',
			txin_index => 7
		},
		intermediary => {
			internal_pubkey => 'ee4fe085983462a184015d1f782d6a5f8b9c2b60130aff050ce221ecf3786592',
			precomputed_used => [],
			sig_hash => 'cd292de50313804dabe4685e83f923d2969577191a3e1d2882220dca88cbeb10',
			sig_msg =>
				'0082020000000065cd1d00e9aa6b8e6c9de67619e6a3924ae25696bb7b694bb677a632a74ef7eadfd4eabf00000000804c8b2000000000225120712447206d7a5238acc7ff53fbe94a3b64539ad291c7cdbc490b7577e4b17df5ffffffff',
			tweak => '9e0517edc8259bb3359255400b23ca9507f2a91cd1e4250ba068b4eafceba4a9',
			tweaked_privkey => '65b6000cd2bfa6b7cf736767a8955760e62b6649058cbc970b7c0871d786346b'
		}
	},
	{
		expected => {
			witness => [
				'bbc9584a11074e83bc8c6759ec55401f0ae7b03ef290c3139814f545b58a9f8127258000874f44bc46db7646322107d4d86aec8e73b8719a61fff761d75b5dd981'
			]
		},
		given => {
			hash_type => 129,
			internal_privkey => '77863416be0d0665e517e1c375fd6f75839544eca553675ef7fdf4949518ebaa',
			merkle_root => 'ab179431c28d3b68fb798957faf5497d69c883c6fb1e1cd9f81483d87bac90cc',
			txin_index => 8
		},
		intermediary => {
			internal_pubkey => 'f9f400803e683727b14f463836e1e78e1c64417638aa066919291a225f0e8dd8',
			precomputed_used => [
				'hash_outputs'
			],
			sig_hash => 'cccb739eca6c13a8a89e6e5cd317ffe55669bbda23f2fd37b0f18755e008edd2',
			sig_msg =>
				'0081020000000065cd1da2e6dab7c1f0dcd297c8d61647fd17d821541ea69c3cc37dcbad7f90d4eb4bc500a778eb6a263dc090464cd125c466b5a99667720b1c110468831d058aa1b82af101000000002b0c230000000022512077e30a5522dd9f894c3f8b8bd4c4b2cf82ca7da8a3ea6a239655c39c050ab220ffffffff',
			tweak => '639f0281b7ac49e742cd25b7f188657626da1ad169209078e2761cefd91fd65e',
			tweaked_privkey => 'ec18ce6af99f43815db543f47b8af5ff5df3b2cb7315c955aa4a86e8143d2bf5'
		}
	}
);

# disable randomness for deterministic signatures
$Bitcoin::Secp256k1::FORCED_SCHNORR_AUX_RAND = "\x00" x 32;

# some random block data to silence locktime verification warnings
my $block_then = btc_block->new(timestamp => 1_000_000_000, height => 1_000);
my $block_now = btc_block->new(timestamp => 2_000_000_000, height => 1_000_000);

# tx/utxo preparation
my $tx = btc_transaction->from_serialized([hex => $base_data{given}{raw_unsigned_tx}]);
foreach my $input_index (0 .. $#{$base_data{given}{utxos_spent}}) {
	my $utxo = $base_data{given}{utxos_spent}[$input_index];
	my $input = $tx->inputs->[$input_index];

	btc_utxo->new(
		txid => $input->utxo_location->[0],
		output_index => $input->utxo_location->[1],
		block => $block_then,
		output => {
			value => $utxo->{amount_sats},
			locking_script => [hex => $utxo->{script_pub_key}],
		},
	)->register;
}

subtest 'should verify signed tx with key path' => sub {
	my $tx = btc_transaction->from_serialized([hex => $base_data{auxiliary}{fully_signed_tx}]);
	$tx->set_block($block_now);
	ok lives { $tx->verify }, 'verification ok';

	# invalidate the signature of the taproot input
	substr $tx->inputs->[0]->witness->[0], 15, 1, "\x00";

	my $err = dies { $tx->verify };
	like $err, qr/marked as invalid/, 'wrong signature ok';
};

foreach my $case_ind (0 .. $#cases) {
	subtest "should pass case #$case_ind" => sub {
		my $case = $cases[$case_ind];

		my $privkey = btc_prv->from_serialized([hex => $case->{given}{internal_privkey}]);
		my $in_ind = $case->{given}{txin_index};
		$privkey->sign_transaction(
			$tx,
			signing_index => $in_ind,
			sighash => $case->{given}{hash_type},
			($case->{given}{merkle_root} ? (script_tree => [{hash => [hex => $case->{given}{merkle_root}]}]) : ()),
		);

		is [map { to_format [hex => $_] } @{$tx->inputs->[$in_ind]->witness // []}], $case->{expected}{witness},
			"witness of index $in_ind ok";
	};
}

done_testing;

