(* Copyright (C) 1995, Digital Equipment Corporation                         *)
(* Digital Internal Use Only                                                 *)
(* All rights reserved.                                                      *)
(*                                                                           *)
(* Last modified on Thu Jun  8 23:04:26 PDT 1995 by najork                   *)
(*       Created on Tue Jan 17 16:48:42 PST 1995 by najork                   *)


MODULE WinScrnPaintOp;

IMPORT PaintOp, ScreenType, ScrnPaintOp, TrestleClass, VBTClass, WinGDI, 
       WinScreenType, WinScreenTypePrivate, Word;


PROCEDURE NewOracle (st: WinScreenType.T): ScrnPaintOp.Oracle =
  BEGIN
    RETURN NEW(Oracle, st := st)
  END NewOracle;


TYPE
  Oracle = ScrnPaintOp.Oracle BRANDED OBJECT
    st: WinScreenType.T;
  OVERRIDES
    opaque      := Opaque;
    bgfg        := BgFg;
    swap        := Swap;
    transparent := Transparent;
    copy        := Copy;
    builtIn     := BuiltIn;    
  END;


PROCEDURE Opaque (self: Oracle; pix: ScrnPaintOp.Pixel): ScrnPaintOp.T =
  BEGIN
    RETURN NewPaintOp (self.st, Op {Mode.Opaq, pix}, Op {Mode.Opaq, pix}, pix);
  END Opaque;


PROCEDURE Transparent (self: Oracle): ScrnPaintOp.T =
  BEGIN
    RETURN NewPaintOp (self.st, Op {Mode.Tran, 0}, Op {Mode.Tran, 0});
  END Transparent;


PROCEDURE Copy (self: Oracle): ScrnPaintOp.T =
  BEGIN
    RETURN NewPaintOp (self.st, Op {Mode.Copy, 0}, Op {Mode.Copy, 0});
  END Copy;


PROCEDURE Swap (self: Oracle; p, q: ScrnPaintOp.Pixel): ScrnPaintOp.T =
  VAR 
    pix := Word.Xor(p, q);
  BEGIN
    (* "p = q" is a special case, which can be handled 
       much more efficiently by calling "Transparent". *)
    IF p = q THEN 
      RETURN Transparent(self);  
    END;
    RETURN NewPaintOp (self.st, Op {Mode.Swap, pix}, Op {Mode.Swap, pix});
  END Swap;


PROCEDURE BgFg (self: Oracle; bg, fg: ScrnPaintOp.T): ScrnPaintOp.T
    RAISES {ScrnPaintOp.Failure} =
  VAR
    st := self.st;
  BEGIN
    LOCK st.trsl DO
      IF bg.id < 0 OR bg.id >= st.opcount OR 
         fg.id < 0 OR fg.id >= st.opcount THEN
        RAISE ScrnPaintOp.Failure;
      END;
    END;
    RETURN NewPaintOp(st, st.optable[bg.id].bop, st.optable[fg.id].fop);
  END BgFg;


PROCEDURE BuiltIn (self: Oracle; op: PaintOp.Predefined): ScrnPaintOp.T =
  VAR 
    b    := self.st.bg;
    f    := self.st.fg;
    back := Op {Mode.Opaq, b};
    fore := Op {Mode.Opaq, f};
    swap := Op {Mode.Swap, Word.Xor (b, f)};
    tran := Op {Mode.Tran, 0};
  BEGIN
    CASE op OF
    | PaintOp.Bg.op              => RETURN NewPaintOp (self.st, back, back, b);
    | PaintOp.BgFg.op            => RETURN NewPaintOp (self.st, back, fore);
    | PaintOp.BgTransparent.op   => RETURN NewPaintOp (self.st, back, tran);
    | PaintOp.BgSwap.op          => RETURN NewPaintOp (self.st, back, swap);
    | PaintOp.FgBg.op            => RETURN NewPaintOp (self.st, fore, back);
    | PaintOp.Fg.op              => RETURN NewPaintOp (self.st, fore, fore, f);
    | PaintOp.FgTransparent.op   => RETURN NewPaintOp (self.st, fore, tran);
    | PaintOp.FgSwap.op          => RETURN NewPaintOp (self.st, fore, swap);
    | PaintOp.TransparentBg.op   => RETURN NewPaintOp (self.st, tran, back);
    | PaintOp.TransparentFg.op   => RETURN NewPaintOp (self.st, tran, fore);
    | PaintOp.Transparent.op     => RETURN NewPaintOp (self.st, tran, tran);
    | PaintOp.TransparentSwap.op => RETURN NewPaintOp (self.st, tran, swap);
    | PaintOp.SwapBg.op          => RETURN NewPaintOp (self.st, swap, back);
    | PaintOp.SwapFg.op          => RETURN NewPaintOp (self.st, swap, fore);
    | PaintOp.SwapTransparent.op => RETURN NewPaintOp (self.st, swap, tran);
    | PaintOp.Swap.op            => RETURN NewPaintOp (self.st, swap, swap);
    | PaintOp.Copy.op            => RETURN Copy(self);
    END;
  END BuiltIn;


PROCEDURE ToBinaryRasterOp (bop, fop: Op): INTEGER =
  BEGIN
    CASE bop.mode OF
    | Mode.Copy =>
      CASE fop.mode OF
      | Mode.Copy => RETURN WinGDI.R2_COPYPEN;
      | Mode.Tran => RETURN WinGDI.R2_NOP;  (* illegal combination *)
      | Mode.Opaq => RETURN WinGDI.R2_NOP;  (* illegal combination *)
      | Mode.Swap => RETURN WinGDI.R2_NOP;  (* illegal combination *)
      END;
    | Mode.Tran =>
      CASE fop.mode OF
      | Mode.Copy => RETURN WinGDI.R2_NOP;  (* illegal combination *)
      | Mode.Tran => RETURN WinGDI.R2_NOP;
      | Mode.Opaq => RETURN WinGDI.R2_COPYPEN;
      | Mode.Swap => RETURN WinGDI.R2_XORPEN;
      END;
    | Mode.Opaq =>
      CASE fop.mode OF
      | Mode.Copy => RETURN WinGDI.R2_NOP;  (* illegal combination *)
      | Mode.Tran => RETURN WinGDI.R2_NOP;  
      | Mode.Opaq => RETURN WinGDI.R2_COPYPEN;
      | Mode.Swap => RETURN WinGDI.R2_XORPEN;
      END;
    | Mode.Swap =>
      CASE fop.mode OF
      | Mode.Copy => RETURN WinGDI.R2_NOP;  (* illegal combination *)
      | Mode.Tran => RETURN WinGDI.R2_NOP;  
      | Mode.Opaq => RETURN WinGDI.R2_COPYPEN;
      | Mode.Swap => RETURN WinGDI.R2_XORPEN;
      END;
    END;
  END ToBinaryRasterOp;


PROCEDURE ToTernaryRasterOp (bop, fop: Op): INTEGER =
  BEGIN
    CASE bop.mode OF
    | Mode.Copy =>
      CASE fop.mode OF
      | Mode.Copy => RETURN WinGDI.SRCCOPY;
      | Mode.Tran => RETURN 16_00AA0029;       (* illegal combination *)
      | Mode.Opaq => RETURN 16_00AA0029;       (* illegal combination *)
      | Mode.Swap => RETURN 16_00AA0029;       (* illegal combination *)
      END;
    | Mode.Tran =>
      CASE fop.mode OF
      | Mode.Copy => RETURN 16_00AA0029;       (* illegal combination *)
      | Mode.Tran => RETURN 16_00AA0029;       (* works! *)
      | Mode.Opaq => RETURN 16_00E20746;       (* works! *)
      | Mode.Swap => RETURN 16_006A01E9;       (* works! *)
      END;
    | Mode.Opaq =>
      CASE fop.mode OF
      | Mode.Copy => RETURN 16_00AA0029;       (* illegal combination *)
      | Mode.Tran => RETURN 16_00B8074A;       (* works! *)
      | Mode.Opaq => RETURN WinGDI.SRCCOPY;
      | Mode.Swap => RETURN WinGDI.NOTSRCCOPY;
      END;
    | Mode.Swap =>
      CASE fop.mode OF
      | Mode.Copy => RETURN 16_00AA0029;       (* illegal combination *)
      | Mode.Tran => RETURN 16_009A0709;  (* currently testing *)
      | Mode.Opaq => RETURN 16_00AA0029;       
      | Mode.Swap => RETURN WinGDI.SRCINVERT; 
      END;
    END;
  END ToTernaryRasterOp;


PROCEDURE NewPaintOp (VAR      st  : WinScreenType.T;
                      READONLY bop : Op;
                      READONLY fop : Op;
                               pix := -1): ScrnPaintOp.T =
  VAR 
    res := NEW(ScrnPaintOp.T, pix := pix);
    rec : OpRecord;
  BEGIN
    rec.bop  := bop;
    rec.fop  := fop;
    rec.brop := ToBinaryRasterOp  (bop, fop);
    rec.trop := ToTernaryRasterOp (bop, fop);

    LOCK st.trsl DO
      WITH n = NUMBER(st.optable^) DO
        IF n = st.opcount THEN
          WITH new = NEW(REF ARRAY OF OpRecord, 2 * n) DO
            SUBARRAY (new^, 0, n) := st.optable^;
            st.optable := new;
          END;
        END;
      END;
      res.id := st.opcount;
      st.optable[res.id] := rec;
      INC(st.opcount);
    END;
    RETURN res
  END NewPaintOp;


BEGIN
END WinScrnPaintOp.
