(* Copyright (C) 1993, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)

MODULE BoardServerX;

IMPORT Board, BoardServer, SharedBoardTbl,
       BoardX, StableBoardX, SharedBoard, StableError;

REVEAL T = Public BRANDED OBJECT
    mu: MUTEX;
    boards: SharedBoardTbl.Default;
  OVERRIDES
    init := Init;
    create := Create;
    open := Open;
    save := Save;
    close := Close;
    remove := Remove;
  END;

CONST ExpectedBoards = 10;

PROCEDURE Init (bs: T): T =
  BEGIN
    bs.mu := NEW (MUTEX);
    bs.boards := NEW (SharedBoardTbl.Default).init (ExpectedBoards);
    RETURN bs;
  END Init;

(* delete this function?  make a reinitialization function instead? *)
PROCEDURE Create (bs: T; boardName: TEXT): Board.T 
        RAISES {BoardServer.Failed} =
  BEGIN
    RETURN Open(bs, boardName);
  END Create;

PROCEDURE Open (bs: T; boardName: TEXT): Board.T 
        RAISES {BoardServer.Failed} =
  VAR board: SharedBoard.T;
      recovered: BOOLEAN;
  BEGIN
    TRY
      LOCK bs.mu DO
        IF bs.boards.get (boardName, board) THEN (* in-memory *)
          RETURN board;
        ELSE (* load board *)
          board := NEW (SharedBoard.T).init (boardName, recovered);
          IF (NOT recovered) THEN
            board.initS();
          END;
          board.initT();
          EVAL bs.boards.put (boardName, board);
          RETURN board;
        END;
      END;
    EXCEPT
      StableError.E => 
      RAISE BoardServer.Failed ("Could not open " & boardName);
    END;
  END Open; 

PROCEDURE Save (bs: T; boardName: TEXT) 
        RAISES {BoardServer.Failed} =
  VAR board: SharedBoard.T;
  BEGIN
    TRY
      LOCK bs.mu DO
        IF bs.boards.get (boardName, board) THEN
          StableBoardX.Checkpoint(board);
        ELSE
          RAISE BoardServer.Failed ("Board not loaded");
        END;
      END;
    EXCEPT
      StableError.E => 
      RAISE BoardServer.Failed ("Error saving board " & boardName);
    END;
  END Save;

PROCEDURE Close (bs: T; boardName: TEXT) 
        RAISES {BoardServer.Failed} =
  VAR board: SharedBoard.T;
  BEGIN
    TRY
      LOCK bs.mu DO
        IF bs.boards.get (boardName, board) THEN
          IF NOT BoardX.Busy (board) THEN
            EVAL bs.boards.delete (boardName, board);
            StableBoardX.Checkpoint(board);
            BoardX.Quit (board);
          END;
        ELSE
          RAISE BoardServer.Failed ("Board not loaded");
        END;
      END;
    EXCEPT
      StableError.E => 
      RAISE BoardServer.Failed ("Error saving board " & boardName);
    END;
  END Close;

PROCEDURE Remove (bs: T; boardName: TEXT) 
    RAISES {BoardServer.Failed} =
  VAR board: SharedBoard.T;
  BEGIN
    TRY
      LOCK bs.mu DO
        IF bs.boards.get (boardName, board) THEN
          IF BoardX.Busy (board) THEN 
            RAISE BoardServer.Failed ("Board is busy")
          END;
          EVAL bs.boards.delete (boardName, board);
          board.dispose();
        END
      END
    EXCEPT
      StableError.E => 
      RAISE BoardServer.Failed ("Problems deleting " & boardName);
    END;
  END Remove; 

BEGIN
END BoardServerX.
