structure ConnAnalyzer :CONN_ANALYZER = struct

structure SysIO = System.Unsafe.SysIO

val dummyEOF = AMLLrVals.Tokens.EOF(0,0)
val dummySEMI = AMLLrVals.Tokens.SEMICOLON(0,0)

exception Quit

val noPervasives = ref false

fun analyzeFile (filename:string) =
 let val source'stream = open_in filename 
           handle Io s => (outputc std_err ("? sml-conn: "^s^"\n");raise Quit)
     fun cleanup () = close_in source'stream in
 let val (context as {sourceStream,linePos,lineNum,anyErrors,...}) =
           ErrorMsg.newSource(filename,source'stream,false,std_err,NONE)
     val complain = ErrorMsg.error context
     fun parseerror(s,p1,p2) = complain (p1,p2) ErrorMsg.COMPLAIN s
     val lexarg = {comLevel = ref 0, lineNum = lineNum,
                   linePos = linePos, charlist = ref (nil : string list),
                   stringstart = ref 0, err = complain}
    val startingLexer' = AML_Lex.makeLexer (inputc sourceStream) lexarg
    val startingLexer  = LrParser.Stream.streamify startingLexer'
    val lookahead = 30

    fun do'parse (lexer) =
       let val (nextToken, restLexer) = LrParser.Stream.get lexer
       in linePos := [hd(!linePos)];
            if AMLParser.sameToken(nextToken,dummySEMI) then do'parse restLexer
             else if AMLParser.sameToken(nextToken,dummyEOF) then ()
             else 
                let val (_, lexer') =
                      AMLParser.parse (lookahead, lexer, parseerror, complain)
                 in if !anyErrors then raise Quit else do'parse lexer'
                end handle LrParser.ParseError => raise Quit
          end
 in
   do'parse startingLexer;
   cleanup()
 end
  handle any => (cleanup(); raise any)
end

fun analyze (doPrint:bool) (pathname:string) :unit =
 (Namespace.init (!noPervasives);
  analyzeFile pathname;
  if (not doPrint) then () else
        (print "source sml \""; print pathname; print "\"";
         Namespace.printAll std_out)
 ) handle Quit => ()

fun usageError () = output (std_err, "Usage: sml-conn [-n] [-y] filenames\n")

fun prList ([]:string list) = []
  | prList (head::tail) = (print head; print " "; prList tail)

fun connections (filename:string) =
  (analyze false filename; Namespace.connectionLists ())

fun printConnections (filename:string) = (analyze true filename; ())

fun processArguments arguments =
  case arguments of
     [] => usageError ()
   | ("-n" :: tail) => (noPervasives := true; processArguments tail)
   | (head :: tail) =>
       if substring (head, 0, 1) = "-"
         then usageError ()
         else app (analyze true) arguments

fun sml_conn ([],_) = ()
  | sml_conn (head::tail,_) = processArguments tail

fun export () = exportFn ("sml-conn", sml_conn)

end
