(*  Title: 	stringtree
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1988  University of Cambridge
*)


signature STRINGTREE = 
sig
   type 'a tree
   val below: 'a tree -> 'a list
   val compat: 'a tree * string list -> 'a list
   val insert: (string list * 'a) * 'a tree -> 'a tree
   val lookup: 'a tree * string list -> 'a list
   val null: 'a tree
end;


functor StringtreeFun () : STRINGTREE = 
struct

(*Trees indexed by string lists: each arc is labelled by a string
  Each node contains a list of items, and arcs to children
  Empty string addresses entire tree 
  Vital to return items the proper order:
    Items stored deeper are more specific and must preceed items stored
    above them;  order must be preserved in items stored at same level.
*)
datatype 'a tree = Stree of 'a list * (string * 'a tree) list

val null = Stree([],[]);

(*add an item to the list at the node addressed by the keys
  create node if not already present*)
fun insert (([],        x), Stree(xs,alist)) = Stree(x::xs, alist)
  | insert ((key::keys, x), Stree(xs,alist)) =
      let fun newpair tr = (key, insert((keys,x), tr)) 
	  fun inslist [] = [ newpair null ]
	    | inslist((keyi: string, tri)::alist) =
		if key=keyi then newpair tri :: alist
		else if key<keyi then (*absent, insert in alist*)
		    newpair null :: (keyi,tri) :: alist
		else (keyi,tri) :: inslist alist
      in  Stree(xs, inslist alist)  end;

(*Return the list of items at the given node, [] if no such node*)
fun lookup (Stree(xs,alist), []) = xs
  | lookup (Stree(xs,alist), key::keys) =
       (case assoc(alist,key) of 
	   None =>  []
	 | Some tr' => lookup(tr',keys));

(*Return the list of all items in the tree*)
fun below (Stree(xs,alist)) =
      let fun bel [] = []
	    | bel ((_,tr)::alist) = below tr  @  bel alist
      in  bel alist @ xs  end;

(*Return all items with compatible addresses:
  those where one address is a prefix of the other*)
fun compat (tr  ,  []) = below tr
  | compat (Stree(xs,alist), key::keys) = 
       (case assoc(alist,key) of 
	   None =>  xs
	 | Some tr' => compat(tr',keys) @ xs);

end;
