(*  Title: 	HOL/prover
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1989  University of Cambridge

Functor for constructing theorem provers from sets of rules

Rules must be classified as intr, elim, safe, hazardous, ...
*)


signature RULE_TAB =
sig
  structure Pure: PURE
  val type_rls: Pure.thm list
  val pairs: (Pure.thm * string) list
end;

signature PROVER =
sig
  structure HOL_Rule: HOL_RULE
  local open HOL_Rule.Pure
  in
  val best_tac : thm list -> tactic
  val comp_step_tac : thm list -> tactic
  val comp_tac : thm list -> tactic
  val determ_prop_tac : int -> tactic
  val expand_tac : int -> tactic
  val fast_tac : thm list -> int -> tactic
  val getrules : string -> thm list
  val haz_brls : (bool * thm) list
  val haz_cintr_rls : thm list
  val haz_elim_rls : thm list
  val haz_intr_rls : thm list
  val onestep_tac : int -> tactic
  val rigid_safep_tac : int -> tactic
  val safe0_brls : (bool * thm) list
  val safep_brls : (bool * thm) list
  val safe_brls : (bool * thm) list
  val safe_elim_rls : thm list
  val safe_intr_rls : thm list
  val step_tac : thm list -> int -> tactic
  val tp_tac : thm list -> tactic
  val typechk_tac: thm list -> tactic
end;
end;


functor ProverFun (structure HOL_Lemmas: HOL_LEMMAS and Rule_tab: RULE_TAB
	sharing HOL_Lemmas.HOL_Rule.Pure = Rule_tab.Pure)
    : PROVER = 
struct
structure HOL_Rule = HOL_Lemmas.HOL_Rule;

local  open  HOL_Rule.Pure  HOL_Rule  HOL_Lemmas 
in

	

val getrules = keyfilter Rule_tab.pairs;

(*Safe rules*)
val safe_intr_rls = getrules "safe_intr";

val safe_elim_rls = getrules "safe_elim";


(*Hazardous (unsafe) rules: require delay or lead to infinite regress*)
val haz_intr_rls = getrules "haz_intr";
val haz_cintr_rls = getrules "haz_cintr";
val haz_elim_rls = getrules "haz_elim";


val safe_brls = sort lessb 
   ((true,asm_rl) :: joinrules(safe_intr_rls, safe_elim_rls));

(*Note that all_elim precedes exists_intr, like in LK ????? *)
val haz_brls = sort lessb (joinrules(haz_intr_rls, haz_elim_rls));

(*0 subgoals vs 1 or more*)
val (safe0_brls, safep_brls) =
    partition (apl(0,op=) o subgoals_of_brl) safe_brls;



(*Try solving subgoal provided its CONCLUSION is rigid! *) 
val rigid_safe0_tac = SUBGOAL(fn (prem,i) =>
    if  is_rigid_prop (Logic.strip_assums_concl prem)
    then  biresolve_tac safe0_brls i  else  no_tac);


(*Try reducing subgoal provided all assumptions and concl are rigid! *) 
val rigid_safep_tac = SUBGOAL(fn (prem,i) =>
    if  is_rigid_prop (Logic.strip_assums_concl prem)  andalso
	forall (is_elem orf is_rigid_prop) (Logic.strip_assums_hyp prem)
    then  biresolve_tac safep_brls i  else  no_tac);


(** Complete (?) but slow prover *)

(*Either simple step (which might solve goal completely!) or a hazardous rule. 
  Should expand all quantifiers, not just pick one!  Redundant alternatives! *)
val expand_tac = IF_PROP
   (rigid_safe0_tac  APPEND'  contr_tac  APPEND'  
    (resolve_tac haz_elim_rls  THEN'  assume_tac)  APPEND'  
    biresolve_tac (joinrules(haz_cintr_rls,[])));

(*NOT COMPLETE BECAUSE contr_tac MAY INSTANTIATE VARIABLES****
  ***NEED A RIGID VERSION OF contr_tac TO USE IN expand_tac ***
  ***GENERAL VERSION TO BE CALLED IN comp_step_tac ***)


(*Attack subgoal i using propositional reasoning*)
val determ_prop_tac = IF_PROP
    (DETERM o REPEAT1 o
	(eq_assume_tac ORELSE' eq_mp_tac ORELSE' rigid_safep_tac));


(*Perform all obvious type checking -- for cleaning up the proof state*)
fun typechk_tac asms = 
    REPEAT (SOMEGOAL (determ_type_tac (asms@Rule_tab.type_rls)));

(*Deterministic Type/Propositional tactic*)
fun tp_tac asms =   REPEAT (FIRSTGOAL determ_prop_tac)
               THEN typechk_tac asms;

(*Single step for the "complete" prover.
  Type checking is done first to avoid waste if backtracking occurs
  afterwards*)
fun comp_step_tac asms =
    tp_tac asms  THEN
    COND (has_fewer_prems 1) all_tac
	 (FIRSTGOAL expand_tac  ORELSE
	  FIRSTGOAL (assume_tac  APPEND'  resolve_tac (asms@type_rls)));


(*Resolution with the safe rules is presumed deterministic,
  except for the axioms (like reflexivity and assumption!!) *)
val onestep_tac =
    rigid_safe0_tac  ORELSE'  mp_tac  ORELSE' 
    (DETERM o rigid_safep_tac);


(*Single step for the incomplete prover. 
  FAILS unless it makes reasonable progress.
  Can use asms as introduction rules (via joinrules) *)
fun step_tac asms = 
    let val btac = biresolve_tac (joinrules(asms, []) @ haz_brls);
	fun retac i = REPEAT1 (onestep_tac i)  ORELSE  btac i;
	fun tac i = retac i  THEN  typechk_tac asms
    in  tac  end;


(*assume_tac must be called as a last resort to solve trivial typing
  subgoals.  But step_tac does not call it since step_tac is also for
  interactive use, where a call to assume_tac may not be wanted.*)

(*Incomplete but fast.  Fails unless it solves one goal!
  Can use asms as introduction rules *)
fun fast_tac asms i = typechk_tac asms  THEN
		      DEPTH_SOLVE_1 (step_tac asms i  ORELSE  assume_tac i);

(*Incomplete but fast.  Fails unless it solves ALL goals!
  Can use asms as introduction rules *)
fun best_tac asms = typechk_tac asms  THEN
    BEST_FIRST (has_fewer_prems 1, size_of_state) 
	(step_tac asms 1  ORELSE  assume_tac 1);

(*[Less in] complete for first order logic. Fails unless it solves ALL goals!*)
fun comp_tac asms = 
    BEST_FIRST (has_fewer_prems 1, size_of_state) (comp_step_tac asms);

end;
end;

