(*  Title: 	Syntx
    Author: 	Tobias Nipkow
*)

signature SYNTAX =
sig
  (*Required components*)
  structure Symtab: SYMTAB
  type syntax
  val empty: syntax
  val pure: syntax
  val merge: syntax * syntax -> syntax
  val read: syntax -> typ -> string -> term
  val prin: syntax -> term -> unit
  val print_top_level: syntax -> term -> unit
  val scan_varname: string list -> (string*int) * string list
end;

signature PURE_SYNTAX =
sig
  include SYNTAX
  include MISC_SYNTAX
  include SYNTAX_DEF0
  val extend: syntax ->
	{logical_types: typ list, mixfix: mixfix list,
	 parse_translation: (string * (term list -> term)) list,
	 print_translation: (string * (term -> term)) list} -> syntax
  val print_syntax: syntax -> unit
end;

functor SYNTAX_FUN(structure Syntax_Def:SYNTAX_DEF and Parser:PARSER and
		   Pure_Ext:PURE_EXT and Printer:PRINTER
        sharing Syntax_Def.XGram = Parser.XGram = Printer.XGram
	and Syntax_Def = Pure_Ext.Syntax_Def)
	: PURE_SYNTAX =
struct

structure Syntax_Def = Syntax_Def;
structure Symtab = Syntax_Def.XGram.Symtab
open Syntax_Def
local open Parser.XGram Parser.Parse_Tree.Lexicon in

datatype Tables = Tab of {Gram: Parser.Gram, Lexicon: Lexicon,
			  TrTab: (term list -> term) Symtab.table,
			  PrintTab: Printer.PrintTab};

fun all_strings(opl:string Prod list) : string list =
    flat(map (fn Prod(_,syn,_,_) => terminals syn) opl);

fun str_to_tok(opl:string Prod list, lex:Lexicon) : Token Prod list =
      map (fn Prod(t,syn,s,pa) =>
              Prod(t,translate (hd o tokenize lex) syn,s,pa))
          opl;

fun xgram_to_tables(XGram{Roots=roots,Prods=ops,TrTab1=tr1,TrTab2=tr2}) =
    let val lex = mk_lexicon(all_strings ops);
        val gram = Parser.compile_xgram(roots, str_to_tok(ops,lex),
                                        token_to_string lex);
        val prtab = Printer.mk_print_tab(ops,tr2)
    in Tab{Gram=gram,TrTab=tr1,Lexicon=lex,PrintTab=prtab} end;

type extension =
	{logical_types: typ list, mixfix: mixfix list,
	 parse_translation: (string * (term list -> term)) list,
	 print_translation: (string * (term -> term)) list};

datatype GramGraph = emptyGG | extGG of GramGraphR * extension |
                     mergeGG of GramGraphR * GramGraphR
withtype GramGraphR = GramGraph ref;

datatype syntax = Syntax of GramGraphR * Tables;

fun flatGG ggr (xg,v) = if ggr mem v then (xg,v) else flatGG' ggr (xg,ggr::v)

and flatGG'(ref emptyGG) xgv = xgv |
    flatGG'(ref(extGG(ggr,ext))) xgv =
      let val (xg',v') = flatGG ggr xgv
      in (Syntax_Def.extend xg' ext, v') end |
    flatGG'(ref(mergeGG(ggr1,ggr2))) xgv = flatGG ggr1 (flatGG ggr2 xgv)

fun flattenGG ggr = fst(flatGG ggr (Syntax_Def.empty,[]));

fun mk_tables ggr = Syntax(ggr,xgram_to_tables(flattenGG ggr));

val empty = mk_tables(ref emptyGG);

fun extend (Syntax(ggr,_)) ext = mk_tables(ref(extGG(ggr,ext)));
fun merge(Syntax(ggr1,_),Syntax(ggr2,_)) = mk_tables(ref(mergeGG(ggr1,ggr2)));

val pure = extend empty Pure_Ext.pureExt;

fun read (Syntax(_,Tab{Gram=g,Lexicon=lex,TrTab=trt,...})) ty s =
    let val tokens = tokenize lex s
    in Parser.Parse_Tree.pt_to_term (apl(trt,Symtab.lookup))
         (Parser.parse(g, Syntax_Def.typ_to_nt ty, tokens))
       handle Parser.SYNTAX_ERR tl => error
          ("Syntax error at\n" ^
           space_implode " " (map (token_to_string lex) tl))
    end;

fun prin(Syntax(_,Tab{PrintTab=prtab,...})) = Printer.printer prtab;
fun print_top_level(Syntax(_,Tab{PrintTab=prtab,...})) =
	Printer.printer_nl prtab;

fun print_syntax(Syntax(_,Tab{Gram=g,Lexicon=lex,...})) =
	Parser.print_gram(g,fn t => token_to_string lex t);

val scan_varname = scan_varname;

end;

open Pure_Ext

end;
