(*  Title: 	CTT/ex/arith
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1989  University of Cambridge

Examples about elementary arithmetic: addition, multiplication, etc.
Tests definitions and simplifier.
*)


(** Addition *)

(*typing of add: short and long versions*)

val add_typing = prove_goal arith_thy 
    "[| a:N;  b:N |] ==> a #+ b : N"
 (fn prems=>
  [ (rewrite_goals_tac arith_defs),
    (typechk_tac prems) ]);

val add_typing_long = prove_goal arith_thy 
    "[| a=c:N;  b=d:N |] ==> a #+ b = c #+ d : N"
 (fn prems=>
  [ (rewrite_goals_tac arith_defs),
    (equal_tac prems) ]);


(*computation for add: 0 and successor cases*)

val add_comp0 = prove_goal arith_thy 
    "b:N ==> 0 #+ b = b : N"
 (fn prems=>
  [ (rewrite_goals_tac arith_defs),
    (rew_tac prems) ]);

val add_comp_succ = prove_goal arith_thy 
    "[| a:N;  b:N |] ==> succ(a) #+ b = succ(a #+ b) : N"
 (fn prems=>
  [ (rewrite_goals_tac arith_defs),
    (rew_tac prems) ]); 


(** Multiplication *)

(*typing of mult: short and long versions*)

val mult_typing = prove_goal arith_thy 
    "[| a:N;  b:N |] ==> a #* b : N"
 (fn prems=>
  [ (rewrite_goals_tac arith_defs),
    (typechk_tac([add_typing]@prems))  ]);

val mult_typing_long = prove_goal arith_thy 
    "[| a=c:N;  b=d:N |] ==> a #* b = c #* d : N"
 (fn prems=>
  [ (rewrite_goals_tac arith_defs),
    (equal_tac (prems@[add_typing_long]))  ]);


(*computation for mult: 0 and successor cases*)

val mult_comp0 = prove_goal arith_thy 
    "b:N ==> 0 #* b = 0 : N"
 (fn prems=>
  [ (rewrite_goals_tac arith_defs),
    (rew_tac prems)  ]);

val mult_comp_succ = prove_goal arith_thy 
    "[| a:N;  b:N |] ==> succ(a) #* b = b #+ (a #* b) : N"
 (fn prems=>
  [ (rewrite_goals_tac arith_defs),
    (rew_tac prems)  ]);


(** Difference *)

(*typing of difference*)

val diff_typing = prove_goal arith_thy 
    "[| a:N;  b:N |] ==> a - b : N"
 (fn prems=>
  [ (rewrite_goals_tac arith_defs),
    (typechk_tac prems) ]);

val diff_typing_long = prove_goal arith_thy 
    "[| a=c:N;  b=d:N |] ==> a - b = c - d : N"
 (fn prems=>
  [ (rewrite_goals_tac arith_defs),
    (equal_tac prems) ]);



(*computation for difference: 0 and successor cases*)

val diff_comp0 = prove_goal arith_thy 
    "a:N ==> a - 0 = a : N"
 (fn prems=>
  [ (rewrite_goals_tac arith_defs),
    (rew_tac prems) ]);

(*Note: rec(a, 0, %z w.z) is pred(a). *)

val diff_0_eq_0 = prove_goal arith_thy 
    "b:N ==> 0 - b = 0 : N"
 (fn prems=>
  [ (N_elim_tac "b" 1),
    (rewrite_goals_tac arith_defs),
    (hyp_rew_tac prems) ]);


(*Essential to simplify FIRST!!  (Else we get a critical pair)
  succ(a) - succ(b) rewrites to   pred(succ(a) - b)  *)
val diff_succ_succ = prove_goal arith_thy 
    "[| a:N;  b:N |] ==> succ(a) - succ(b) = a - b : N"
 (fn prems=>
  [ (rewrite_goals_tac arith_defs),
    (hyp_rew_tac prems),
    (N_elim_tac "b" 1),
    (hyp_rew_tac prems) ]);



(*** Simplification *)

val arith_typing_rls =
  [add_typing, mult_typing, diff_typing];

val arith_congr_rls =
  [add_typing_long, mult_typing_long, diff_typing_long];

val congr_rls = arith_congr_rls@standard_congr_rls;

val arith_comp_rls =
  [add_comp0, add_comp_succ,
   mult_comp0, mult_comp_succ,
   diff_comp0, diff_0_eq_0, diff_succ_succ];


print_depth 1;

structure Arith_simp_data: SIMP_DATA =
struct
  structure Pure = Pure
  val refl= refl_elem
  val sym= sym_elem
  val trans= trans_elem
  val refl_red= refl_red
  val trans_red= trans_red
  val red_if_equal= red_if_equal
  val default_rls = arith_comp_rls @ comp_rls
  val routine_tac = routine_tac (arith_typing_rls @ routine_rls)
end;

structure Arith_simp = SimpFun (Arith_simp_data);

fun arith_rew_tac prems = make_rew_tac
    (Arith_simp.norm_tac(congr_rls, prems));

fun hyp_arith_rew_tac prems = make_rew_tac
    (Arith_simp.cond_norm_tac(prove_cond_tac, congr_rls, prems));

print_depth 6;


(**********
  Addition
 **********)

(*Associative law for addition*)
val add_assoc = prove_goal arith_thy 
    "[| a:N;  b:N;  c:N |] ==> (a #+ b) #+ c = a #+ (b #+ c) : N"
 (fn prems=>
  [ (N_elim_tac "a" 1),
    (hyp_arith_rew_tac prems)  (*7 secs*)  ]);


(*Commutative law for addition.  Can be proved using three inductions.
  Must simplify after first induction!!  Orientation of rewrites is delicate*)  
val add_commute = prove_goal arith_thy 
    "[| a:N;  b:N |] ==> a #+ b = b #+ a : N"
 (fn prems=>
  [ (N_elim_tac "a" 1),
    (hyp_arith_rew_tac prems),
    (N_elim_tac "b" 2),
    (resolve_tac [sym_elem] 1),
    (N_elim_tac "b" 1),
    (hyp_arith_rew_tac prems)   ]);


(****************
  Multiplication
 ****************)

(*Commutative law for multiplication
val mult_commute = prove_goal arith_thy 
    "[| a:N;  b:N |] ==> a #* b = b #* a : N"
 (fn prems=>
  [ (N_elim_tac "a" 1),
    (hyp_arith_rew_tac prems),
    (N_elim_tac "b" 2),
    (resolve_tac [sym_elem] 1),
    (N_elim_tac "b" 1),
    (hyp_arith_rew_tac prems)   ]);   NEEDS COMMUTATIVE MATCHING
***************)

(*right annihilation in product*)
val mult_right0 = prove_goal arith_thy 
    "a:N ==> a #* 0 = 0 : N"
 (fn prems=>
  [ (N_elim_tac "a" 1),
    (hyp_arith_rew_tac prems)  ]);

(*right successor law for multiplication*)
val mult_right_succ = prove_goal arith_thy 
    "[| a:N;  b:N |] ==> a #* succ(b) = a #+ (a #* b) : N"
 (fn prems=>
  [ (N_elim_tac "a" 1),
(*swap round the associative law of addition*)
    (hyp_arith_rew_tac (prems @ [ sym_elem RES add_assoc ])),  
(*8 secs, leaves a goal involving a commutative law*)
    (REPEAT (assume_tac 1  ORELSE  
            resolve_tac
             (prems@[add_commute,mult_typing_long,add_typing_long]@
	       intr_long_rls@[refl_elem])   1))  ]);

(*Commutative law for multiplication*)
val mult_commute = prove_goal arith_thy 
    "[| a:N;  b:N |] ==> a #* b = b #* a : N"
 (fn prems=>
  [ (N_elim_tac "a" 1),
    (hyp_arith_rew_tac (prems @ [mult_right0, mult_right_succ])) ]);

(*addition distributes over multiplication*)
val add_mult_dist = prove_goal arith_thy 
    "[| a:N;  b:N;  c:N |] ==> (a #+ b) #* c = (a #* c) #+ (b #* c) : N"
 (fn prems=>
  [ (N_elim_tac "a" 1),
(*swap round the associative law of addition*)
    (hyp_arith_rew_tac (prems @ [ sym_elem RES add_assoc ])) (*11 secs*) ]);


(*Associative law for multiplication*)
val mult_assoc = prove_goal arith_thy 
    "[| a:N;  b:N;  c:N |] ==> (a #* b) #* c = a #* (b #* c) : N"
 (fn prems=>
  [ (N_elim_tac "a" 1),
    (hyp_arith_rew_tac (prems @ [add_mult_dist]))  (*7 secs*) ]);


(************
  Difference
 ************

Difference on natural numbers, without negative numbers
  a - b = 0  iff  a<=b    a - b = succ(c) iff a>b   *)

val diff_self_eq_0 = prove_goal arith_thy 
    "a:N ==> a - a = 0 : N"
 (fn prems=>
  [ (N_elim_tac "a" 1),
    (hyp_arith_rew_tac prems) ]);


val add_right0 = trans_elem RES add_commute RESN (3, [add_comp0]);

(*Addition is the inverse of subtraction: if b<=x then b#+(x-b) = x.
  An example of induction over a quantified formula (a product).
  Uses rewriting with a quantified, implicative inductive hypothesis.*)
val prems =
goal arith_thy 
    "b:N ==> ?a : PROD x:N. Eq(N, b-x, 0) --> Eq(N, b #+ (x-b), x)";
by (N_elim_tac "b" 1);
(*strip one "universal quantifier" but not the "implication"*)
by (resolve_tac intr_rls 3);  
(*case analysis on x in
    (succ(u) <= x) --> (succ(u)#+(x-succ(u)) = x) *)
by (N_elim_tac "kc" 4 THEN assume_tac 4); 
(*Prepare for simplification of types -- the antecedent succ(u)<=x *)
by (resolve_tac [replace_type] 5);
by (resolve_tac [replace_type] 4);
by (arith_rew_tac prems); 
(*10 secs: solves first 0 goal, simplifies others.  Two sugbgoals remain.
  Both follow by rewriting, (2) using quantified induction hyp*)
by (intr_tac[]);  (*strips remaining PRODs*)
by (hyp_arith_rew_tac (prems@[add_right0]));  (*8 secs*)
by (assume_tac 1);
val add_inverse_diff_lemma = uresult();


(*Version of above with premise   b-a=0   i.e.    a >= b.
  Using Prod_elim does not work -- for ?B(?a) is ambiguous.
  Instead, add_inverse_diff_lemma states the desired induction scheme;
    the use of RES below instantiates Vars in Prod_elim automatically. *)
val prems =
goal arith_thy "[| a:N;  b:N;  b-a = 0 : N |] ==> b #+ (a-b) = a : N";
by (resolve_tac [Eq_elim] 1);
by (resolve_tac [ Prod_elim RES (Prod_elim RES add_inverse_diff_lemma) ] 1);
by (REPEAT (resolve_tac (prems@[Eq_intr]) 1));
val add_inverse_diff = result();


(********************
  Absolute difference
 ********************)

(*typing of absolute difference: short and long versions*)

val absdiff_typing = prove_goal arith_thy 
    "[| a:N;  b:N |] ==> a |-| b : N"
 (fn prems=>
  [ (rewrite_goals_tac arith_defs),
    (typechk_tac prems) ]);

val absdiff_typing_long = prove_goal arith_thy 
    "[| a=c:N;  b=d:N |] ==> a |-| b = c |-| d : N"
 (fn prems=>
  [ (rewrite_goals_tac arith_defs),
    (equal_tac prems) ]);

val absdiff_self_eq_0 = prove_goal arith_thy 
    "a:N ==> a |-| a = 0 : N"
 (fn prems=>
  [ (rewrite_goals_tac [absdiff_def]),
    (arith_rew_tac (prems@[diff_self_eq_0])) ]);

val absdiff_comp0 = prove_goal arith_thy 
    "a:N ==> 0 |-| a = a : N"
 (fn prems=>
  [ (rewrite_goals_tac [absdiff_def]),
    (hyp_arith_rew_tac prems) ]);


val absdiff_succ_succ = prove_goal arith_thy 
    "[| a:N;  b:N |] ==> succ(a) |-| succ(b)  =  a |-| b : N"
 (fn prems=>
  [ (rewrite_goals_tac [absdiff_def]),
    (hyp_arith_rew_tac prems) ]);

(*Note how easy using commutative laws can be?  ...not always... *)
val prems =
goal arith_thy "[| a:N;  b:N |] ==> a |-| b = b |-| a : N";
by (rewrite_goals_tac [absdiff_def]);
by (resolve_tac [add_commute] 1);
by (typechk_tac ([diff_typing]@prems));
val absdiff_commute = result();

(*If a+b=0 then a=0.   Surprisingly tedious*)
val prems =
goal arith_thy "[| a:N;  b:N |] ==> ?c : PROD u: Eq(N,a#+b,0) .  Eq(N,a,0)";
by (N_elim_tac "a" 1);
by (resolve_tac [replace_type] 3);
by (arith_rew_tac prems);
by (intr_tac[]);  (*strips remaining PRODs*)
by (resolve_tac [ F_elim RES zero_ne_succ ] 2);
by (biresolve_tac [(true,sym_elem RES Eq_elim)] 3);
by (typechk_tac ([add_typing] @prems));
val add_eq0_lemma = uresult();


(*Version of above with the premise  a+b=0.
  Again, resolution instantiates variables in Prod_elim *)
val prems =
goal arith_thy "[| a:N;  b:N;  a #+ b = 0 : N |] ==> a = 0 : N";
by (resolve_tac [Eq_elim] 1);
by (resolve_tac [Prod_elim RES add_eq0_lemma] 1);
by (resolve_tac [Eq_intr] 3);
by (ALLGOALS (resolve_tac prems));
val add_eq0 = result();


(*Here is a lemma to infer a-b=0 and b-a=0 from a|-|b=0, below. *)
val prems =
goal arith_thy "[| a:N;  b:N;  a |-| b = 0 : N |] ==> ?a : SUM v: Eq(N, a-b, 0) . Eq(N, b-a, 0)";
by (intr_tac[]);
by eqintr_tac;
by (resolve_tac [add_eq0] 2);
by (resolve_tac [add_eq0] 1);
by (resolve_tac [add_commute RS trans_elem] 6);
by (typechk_tac (diff_typing:: map (rewrite_rule [absdiff_def]) prems));
val absdiff_eq0_lem = uresult();

(*if  a |-| b = 0  then  a = b  
  proof: a-b=0 and b-a=0, so b = a+(b-a) = a+0 = a*)
val prems =
goal arith_thy "[| a |-| b = 0 : N;  a:N;  b:N |] ==> a = b : N";
by (resolve_tac [Eq_elim] 1);
by (resolve_tac [absdiff_eq0_lem RS Sum_elim] 1);
by (REPEAT_FIRST (filt_resolve_tac prems 1));
by eqintr_tac;
by (resolve_tac [trans_elem RES sym_elem RES add_inverse_diff ] 1);
by (resolve_tac [Eq_elim] 3  THEN  assume_tac 3);
by (hyp_arith_rew_tac (prems@[add_right0]));
val absdiff_eq0 = result();

(***********************
  Remainder and Quotient
 ***********************)

(*typing of remainder (//): short and long versions*)

val mod_typing = prove_goal arith_thy
    "[| a:N;  b:N |] ==> a//b : N"
 (fn prems=>
  [ (rewrite_goals_tac [mod_def]),
    (typechk_tac (absdiff_typing::prems)) ]);
 
val mod_typing_long = prove_goal arith_thy
    "[| a=c:N;  b=d:N |] ==> a//b = c//d : N"
 (fn prems=>
  [ (rewrite_goals_tac [mod_def]),
    (equal_tac (prems@[absdiff_typing_long])) ]);
 

(*computation for //: 0 and successor cases*)

val mod_comp0 = prove_goal arith_thy "b:N ==> 0//b = 0 : N"
 (fn prems=>
  [ (rewrite_goals_tac [mod_def]),
    (rew_tac(absdiff_typing::prems)) ]);

val mod_comp_succ = prove_goal arith_thy 
"[| a:N; b:N |] ==> succ(a)//b = rec(succ(a//b) |-| b, 0, %x y.succ(a//b)) : N"
 (fn prems=>
  [ (rewrite_goals_tac [mod_def]),
    (rew_tac(absdiff_typing::prems)) ]);


(*typing of quotient: short and long versions*)

val quo_typing = prove_goal arith_thy "[| a:N;  b:N |] ==> a / b : N"
 (fn prems=>
  [ (rewrite_goals_tac [quo_def]),
    (typechk_tac ([absdiff_typing,mod_typing]@prems)) ]);

val quo_typing_long = prove_goal arith_thy
   "[| a=c:N;  b=d:N |] ==> a / b = c / d : N"
 (fn prems=>
  [ (rewrite_goals_tac [quo_def]),
    (equal_tac (prems @ [absdiff_typing_long, mod_typing_long])) ]);

val quo_typing_rls = [mod_typing, quo_typing, absdiff_typing];


(*computation for quotient: 0 and successor cases*)

val quo_comp0 = prove_goal arith_thy "b:N ==> 0 / b = 0 : N"
 (fn prems=>
  [ (rewrite_goals_tac [quo_def]),
    (rew_tac([mod_typing, absdiff_typing] @ prems)) ]);

val quo_comp_succ =
prove_goal arith_thy "[| a:N;  b:N |] ==> succ(a) / b = \
\    rec(succ(a)//b, succ(a / b), %x y. a / b) : N"
 (fn prems=>
  [ (rewrite_goals_tac [quo_def]),
    (rew_tac([mod_typing]@prems)) ]);


(*Version of above with same condition as the // one*)
val quo_comp_succ2 = prove_goal arith_thy
    "[| a:N;  b:N |] ==> \
\    succ(a) / b =rec(succ(a//b) |-| b, succ(a / b), %x y. a / b) : N"
 (fn prems=>
  [ (resolve_tac [ trans_elem RES quo_comp_succ ] 1),
    (rew_tac(quo_typing_rls @ prems @ [mod_comp_succ])),
    (N_elim_tac "succ(a//b)|-|b" 1),
    (rew_tac ([mod_typing, quo_typing, absdiff_typing] @prems)) ]);


(*for case analysis on whether a number is 0 or a successor*)
val iszero_decidable = prove_goal arith_thy
    "a:N ==> rec(a, inl(eq), %ka kb.inr(<ka, eq>)) : \
\		      Eq(N,a,0) + (SUM x:N. Eq(N,a, succ(x)))"
 (fn prems=>
  [ (N_elim_tac "a" 1),
    (resolve_tac [Plus_intr_inr] 3),
    (resolve_tac [Plus_intr_inl] 2),
    eqintr_tac,
    (equal_tac prems) ]);


(*Main Result.  Holds when b is 0 since   a//0 = a     and    a/0 = 0  *)
val prems =
goal arith_thy "[| a:N;  b:N |] ==> a//b  #+  (a/b) #* b = a : N";
by (N_elim_tac "a" 1);
by (arith_rew_tac (quo_typing_rls @ prems @
	[mod_comp0,mod_comp_succ, quo_comp0, quo_comp_succ2])); 
(*5 secs*)
by (resolve_tac [ Eq_elim ] 1);
(*case analysis on   succ(u//b)|-|b  *)
by (res_inst_tac [("a1", "succ(ka // b) |-| b", Aterm)]
	(Plus_elim RES iszero_decidable) 1);
by (biresolve_tac [(true,Sum_elim)] 3);
by (hyp_arith_rew_tac (prems @ quo_typing_rls @
	[mod_comp0,mod_comp_succ, quo_comp0, quo_comp_succ2])); 
(*36 secs*)
(*Replace one occurence of  b  by succ(ka//b).  Clumsy!*)
by (resolve_tac [ trans_elem RES add_typing_long ] 1);
by (biresolve_tac [(true, sym_elem RES absdiff_eq0 RES Eq_elim)] 1);
by (resolve_tac [refl_elem] 3);
by (hyp_arith_rew_tac (prems @ quo_typing_rls)); 
(*4 secs*)
val mod_quo_equality = result();


writeln"Reached end of file.";


(*21 August 88: loaded this file in 262 secs*)
(*2 September 1988: loaded this file in 290 seconds*)
