signature Vector' =
sig

(* CONSTANT VECTORS

   Created by:	Dave Berry, LFCS, University of Edinburgh
		db@lfcs.ed.ac.uk
   Date:	31 Oct 1989

   Based on the Array signatures provided by Edinburgh ML, NJML and Poly ML.

   Maintenance:	Author


   DESCRIPTION

   A Vector is a single-dimensional constant polymorphic array.  The first
   element has the index 0.  Vectors are equal if they contain the same
   elements and these elements admit equality.

   To create a Vector of Vectors, use the "generate" functions.


   SEE ALSO

   RefVector.


   NOTES

   The name was chosen so that there could also be types for matrices (type
   Matrix), variable matrices (type RefMatrix), tensors, etc.  There is
   also scope for a type Array of arbitrary dimension, indexed by an integer
   list of the appropriate length.

   A possible implementation would be a view of a random access file.
   Possibly there should be a ConstVector signature (combining the Const
   and Vector signatures) to make this easier.
*)

  val version: real
   (* version = 0.1 *)


(* TYPES *)

  type 'a Vector


(* CREATORS *)

  val create: Nat -> 'a -> 'a Vector
   (* create n i; create a Vector of n locations, each containing i.
      Raise (Nat ("create", n)) if n < 0. *)

  val generate: Nat -> (int -> 'a) -> 'a Vector
   (* generate n f; create a Vector v of n locations, (v sub 0) to
      (v sub (n-1)) with (v sub i) initialised to (f i).
      Raise (Nat ("generate", n)) if n < 0. *)

  val generate': Nat -> ('b -> 'a * 'b) -> 'b -> 'a Vector
   (* generate' n f base; create a Vector of n locations, (v sub 0) to
      (v sub (n-1)) with (v sub 0) initialised to (# 1 (f base)) and
      (v sub i (i > 0)) initialised to (# 1 (f (# 2 f_i))), where f_i is
      the result of the i^th application of f.
      Raise (Nat ("generate'", n)) if n < 0. *)


(* CONVERTERS *)

  val list: 'a Vector -> 'a list
   (* list v; make a list containing (only) the elements of v, in
      the same order. *)

  val fromList: 'a list -> 'a Vector
   (* from_list l; make a Vector containing (only) the elements of l, in
      the same order. *)


(* OBSERVERS *)

  val empty: 'a Vector -> bool
   (* empty v; returns true if v is empty. *)

  val size: 'a Vector -> Nat
   (* size v; return the number of elements in v. *)


(* SELECTORS *)

  exception Sub of string * int
   (* Sub (fn, n); raised when the function named fn is passed the arguments
      n and l such that (not (0 <= n <= length l)). *)

  (* infix 9 sub *)
  val sub: 'a Vector * int -> 'a
   (* v sub n; return the n+1'th element of v.
      Raise (Sub ("sub", n)) if not (0 <= n <= size v). *)

  val nth: int -> 'a Vector -> 'a
   (* v sub n; return the n+1'th element of v.
      Raise (Sub ("nth", n)) if not (0 <= n <= size v). *)

  exception Extract of int * int
  val extract: int -> int -> 'a Vector -> 'a Vector
   (* extract start finish v; returns the sub-vector of v starting with
      (v sub start) and ending with (v sub (finish - 1)).
      Returns the empty vector if (start = finish).
      Raise (Extract (start, finish)) if not (0 <= start,finish <= size v). *)


(* MANIPULATORS *)

  val rev: 'a Vector -> 'a Vector
   (* rev v; builds a vector containing the elements of v in reverse order. *)

  (* infix 6 ^ *)
  val ^ : 'a Vector * 'a Vector -> 'a Vector
   (* v ^ v'; builds a Vector containing the elements of v' appended to those
      of v. *)


(* REDUCERS *)

  val foldR: ('a -> 'b -> 'b) -> 'b -> 'a Vector -> 'b
   (* foldR f base v; folds using f associating to the right over the
      base element.
      foldR f base [a1,a2,...,an] = op(a1,f(a2,...,f(an,base)...)). *)

  val foldL: ('a -> 'b -> 'b) -> 'b -> 'a Vector -> 'b
   (* foldL f base v; folds using op associating to the left over the
      base element.
      foldL f base [a1,a2,...,an] = f(an,...,f(a2,f(a1,base))...). *)

  exception Empty of string

  val foldR': ('a -> 'a -> 'a) -> 'a Vector -> 'a
   (* foldR' f v; folds using f associating to the right over the
      last element of v.  Raises (Empty "foldR'") if v is empty. *)

  val foldL': ('a -> 'a -> 'a) -> 'a Vector -> 'a
   (* foldL' f v; folds using f associating to the left over the
      first element of v.  Raises (Empty "foldL'") if v is empty. *)

  val pairwise: ('a -> 'a -> bool) -> 'a Vector -> bool
   (* pairwise f v; true if (f (v sub i) (v sub (i + 1))) is true for all
      0 <= i < size v, or if v is empty. *)

(* ITERATORS *)

  val map: ('a -> 'b) -> 'a Vector -> 'b Vector
   (* map f v; builds a new vector by applying f to each element of v. *)

  val apply: ('a -> unit) -> 'a Vector -> unit
   (* apply f v; applies f to each element of v. *)

  val iterate: ('a * int -> 'b) -> 'a Vector -> 'b Vector
   (* iterate f v; builds a new vector by applying f to each element of v
      paired with its index. *)

  val iterateApply: ('a * int -> unit) -> 'a Vector -> unit
   (* iterateApply f v; applies f to each element of v paired
      with its index. *)
end
