loadSig "StringType";
loadSig "StringParse";

signature String =
sig

(* ASCII STRINGS

   Created by:	Dave Berry, LFCS, University of Edinburgh
		db@lfcs.ed.ac.uk
   Date:        4 Oct 1989

   Maintenance:	Author


   DESCRIPTION

   Standard functions on the built-in type "string".

   Many of these functions use an orthogonal naming scheme.  They can be
   grouped as follows:

   f;
     if f is "last", return the last element.
     if f is "sub", return the (n-1)th element.
     if f is "first", return the first element that satisfies a predicate p.
     if f is "all", return the list of elements that satisfy p.
     if f is "prefix", return the prefix of elements that satisfy p.

   dropF ... l; l with (f l) removed.
   splitF ... l; a pair of (f l, dropF l) in the order they appear in l.
   updateF ... l; l with (f l) replaced with a value.
   changeF ... l; l with (f l) replaced with (g (f l)).
   spliceF ... l; l with (f l) replaced with the elements in another list.
   insertF ... l; l with the elements in another list inserted before (f l).
   appendF ... l; l with the elements in another list inserted after (f l).

   The following combinations not provided, because the changeF etc. functions
   operate on an element of a list and the ...Prefix functions return sublists:
     changePrefix, updatePrefix, insertPrefix, appendPrefix, splicePrefix.

   The "last" functions raise "Empty" if l is empty.
   The "sub" functions raise "Sub" if not (0 <= n <= size l - 1).
   The "first" functions (ecept dropFirst) raise "First" if there aren't
        any elements that satisfy p.

   Functions such as  index  and  search  take an integer
   offset.  This is because it's more efficient to index into
   a string than to take a substring.


   NOTES

   Possibly should add revFirst, dropRevFirst, ... and postfix, dropPostfix,
   ... sets of functions.

   Possibly first, ... and prefix, ... should take an integer offset.

   Possibly the integer offset parameters and the revXXX constructs should
   be dropped, relying on the compiler to optimise away intermediate string
   constructions.

   Possibly all these functions should be defined on string lists to
   minimise repeated calls to implode and explode.  Alternatively, we could
   rely on the compiler to optimise them away.

   Possibly there should be a dropExtract function to delete a range
   of characters from a string.

   Possibly should add extractLast for convenience.
*)

  include StringType


(* PERVASIVES *)

  val size: string -> int
  exception Ord
  val ord: string -> int
  exception Chr
  val chr: int -> string
  val explode: string -> string list
  val implode: string list -> string
  val ^ : string * string -> string


(* OBJECTS *)

  structure Object: Object

  (* The Object' structure contains relational operations that
     ignore the case of letters. *)

  structure Object': Object
    sharing type Object.T = Object.T = string

  type T
    sharing type T = string

  include StringParse

  val fixedWidth: bool

  (* The comparison functions use lexicographic ordering based on the
     ASCII ordering. *)

  (* The following comparisons are those from the Object substructure,
     in which case differences are significant. *)

  val lt: string -> string -> bool

  val le: string -> string -> bool

  val gt: string -> string -> bool

  val ge: string -> string -> bool

  val eq: string -> string -> bool

  val ne: string -> string -> bool

  (* The following comparisons are those from the Object' substructure,
     in which case differences are ignored. *)

  val lt': string -> string -> bool

  val le': string -> string -> bool

  val gt': string -> string -> bool

  val ge': string -> string -> bool

  val eq': string -> string -> bool

  val ne': string -> string -> bool


(* CREATORS *)

  val create: Nat -> string -> string
   (* create n s; returns a string composed of n copies of s.
      Raises (Nat ("create", n)) if n < 0. *)


(* CONVERTERS *)

  val fromFile: string -> string
   (* fromFile name; read the contents of name into a string.
      Raises the pervasive exception Io if something goes wrong. *)

  val file: string -> string -> unit
   (* file name s; write s to the new file called name.
      Raises the pervasive exception Io if something goes wrong. *)


(* OBSERVERS *)

  val exists: (string -> bool) -> string -> bool
   (* exists p s; true if there exists a character in s satisfying p *)

  val forAll: (string -> bool) -> string -> bool
   (* forAll p s; true if every character in s satisfies p *)


  exception Sub of string * int
   (* Sub (fn, n); raised if the function named fn is called with an out of
      range argument n. *)

  val prefixes: string -> string -> int -> bool
   (* prefixes s1 s2 n; returns true if s1 is a prefix of the substring of
      s2 starting at (s2 sub n).   Raises (Sub ("prefixes", n)) if not
      (0 <= n < size s2). *)

  val postfixes: string -> string -> int -> bool
   (* postfixes s1 s2 n; returns true if s1 is a postfix of the substring of
      s2 ending at (s2 sub n).   Raises (Sub ("postfixes", n)) if not
      (0 <= n < size s2). *)

  val index: (string -> bool) -> string -> int -> (int, unit) Result
   (* index p s n; returns the position in s of the first character
      after (s sub n) (inclusive) that satisfies p.
      Raises (Sub ("index", n)) if not (0 <= n < size s2). *)

  val revIndex: (string -> bool) -> string -> int -> (int, unit) Result
   (* revIndex p s n; returns the position in s of the last character
      before (s sub n) (inclusive) that satisfies p.
      Raises (Sub ("revIndex", n)) if not (0 <= n < size s2). *)

  val search: string -> string -> int -> (int, unit) Result
   (* search s' s n; find the first occurrence of s' in s after (s sub n)
      (inclusive) and return the index of the first character in that
      substring.  Raises (Sub ("search", n)) if not (0 <= n < size s). *)

  val revSearch: string -> string -> int -> (int, unit) Result
   (* revSearch s' s n; find the last occurrence of s' in s before (s sub n)
      (inclusive) and return the index of the first character in that
      substring.  Raises (Sub ("revSearch", n)) if not (0 <= n < size s). *)

  val occurs: string -> string -> int -> bool
   (* occurs s' s n; return true if s' occurs in s after (s sub n) (inclusive).
      Raises (Sub ("occurs", n)) if not (0 <= n < size s). *)

  val revOccurs: string -> string -> int -> bool
   (* revOccurs s' s n; return true if s' occurs in s before (s sub n)
      (inclusive).  Raises (Sub ("revOccurs", n)) if not (0 <= n < size s). *)


(* MANIPULATING STRING TYPE *)

  val upper: string -> string
   (* upper s; if the first character in s is a lower case letter,
      returns s with that letter in upper case.  Otherwise leaves s unchanged.
      Raises (Empty "upper") if s is empty. *)

  val lower: string -> string
   (* lower s; if the first character in s is an upper case letter,
      returns s with that letter in lower case.  Otherwise leaves s unchanged.
      Raises (Empty "lower") if s is empty. *)

  val ascii: string -> string
   (* ascii s; if the first character in s is not an ascii character,
      returns s with the top bit stripped from that character.  Otherwise
      leaves s unchanged.  Raises (Empty "ascii") if s is empty. *)

  val control: string -> string
   (* control s; if the first character in s is in the range @ .. _ , returns
      s with that character replaced with the corresponding control character.
      Otherwise leaves s unchanged.  Raises (Empty "ascii") if s is empty. *)


(* MANIPULATING THE NTH ELEMENT *)

  (* infix 9 sub *)
  val sub: string * int -> string
   (* s sub n; return the nth character of s.
      Raises (Sub ("sub", n)) if not (0 <= n <= size l - 1). *)

  val nth: int -> string -> string
   (* nth n s; return the nth character of s.
      Raises (Sub ("nth", n)) if not (0 <= n <= size l - 1). *)

  exception Char of string * string

  val updateSub: int -> string -> string -> string
   (* updateSub n s' s; returns s with (s sub n) replaced by s'.
      Raises (Char ("updateSub", s')) if (size s' <> 1).
      Raises (Sub ("updateSub", n)) if not (0 <= n <= size s - 1). *)

  val changeSub: int -> (string -> string) -> string -> string
   (* changeSub n f s; returns s with (s sub n) replaced by (f (s sub n)).
      Raises (Sub ("changeSub", n)) if not (0 <= n <= size s - 1). *)

  val insertSub: int -> string -> string -> string
   (* insertSub n s' s; returns s with s' inserted before (s sub n).
      Raises (Sub ("insertSub", n)) if not (0 <= n <= size s - 1). *)

  val appendSub: int -> string -> string -> string
   (* appendSub n s' s; returns s with s' appended after (s sub n).
      Raises (Sub ("appendSub", n)) if not (0 <= n <= size s - 1). *)

  val spliceSub: int -> string -> string -> string
   (* spliceSub n s' s; returns s with (s sub n) replaced by s'.
      Unlike updateSub, s' may be any length.
      Raises (Sub ("spliceSub", n)) if not (0 <= n <= size s - 1). *)


(* ACCESSING A RANGE OF ELEMENTS *)

  exception Extract of int * int

  val extract: int -> int -> string -> string
   (* extract s start finish; 0 <= start,finish <= size (s); returns the
      substring starting with (s sub start) and ending with
      (s sub (finish - 1)).  Returns "" if (start = finish).  Raises
      (Extract (start, finish)) if not (0 <= start <= finish <= size l - 1). *)


(* MANIPULATING THE FIRST ELEMENT THAT SATISFIES A PREDICATE *)

  exception First of string

  val first: (string -> bool) -> string -> string
   (* first p s; returns the first character in s satisfying p.
      Raises (First "first") if p doesn't hold for any character in s. *)

  val dropFirst: (string -> bool) -> string -> string 
   (* dropFirst p s; returns s without the first of its characters (if any)
      that satisfy p. *)

  val splitFirst: (string -> bool) -> string -> (string * string)
   (* splitFirst p s;
      returns (s sub 0 ^ ... ^ s sub (n - 1), extract n (size s - 1) s),
      where (s sub n) is the first character in s that satisfies p.
      Raises (First "splitFirst") if p doesn't hold for any character in s. *)

  val updateFirst: (string -> bool) -> string -> string -> string
   (* updateFirst p s' s; returns s with (first p s) replaced by s'.
      Raises (First "updateFirst") if there is no (first p s).
      Raises (Char ("updateFirst", s')) if (size s' <> 1). *)

  val changeFirst: (string -> bool) -> (string -> string) -> string -> string
   (* changeFirst p f s; returns s with (first p s) replaced by
      (f (first p s)).  Raises (First "changeFirst") if there is no
      (first p s).  *)

  val insertFirst: (string -> bool) -> string -> string -> string
   (* insertFirst p s' s; returns s with s' inserted before
      (first p l).  Raises (First "insertFirst") if there is no (first p l). *)

  val appendFirst: (string -> bool) -> string -> string -> string
   (* appendFirst p s' s; returns s with s' appended after
      (first p l).  Raises (First "insertFirst") if there is no (first p l). *)

  val spliceFirst: (string -> bool) -> string -> string -> string
   (* spliceFirst p s' s; returns s with (first p s) replaced by s'.
      Raises (First "spliceFirst") if there is no (first p s).
      Unlike updateFirst, s' may be any length. *)


(* TAKING A PREFIX OF ELEMENTS THAT SATISFY A PREDICATE *)

  val prefix: (string -> bool) -> string -> string
   (* getPrefix p s; returns the largest prefix of s in which each character
      satisfies p. *)

  val dropPrefix: (string -> bool) -> string -> string
   (* dropPrefix p s; returns s without the largest prefix in which
      every character satisfies p. *)

  val splitPrefix: (string -> bool) -> string -> string * string
   (* splitPrefix p s: returns the pair composed of the largest prefix
      of s that satisfies p and what remains of s without it. *)


(* MANIPULATING ALL ELEMENTS THAT SATISFY A PREDICATE *)

  val all: (string -> bool) -> string -> string
   (* all p s: returns a string formed from the characters in s that
      satisfy p. *)

  val dropAll: (string -> bool) -> string -> string
   (* dropAll p s: returns a string formed from the characters in s that don't
      satisfy p. *)

  val splitAll: (string -> bool) -> string -> (string * string)
   (* splitAll p s; returns (all p s, dropAll p s). *)

  val updateAll: (string -> bool) -> string -> string -> string
   (* updateAll p s' s; returns s with each character that satisfies p
      replaced by s'.  Raises (Char ("updateAll", s')) if (size s' <> 1). *)

  val changeAll: (string -> bool) -> (string -> string) -> string -> string
   (* changeAll p f s; returns s with each character c that satisfies p
      replaced by (f c). *)

  val insertAll: (string -> bool) -> string -> string -> string
   (* insertAll p s' s; returns s with s' inserted before each
      character in s that satisfies p. *)

  val appendAll: (string -> bool) -> string -> string -> string
   (* appendAll p s' s; returns s with s' appended after each
      character in s that satisfies p. *)

  val spliceAll: (string -> bool) -> string -> string -> string
   (* spliceAll p s' s; returns s with each character that satisfies p
      replaced by s'. *)


(* OTHER MANIPULATORS *)

  val subst: string -> string -> string -> string
   (* subst c s' s; replaces all occurrences of c in s by s'.
      Raises (Char ("subst", s')) if size c > 0. *)

  val subst': string -> string -> string -> string
   (* subst c s' s; replaces all occurrences of c in s, ignoring case, by s'.
      Raises (Char ("subst'", s')) if size c > 0. *)

  val dropRepeats': string -> string
   (* dropRepeats' s; removes duplicated characters from l. *)

  val rev: string -> string
   (* rev s; returns the reflection of s. E.g. rev "abc" = "cba". *)

  val showAscii: string -> string
   (* showAscii s; returns s with all characters expanded to the form they
      would have in a string literal.  E.g. showAscii "\n" = "\\n".  *)

  val padL: string -> int -> string -> string
   (* padL c w s; pads the string s with character c on the
      left until its size is w.  Raises (Char ("padL", c)) if (size c <> 1). *)

  val padR: string -> int -> string -> string
   (* padR c w s; pads the string s with character c on the
      right until its size is w.  Raises (Char ("padR", c)) if (size c <> 1). *)

  val padC: string -> int -> string -> string
   (* padC c w s; pads the string s with character c to centre the existing
      satring in a new string of size w.
      Raises (Char ("padC", c)) if (size c <> 1). *)

  val truncL: int -> string -> string
   (* truncL w s; truncates the string s by removing characters on the left
      until its size is w. *)

  val truncR: int -> string -> string
   (* truncR w s; truncates the string s by removing characters on the right
      until its size is w. *)

  val truncC: int -> string -> string
   (* truncC w s; truncates the string s by removing characters equally from
      both ends until its size is w. *)

  val dropL: string -> string -> string
   (* dropL c s; removes occurrence of the character c on the left of s.
      Raises (Char ("dropL", c)) if (size c <> 1). *)

  val dropR: string -> string -> string
   (* dropR c s; removes occurrence of the character c on the right of s.
      Raises (Char ("dropR", c)) if (size c <> 1). *)

  val words: string -> string -> string list
   (* words sep s; splits s into words separated by one or more occurrences
      of the characters in sep. *)

  val words': string -> string -> string -> string list
   (* words sep singles s; splits s into words that either contain just one
      of the characters in singles or that are separated by one or more
      occurrences of the characters in sep. *)


(* ITERATORS *)

  val map: (string -> string) -> string -> string
   (* map f s; builds a string by applying f to each character in s. *)

  val apply: (string -> unit) -> string -> unit
   (* apply f s; applies f to each character in s. *)

  val mapAll: (string -> bool) -> (string -> string) -> string -> string
   (* map p f s; builds a string by applying f to each character in s
      that satisfies p. *)

  val applyAll: (string -> bool) -> (string -> unit) -> string -> unit
   (* applyAll p f s; applies f to each character in s that satisfies p. *)
end
