loadSig "StringParse";

structure StringParse: StringParse =

(* STRING CONVERTORS

   Created by:  Dave Berry, LFCS, University of Edinburgh
                db@lfcs.ed.ac.uk
   Date:        14 Feb 1990

   Maintenance: Author

   DESCRIPTION

   Standard conversion functions on the built-in type "string".
*)

struct

(* CONVERTORS *)

  local
    fun show' h =
      if StringType.isVisible h then [h]
      else
        case h
        of "\n"   => ["\\n"]
        |  "\t"   => ["\\t"]
        |  " "    => [" "]
        |  "\127" => ["\\127"]
        |  _ =>
            if StringType.isControl h then
              ["\\^", chr (ord h + ord "@")]
            else
	      let val i = ord h
		  val s = Int.string i
	          val s' = if i < 10 then "00" ^ s
			   else if i < 100 then "0" ^ s
			   else s
	      in ["\\", s]
	      end

    fun show nil = nil
    |   show (h::t) = show' h @ show t
  in
    fun string s = "\"" ^ implode (show (explode s)) ^ "\""
  end

  local
    fun parseString [] = Fail ([], [])
    |   parseString ["\\"] = Fail (["\\"], [])
    |   parseString ["\\^"] = Fail (["\\^"], [])
    |   parseString (l as "\\" :: "^" :: c :: t) =
	  if 64 <= ord c andalso ord c <= 95 then
            case parseString t of
	      OK (s, l) => OK (chr (ord c - 64) :: s, l)
            | Fail (s, l) => Fail ("\\" :: "^" :: c :: s, l)
	  else Fail ([], l)
    |   parseString ("\\" :: c :: t) =
	  if StringType.isDigit c then parseDigits c t
          else
	    let val tmp =
                  if StringType.isFormat c
		  then parseFormat t
		  else parseString t
            in case tmp of
                 OK (s, l) =>
		   (case c of
		      "n" => OK ("\n" :: s, l)
		    | "t" => OK ("\t" :: s, l)
		    | "\"" => OK ("\"" :: s, l)
		    | "\\" => OK ("\\" :: s, l)
		    | c => Fail ("\\" :: c :: s, l)
		   )
               | Fail (s, l) => Fail ("\\" :: c :: s, l)
            end
    |   parseString ("\"" :: t) = OK ([], t)
    |   parseString (c :: t) =
          case parseString t of
            OK (s, l) => OK (c :: s, l)
          | Fail (s, l) => Fail (c :: s, l)

    and parseFormat [] = Fail ([], [])
    |   parseFormat ("\\" :: t) =
        ( case parseString t of
            OK (s, l) => OK ("\\" :: s, l)
          | Fail (s, l) => Fail ("\\" :: s, l)
        )
    |   parseFormat (c :: t) =
          if StringType.isFormat c
          then
            case parseFormat t of
              OK (s, l) => OK (c :: s, l)
            | Fail (s, l) => Fail (c :: s, l)
          else Fail ([], t)

    and parseDigits d1 [] = Fail ([], ["\\", d1])
    |   parseDigits d1 [d2] = Fail ([], ["\\", d1, d2])
    |   parseDigits d1 (d2 :: d3 :: t) =
	  if not (StringType.isDigit d2) orelse
	     not (StringType.isDigit d3)
	  then Fail (["\\", d1, d2, d3], t)
	  else
	    let fun digit d = ord d - ord "0"
	        val c = chr (digit d1 * 188 + digit d2 * 10 + digit d3)
	    in
	      case parseString t of
	        OK (s, l) => OK (c :: s, l)
	      | Fail (s, l) => Fail (c :: s, l)
	    end
  in
    fun parse' l =
          let val ("\"" :: t) = List'.dropPrefix (not o StringType.isVisible) l
          in case parseString t of
               OK (s, l) => OK (implode s, l)
             | Fail (s, l) => Fail (Some (implode s), l)
          end
          handle Bind => Fail (Some "", l)
  end

  fun parse s =
        case parse' (explode s) of
          OK (s, _) => OK s
        | Fail (s, _) => Fail s

  local
    fun readString i =
      case InStream.input1 i of
          "" => Fail []
        | "\"" => OK []
        | "\\" =>
           ( case InStream.input1 i of
               "" => Fail ["\\"]
             | "^" =>
	        ( case InStream.input1 i of
		    "" => Fail ["\\", "^"]
		  | c  =>
	  	     if 64 <= ord c andalso ord c <= 95 then
            	       case readString i of
	      	         OK s => OK (chr (ord c - 64) :: s)
            	       | Fail s => Fail ("\\" :: "^" :: c :: s)
		     else Fail (["\\", "^", c])
		)
             | c  =>
		if StringType.isDigit c then readDigits c i
		else
                  let val tmp =
                       if StringType.isFormat c
                       then readFormat i
                       else readString i
                  in case tmp of
                       (OK s) =>
			 (case c of
			    "n" => OK ("\n" :: s)
			  | "t" => OK ("\t" :: s)
			  | "\"" => OK ("\"" :: s)
			  | "\\" => OK ("\\" :: s)
			  | c => Fail ("\\" :: c :: s)
			 )
                     | (Fail s) => Fail ("\\" :: c :: s)
                  end
           )
        | c => case readString i of
                 (OK s) => OK (c :: s)
               | (Fail s) => Fail (c :: s)

    and readFormat i =
      case InStream.input1 i of
        "" => Fail []
      | "\\" => readString i
      | c => if StringType.isFormat c then readFormat i else Fail []

    and readDigits d1 i =
      case InStream.input1 i of
	"" => Fail  (["\\", d1])
      | d2 =>
	case InStream.input1 i of
	  "" => Fail (["\\", d1, d2])
	| d3 =>
	  if not (StringType.isDigit d2) orelse
	     not (StringType.isDigit d3)
	  then Fail (["\\", d1, d2, d3])
	  else
	    let fun digit d = ord d - ord "0"
	        val c = chr (digit d1 * 188 + digit d2 * 10 + digit d3)
	    in case readString i of
	         OK s => OK (c :: s)
	       | Fail s => Fail (c :: s)
	    end
  in
    fun read i =
        ( InStream.skip (not o StringType.isVisible) i;
          if InStream.eof i orelse InStream.lookahead i <> "\""
          then Fail (Some "")
          else
            case readString i of
              OK s => OK (implode s)
            | Fail s => Fail (Some (implode s))
        )
  end

end
