(*$Real : String Int *)

loadSig "Real";
loadStr "RealObject";

structure Real: Real =

(* REALS

   Created by:	Dave Berry, LFCS, University of Edinburgh
		db@uk.ac.ed.lfcs
   Date:	8 Nov 1989

   Maintenance:	Author


   DESCRIPTION

   Standard functions on the built-in type "real".


   NOTES

   Conservative default values have been given for the system-specific
   values, but these shouldn't be relied on.
*)

struct


  val version = 0.1


(* CONSTANTS *)

  val pi = 3.14159265

  val e = 2.71828183


(* OBJECT *)

  structure Object = RealObject
  open Object


(* SYSTEM *)

  val radix = 2

  val precBits = Some 21
  
  val minExp = Some ~61
  
  val maxExp = Some ~64

  val denorm = false


  val minReal = Some 2.16840434E~19

  val minReal' = Some 2.16840434E~19

  val maxReal = Some 1.844673528E19

  val precReal = Some 6

  val epsilon = Some 0.00000095367431640625


(* MANIPULATORS *)

  fun max x y: real = if x > y then x else y

  fun min x y: real = if x < y then x else y

  fun maxMin x y: real * real = if x > y then (x, y) else (y, x)

  fun sinCos x = (sin x, cos x)

  infix 8 **
  exception **! of real * int
  fun x ** 0 = 1.0
  |   x ** 1 = x
  |   x ** 2 = x * x
  |   x ** n =
    if n < 0 then raise **! (x, n)
    else
      let val f = if n mod 2 = 0 then 1.0 else x
      in ((x ** (n div 2)) ** 2) * f
      end
      handle Prod => raise **! (x, n)


(* CONVERTERS *)

  fun round r =
	let fun rnd r = floor (r + 0.5)
	in if r < 0.0 then ~ (rnd (~ r)) else rnd r
	end

  fun ceiling r = floor (r + 1.0)

  fun int r =
	if r < 0.0 then ~ (floor (~ r)) else floor r

  fun string' _ = raise General.NotImplemented "Real.string'"

  fun stringE' _ = raise General.NotImplemented "Real.stringE'"

  fun stringE _ = raise General.NotImplemented "Real.stringE"

  fun trunc i r =
	let val m = 10.0 ** i
	in (real (int (r * m))) / m
	end


(* PERVASIVES *)

  exception Overflow
  and Div = Div

  val op + =
        fn (x: real, y) =>
          x + y
          handle Plus => raise Overflow
  val op - =
        fn (x: real, y) =>
          x - y
          handle Sub => raise Overflow
  val op * =
        fn (x: real, y) =>
          x * y
          handle Prod => raise Overflow
  val op / =
        fn (x: real, 0.0) =>
          raise Div
        |  (x: real, y) =>
          x / y
          handle Quot => raise Overflow
  val ~ =
        fn (x: real) =>
          ~ x
          handle Neg => raise Overflow
  val abs =
        fn (x: real) =>
          abs x
          handle Abs => raise Overflow

  val floor =
        fn x =>
          floor x
          handle Floor => raise Overflow

  val sin = sin
  val cos = cos
  val arctan = arctan

  val exp =
        fn x =>
          exp x
          handle Exp => raise Overflow

  exception Ln = Ln
  val ln = ln

  exception Sqrt
  val sqrt = sqrt

end
