(*$MonoVector : Object Vector GeneralTypes InStreamType OutStreamType *)

loadSig "MonoVector";

functor MonoVector (

(* CONSTANT VECTORS OF A GIVEN TYPE

   Created by:	Dave Berry, LFCS, University of Edinburgh
		db@lfcs.ed.ac.uk
   Date:	21 Dec 1989

   Maintenance: Author


   DESCRIPTION

   A constant vector is a unique sequence of objects that can't be changed.
   Vector is an equality type.  There is only one empty Vector.

   This is a reference implementation.  Most systems will provide most of
   the following as built-in functions.  Any sensible implementation will
   provide constant time access to each element.


   SEE ALSO

   Vector, RefVector.
*)

  structure Object: Object
): MonoVector =
struct

  val version = 0.1



(* TYPES *)

  type Object = Object.T

  type MonoVector = Object Vector.Vector


(* CREATORS *)

  val create = Vector.create

  val generate = Vector.generate

  val generate' = Vector.generate'


(* ITERATORS *)

  val map = Vector.map

  val apply = Vector.apply

  val iterate = Vector.iterate

  val iterateApply = Vector.iterateApply


(* OBSERVERS *)

  val size = Vector.size

  fun empty v = (size v = 0)

  fun eq v v' = Vector.eq Object.eq v v'
  fun ne v v' = Vector.ne Object.ne v v'
  fun lt v v' = Vector.lt Object.lt v v'
  fun le v v' = Vector.le Object.le v v'
  fun gt v v' = Vector.gt Object.gt v v'
  fun ge v v' = Vector.ge Object.ge v v'


(* CONVERTORS *)

  val list = Vector.list

  val fromList = Vector.fromList

  fun stringSep start finish sep v =
        Vector.stringSep start finish sep Object.string v

  fun string v =
	if Object.fixedWidth
	then stringSep "" "" "" v
	else stringSep "" "" " " v

  exception Sep of string * string * string * string

  (* The parse, parse' and read functions assume that entries have a fixed
     width or are separated by formatting characters. *)

  fun parseSepN' start finish sep n l =
        Vector.parseSepN' start finish sep Object.parse' n l
        handle Vector.Sep x => raise Sep x

  fun parseSep' start finish sep l =
        Vector.parseSep' start finish sep Object.parse' l
        handle Vector.Sep x => raise Sep x

  fun parseN' n l =
        if n < 0 then raise General.Nat ("parseN'", n)
	else if Object.fixedWidth
        then parseSepN' "" "" "" n l
        else parseSepN' "" "" " " n l

  fun parse' l =
	if Object.fixedWidth
        then parseSep' "" "" "" l
        else parseSep' "" "" " " l

  fun parseSepN start finish sep n s =
        if n < 0 then raise General.Nat ("parseSepN", n)
        else
          case parseSepN' start finish sep n (explode s) of
            OK (v, _) => OK v
          | Fail (x, _) => Fail x

  fun parseSep start finish sep s =
        case parseSep' start finish sep (explode s) of
          OK (v, _) => OK v
        | Fail (x, _) => Fail x

  fun parseN n s =
        if n < 0 then raise General.Nat ("parseN", n)
	else if Object.fixedWidth
        then parseSepN "" "" "" n s
        else parseSepN "" "" " " n s

  fun parse s =
	if Object.fixedWidth
        then parseSep "" "" "" s
        else parseSep "" "" " " s

  fun readSepN start finish sep n i =
        Vector.readSepN start finish sep Object.read n i
        handle Vector.Sep x => raise Sep x

  fun readSep start finish sep i =
        Vector.readSep start finish sep Object.read i
        handle Vector.Sep x => raise Sep x

  fun readN n i =
        if n < 0 then raise General.Nat ("readN", n)
	else if Object.fixedWidth
        then readSepN "" "" "" n i
        else readSepN "" "" " " n i

  fun read i =
	if Object.fixedWidth
        then readSep "" "" "" i
        else readSep "" "" " " i

  fun fromFile name =
	let fun readList i =
		  case Object.read i
		  of Fail None => (InStream.closeIn i; [])
		  |  Fail (Some x) => (InStream.closeIn i; [x])
		  |  OK x => x :: readList i
	in fromList (readList (InStream.openIn name))
	end

  fun file name v =
	let val os = OutStream.openOut name
	    fun out s = OutStream.output' os s
	in Vector.apply (out o Object.string) v;
	   OutStream.closeOut os
	end


(* SELECTORS *)

  exception Sub of string * int
  val sub = Vector.sub
	    handle Vector.Sub x => raise Sub x
  infix 9 sub;

  fun nth n v = v sub n
		handle Sub _ => raise Sub ("nth", n)

  exception Extract of int * int
  fun extract start finish v =
	Vector.extract start finish v
	handle Vector.Extract x => raise Extract x


(* MANIPULATORS *)

  val rev = Vector.rev

  infix 6 ^
  val op ^ = Vector.^


(* REDUCERS *)

  val foldL = Vector.foldL

  val foldR = Vector.foldR

  exception Empty of string

  fun foldR' f v =
	if empty v then raise Empty "foldR'"
        else Vector.foldR' f v

  fun foldL' f v =
	if empty v then raise Empty "foldL'"
        else Vector.foldL' f v

  val pairwise = Vector.pairwise
end
