(* Gene Rollins
   School of Computer Science
   Carnegie-Mellon University
   Pittsburgh, PA 15213
   rollins@cs.cmu.edu *)

functor FileListFun (structure DirFile :DIRFILE
                     structure Pathname :PATHNAME) :FILELIST = struct

datatype limitation =
        SKIPFILES of string list
      | SKIPDIRS of string list
      | EXTS of string list

datatype fileListDescription =
        FILES of string list
      | DIRS of string list
      | RECDIRS of string list
      | LIMIT of (fileListDescription * limitation)

fun check'ext name (ext :string, result :bool) =
  result orelse ((Pathname.extension name) = ext)

fun check'ext'list (ext'list:string list) (name:string, result:string list) =
  if (ext'list = []) orelse (fold (check'ext name) ext'list false)
    then name::result else result

fun isEqual name (name2 :string, result :bool) = result orelse (name = name2)

fun isInList lst name = fold (isEqual name) lst false

fun filter'skips lst (name:string, result:string list) =
  if isInList lst name then result else name::result

fun filter names skips exts :string list =
  let val exts'only = fold (check'ext'list exts) names [] in
    fold (filter'skips skips) exts'only []
  end

fun translate (fs :fileListDescription) (skipfiles :string list)
              (skipdirs :string list) (exts :string list) :string list =
  let fun maybe'cons'file name lst =
        if ((exts = []) orelse (fold (check'ext name) exts false))
           andalso (not (isInList skipfiles name))
          then name::lst else lst
      fun maybe'cons'dir name lst =
        if isInList skipdirs name then lst else name::lst
      fun pathlists dirname filename filetype (dirlst, filelst) =
        if (ordof(filename,0) = (ord ".")) then (dirlst, filelst)
          else
            let val pathname = Pathname.mergeDirFile dirname filename in
              case filetype of
                 DirFile.FILE => (dirlst, maybe'cons'file pathname filelst)
               | DirFile.SYMLINK => (dirlst, maybe'cons'file pathname filelst)
               | DirFile.DIR => (maybe'cons'dir pathname dirlst, filelst)
            end
      fun dirpathlists (dirname :string, result) =
        DirFile.fold pathlists [DirFile.FOLLOWDIRS] dirname result
  in case fs of
     (FILES names) =>
       filter names skipfiles exts
   | (DIRS dirs) =>
       let val (dirlst,filelst) = fold dirpathlists dirs ([],[]) in
         filelst
       end
   | (RECDIRS dirs) =>
       let val (dirlst,filelst) = fold dirpathlists dirs ([],[])
           val morefiles =
             if dirlst = [] then []
               else translate (RECDIRS dirlst) skipfiles skipdirs exts
       in filelst @ morefiles end
   | (LIMIT (fset :fileListDescription, (SKIPFILES skip'list))) =>
       translate fset (skip'list @ skipfiles) skipdirs exts
   | (LIMIT (fset :fileListDescription, (SKIPDIRS skip'list))) =>
       translate fset skipfiles (skip'list @ skipdirs) exts
   | (LIMIT (fset :fileListDescription, (EXTS ext'list))) =>
       translate fset skipfiles skipdirs (ext'list @ exts)
  end

fun create (f:fileListDescription) :string list = translate f [] [] []

end
