/* 
   Copyright (C) 1990 C van Reewijk, email: dutentb.uucp!reeuwijk

This file is part of GLASS.

GLASS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

GLASS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GLASS; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* File: tmfn.c
 *
 * Handle function expressions.
 */

/* Standard UNIX libraries and functions */
#include <stdio.h>
#include <ctype.h>

extern char *getenv();
extern char *calloc();

/* tm library */
#include <tmc.h>

/* local definitions */
#include "tmdefs.h"

#include "refex.h"
#include "tmds.h"
#include "tmstring.h"
#include "debug.h"
#include "tmerror.h"
#include "tmfn.h"
#include "tmglobal.h"
#include "tmmisc.h"
#include "tmvar.h"

/***************************************************************
 *   tm functions                                              *
 ***************************************************************/

/* implementation of functions. Each function is given a parameter
   string and an origin string for error messages
 */

/* -- numerical functions -- */

/* max */
static char *fnmax( sl )
 string_list sl;
{
    register int max;
    register int n;
    register unsigned int ix;

    if( sl->sz<1 ){
	line_error( BADPARNO );
	return( newintstr( 0 ) );
    }
    max = atoi( sl->arr[0] );
    for( ix=0; ix<sl->sz; ix++ ){
	cknumpar( sl->arr[ix] );
	n = atoi( sl->arr[ix] );
	if( n>max )
	    max = n;
    }
    return( newintstr( max ) );
}

/* min */
static char *fnmin( sl )
 string_list sl;
{
    register int min;
    register int n;
    register unsigned int ix;

    if( sl->sz<1 ){
	line_error( BADPARNO );
	return( newintstr( 0 ) );
    }
    min = atoi( sl->arr[0] );
    for( ix=0; ix<sl->sz; ix++ ){
	cknumpar( sl->arr[ix] );
	n = atoi( sl->arr[ix] );
	if( n<min )
	    min = n;
    }
    return( newintstr( min ) );
}

/* addition */
static char *fnplus( sl )
 string_list sl;
{
    register int sum;
    register unsigned int ix;

    sum = 0;
    for( ix=0; ix<sl->sz; ix++ ){
	cknumpar( sl->arr[ix] );
	sum += atoi( sl->arr[ix] );
    }
    return( newintstr( sum ) );
}

/* subtraction */
static char *fnsubtract( sl )
 string_list sl;
{
    int a;

    if( sl->sz != 2 ){
	line_error( BADPARNO );
	return( new_string( "" ) );
    }
    cknumpar( sl->arr[0] );
    cknumpar( sl->arr[1] );
    a = atoi( sl->arr[0] ) - atoi( sl->arr[1] );
    return( newintstr( a ) );
}

/* multiplication */
static char *fntimes( sl )
 string_list sl;
{
    register int prod;
    unsigned int ix;

    prod = 1;
    for( ix=0; ix<sl->sz; ix++ ){
	cknumpar( sl->arr[ix] );
	prod *= atoi( sl->arr[ix] );
    }
    return( newintstr( prod ) );
}

/* division */
static char *fndiv( sl )
 string_list sl;
{
    int a;

    if( sl->sz != 2 ){
	line_error( BADPARNO );
	return( new_string( "" ) );
    }
    cknumpar( sl->arr[0] );
    cknumpar( sl->arr[1] );
    a = atoi( sl->arr[0] ) / atoi( sl->arr[1] );
    return( newintstr( a ) );
}

/* modulus */
static char *fnmod( sl )
 string_list sl;
{
    int a;

    if( sl->sz != 2 ){
	line_error( BADPARNO );
	return( new_string( "" ) );
    }
    cknumpar( sl->arr[0] );
    cknumpar( sl->arr[1] );
    a = atoi( sl->arr[0] ) % atoi( sl->arr[1] );
    return( newintstr( a ) );
}

/* -- comparison functions -- */

/* < */
static char *fnless( sl )
 string_list sl;
{
    bool b;

    if( sl->sz != 2 ){
	line_error( BADPARNO );
	return( new_string( "" ) );
    }
    cknumpar( sl->arr[0] );
    cknumpar( sl->arr[1] );
    b = atoi( sl->arr[0] ) < atoi( sl->arr[1] );
    return( newboolstr( b ) );
}

/* <= */
static char *fnlesseq( sl )
 string_list sl;
{
    bool b;

    if( sl->sz != 2 ){
	line_error( BADPARNO );
	return( new_string( "" ) );
    }
    cknumpar( sl->arr[0] );
    cknumpar( sl->arr[1] );
    b = atoi( sl->arr[0] ) <= atoi( sl->arr[1] );
    return( newboolstr( b ) );
}

/* > */
static char *fngreater( sl )
 string_list sl;
{
    bool b;

    if( sl->sz != 2 ){
	line_error( BADPARNO );
	return( new_string( "" ) );
    }
    cknumpar( sl->arr[0] );
    cknumpar( sl->arr[1] );
    b = atoi( sl->arr[0] ) > atoi( sl->arr[1] );
    return( newboolstr( b ) );
}

/* >= */
static char *fngreatereq( sl )
 string_list sl;
{
    bool b;

    if( sl->sz != 2 ){
	line_error( BADPARNO );
	return( new_string( "" ) );
    }
    cknumpar( sl->arr[0] );
    cknumpar( sl->arr[1] );
    b = atoi( sl->arr[0] ) >= atoi( sl->arr[1] );
    return( newboolstr( b ) );
}

/* == */
static char *fneq( sl )
 string_list sl;
{
    bool b;

    if( sl->sz != 2 ){
	line_error( BADPARNO );
	return( new_string( "" ) );
    }
    cknumpar( sl->arr[0] );
    cknumpar( sl->arr[1] );
    b = atoi( sl->arr[0] ) == atoi( sl->arr[1] );
    return( newboolstr( b ) );
}

/* != */
static char *fnneq( sl )
 string_list sl;
{
    bool b;

    if( sl->sz != 2 ){
	line_error( BADPARNO );
	return( new_string( "" ) );
    }
    cknumpar( sl->arr[0] );
    cknumpar( sl->arr[1] );
    b = atoi( sl->arr[0] ) != atoi( sl->arr[1] );
    return( newboolstr( b ) );
}

/* strcmp */
static char *fnstrcmp( sl )
 string_list sl;
{
    int cmp;

    if( sl->sz != 2 ){
	line_error( BADPARNO );
	return( new_string( "" ) );
    }
    cmp = strcmp( sl->arr[0] , sl->arr[1] );
    if( cmp == 0 ) return( new_string( "0" ) );
    if( cmp < 0 ) return( new_string( "-1" ) );
    return( new_string( "1" ) );
}

/* eq */
static char *fnstreq( sl )
 string_list sl;
{
    int cmp;

    if( sl->sz != 2 ){
	line_error( BADPARNO );
	return( new_string( "" ) );
    }
    cmp = strcmp( sl->arr[0] , sl->arr[1] );
    return( newboolstr( cmp == 0 ) );
}

/* neq */
static char *fnstrneq( sl )
 string_list sl;
{
    int cmp;

    if( sl->sz != 2 ){
	line_error( BADPARNO );
	return( new_string( "" ) );
    }
    cmp = strcmp( sl->arr[0] , sl->arr[1] );
    return( newboolstr( cmp != 0 ) );
}

/* -- string functions -- */

/* len */
static char *fnlen( sl )
 string_list sl;
{
    return( newintstr( (int) sl->sz ) );
}

/* strpad */
static char *fnstrpad( sl )
 string_list sl;
{
    string w;
    register string wp;
    unsigned int len;
    string pw;
    string buf;
    register string bufp;

    if( sl->sz != 3 ){
	line_error( BADPARNO );
	return( new_string( "" ) );
    }
    buf = new_string( "" );
    w = sl->arr[0];
    pw = sl->arr[2];
    if( *pw == '\0' ){
	line_error( BADPADDING );
	return( buf );
    }
    cknumpar( sl->arr[1] );
    len = atoi( sl->arr[1] );
    buf = ckrealloc( buf, len+1 );
    wp = w;
    bufp = buf;
    while( len!=0 && *wp != '\0' ){
	*bufp++ = *wp++;
	len--;
    }
    wp = pw;
    while( len!=0 ){
	if( *wp == '\0' ){
	    wp = pw;
	}
	*bufp++ = *wp++;
	len--;
    }
    *bufp = '\0';
    return( buf );
}

/* strlen */
static char *fnstrlen( sl )
 string_list sl;
{
    int l;

    if( sl->sz<1 ){
	l = 0;
    }
    else {
	l = strlen( sl->arr[0] );
    }
    return( newintstr( l ) );
}

/* toupper */
static char *fntoupper( sl )
 string_list sl;
{
    char *np;
    string buf;

    if( sl->sz<1 ){
	return( new_string( "" ) );
    }
    buf = new_string( sl->arr[0] );
    np = buf;
    while( *np != '\0' ){
	if( islower( *np ) ) *np -= ('a' - 'A');
	np++;
    }
    return( buf );
}

/* tolower */
static char *fntolower( sl )
 string_list sl;
{
    char *np;
    string buf;

    if( sl->sz < 1 ){
	return( new_string( "" ) );
    }
    buf = new_string( sl->arr[0] );
    np = buf;
    while( *np != '\0' ){
	if( isupper( *np ) ) *np += ('a' - 'A');
	np++;
    }
    return( buf );
}

/* strindex <c> <word> */
static char *fnstrindex( sl )
 string_list sl;
{
    int n;
    char *ixp;

    if( sl->sz != 2 ){
	line_error( BADPARNO );
	return( new_string( "" ) );
    }
    ixp = index( sl->arr[1], sl->arr[0][0] );
    n = ( ixp == CHARNIL ? 0 : 1 + (ixp - sl->arr[1]) );
    return( newintstr( n ) );
}

/* index <elm> <list> */
/* Note that due to a coincidence the index in the string list
 * the correct index to return.
 */
static char *fnindex( sl )
 string_list sl;
{
    register unsigned int ix;
    string estr;

    if( sl->sz<1 ){
	line_error( NOEXPR );
	return( new_string( "0" ) );
    }
    estr = sl->arr[0];
    ix = 1;
    for( ix=1; ix<sl->sz; ix++ ){
	if( strcmp( estr, sl->arr[ix] ) == 0 ) break;
    }
    if( ix>=sl->sz ) ix=0;
    return( newintstr( (int) ix ) );
}

/* seplist <str> <list> */
static char *fnseplist( sl )
 string_list sl;
{
    register string ans;
    register unsigned int ix;
    string_list nl;

    if( sl->sz<1 ){
	line_error( NOEXPR );
	return( new_string( "" ) );
    }
    nl = new_string_list();
    for( ix=1; ix<sl->sz; ix++ ){
	app_string_list( nl, new_string( sl->arr[ix] ) );
    }
    ans = sepstrings( nl, sl->arr[0] );
    rfre_string_list( nl );
    return( ans );
}

/* prefix <pf> <list> */
static char *fnprefix( sl )
 string_list sl;
{
    string pfstr;
    string ans;
    string buf;
    unsigned int len;
    int maxlen;
    string_list nl;
    register unsigned int ix;

    buf = new_string( "" );
    if( sl->sz<1 ){
	line_error( NOEXPR );
	return( buf );
    }
    pfstr = sl->arr[0];
    maxlen = 0;
    for( ix=1; ix<sl->sz; ix++ ){
	len = strlen( sl->arr[ix] );
	if( len>maxlen ) maxlen = len;
    }
    len = 1 + strlen( pfstr ) + maxlen;
    buf = ckrealloc( buf, len );
    nl = new_string_list();
    for( ix=1; ix<sl->sz; ix++ ){
	(void) sprintf( buf, "%s%s", pfstr, sl->arr[ix] );
	app_string_list( nl, new_string( buf ) );
    }
    fre_string( buf );
    ans = flatstrings( nl );
    rfre_string_list( nl );
    return( ans );
}

/* suffix <sf> <list> */
static char *fnsuffix( sl )
 string_list sl;
{
    string sfstr;
    string ans;
    string buf;
    unsigned int len;
    int maxlen;
    string_list nl;
    register unsigned int ix;

    buf = new_string( "" );
    if( sl->sz<1 ){
	line_error( NOEXPR );
	return( buf );
    }
    sfstr = sl->arr[0];
    maxlen = 0;
    for( ix=1; ix<sl->sz; ix++ ){
	len = strlen( sl->arr[ix] );
	if( len>maxlen ) maxlen = len;
    }
    len = 1 + strlen( sfstr ) + maxlen;
    buf = ckrealloc( buf, len );
    nl = new_string_list();
    for( ix=1; ix<sl->sz; ix++ ){
	(void) sprintf( buf, "%s%s", sl->arr[ix], sfstr );
	app_string_list( nl, new_string( buf ) );
    }
    fre_string( buf );
    ans = flatstrings( nl );
    rfre_string_list( nl );
    return( ans );
}

static int sortcmp( s1, s2 )
 char **s1;
 char **s2;
{
     return( strcmp( *s1, *s2 ) );
}

/* sort e1..en */
static char *fnsort( sl )
 string_list sl;
{
    register char *ans;

    qsort( (char *) sl->arr, sl->sz, sizeof( sl->arr[0] ), sortcmp );
    ans = flatstrings( sl );
    return( ans );
}

/* rev e1..en */
static char *fnrev( sl )
 string_list sl;
{
    register unsigned int lix;
    register unsigned int rix;
    register string tmp;
    register char *ans;

    lix = 0;
    rix = sl->sz;
    while( lix<rix ){
	rix--;
	tmp = sl->arr[lix];
	sl->arr[lix] = sl->arr[rix];
	sl->arr[rix] = tmp;
	lix++;
    }
    ans = flatstrings( sl );
    return( ans );
}

/* comm a "" b */
static char *fncomm( sl )
 string_list sl;
{
    unsigned int aix;
    register unsigned int bix;
    register unsigned int sepix;
    register string tofind;
    register char *ans;
    string_list nl;
    bool takeit;

    sepix = 0;
    while( sepix<sl->sz && sl->arr[sepix][0] != '\0' ) sepix++;
    if( sepix>=sl->sz ){
	line_error( NOSEP );
	return( new_string( "" ) );
    }
    nl = new_string_list();
    for( aix=0; aix<sepix; aix++ ){
	takeit = FALSE;
	tofind = sl->arr[aix];
	for( bix=sepix+1; bix<sl->sz; bix++ ){
	    if( strcmp( tofind, sl->arr[bix] ) == 0 ){
		takeit = TRUE;
		break;
	    }
	}
	if( takeit ){
	    app_string_list( nl, new_string( tofind ) );
	}
    }
    ans = flatstrings( nl );
    rfre_string_list( nl );
    return( ans );
}

/* excl a "" b */
static char *fnexcl( sl )
 string_list sl;
{
    unsigned int aix;
    register unsigned int bix;
    register unsigned int sepix;
    register string tofind;
    register char *ans;
    string_list nl;
    bool takeit;

    sepix = 0;
    while( sepix<sl->sz && sl->arr[sepix][0] != '\0' ) sepix++;
    if( sepix>=sl->sz ){
	line_error( NOSEP );
	return( new_string( "" ) );
    }
    nl = new_string_list();
    for( aix=0; aix<sepix; aix++ ){
	takeit = TRUE;
	tofind = sl->arr[aix];
	for( bix=sepix+1; bix<sl->sz; bix++ ){
	    if( strcmp( tofind, sl->arr[bix] ) == 0 ){
		takeit = FALSE;
		break;
	    }
	}
	if( takeit ){
	    app_string_list( nl, new_string( tofind ) );
	}
    }
    ans = flatstrings( nl );
    rfre_string_list( nl );
    return( ans );
}

/* uniq e1..en */
static char *fnuniq( sl )
 string_list sl;
{
    register string_list nl;
    register string ans;
    register string prevstr;
    register unsigned int ix;

    qsort( (char *) sl->arr, sl->sz, sizeof( sl->arr[0] ), sortcmp );
    nl = new_string_list();
    prevstr = stringNIL;
    for( ix=0; ix<sl->sz; ix++ ){
	if( prevstr == stringNIL || strcmp( sl->arr[ix], prevstr ) != 0 ){
	    app_string_list( nl, rdup_string( sl->arr[ix] ) );
	    prevstr = sl->arr[ix];
	}
    }
    ans = flatstrings( nl );
    rfre_string_list( nl );
    return( ans );
}

/* filt findpat newpat e1..en
   find all elements matching 'findpat' and replace them
   by 'newpat'. Do not copy elements that don't match.
 */
static char *fnfilt( sl )
 string_list sl;
{
    string ans;
    string errm;
    char buf[STRBUFSIZE];
    register unsigned int ix;
    register string_list nl;

    if( sl->sz < 2 ){
	line_error( MISSINGPAR );
	return( new_string( "" ) );
    }
    errm = ref_comp( sl->arr[0] );
    if( errm != stringNIL ){
	(void) strcpy( errarg, errm );
	line_error( BADRE );
	return( new_string( "" ) );
    }
    nl = new_string_list();
    for( ix=2; ix<sl->sz; ix++ ){
	if( ref_exec( sl->arr[ix] ) ){
	    ref_subs( sl->arr[1], buf );
	    app_string_list( nl, new_string( buf ) );
	}
    }
    ans = flatstrings( nl );
    rfre_string_list( nl );
    return( ans );
}

/* subs findpat newpat e1..en
   find all elements matching 'findpat' and replace them
   by 'newpat'. Copy elements that don't match.
 */
static char *fnsubs( sl )
 string_list sl;
{
    string ans;
    string errm;
    char buf[STRBUFSIZE];
    register unsigned int ix;
    register string_list nl;

    if( sl->sz < 2 ){
	line_error( MISSINGPAR );
	return( new_string( "" ) );
    }
    errm = ref_comp( sl->arr[0] );
    if( errm != stringNIL ){
	(void) strcpy( errarg, errm );
	line_error( BADRE );
	return( new_string( "" ) );
    }
    nl = new_string_list();
    for( ix=2; ix<sl->sz; ix++ ){
	if( ref_exec( sl->arr[ix] ) ){
	    ref_subs( sl->arr[1], buf );
	    app_string_list( nl, new_string( buf ) );
	}
	else {
	    app_string_list( nl, new_string( sl->arr[ix] ) );
	}
    }
    ans = flatstrings( nl );
    rfre_string_list( nl );
    return( ans );
}

/* rmlist findpat e1..en
   find all elements matching 'findpat' and delete them.
   Copy elements that don't match.
 */
static char *fnrmlist( sl )
 string_list sl;
{
    string ans;
    string errm;
    register unsigned int ix;
    register string_list nl;

    if( sl->sz < 1 ){
	line_error( MISSINGPAR );
	return( new_string( "" ) );
    }
    errm = ref_comp( sl->arr[0] );
    if( errm != stringNIL ){
	(void) strcpy( errarg, errm );
	line_error( BADRE );
	return( new_string( "" ) );
    }
    nl = new_string_list();
    for( ix=1; ix<sl->sz; ix++ ){
	if( !ref_exec( sl->arr[ix] ) ){
	    app_string_list( nl, new_string( sl->arr[ix] ) );
	}
    }
    ans = flatstrings( nl );
    rfre_string_list( nl );
    return( ans );
}

/* first */
static char *fnfirst( sl )
 string_list sl;
{
    string ans;

    if( sl->sz <1 ){
	ans = new_string( "" );
    }
    else {
	ans = new_string( sl->arr[0] );
    }
    return( ans );
}

/* shift */
static char *fnshift( sl )
 string_list sl;
{
    register unsigned int ix;
    register string_list nl;
    register string ans;

    nl = new_string_list();
    for( ix=1; ix<sl->sz; ix++ ){
        app_string_list( nl, new_string( sl->arr[ix] ) );
    }
    ans = flatstrings( nl );
    rfre_string_list( nl );
    return( ans );
}

/* -- logic functions -- */

/* and */
static char *fnand( sl )
 string_list sl;
{
    register bool flag;
    register unsigned int ix;

    flag = TRUE;
    for( ix=0; ix<sl->sz; ix++ ){
	flag = istruestr( sl->arr[ix] );
	if( !flag ) break;
    }
    return( newboolstr( flag ) );
}

/* or */
static char *fnor( sl )
 string_list sl;
{
    register bool flag;
    register unsigned int ix;

    flag = FALSE;
    for( ix=0; ix<sl->sz; ix++ ){
	flag = istruestr( sl->arr[ix] );
	if( flag ) break;
    }
    return( newboolstr( flag ) );
}

/* not */
static char *fnnot( sl )
 string_list sl;
{
    bool a;

    if( sl->sz < 1 ){
	a = TRUE;
    }
    else {
	a = isfalsestr( sl->arr[0] );
    }
    return( newboolstr( a ) );
}

/* -- datastructure access & file name access -- */

/* construct a list of types */
static char *fntypelist( sl )
 string_list sl;
{
    ds d;
    string vp;
    string ans;
    register unsigned int ix;
    register string_list nl;

    if( sl->sz!=0 ){
	line_error( BADPARNO );
    }
    nl = new_string_list();
    for( ix = 0; ix< allds->sz; ix++ ){
	d = allds->arr[ix];
	switch( d->tag ){
	    case TAGDsCons:
		vp = d->DsCons.ctypename;
		break;

	    case TAGDsTuple:
		vp = d->DsTuple.ttypename;
		break;

	    default:
		(void) sprintf( errarg, "%d", d->tag );
		crash( BADTAG );
	}
	app_string_list( nl, new_string( vp ) );
    }
    ans = flatstrings( nl );
    rfre_string_list( nl );
    return( ans );
}

/* construct a list of constructor types */
static char *fnctypelist( sl )
 string_list sl;
{
    ds d;
    string vp;
    string ans;
    register unsigned int ix;
    register string_list nl;

    if( sl->sz!=0 ){
	line_error( BADPARNO );
    }
    nl = new_string_list();
    for( ix = 0; ix< allds->sz; ix++ ){
	d = allds->arr[ix];
	switch( d->tag ){
	    case TAGDsCons:
		vp = d->DsCons.ctypename;
		app_string_list( nl, new_string( vp ) );
		break;

	    case TAGDsTuple:
		break;

	    default:
		(void) sprintf( errarg, "%d", d->tag );
		crash( BADTAG );
	}
    }
    ans = flatstrings( nl );
    rfre_string_list( nl );
    return( ans );
}

/* construct a list of tuple types */
static char *fnttypelist( sl )
 string_list sl;
{
    ds d;
    string vp;
    string ans;
    register unsigned int ix;
    register string_list nl;

    if( sl->sz!=0 ){
	line_error( BADPARNO );
    }
    nl = new_string_list();
    for( ix = 0; ix< allds->sz; ix++ ){
	d = allds->arr[ix];
	switch( d->tag ){
	    case TAGDsTuple:
		vp = d->DsTuple.ttypename;
		app_string_list( nl, new_string( vp ) );
		break;

	    case TAGDsCons:
		break;

	    default:
		(void) sprintf( errarg, "%d", d->tag );
		crash( BADTAG );
	}
    }
    ans = flatstrings( nl );
    rfre_string_list( nl );
    return( ans );
}

/* Given a list of types 'dl', search for type with name 't'. */ 
static ds findtype( dl, t )
 ds_list dl;
 char *t;
{
    register ds d;
    register unsigned int ix;

    for( ix = 0; ix < dl->sz; ix++ ){
	d = dl->arr[ix];
	switch( d->tag ){
	    case TAGDsCons:
		if( strcmp( d->DsCons.ctypename, t ) == 0 ){
		    return( d );
		}
		break;

	    case TAGDsTuple:
		if( strcmp( d->DsTuple.ttypename, t ) == 0 ){
		    return( d );
		}
		break;

	    default:
		(void) sprintf( errarg, "%d", d->tag );
		crash( BADTAG );
	}
    }
    (void) strcpy( errarg, t );
    line_error( NOSUCHTYPE );
    return( dsNIL );
}

/* Given a list of constructors 'dl', search for
   constructor with name 'nm'.
 */ 
static constructor findconstructor( cl, nm )
 constructor_list cl;
 char *nm;
{
    register unsigned int ix;
    register constructor c;

    for( ix=0; ix<cl->sz; ix++ ){
	c = cl->arr[ix];
	if( strcmp( c->conname, nm ) == 0 )
	    return( c );
    }
    (void) strcpy( errarg, nm );
    line_error( NOSUCHCONS );
    return( constructorNIL );
}

/* Given a list of constructor elements 'el',
   search for constructor element with name 'nm'.
 */ 
static field findfield( el, nm )
 field_list el;
 char *nm;
{
    register unsigned int ix;
    register field e;

    for( ix=0; ix<el->sz; ix++ ){
	e = el->arr[ix];
	if( strcmp( e->sename, nm ) == 0 )
	    return( e );
    }
    (void) strcpy( errarg, nm );
    line_error( NOSUCHELM );
    return( fieldNIL );
}

/* Construct a list of fields for given tuple type.
 * A constructor type has an empty list.
 */
static char *fntelmlist( sl )
 string_list sl;
{
    ds d;
    field_list el;
    field e;
    string vp;
    string ans;
    register unsigned int ix;
    register string_list nl;

    if( sl->sz != 1 ){
	line_error( BADPARNO );
	return( new_string( "" ) );
    }
    d = findtype( allds, sl->arr[0] );
    if( d == dsNIL || d->tag != TAGDsTuple ){
	return( new_string( "" ) );
    }
    nl = new_string_list();
    el = d->DsTuple.tuplefields;
    for( ix=0; ix<el->sz; ix++ ){
	e = el->arr[ix];
	vp = e->sename;
	app_string_list( nl, new_string( vp ) );
    }
    ans = flatstrings( nl );
    rfre_string_list( nl );
    return( ans );
}

/* Given a tuple type name and element name, return the
   type name of the given element.
 */
static char *fnttypename( sl )
 string_list sl;
{
    ds d;
    field_list el;
    field e;

    if( sl->sz != 2 ){
	line_error( BADPARNO );
	return( new_string( "" ) );
    }
    d = findtype( allds, sl->arr[0] );
    if( d == dsNIL || d->tag != TAGDsTuple ){
	(void) strcpy( errarg, sl->arr[0] );
	line_error( NOTATUPLE );
	return( new_string( "" ) );
    }
    el = d->DsTuple.tuplefields;
    e = findfield( el, sl->arr[1] );
    if( e == fieldNIL ) return( new_string( "" ) );
    return( new_string( e->setype ) );
}

/* Given a type name and element name, return the
   type class of the given element.

   Possible type classes are: `single' and `list'.
 */
static char *fnttypeclass( sl )
 string_list sl;
{
    ds d;
    field_list el;
    field e;

    if( sl->sz != 2 ){
	line_error( BADPARNO );
	return( new_string( "" ) );
    }
    d = findtype( allds, sl->arr[0] );
    if( d == dsNIL || d->tag != TAGDsTuple ){
	(void) strcpy( errarg, sl->arr[0] );
	line_error( NOTATUPLE );
	return( new_string( "" ) );
    }
    el = d->DsTuple.tuplefields;
    e = findfield( el, sl->arr[1] );
    if( e == fieldNIL ) return( new_string( "" ) );
    return( new_string( ( e->listlev == 1 ? "list" : "single" ) ) );
}

/* construct a list of constructors for given type */
static char *fnconslist( sl )
 string_list sl;
{
    ds d;
    constructor_list cl;
    constructor c;
    string vp;
    string ans;
    register unsigned int ix;
    register string_list nl;

    if( sl->sz != 1 ){
	line_error( BADPARNO );
	return( new_string( "" ) );
    }
    d = findtype( allds, sl->arr[0] );
    if( d == dsNIL || d->tag != TAGDsCons ) return( new_string( "" ) );
    cl = d->DsCons.conslist;
    nl = new_string_list();
    for( ix=0; ix<cl->sz; ix++ ){
	c = cl->arr[ix];
	vp = c->conname;
	app_string_list( nl, new_string( vp ) );
    }
    ans = flatstrings( nl );
    rfre_string_list( nl );
    return( ans );
}

/* construct a list of fields for given type */
static char *fncelmlist( sl )
 string_list sl;
{
    ds d;
    constructor_list cl;
    constructor c;
    field_list el;
    field e;
    string vp;
    string ans;
    register unsigned int ix;
    register string_list nl;

    if( sl->sz != 2 ){
	line_error( BADPARNO );
	return( new_string( "" ) );
    }
    d = findtype( allds, sl->arr[0] );
    if( d == dsNIL || d->tag != TAGDsCons ){
	(void) strcpy( errarg, sl->arr[0] );
	line_error( NOTACONS );
	return( new_string( "" ) );
    }
    cl = d->DsCons.conslist;
    c = findconstructor( cl, sl->arr[1] );
    if( c == constructorNIL ){
	return( new_string( "" ) );
    }
    el = c->confields;
    nl = new_string_list();
    for( ix=0; ix<el->sz; ix++ ){
	e = el->arr[ix];
	vp = e->sename;
	app_string_list( nl, new_string( vp ) );
    }
    ans = flatstrings( nl );
    rfre_string_list( nl );
    return( ans );
}

/* Given a type name, constructor name and element name, return the
   type name of the given element.

   It is not possible to determine whether it is a list of elements
   or not from this list.
 */
static char *fnctypename( sl )
 string_list sl;
{
    ds d;
    constructor_list cl;
    constructor c;
    field_list el;
    field e;

    if( sl->sz != 3 ){
	line_error( BADPARNO );
	return( new_string( "" ) );
    }
    d = findtype( allds, sl->arr[0] );
    if( d == dsNIL || d->tag != TAGDsCons ){
	(void) strcpy( errarg, sl->arr[0] );
	line_error( NOTACONS );
	return( new_string( "" ) );
    }
    cl = d->DsCons.conslist;
    c = findconstructor( cl, sl->arr[1] );
    if( c == constructorNIL ){
	return( new_string( "" ) );
    }
    el = c->confields;
    e = findfield( el, sl->arr[2] );
    if( e == fieldNIL ) return( new_string( "" ) );
    return( new_string( e->setype ) );
}

/* given a type name, constructor name and element name, return the
   type class of the given element.

   Possible type classes are:
    single
    list
 */
static char *fnctypeclass( sl )
 string_list sl;
{
    ds d;
    constructor_list cl;
    constructor c;
    field_list el;
    field e;

    if( sl->sz != 3 ){
	line_error( BADPARNO );
	return( new_string( "" ) );
    }
    d = findtype( allds, sl->arr[0] );
    if( d == dsNIL || d->tag != TAGDsCons ){
	(void) strcpy( errarg, sl->arr[0] );
	line_error( NOTACONS );
	return( new_string( "" ) );
    }
    cl = d->DsCons.conslist;
    c = findconstructor( cl, sl->arr[1] );
    if( c == constructorNIL ){
	return( new_string( "" ) );
    }
    el = c->confields;
    e = findfield( el, sl->arr[2] );
    if( e == fieldNIL ) return( new_string( "" ) );
    return( new_string( ( e->listlev == 1 ? "list" : "single" ) ) );
}

/* The recursive code for 'fndeptype'.
 * Given a type with name 'tnm', search for it in the list of defined
 * types, and if it occurs, mark it as used in the array depsingle.
 * Also mark it in deplist if the boolean 'islist' is TRUE.
 *
 * If no mark had been set before, recursively mark all types used in
 * all constructors in that type as used.
 */
static void markdep( tnm, dl, depsingle, deplist, islist )
 char *tnm;
 ds_list dl;
 bool *depsingle;
 bool *deplist;
 bool islist;
{
    register unsigned int tix;
    register bool recurse;
    bool subislist;
    bool found;
    ds d;

    found = FALSE;
    for( tix = 0; tix<dl->sz; tix++ ){
	d = dl->arr[tix];
	switch( d->tag ){
	    case TAGDsCons:
		if( strcmp( d->DsCons.ctypename, tnm ) == 0 ){
		    found = TRUE;
		}
		break;

	    case TAGDsTuple:
		if( strcmp( d->DsTuple.ttypename, tnm ) == 0 ){
		    found = TRUE;
		}
		break;

	    default:
		(void) sprintf( errarg, "%d", d->tag );
		crash( BADTAG );
	}
	if( found ) break;
    }
    if( !found ) return;
    recurse = (!depsingle[tix] && !deplist[tix] );
    depsingle[tix] = TRUE;
    if( islist ){
	deplist[tix] = TRUE;
    }
    if( !recurse ) return;
    switch( d->tag ){
	case TAGDsCons:
	{
	    register unsigned int cix;
	    constructor_list cl;
	    constructor c;
	    register unsigned int eix;
	    field_list el;
	    field e;

	    cl = d->DsCons.conslist;
	    for( cix=0; cix<cl->sz; cix++ ){
		c = cl->arr[cix];
		el = c->confields;
		for( eix=0; eix<el->sz; eix++ ){
		    e = el->arr[eix];
		    subislist = (e->listlev == 1 );
		    markdep( e->setype, dl, depsingle, deplist, subislist );
		}
	    }
	    break;
	}

	case TAGDsTuple:
	{
	    register unsigned int eix;
	    field_list el;
	    field e;

	    el = d->DsTuple.tuplefields;
	    for( eix=0; eix<el->sz; eix++ ){
		e = el->arr[eix];
		subislist = (e->listlev == 1 );
		markdep( e->setype, dl, depsingle, deplist, subislist );
	    }
	    break;
	}

	default:
	    (void) sprintf( errarg, "%d", d->tag );
	    crash( BADTAG );
    }
}

/* Given a class name, and a list of type names, determine which types and
   lists of types are dependent on these types.
   If class name is 'single' return the list of types that required for
   the given types.
   If class name is 'single' return the list of type lists that required for
   the given types.
   The top level types are assumed to be 'single'.
 */
static char *fndeptype( sl )
 string_list sl;
{
    bool *depsingle;
    bool *deplist;
    bool islist;
    string ans;
    register unsigned int ix;
    register string_list nl;

    if( sl->sz<1 ){
	line_error( NOEXPR );
	return( new_string( "" ) );
    }
    if( strcmp( sl->arr[0], "single" ) == 0 ){
	islist = FALSE;
    }
    else if( strcmp( sl->arr[0], "list" ) == 0 ){
	islist = TRUE;
    }
    else {
	(void) strcpy( errarg, sl->arr[0] );
	line_error( BADPAR );
	return( new_string( "" ) );
    }
    /* The +1 below is to ensure no calloc of size 0 */
    depsingle = (bool *) ckmalloc( (allds->sz+1)*sizeof(bool) );
    deplist = (bool *) ckmalloc( (allds->sz+1)*sizeof(bool) );
    for( ix=0; ix<allds->sz; ix++ ){
	depsingle[ix] = deplist[ix] = FALSE;
    }
    for( ix=0; ix<sl->sz; ix++ ){
	markdep( sl->arr[ix], allds, depsingle, deplist, FALSE );
    }
    nl = new_string_list();
    for( ix=0; ix<allds->sz; ix++ ){
	if( ( islist && deplist[ix] ) || ( !islist && depsingle[ix] ) ){
	    char *vp;
	    ds d;

	    d = allds->arr[ix];
	    switch( d->tag ){
		case TAGDsCons:
		    vp = d->DsCons.ctypename;
		    break;

		case TAGDsTuple:
		    vp = d->DsTuple.ttypename;
		    break;

		default:
		    (void) sprintf( errarg, "%d", d->tag );
		    crash( BADTAG );
	    }
	    app_string_list( nl, new_string( vp ) );
	}
    }
    free( (char *) depsingle );
    free( (char *) deplist );
    ans = flatstrings( nl );
    rfre_string_list( nl );
    return( ans );
}

static char *fndsfilename( sl )
 string_list sl;
{

    if( sl->sz!=0 ){
	line_error( BADPARNO );
    }
    return( new_string( dsfilename ) );
}

static char *fntplfilename( sl )
 string_list sl;
{

    if( sl->sz!=0 ){
	line_error( BADPARNO );
    }
    return( new_string( tplfilename ) );
}

static char *fntpllineno( sl )
 string_list sl;
{

    if( sl->sz != 0 ){
	line_error( BADPARNO );
    }
    return( newintstr( tpllineno ) );
}

static char *fndefined( sl )
 string_list sl;
{
    char *v;

    if( sl->sz<1 ){
	return( new_string( "0" ) );
    }
    v = getvar( sl->arr[0] );
    return( newboolstr( v != CHARNIL ) );
}

/* -- UNIX interface -- */

/* access of environment variables */
static char *fngetenv( sl )
 string_list sl;
{
    char *v;
    register string_list nl;
    string ans;

    if( sl->sz<1 ){
	line_error( BADPARNO );
	return( "" );
    }
    v = getenv( sl->arr[0] );
    if( v == CHARNIL ){
	register unsigned int ix;

	nl = new_string_list();
	for( ix=1; ix<sl->sz; ix++ ){
	    app_string_list( nl, new_string( sl->arr[ix] ) );
	}
	ans = flatstrings( nl );
	rfre_string_list( nl );
	return( ans );
    }
    return( new_string( v ) );
}

/***************************************************************
 *                                                             *
 *   function table                                            *
 *                                                             *
 ***************************************************************/

typedef char *(*charfn)();

/* structure to describe an entry in the table of functions */
struct fnentry {
    char *fnname;
    charfn fncode;
};

/* table of functions. It is terminated by an entry with an
   empty function name
 */
struct fnentry fntab[] = {
     { "!=", fnneq },
     { "%", fnmod },
     { "*", fntimes },
     { "+", fnplus },
     { "-", fnsubtract },
     { "/", fndiv },
     { "<", fnless },
     { "<=", fnlesseq },
     { "==", fneq },
     { ">", fngreater },
     { ">=", fngreatereq },
     { "and", fnand },
     { "celmlist", fncelmlist },
     { "comm", fncomm },
     { "conslist", fnconslist },
     { "ctypeclass", fnctypeclass },
     { "ctypelist", fnctypelist },
     { "ctypename", fnctypename },
     { "defined", fndefined },
     { "deptype", fndeptype },
     { "dsfilename", fndsfilename },
     { "eq", fnstreq },
     { "excl", fnexcl },
     { "filt", fnfilt },
     { "first", fnfirst },
     { "getenv", fngetenv },
     { "index", fnindex },
     { "len", fnlen },
     { "max", fnmax },
     { "min", fnmin },
     { "neq", fnstrneq },
     { "not", fnnot },
     { "or", fnor },
     { "prefix", fnprefix },
     { "rev", fnrev },
     { "rmlist", fnrmlist },
     { "seplist", fnseplist },
     { "shift", fnshift },
     { "sort", fnsort },
     { "strcmp", fnstrcmp },
     { "strindex", fnstrindex },
     { "strlen", fnstrlen },
     { "strpad", fnstrpad },
     { "subs", fnsubs },
     { "suffix", fnsuffix },
     { "telmlist", fntelmlist },
     { "tolower", fntolower },
     { "toupper", fntoupper },
     { "tplfilename", fntplfilename },
     { "tpllineno", fntpllineno },
     { "ttypeclass", fnttypeclass },
     { "ttypelist", fnttypelist },
     { "ttypename", fnttypename },
     { "typelist", fntypelist },
     { "uniq", fnuniq },
     { "", fnplus }
 };

/* Given a function string 'f', evaluate the function
   by looking up the function name in a list and delegating evaluation to
   that function.
 */
char *evalfn( f )
 char *f;
{
    char *fnname;
    struct fnentry *fp;
    char *par;
    char *ans;
    string_list sl;

    if( fntr ){
	fprintf( tracestream, "evaluating function ${%s}\n", f );
    }
    par = scanword( f, &fnname );
    if( fnname == CHARNIL ){
	line_error( NONAME );
	return( new_string( "" ) );
    }
    fp = fntab;
    while( fp->fnname[0] != '\0' ){
	if( strcmp( fp->fnname, fnname ) == 0 ) break;
	fp++;
    }
    if( fp->fnname[0] != '\0' ){
	sl = chopstring( par );
	ans = (*fp->fncode)( sl );
	rfre_string_list( sl );
    }
    else {
	(void) strcpy( errarg, fnname );
	line_error( NOSUCHFN );
	ans = new_string( "" );
    }
    fre_string( fnname );
    if( fntr ){
	fprintf( tracestream, "function value: '%s'\n", ans );
    }
    return( ans );
}
