
/*

    This file is a part of the GLASS source distribution 
    and therefore subjected to the copy notice below. 
    
    Copyright (C) 1989,1990  S.J. Klaver, R Doesborg
              email: simon@sagan.nl

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation version 1

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* File: wdisplayprocs.c
   Authors: R. Doesborg, S.J. Klaver
*/

#include <stdio.h>
#include <curses.h>
#include "gconst.h"
#include "errnum.h"
#include "boxdefs.h"
#include "tctypes.h"
#include "globalvars.h"

extern char *malloc();

/* ATTR means if your curses implementation knows
   screen attributes
*/
#ifdef ATTR
#define WREVERSE_ON(win)     wattron(win, A_REVERSE)
#define WREVERSE_OFF(win)    wattroff(win, A_REVERSE)
#define WBOLD_ON(win)        wattron(win, A_BOLD)
#define WBOLD_OFF(win)       wattronff(win, A_BOLD)
#else
#define WREVERSE_ON(win)     wstandout(win)
#define WREVERSE_OFF(win)    wstandend(win)
#define WBOLD_ON(win)        wstandout(win)
#define WBOLD_OFF(win)       wstandend(win)
#endif

#define MAXLINES 25
static int delay_buffer[MAXLINES];
static int bufpoint = 0;


void wdelayed_delete (win, line)
  int line;
  WINDOW *win;
{
  if (bufpoint<=MAXLINES) {
    delay_buffer[bufpoint] = line;
    bufpoint++;
  }
  else {
    wmove(win, line, 0); wclrtoeol(win);
  }
}

void del_delay_buf (win)
  WINDOW *win;
{
  while (bufpoint > 0) {
    bufpoint--;
    wmove(win, delay_buffer[bufpoint], 0);
    wclrtoeol(win);
  }
}

/* first time setup of curses */
void init_screen ()
{
  freopen("/dev/tty", "r+", stdin);
  initscr();
  nonl();
  crmode();  
  noecho();
  refresh();
}

void wclear_screen (win)
  WINDOW *win;
{
  wclear(win); wrefresh(win);
}

/* clear screen win and exit curses */
void wexit_screen (win)
  WINDOW *win;
{
  wclear_screen(win);
  endwin();
}

void switchtounbuffered ()
{
  nonl();
  crmode();  
  noecho();
}

void switchtobuffered ()
{
  nl();
  echo();
  nocrmode(); 
}

void wdraw_box (win, y0, x0, y1, x1)
  WINDOW *win;
  int y0, x0, y1, x1;
{
  int i;
  mvwaddch (win, y0, x0, BOXTL);
  mvwaddch (win, y0, x1, BOXTR);
  mvwaddch (win, y1, x0, BOXBL);
  mvwaddch (win, y1, x1, BOXBR);
  for (i=x0+1; i<=x1-1; i++) {
    mvwaddch (win, y0, i, BOXH);
    mvwaddch (win, y1, i, BOXH);
  }
  for (i=y0+1; i<=y1-1; i++) {
    mvwaddch (win, i, x0, BOXV);
    mvwaddch (win, i, x1, BOXV);
  }
}


/* str is a buffer, already allocated, that contains the
   default value.  */
void wgetstr_edit (win, edit_buf, maxlen)
  WINDOW *win;
  char *edit_buf;
  int maxlen;
{
  int x, y, c, len = 0;

  getyx(win, y, x);
  mvwaddstr (win, y, x, edit_buf);
  len = strlen(edit_buf);
  wmove(win, y, x+len);
  wrefresh(win);
  
  c = wgetch (win);

  while ((c != KEY_RETURN) && (c != 13) && (c != 10)) {   /* <return> */
    if ((c == 127) || (c == 8)) {    /* <backspace> or <erase> */
      if (len > 0) {                 /* not at beginning of string */
        len = len - 1;
        edit_buf[len] = (char)0;
        mvwaddch(win, y, x+len, ' '); 
        wmove(win, y, x+len);
        wrefresh(win);
      }
    }
    else 
    if (c == 21) {                 /* <erase line> */
      while (len > 0) {
        len = len - 1;
        mvwaddch(win, y, x+len, ' '); 
      }    
      /* len == 0 */
      edit_buf[0] = (char)0;
      wmove(win, y, x);
      wrefresh(win);
    }
    else 
    if (  (c < 127)                 /* character, len < maxlen */
       && (c >= 32)
       && (len < maxlen)) {  
      edit_buf[len] = (char)c;
      mvwaddch(win, y, x+len, (char)c);
      len = len + 1;
      edit_buf[len] = (char)0;
      wrefresh(win);
    }
    c = wgetch (win);
  }
}


int werase_word (win, row, column, len)
  WINDOW *win;
  int row, column, len;
{
  int i;
  for (i=0; i<len; i++) 
    mvwaddch (win, row, column+i, ' ');
  return(0);
}

int wdisplay_string (win, str, row, column, attr)
  WINDOW *win;
  char *str;
  int row;
  int column;
  int attr;
{
  if (strlen(str)+column > LINELENGTH)
     return(1);
  if (attr==REVERSE)
    WREVERSE_ON(win); 
  else if (attr==BOLD)
    WBOLD_ON(win); 
  mvwaddstr(win, row, column, str);
  if (attr==REVERSE)
    WREVERSE_OFF(win);
  else if (attr==BOLD)
    WBOLD_OFF(win);
  del_delay_buf(win);   /* oppassen met deze */
  wrefresh(win);
  return(0);
}


/* Displays keywords.  If the display attribute is NORMAL, the
   identifying character is displayed in BOLD.
*/
int wdisplay_keyword (win, str, row, column, attr, idchar, idpos)
  WINDOW *win;
  int row, column, attr, idpos;
  char *str;
  char idchar;
{ 
  int err;
  err = wdisplay_string (win, str, row, column, attr);
  if (attr == NORMAL) {
    wmove (win, row, column+idpos-1); 
    WBOLD_ON(win);
    waddch (win, idchar);
    WBOLD_OFF(win);
  }
  return(err);
}

void wdisplay_error (win, errnum)
  WINDOW *win;
  int errnum;
{
  wmove(win, 23, 1);
  switch (errnum) {
    case errNOQUST:
      waddstr(win, "Cannot go to higher menu (use Exit to quit program)");
      break;
    case errILLKEY:
      waddstr(win, "This key is not defined in this menu");
      break;
    default:
      waddstr(win, "Unknown errormessage");
      break;
  }
  wrefresh(win);
  wdelayed_delete (win, 23);
}

int wget_key (win)
  WINDOW *win;
{
  int key1, key2, key3;
  int result, t;

  wmove (win, 23,1); wrefresh(win);

  /* the following section reads a character and does a shell  */
  /* escape if ! is pressed                                    */

  do {
    key1 = wgetch(win);
    if (key1 == '!') {
      endwin();
      printf ("\n\nLogout to return to GLUE.\n"); 
      if (cshell) 
        system ("csh");
      else
        system ("sh");
      printf ("\nBack in GLUE.\n");
      wrefresh(curscr); 
      crmode(); noecho(); nonl();
      key1 = wgetch(win);
      break;
    }
  }
  while (key1 == '!');
  /* key1 <> '!' */

  if (key1 == KEY_ESCAPE) {
    /* nodelay(win, 1); */
    for (t=0; ((key2 = wgetch(win)) == ERR) && t<100; t++);
    if (key2 == ERR)
      result = key1;
    else {
      for (t=0; ((key3 = wgetch(win)) == ERR) && t<100; t++);
      if (key3 == ERR)
        result = ERR;
      else {
        if (key2 == '[') {
          switch (key3) {
            case 'A':
              result = KEY_UP;
              break;
            case 'B':
              result = KEY_DOWN;
              break;
            case 'C':
              result = KEY_RIGHT;
              break;
            case 'D':
              result = KEY_LEFT;
              break;
            default:
              result = ERR;
              break;
          }
        }
        else result = ERR;
      }
    }
  /* nodelay(win, 0); */
  }
  else result = key1;
  return(result);
}
