/*
  $Id: oiladt2a.c,v 1.15 90/04/27 16:00:36 swain Exp $
Copyright, 1989, The Regents of the University of Colorado
*/

#include <stdio.h>

#include "oiladt2.h"


/**********************************************
  OilNewID - allocates a object and assigns the ptr
      may define the object's ID if the compilation switch 'WithIDs' is
      set.
  */
#ifdef WithIDs
#define OilNewID(ptr,type,str) ptr=OilNew(type,str); ptr->sID= OilMasterID++;
  int OilMasterID=1;	/* counter for debugging  and output formatting */
#else
#define OilNewID(ptr,type,str) ptr=OilNew(type,str); 
#endif


/**********************************************
  These state variables are used in the minimum cost coercion sequence
      algorithm for the balancing of 2 types.
  */
tOilCoercionSeq vOilCoercionSeq;/* which is the coercion seq selected */
tOilTypeSet vOilCoercionElement;/* which is the set element selected */
tOilType vOilCoercionType;	 /* what type is the result of balancing */
int vOilCost;	    		 /* cost of identifying a type */

extern tOilCoercionSeq OilCoerce(); /* ( t1, t2 ) */


/**********************************************
  OilNewArgSig - creates an empty signature
  */
tOilArgSig OilNewArgSig(){ return(nil); }


/**********************************************
  OilAddArgSig - adds a new argument type to the front of the signature
  */
tOilArgSig OilAddArgSig( arg, sig )
        tOilType arg;
        tOilArgSig sig; {
    tOilArgSig ps;

    /* search for like signature */
    for ( ps=arg->sSigs; ps!=nil; ps= ps->sSameType )
      if (ps->sNext == sig) return( ps );

    /* otherwise construct a new one */
    OilNewID(ps,tOilArgSig,Ssig);
    ps->sNext= sig;
    ps->sArg= arg;

    ps->sSameType= arg->sSigs;
    arg->sSigs= ps;

    return( ps );
    }


/********************************
  OilNewSetSig - create an empty set signature
  */
tOilSetSig OilNewSetSig(){ return(nil); }


/********************************
  OilAddSetSig - add a type-set to the front of a set signature
  */
tOilSetSig OilAddSetSig( arg, sig )
        tOilTypeSet arg;
        tOilSetSig sig; {
    tOilSetSig ps;
    OilNewID(ps,tOilSetSig,SsetSig);
    ps->sNext= sig;
    ps->sResultTypes= arg;
    return( ps );
    }


/********************************
  OilNewOp - define a new operator 'id' with signature 'sig' and cost 'cost'
  */
tOilOp OilNewOp( id, sig, cost )
        oilName id;
        int cost;
        tOilArgSig sig; {
    tOilOp op;
    OilNewID(op,tOilOp,Sop);
    op->sName= id;	/* name of operator */
    op->sCost= cost;
    op->sArgs= sig;
    op->sSameType = nil;
    op->sIndOps= nil;
    return( op );
    }

tOilOp  OilErrorOp()
  {
      return(nil);
  }


/********************************
  OilAddIdentification - add the possible identification of 'op' by indication
          'id'
  */
BOOL OilAddIdentification( id, op )
        tOilOp op;
        tOilOp id; {
    tOilOpIdentification pid;
    OilNewID(pid,tOilOpIdentification,SopIdent);  /* create new identification */
    pid->sNext= id->sIndOps;	/* link in new identification */
    pid->sOp= op;
    id->sIndOps= pid;
    return(TRUE);
    }
 
extern BOOL OilIsValidCS();

/******************************** 
  OilAddCoercionCost - define a coercion from 'srctype' to 'destype' by 
      operator 'cop' with cost 'cs'
  */
void OilAddCoercionCost( srctype, destype, cop, cs )
    tOilCoercionSeq cs;
    tOilType srctype, destype;
    tOilOp cop;
  {
    tOilCoercionSeq pc, destcc;
    int cost;

    /* determine cost */
    cost= cop->sCost;
    if (cs!=nil) cost+= cs->sCost;

    /* look for an existing coercion sequence */
    destcc= OilCoerce( srctype, destype);
    
    /* construct the coercion cost for src type */
    if (OilIsValidCS(destcc)) {
	if (destcc==nil) return;
	/* if there exists a cheaper coercion sequence stop here */
	if (cost>=destcc->sCost) return;
            else pc=destcc;	/* otherwise replace the existing one */
		}
	else {
	        /* if no seq. exists then create it */
		OilNewID(pc,tOilCoercionSeq,ScoercionSeq);
    		pc->sNextCost= srctype->sCoercionCosts;
    		srctype->sCoercionCosts= pc;
		}
    pc->sResultType= destype;
    pc->sCoercionOp= cop;
    pc->sCoercionSeq= cs;
    pc->sCost= cost;

    /* now do transitive closure of reachable types  from destination
	adding new coercions onto the end of the current list */
    for (destcc= destype->sCoercionCosts;
	destcc != nil;
	destcc= destcc->sNextCost ) 
		OilAddCoercionCost( srctype, destcc->sResultType,
			cop, destcc );

    /* do transitive closure of those types which can reach here */
    for ( cop= srctype->sOps;
	cop != nil;
	cop= cop->sSameType )
		OilAddCoercionCost( cop->sArgs->sNext->sArg, destype,
			cop, pc );

  }  /* end of OilAddCoercionCost */ 


/********************************
  OilAddCoercion - add a coercion operator 'cop' to the coercion lattice
  */
void OilAddCoercion( cop )
        tOilOp cop; { 
    tOilType destype, srctype;
 
    /* select src vs dest types from signature */
    srctype= cop->sArgs->sNext->sArg;
    destype= cop->sArgs->sArg;
 
    OilAddCoercionCost( srctype, destype, cop, nil );

    /* add coercion onto coercions which can reach here */
    cop->sSameType= destype->sOps;
    destype->sOps= cop;
  }  /* end of OilAddCoercion */
 

/********************************
  OilNewType - define a new type with name 'id'
  */

tOilType OilNewType( id )
        oilName id; {
    tOilType nt;
    OilNewID(nt,tOilType,Stype);
    nt->sName= id;              /* name for type */
    
    nt->sCoercionCosts= nil;    /* no coercions to start */
    nt->sSigs= nil;             /* no signatures */
    nt->sResultSets= nil;       /* no result sets */
    nt->sOps= nil;              /* no coercions can pt here */
    return(nt);
    }	/* end of Oil New Type */

tOilType OilErrorType()
  {
      return(nil);
  }

/********************************
  OilIsCoercibleTo - test if type 't1' can be coerced to type 't2'
  */
BOOL OilIsCoercibleTo( t1, t2 )
        tOilType t1, t2;
    {
    tOilCoercionSeq pcc;

    if (t1==t2 || t2==nil || t1==nil) return(TRUE);
    for (pcc=t1->sCoercionCosts; pcc!=nil; pcc=pcc->sNextCost )
    	if (pcc->sResultType==t2) return(TRUE);
    return( FALSE );
    }


/********************************
  OilSetIncludes - test if type set 's1' includes type 't2'
  */
BOOL OilSetIncludes( s1, t2 )
        tOilTypeSet s1;
        tOilType t2; {
    tOilCoercionSeq pcc;
    tOilType pt;

    if (s1==nil) { vOilCoercionSeq=nil; vOilCoercionElement=nil;
		   return(TRUE);}	/* error set includes all */
    /* validate that type 't2' is in the type set 's1'*/
    for( ;s1!=nil; s1= s1->sNext ) {
      pt= s1->sBaseType;
      vOilCoercionElement= s1;
      vOilCoercionSeq= nil;
      if (pt==t2 || t2==nil) return(TRUE);
      for (pcc=pt->sCoercionCosts; pcc!=nil; pcc=pcc->sNextCost ){
    	if (pcc->sResultType==t2) { vOilCoercionSeq= pcc; return(TRUE);}
    	}}
    return( FALSE );
    }


/********************************
  OilAddResultType -  add a new type to the set of possible result types
  */
tOilTypeSet OilAddResultType( t, set, cost )
        tOilType t;
        tOilTypeSet set;
        int cost;
    {
    tOilTypeSet prs=nil;
    tOilTypeSet next;
    if (t==nil) return nil;	/* error type => error set */
    for( next= t->sResultSets;
	(next!=nil) && (prs==nil);
	next=next->sSameType )
      if ((next->sNext==set)&&(next->sCost==cost)) prs=next;

    if (prs==nil) {
        OilNewID(prs,tOilTypeSet,StypeSet);
        prs->sBaseType= t;
        prs->sNext= set;
        prs->sCost= cost;

	prs->sSameType= t->sResultSets;	/* link in new result set */
	t->sResultSets= prs;
    }
    return( prs );
    }

/********************************
  OilEmptyTS, OilHeadTS & OilTailTS - access root elements of type sets
  */
BOOL OilEmptyTS( set )
        tOilTypeSet set;
    {
	return( set==nil );
    }
tOilType OilHeadTS( set )
        tOilTypeSet set;
    {
            return( (set==nil)?nil:set->sBaseType );
    }
tOilTypeSet OilTailTS( set )
        tOilTypeSet set;
    {
            return( (set==nil)?nil:set->sNext );
    }

/********************************
  OilCanIdSigs -  Check the sets of possible actual argument types to see
      if they might be coerced to the corresponding formal type
  */
BOOL OilCanIdSigs( src, dest )
	tOilSetSig src;
	tOilArgSig dest; {
    vOilCost= 0;
    for (dest=dest->sNext; dest!=nil; dest= dest->sNext){
	if (src==nil) return(FALSE);
	if (OilSetIncludes( src->sResultTypes, dest->sArg)){
		src=src->sNext;
		if (vOilCoercionSeq!=nil) vOilCost+= vOilCoercionSeq->sCost;
		}
	  else return(FALSE);
	}
    return(src==nil);  /* match only if sigs same length */
    }


/********************************
  OilcErrorCoercion -  This constant is used to denote an invalid coercion
      sequence
  */
struct ScoercionSeq OilcErrorCoercion= { nil, nil, nil, nil, 0 };


/********************************
  OilIsValidCS - check for valid coercion
  */
BOOL OilIsValidCS( ccs )
        tOilCoercionSeq ccs; {
  return (ccs != &OilcErrorCoercion);
  }

/*****************************************************
  OilHeadCS - returns the first operator in the coercion sequence
  */
tOilOp  OilHeadCS( cs )
    tOilCoercionSeq cs;
    {
	return(cs->sCoercionOp);
    }

/*****************************************************
  OilTailCS - returns the 'rest' of the coercion sequence
  */
tOilCoercionSeq  OilTailCS( cs )
    tOilCoercionSeq cs;
    {
        return(cs->sCoercionSeq);
    }

/*****************************************************
  OilEmptyCS - returns true for and empty coercion sequence
  */
BOOL OilEmptyCS( cs )
    tOilCoercionSeq cs;
    {
        return(cs==nil);
    }




/********************************
  OilIsValidOp - check for valid operator
  */
BOOL OilIsValidOp( op )
        tOilOp op; {
  return( op!=nil );
  }


/********************************
  OilCoerce - return coercion sequence for coercing 't1' to 't2'
  */
tOilCoercionSeq OilCoerce( t1, t2 )
	tOilType t1, t2;
  {
  tOilCoercionSeq cc;
  if (t1==t2 || t1==nil || t2==nil) return( nil );
  for (cc=t1->sCoercionCosts; cc!=nil; cc=cc->sNextCost)
  	if (cc->sResultType == t2) return(cc);
  return(&OilcErrorCoercion);
  }

tOilCoercionSeq  OilErrorCS()
  {
      return(&OilcErrorCoercion);
  }

/********************************
  OilIdResultSet - find all possible result types which might be indicated
      by 'ind' with arguments 'setsig'. (PS only the minimum coercion costs
      for each possible result is used )
  */
tOilTypeSet OilIdResultSet( ind, setsig )
	tOilOp ind;
	tOilSetSig setsig; {
    tOilOpIdentification pid;
    tOilTypeSet rts=nil;
    for( pid= ind->sIndOps; pid!= nil; pid=pid->sNext ){
	if (OilCanIdSigs( setsig, pid->sOp->sArgs))
		rts= OilAddResultType( pid->sOp->sArgs->sArg, rts, vOilCost );
	}
    return( rts );
    }


/********************************
  OilIdOpTSn - find the minimum operator cost which might be indicated
      by 'ind' with arguments 'setsig' and result type 'rt'
  */
tOilOp OilIdOpTSn( ind, setsig, rt )
	tOilType rt;
	tOilOp ind;
	tOilSetSig setsig; {
    tOilOpIdentification pid;
    int cost=OilMaxCost;
    tOilOp op=nil;
    for( pid= ind->sIndOps; pid!= nil; pid=pid->sNext ){
    	int tcost;
    	tOilCoercionSeq cs;
    	if (OilIsValidCS( cs=OilCoerce( pid->sOp->sArgs->sArg, rt))) {
    	  tcost= pid->sOp->sCost;
    	  if (cs!=nil) tcost+= cs->sCost;
	  if (OilCanIdSigs( setsig, pid->sOp->sArgs))
	    if( cost> (tcost+= vOilCost)) {
		cost= tcost;
		op= pid->sOp;
		}
	  }  /* if 'OilIsValidCS */
	}  /* for 'pid */
    return( op );
    }  /* end of Id Op Type Set */


/********************************
  OilTypeToSet - return the set of possible types which 't' can be coerced to
  */
tOilTypeSet OilTypeToSet( t )
        tOilType t; {
    return( OilAddResultType( t, nil, 0 ) );
    }


/********************************
  OilBalance - select the minimum cost coercion sequence common to the two
      type sets
  */
tOilType OilBalance( ts1, ts2 )
        tOilTypeSet ts1, ts2; {
    tOilType rt;
    int cost;

    cost= OilMinimumCoercionSeq( ts1, ts2, OilMaxCost );
    rt= vOilCoercionType;
    if(cost>OilMinimumCoercionSeq( ts2, ts1, cost )){
      rt=vOilCoercionType;
      }
    return( rt );
    }  /* end of OilBalance */


/********************************
  OilMinimumCoercionSeq - calculate the minimum coercion cost sequence between
      ts1 and ts2
  */
int OilMinimumCoercionSeq( ts1, ts2, cost )
	tOilTypeSet ts1, ts2;
	int cost;
  {
    tOilType rt=nil;
    tOilCoercionSeq seq=nil, cc;
    tOilTypeSet rs;

    for (rs=ts1; rs!=nil; rs=rs->sNext){ int tcost;
      tcost= OilMinimumCoercionCost(ts2,rs->sBaseType,cost);
      if (cost>tcost){
              seq= vOilCoercionSeq;
              rt= vOilCoercionType;
              cost= tcost;
          }
        else if (cost > rs->sCost)	/* quick rejection */
          for (cc=rs->sBaseType->sCoercionCosts;
		cc!=nil;
		cc=cc->sNextCost){
            int basecost= rs->sCost;
            int tmpcost;
	    tmpcost= OilMinimumCoercionCost(ts2,
            				cc->sResultType,
            				cost-basecost);
            if(cost> (basecost+=tmpcost)){
              seq= vOilCoercionSeq;
              rt= vOilCoercionType;
              cost= basecost;
              }  /* end of if'cost */
            }  /* end of for'cc */
      } /* end of for'rs */
    vOilCoercionSeq= seq;
    vOilCoercionType= rt;
    return( cost );
    }  /* end of Minimum Coercion Seq */


/********************************
  OilMinimumCoercionCost - 
  */ 
int OilMinimumCoercionCost( ts, t, cost )
	tOilTypeSet ts;
	tOilType t;
	int cost;
  {
    tOilType rt=nil;
    tOilCoercionSeq seq=nil;

      if (OilSetIncludes(ts,t))
          { int tcost;
            seq= vOilCoercionSeq;
            tcost= vOilCoercionElement->sCost;
            if (vOilCoercionSeq==nil){
                if (tcost<cost) {rt= vOilCoercionElement->sBaseType; cost= tcost;}}
              else { tcost+= vOilCoercionSeq->sCost;
                if(tcost<cost) { rt= vOilCoercionSeq->sResultType;
                  cost= tcost; }
              	}
            }
    vOilCoercionSeq= seq;
    vOilCoercionType= rt;
    return( cost );
    }  /* end of Minimum Coercion Cost */


/********************************
  OilNewClass - define a new class with 'numarg' parameters
  */
tOilClass OilNewClass( id, numArg )
        oilName id;
        int numArg; {
  tOilClass nc;
  OilNewID(nc,tOilClass,Sclass);
  nc->sName= id;
  nc->sNumArg= numArg;
  nc->sClassOps= nil;
  return( nc );
  }


/********************************
  OilNewClassSigArg - return and empty class signature
  */
tOilClassArgSig OilNewClassSigArg() { return(nil); }



/********************************
  This variable holds the list of existing class argument signatures
  */

tOilClassArgSig vOilClassArgSig=nil;


/********************************
  OilAddClassSigArg - add an argument to a class signature
      td - is arg a class ref/ specific type ref / arg ref
      st - the specific type referenced
      pi - the parameter index ref'd
      cs - the root class argument signature
  */
tOilClassArgSig OilAddClassSigArg( td, st, pi, cs )
	tOilClassSigArgDesc td;
	tOilType st;
        int pi;
	tOilClassArgSig cs;
  {
    tOilClassArgSig  next;
    tOilClassArgSig  ntd=nil;

    for(next= vOilClassArgSig;
	next!=nil &&ntd==nil;
	next= next->sAnother )
	    if ((next->sDescPattern==td)&&
		(next->sNext==cs) &&
		 ((td==eSpecTypeRef)
		       ?(next->sRefdType==st)
		       :(next->sParamIndex==pi)))
		    ntd= next;
    if (ntd==nil) {
	OilNewID(ntd,tOilClassArgSig,SclassArgSig);
	ntd->sAnother= vOilClassArgSig;
	ntd->sNext= cs;
	ntd->sDescPattern= td;
	ntd->sRefdType= (td==eSpecTypeRef)?st:nil;
	ntd->sParamIndex= pi;
	vOilClassArgSig= ntd;
    }
    return( ntd );
  }


/********************************
  OilAddClassOpId - add an identification to class operator 'op'
      ind - the indication
      op - the class operator
  */
BOOL OilAddClassOpId( ind, op )
	tOilOp ind;
        tOilClassOp op;
  {
    tOilClassOpId opid;
    OilNewID(opid,tOilClassOpId,SclassOpId);
    opid->sNext= op->sIdents;
    op->sIdents= opid;

    opid->sOpInd= ind;
    return(TRUE);
  }
/********************************
  OilAddClassOp - adda new operator to class 'c'
      id - name of operator
      sig - the argument signature
      cost - the cost of operator
  */
tOilClassOp OilAddClassOp( id, sig, cost, c )
        oilName id;
	tOilClassArgSig sig;
	int cost;
        tOilClass c;
  {
    tOilClassOp op;
    OilNewID(op,tOilClassOp,SclassOp);
    op->sNext= c->sClassOps;
    c->sClassOps= op;

    op->sName= id;
    op->sArgDesc= sig;
    op->sCost= cost;
    op->isCoercion= FALSE;
    op->sIdents= nil;
    return(op);
  }


/********************************
  OilAddClassCoercion - define coercion for the class 'c'
  */
BOOL OilAddClassCoercion( op )
        tOilClassOp op;
  {
    op->isCoercion= TRUE;
    return(TRUE);
  }


/********************************
  OilArgRef - get the ref'd argument
  */
tOilType OilArgRef( index, argSig )
        int index;
        tOilArgSig argSig ; {
  if (argSig == nil) return( nil );
  if (index==0) return( argSig->sArg );
  return( OilArgRef(index-1,argSig->sNext) );
  }


/********************************
  OilBuildClassSigInst - build the argument signature from the class signature
  */
tOilArgSig OilBuildClassSigInst( argSig, tdl )
        tOilArgSig argSig;
	tOilClassArgSig tdl; {
    if (tdl==nil) return(nil);
      else {
	tOilType t;
	switch (tdl->sDescPattern) {
	  case eClassRef: t= argSig->sArg; break;
	  case eElementRef: t= OilArgRef(tdl->sParamIndex, argSig); break;
	  case eSpecTypeRef: t= tdl->sRefdType; break;
	  }
	return( OilAddArgSig( t,
			OilBuildClassSigInst( argSig, tdl->sNext )
			));
      }
    } /* end of OilBuildClassSigInst */


/********************************
  OilDefClassOpId - add the defined class operator to the possible
  identification.
  */
void OilDefClassOpId( opid, op )
	tOilClassOpId opid;
	tOilOp op ; {
  if (opid==nil) return;
  OilDefClassOpId( opid->sNext, op );
  OilAddIdentification( opid->sOpInd, op );
  } /* end of OilDefClassOpId */


/********************************
  OilDefClassOp - build an instance of the class operator 'co' using the 
      class type signature
  */
tOilOp OilDefClassOp( co, argSig )
	tOilClassOp co;
        tOilArgSig argSig ; { 
  tOilOp  op;
  op= OilNewOp( co->sName,
	OilBuildClassSigInst( argSig, co->sArgDesc ),
	co->sCost
	);
  if (co->isCoercion==TRUE) OilAddCoercion(op);
  OilDefClassOpId(co->sIdents,op);
  return( op );
  } /* end of Build Class Op */

    
/********************************
  OilNewTypeInClass - define a new type of class 'c' with element type 'et'
      with name 'id'
  */
tOilType OilNewTypeInClass( c, as, id )
	tOilClass c;
	tOilArgSig as;
	oilName id; {
    tOilType nt;
    tOilArgSig ts;		/* the completed argument signature */
    tOilClassOp pco;

    nt= OilNewType( id );
    ts= OilAddArgSig( nt, as );

    /* establish the operators */
    for (pco=c->sClassOps; pco!=nil; pco=pco->sNext ){
        OilDefClassOp( pco, ts );
        }
    return(nt);
    }	/* end of OilNewTypeInClass */



/********************************
  OilGetArgType - Get type of argument #n of operator 'op'
  */
tOilType OilGetArgType( op, n )
	tOilOp op; int n;
  {
  tOilArgSig s;
  for (s= (op==nil) ? nil : op->sArgs;
  	n>0&&s!=nil;
  	n--)
  		s=s->sNext;
  return( (s!=nil) ? s->sArg : nil);
   
  }

	/* the end of oiladt2a.c */
