static char rcsid[] = "$Id: symbinfo.c,v 1.4 1992/08/10 22:20:26 waite Exp $";
/* Symbol information database */
/* Copyright (c) 1991, The Regents of the University of Colorado */

#include "system.h"
#include "deftbl.h"
#include "prop.h"
#include "envmod.h"
#include "obstack.h"
#include "rdsymb.h"
#include "idn.h"

static ENVIRONMENT *env;	/* Contour model for symbol definition */

static struct obstack UsedSym;	/* Symbols used in each program */
static int NoSymbol;		   /* (array terminated by NoSymbol) */

static DefTableKey *Programs;	/* Programs in most recent contour */
static struct obstack ProgStk;	   /* (array terminated by NoKey) */


static DefTableKey PgmKey;	/* Program currently being examined */

/***/
void
SymbolInfo(string, external)
char *string; int external;
/* Obtain information about a single symbol
 *    On entry-
 *       string=character form of the symbol
 *       external=0 if the symbol is defined in the current file
 *                nonzero otherwise
 ***/
{
   int type = 1, Symbol;

   mkidn(string, strlen(string), &type, &Symbol);

   if (external) obstack_grow(&UsedSym, &Symbol, sizeof(int));
   else SetProgram(DefineIdn(env, Symbol), PgmKey);
}

/***/
void
AddFiles(obj, src)
char **obj, **src;
/* Add symbol information to the data base
 *    On entry-
 *       obj=array of object file names, terminated by null
 *       src=array of source file names parallel to obj
 ***/
{
   env = NewScope(env);
   obstack_free(&ProgStk, Programs);

   while (*obj) {
      PgmKey = NewKey();
      obstack_grow(&ProgStk, &PgmKey, sizeof(DefTableKey));
      SetSource(PgmKey,*src++);
      SetObject(PgmKey,*obj);
      read_symbols(*obj++);
      SetUses(PgmKey,(int *)obstack_copy(&UsedSym, &NoSymbol, sizeof(int)));
   }

   PgmKey = NoKey;
   Programs = (DefTableKey*)obstack_copy(&ProgStk,&PgmKey,sizeof(DefTableKey));
}


static FILE *source;	/* Source file names for required programs */
static FILE *object;	/* Object file names for required programs */

/**/
static void
NeedProgram(pkey)
DefTableKey pkey;
/* Make certain a program is in the set of required programs
 *    On entry-
 *       pkey=definition table key for the required program
 *    On exit-
 *       All programs defining symbols used in this program are in the set
 *          of required programs
 **/
{
   int *sym;

   if (GetMember(pkey)) return;

   SetMember(pkey);
   (void)fprintf(source, "%s\n", GetSource(pkey, "Missing file name"));
   (void)fprintf(object, "%s\n", GetObject(pkey, "Missing file name"));

   if ((sym = GetUses(pkey, (int *)NULL)) == (int *)NULL) return;

   while (*sym != NoSymbol) {
      DefTableKey key;

      if ((key = GetProgram(KeyInEnv(env, *sym), NoKey)) != NoKey)
         NeedProgram(key);
      sym++;
   }
}

/***/
void
OutFiles(symfile)
char *symfile;
/* Output the source and object file names of the required programs
 *    On entry-
 *       symfile defines the symbols that must be satisfied
 ***/
{
   char string[BUFSIZ+1];
   DefTableKey *p;
   FILE *symbols;

   if ((source = fopen("source", "w")) == NULL) {
      (void)fputs("Could not open file source\n", stderr);
      exit(1);
   }

   if ((object = fopen("object", "w")) == NULL) {
      (void)fputs("Could not open file object\n", stderr);
      exit(1);
   }

   if ((symbols = fopen(symfile, "r")) == NULL) {
      (void)fprintf(stderr, "Could not open file %s\n", symfile);
      exit(1);
   }
   *string = '_';
   while (fgets(string+1, BUFSIZ, symbols)) {
      int l = strlen(string), type = 1, Symbol;
      DefTableKey program;

      string[l-1] = '\0';	/* Remove the newline from the string */
      mkidn(string+1, l-2, &type, &Symbol);
      program = GetProgram(KeyInEnv(env, Symbol), NoKey);
      if (program != NoKey) NeedProgram(program);
      else {
         mkidn(string, l-1, &type, &Symbol);
         program = GetProgram(KeyInEnv(env, Symbol), NoKey);
         if (program != NoKey) NeedProgram(program);
         else fprintf(stdout, "\"%s\" is not an entry point\n", string);
      }
   }

   /* By definition, all programs on the innermost contour are required */
   for (p = Programs; *p != NoKey; p++) NeedProgram(*p);
}

/***/
void
SymbInit()
/* Initialize the symbol data base
 ***/
{
   env = NewEnv();
   obstack_init(&UsedSym);
   obstack_init(&ProgStk);
				/* Establish the Non-existent symbol */
   NoSymbol = stostr(" ", 1);
				/* Establish the invariant for Programs */
   PgmKey = NoKey;
   Programs = (DefTableKey*)obstack_copy(&ProgStk,&PgmKey,sizeof(DefTableKey));
}
