#include <string.h>
#include <stdio.h>

#include "comar.h"
#include "cmrlib.h"
#include "comar_ins.h"
#include "comarMacros.h"
#include "comar_func.h"
#include "name.h"
#include "err.h"        /* definition of message(), ... */
#include "csm.h"       /* definition of string[] */
#include "source.h"  /* definition of SourceName */

/**************************************************************************/

char    *ERRTXT[5] =
{
    "Nonterminal expected: symbol is also used as terminal or meta symbol.",
    "Terminal expected: symbol is also used as nonterminal or meta symbol.",
    "Meta symbol expected: symbol is also used as nonterminal or terminal.",
    "It can't be allocated new storage!",
    "Write Error: output file can't be open!"
};

#define ERR0    0
#define ERR1    1
#define ERR2    2
#define ERR3    3
#define ERR4    4

POSITION        NULLPOS = {0,0};


/**************************************************************************/

typedef enum { UNDEF, TERM, NON, UNKNOWN, META }  tag_type;

typedef struct {
	tag_type	tag;
	union {
		p_term	term;
		p_nterm	non;
		int	*unknown;
		SEQunit	meta;
	}	info;
} * tab_type;

typedef struct {
	p_other	conn;
	p_other	reduce;
	p_other	noreduce;
	} * oth_type;

static	tab_type	TAB;
static	oth_type	OTHER;
static	SID		MAX;

/**************************************************************************/

/* the following functions can be classified in four categories. Three
 * of them are associated with one CHAIN (symb, unknown, list):
 *	0. Associated with no chain: functions for output and others
 *
 *	1. CHAIN symb: It collects all Symbols and is used to build up
 *		the COMAR symbol table.
 *	2. CHAIN unknown: Starting with the initialization of the internal
 *		symbol table TAB it collects all objects with their
 *		properties. At the end all objects are classified in
 *		terminals, nonterminals and others and inserted in the
 *		COMAR-definitiontable.
 *	3. CHAIN list: It inserts the productions in the
 *		COMAR-definitiontable.
 */


/*--------------------------------------------------------------------*/
/* CATEGORIE 0: functions for inserting property names and for output */
/*--------------------------------------------------------------------*/

static	String genfilename(name, ext)
String  name, ext;
{
  int l1, l2;
  char *result;
 
  l1 = strlen(name);
  l2 = strlen(ext);
 
  if ( (result = malloc( (l1+l2+1)*sizeof(char) ))  == (char *)NULL )
        message(FATAL, ERRTXT[ERR3], 0, &NULLPOS);
 
  (void)strncpy(result, name, l1+1);
  (void)strncat(result, ext, l2);
 
  return(result);
}               /* end of genfilename() */

void	write_results()
{  FILE *fp;
 
     if ((fp = fopen(genfilename(SourceName,".cmr"),"w")) == NULL)
        {
                message(FATAL, ERRTXT[ERR4], 0 , &NULLPOS);
                return;
        }
     OUTPUT(fp, CMR);
     fclose(fp);
     return;
}

/*---------------------------------------------------*/
/* CATEGORIE 1: functions used to compute CHAIN symb */
/*---------------------------------------------------*/

/* global variables for the COMAR symboltable indices of some properties
 * defined with ins_property_name() and used in ins_term_with_prop()
 * or ins_other_with_prop().
 */
static	SID	CODE_SID = 0, STOP_SID = 0, SEPA_SID = 0,
		BRACKET_SID = 0, SKIP_SID = 0, GRAMNAME_SID = 0,
		CONNECT_SID = 0, REDUCE_SID = 0, NOREDUCE_SID = 0;

SID	ins_property_name(prop)
proptype	prop;
{
  p_string	str;

      switch ( prop )
      {
	/* Property names of terminals */
	case	P_CODE	 : if ( CODE_SID == 0 )
			     {
				str = cmrlib_newstring(CODE);
			   	CODE_SID = str->sid;
			     }
			   break;
	case	P_STOP	 : if ( STOP_SID == 0 )
			     {
				str = cmrlib_newstring(STOP);
			   	STOP_SID = str->sid;
			     }
			   break;
	case	P_SEPA	 : if ( SEPA_SID == 0 )
			     {
				str = cmrlib_newstring(SEPA);
			   	SEPA_SID = str->sid;
			     }
			   break;
	case	P_BRACKET: if ( BRACKET_SID == 0 )
			     {
				str = cmrlib_newstring(BRACKET);
			   	BRACKET_SID = str->sid;
			     }
			   break;
	case	P_SKIP	 : if ( SKIP_SID == 0 )
			     {
				str = cmrlib_newstring(SKIP);
			   	SKIP_SID = str->sid;
			     }
			   break;

	/* Property names of other-objects */
	case P_CONNECT	: if ( CONNECT_SID == 0 )
			     {
				str = cmrlib_newstring(CONNECTION);
			  	CONNECT_SID = str->sid;
			     }
			  break;
	case P_GRAMMAR	: if ( GRAMNAME_SID == 0 )
			     {
				str = cmrlib_newstring(GRAMNAME);
			  	GRAMNAME_SID = str->sid;
			     }
			  break;
	case P_REDUCE	: if ( REDUCE_SID == 0 )
			     {
				str = cmrlib_newstring(REDUCE);
			  	REDUCE_SID = str->sid;
			     }
			  break;
	case P_NOREDUCE	: if ( NOREDUCE_SID == 0 )
			     {
				str = cmrlib_newstring(NOREDUCE);
			  	NOREDUCE_SID = str->sid;
			     }
	default		: /* Is not possible */
			  break;
      }  /* of switch */
  return(0);
}
 

SID	ins_symbol(str, mode)
String	str;
int	mode;
{
  symb_entry	symb;

  if ( mode == Kp_string )
	symb.Vp_string = cmrlib_newstring(str);
  else /* Kp_name */
	symb.Vp_name = cmrlib_newname(str);

  return( sidOfsymb_entry(symb) );
}

/*------------------------------------------------------*/
/* CATEGORIE 2: functions used to compute CHAIN unknown */
/*------------------------------------------------------*/

void	init_internal_sid_tab()
{
  int 	i;

  MAX = cmrlib_getmaxsid(CMR->symbols);
  if ( (TAB = (tab_type)malloc((MAX+1) * sizeof(*TAB))) == (tab_type)NULL )
    {
	message(FATAL, ERRTXT[ERR3], 0 , &NULLPOS);
	exit(1);
    }
  if ( (OTHER = (oth_type)calloc((MAX+1), sizeof(*OTHER))) == (oth_type)NULL)
    {
	message(FATAL, ERRTXT[ERR3], 0 , &NULLPOS);
	exit(1);
    }

  for (i=0; i<=MAX; i++)
	TAB[i].tag = UNDEF;

  return;
}

/* Try to insert symb as terminal */
void	ins_term(symb, pos)
SID	symb;
POSITION	*pos;
{
  switch ( TAB[symb].tag )
    {
	/* already used: now symbol can be classified as terminal */
	case UNKNOWN	:
	/* not defined => can be inserted as terminal */
	case UNDEF	: TAB[symb].info.term = cmrlib_newterm(symb);
			  TAB[symb].tag = TERM;

	/* already defined as terminal => nothing to do */
	case TERM	: break;

	/* NON or META */
	default		: message(ERROR, ERRTXT[ERR1], 0 , pos);
			  break;

    }  /* of switch */

  return;
}


void	ins_nonterm(symb, pos)
SID	symb;
POSITION	*pos;
{
  switch ( TAB[symb].tag )
    {
	/* already used: now symbol can be classified as nonterminal */
	case UNKNOWN	:
	/* not defined => can be inserted as nonterminal */
	case UNDEF	: TAB[symb].info.non = cmrlib_newnterm(symb);
			  TAB[symb].tag = NON;

	/* already defined as nonterminal => nothing to do */
	case NON	: break;

	/* TERM or META */
	default		: message(ERROR, ERRTXT[ERR0], 0 , pos);
			  break;

    }  /* of switch */

  return;
}


void	ins_unknown(symb)
SID	symb;
{
  symb_entry	new;

  switch ( TAB[symb].tag )
    {
	/* not defined => must be inserted as not classified symbol */
	case UNDEF	: TAB[symb].tag = UNKNOWN;
			  break;
	/* UNKNOWN, TERM, NON or META */
	default		: break;

    }  /* of switch */

  return;
}


void	ins_meta(symb, pos, list)
SID	symb;
POSITION	*pos;
SEQunit	list;
{
  symb_entry	new;

  switch ( TAB[symb].tag )
    {
	/* not defined => can be inserted as meta symbol */
	case UNDEF	:
	/* already used: now symbol can be classified as meta symbol */
	case UNKNOWN	: TAB[symb].tag = META;
			  TAB[symb].info.meta = list;
			  break;
	/* TERM or NON */
	default		: message(ERROR, ERRTXT[ERR2], 0 , pos);
			  break;
    }  /* of switch */

  return;
}


void	ins_termlist()
{
  int 	i;

  for (i=0; i<=MAX; i++)
    if ( TAB[i].tag == UNKNOWN )
	ins_term((SID)i, NULLPOS);
  return;
}

void	ins_term_with_prop(symb, pos, prop, propval)
SID	symb;
POSITION	*pos;
proptype	prop;
int		propval;
{
  def_entry	def;

  ins_term(symb, pos);

  if ( TAB[symb].tag == TERM )
  {
    def = p_termTodef_entry(TAB[symb].info.term);
    switch ( prop )
      {
	case	P_CODE	 : 
		(void)cmrlib_newpropval(def,
					CODE_SID,
					p_nvalTovalue(Mkp_nval(propval)));
			   break;
	case	P_STOP	 : (void)cmrlib_newprop(def,STOP_SID);
			   break;

	case	P_SEPA	 : (void)cmrlib_newprop(def,SEPA_SID);
			   break;

	case	P_BRACKET: (void)cmrlib_newprop(def,BRACKET_SID);
			   break;

	case	P_SKIP	 : (void)cmrlib_newprop(def,SKIP_SID);
			   break;

	default		: /* Is not possible */
			  break;
      }  /* of switch */
  } /* if */

  /* else: ins_term has already reported an error. */

  return;
}


void	ins_other_with_prop(symb, pos, prop)
SID	symb;
POSITION	*pos;
proptype	prop;
{
def_entry	def;
value		val;

 switch ( prop )
    {
	case P_CONNECT: if ( OTHER[symb].conn == (p_other)NULL )
			{
			   def.Vp_other =
				OTHER[symb].conn = cmrlib_newother(symb);
			   (void)cmrlib_newprop(def, CONNECT_SID);
			}
			break;
	/* This can be called at most once because of syntactically reasons */
	case P_GRAMMAR: def.Vp_other = cmrlib_newother(symb);
			(void)cmrlib_newprop(def, GRAMNAME_SID);
			break;
	case P_REDUCE:  if ( OTHER[symb].reduce == (p_other)NULL )
			{
			   def.Vp_other =
				OTHER[symb].reduce = cmrlib_newother(symb);
			   ins_term(symb, pos);
			   if ( TAB[symb].tag == TERM )
			     {
				val.Vp_dval = Mkp_dval(TAB[symb].info.term->did);
				(void)cmrlib_newpropval(def,REDUCE_SID,val);
			     }
			   /* else: ins_term has already reported an error */
			}
			break;
	case P_NOREDUCE: if ( OTHER[symb].noreduce == (p_other)NULL )
			{
			   def.Vp_other =
				OTHER[symb].noreduce = cmrlib_newother(symb);
			   ins_term(symb, pos);
			   if ( TAB[symb].tag == TERM )
			     {
				val.Vp_dval = Mkp_dval(TAB[symb].info.term->did);
				(void)cmrlib_newpropval(def,NOREDUCE_SID,val);
			     }
			   /* else: ins_term has already reported an error */
			}
	default: /* is not possible ! */
			break;
    }  /* of switch */
  return;
} /* ins_other_with_prop() */


/*---------------------------------------------------*/
/* CATEGORIE 3: functions used to compute CHAIN list */
/*---------------------------------------------------*/

SEQunit	ins_prod(name, lhs, rhs)
SID	name, lhs;
SEQunit		rhs;
{
  if ( TAB[lhs].tag == NON )
	(void)cmrlib_newprod(name, TAB[lhs].info.non->did, rhs);
  /* else: error has been already reported. */

  return( nullSEQunit() );
}


SEQunit	make_list(l1, l2, opr)
SEQunit		l1, l2;
ebnftype	opr;
{
  SEQunit	res;
  unit	un;
  res = nullSEQunit();

  switch ( opr )
  {
    case ALT	: un = p_altTounit(Mkp_alt(l1,l2));
		  break;
    case OPT	: un = p_optTounit(Mkp_opt(l1));
		  break;
    case STAR	: un = p_starTounit(Mkp_star(l1));
		  break;
    case PLUS	: un = p_plusTounit(Mkp_plus(l1));
		  break;
    case DELREP	: un = p_delrepTounit(Mkp_delrep(l1,l2));
		  break;
    default	: un = p_elunitTounit((p_elunit)NULL); /* Is not  possible. */
  } /* of switch */

  return( appendfrontSEQunit(res,un) );
}


SEQunit	make_elem_for_others(symb, el_mode)
SID	symb;
proptype	el_mode;
{
  SEQunit	res;
  unit	un;

  res = nullSEQunit();

  switch ( el_mode )
  {
    case P_CONNECT : un = p_elunitTounit(Mkp_elunit(OTHER[symb].conn->did));
		     break;
    case P_REDUCE  : un = p_elunitTounit(Mkp_elunit(OTHER[symb].reduce->did));
		     break;
    case P_NOREDUCE: un = p_elunitTounit(Mkp_elunit(OTHER[symb].noreduce->did));
		     break;
    default:  un = p_elunitTounit((p_elunit)NULL);/* Is not  possible. */
  } /* of switch */

  return( appendfrontSEQunit(res, un) );
}


SEQunit	make_elem(symb)
SID	symb;
{
  SEQunit	res;
  unit	un;
  res = nullSEQunit();

  switch ( TAB[symb].tag )
  {
    case TERM	: un = p_elunitTounit(Mkp_elunit(TAB[symb].info.term->did));
		  return(appendfrontSEQunit(res,un));
    case NON	: un = p_elunitTounit(Mkp_elunit(TAB[symb].info.non->did));
		  return(appendfrontSEQunit(res,un));
    case META	: return( TAB[symb].info.meta );

    default	: break; /* This can't be! Internal Error! */
  } /* of switch */

  return( res );
}


SEQunit concatMETAunit(l1, l2)
SEQunit l1, l2;
{
        SEQunit    help;
 
        if ( l1 == (SEQunit)NULL )
                return(l2);
        if ( l2 == (SEQunit)NULL )
                return(l1);
        return( make_list(l1, l2, ALT) );
}
