/* 
 * $Id: array.c,v 2.0 1992/09/23 08:38:57 toh-hei Exp $
 *
 * Copyright (c) 1992 Kimura Laboratory, Department of Information Science,
 * Tokyo Institute of Technology.  All Rights Reserved.
 *
 */

/*
 * Data Structure
 *
 *	    -----------------+-----+
 *	      |     buf_sz-1 |  /  |
 *	      |              +-----+
 *	      |              :     :
 *	      |              +-----+
 *	      |    base+size |  /  |
 *	      |              +-----+-----------------
 *	      |  base+size-1 |  O  | low+size-1   |
 *	      |              +-----+              |
 *	   buf_sz            :     :              |
 *	      |              +-----+              |
 *	      |   base-low+k |  O  | k          size 
 *	      |              +-----+              |
 *	      |              :     :              |
 *	      |              +-----+              |
 *	      |         base |  O  | low          |
 *	      |  ------------+-----+-----------------
 *	      |    |  base-1 |  /  |
 *	      |    |         +-----+
 *	      |  base        :     :
 *	      |    |         +-----+
 *	      |    |       0 |  /  |
 *	    -----------------+-----+
 *	                       buf
 *	The world of C array         The world of CLU array
 */

#include <clu2c.h>
#include <type.h>
#include <glo.h>

#define INIT_BASE	3
#define INIT_SIZE	8

/* define for array$elements */
#define ALLOC_SZ_ELM     4          /* allocate size for iter array$elements */
#define ARG_A_ELM    ((array)(*ivarp)[1])     /* argment a in array$elements */
#define LV_I_ELM     (*ivarp)[2]    /* local variable i in array$elements */
#define LV_SZ_ELM    (*ivarp)[3]    /* local variable sz in array$elements */

/* define for array$indexes */
#define ALLOC_SZ_IND     4          /* allocate size for iter array$indexes */
#define ARG_A_IND    ((array)(*ivarp)[1])     /* argment a in array$indexes */
#define LV_LO_IND    (*ivarp)[2]    /* local variable lo in array$indexes */
#define LV_HI_IND    (*ivarp)[3]    /* local variable hi in araay$indexes */

extern int (*parm_call(type_parm, char*))();
extern int (*op_table())();
extern string get_subparam();

int array_create(int l)

{
  array res;

  res = (array) malloc(sizeof(struct array_rep));	/* make rep */
  res->low = l;
  res->size = 0;
  res->base = INIT_BASE;
  res->buf_sz = INIT_SIZE;
  res->buf = (elt *) malloc(INIT_SIZE * sizeof(elt));
	
  retval_area[0] = (elt) res;
  return(RET);
}

int array_new()
{
  array res;

  res = (array) malloc(sizeof(struct array_rep));	/* make rep */
  res->low = 1;
  res->size = 0;
  res->base = INIT_BASE;
  res->buf_sz = INIT_SIZE;
  res->buf = (elt *) malloc(INIT_SIZE * sizeof(elt));
	
  retval_area[0] = (elt) res;
  return(RET);
}

int array_predict(int l, int sz)
{
  array res;
  int real_size;

  real_size = (abs(sz) <= INIT_SIZE) ? INIT_SIZE: abs(sz);

  res = (array) malloc(sizeof(struct array_rep));	/* make rep */
  res->low = l;
  res->size = 0;
  res->buf_sz = real_size;
  res->buf = (elt *) malloc(real_size * sizeof(elt));

  if (sz >= 0)
    res->base = 0;
  else
    res->base = real_size - 1;

  retval_area[0] = (elt) res;
  return(RET);
}

int array_fill(int l, int sz, elt e)
{
  array res;
  int i;

  if (sz < 0) {
    signame  = "negative_size";
    return(SIG);
  }

  res = (array) malloc(sizeof(struct array_rep));	/* make rep */
  res->low = l;
  res->size = sz;
  res->base = 0;
  res->buf_sz = sz;	
  res->buf = (elt *) malloc(sz * sizeof(elt));

  for(i = 0; i < sz; i++)				/* insert to buf */
    res->buf[i] = (elt) e;
	
  retval_area[0] = (elt) res;
  return(RET);
}

int array_high(array a)
{
  retval_area[0] = (elt) (a->low + a->size - 1);
  return(RET);
}

int array_low(array a)
{
  retval_area[0] = (elt) (a->low);
  return(RET);
}

int array_size(array a)
{
  retval_area[0] = (elt) (a->size);
  return(RET);
}

int array_empty(array a)
{
  retval_area[0] = (elt) (a->size == 0);
  return(RET);
}

int array_set_low(array a, int l)
{
  a->low = l;
  return(RET);
}

int array_trim(array a, int low1, int size)
{
  int d;

  if (low1 < a->low || low1 > a->low + a->size) {
    signame = "bounds";
    return(SIG);
  }
  if (size < 0){
    signame = "negative_size";
    return(SIG);
  }
  d = low1 - a->low;
  a->base += d;
  a->low = low1;
  a->size -= d;
  if (a->size > size)
    a->size = size;
  return(RET);
}
    

int array_fetch(array a, int i)
{
  int low;

  low = a->low;
  if (i < low || i > low + a->size - 1) {
    signame = "bounds";
    return(SIG);
  }
  retval_area[0] = (elt) (a->buf[i - low + a->base]);
  return(RET);
}

int array_store(array a, int i, elt e)
{
  int low;

  low = a->low;
  if (i < low || i > low + a->size - 1) {
    signame = "bounds";
    return(SIG);
  }
  a->buf[i - low + a->base] = (elt) e;
  return(RET);
}

int array_addh(array a, elt e)
{
  int sz, i, j;
  elt *new_buf;

  sz = a->size;
  if (a->base + sz + 1 >= a->buf_sz)		/* when no space */
    if (sz < a->base) {
      i = a->base / 2;				/* shift base to left */
      if (i <= 0) { fprintf(stderr, "offset error in int$addh"); exit(1);}
      for (j = a->base; j <= a->base + sz - 1; j++)
        a->buf[j-i] = a->buf[j];		/* shift element to left */
      a->base -= i;
    }
    else {
      new_buf = (elt *) malloc(a->buf_sz*2*sizeof(elt)); /* make new buf */
      a->buf_sz = a->buf_sz * 2;
      for (i = a->base; i <= a->base + sz - 1; i++)
        new_buf[i] = a->buf[i];
      a->buf = new_buf;
    }
  a->buf[a->base + sz] = e;
  a->size = sz + 1;
  return(RET);
}

int array_addl(array a, elt e)
{
  int i,j;
  elt *new_buf;

  if (a->base ==  0)               /* when no space */
    if (a->size*2 < a->buf_sz) {
      i = (a->buf_sz - a->size) / 2;      /* shift base to right */
      if (i <= 0) { fprintf(stderr, "offset error in int$addl"); exit(1);}
      for(j = a->base + a->size - 1; j >= a->base; j--)
	a->buf[j+i] = a->buf[j];
      a->base += i;
    }
    else {
      new_buf = (elt *) malloc(a->buf_sz*2*sizeof(elt)); /* make new buf */
      a->buf_sz = a->buf_sz * 2;
      for (i = a->base; i <= a->base + a->size - 1; i++)
        new_buf[i+a->size] = a->buf[i];
      a->base += a->size;
      a->buf = new_buf;      
    }
  a->base--;
  a->buf[a->base] = (elt) e;
  a->size++;
  a->low--;
  return(RET);
}

int array_remh(array a)
{
  if (a->size <= 0) {
    signame = "bounds";
    return(SIG);	
  }
  a->size--;  
  retval_area[0] = (elt) a->buf[a->base + a->size];
  return(RET);
}

int array_reml(array a)
{
  if (a->size <= 0) {
    signame = "bounds";
    return(SIG);	
  }
  a->size--;
  a->low++;
  retval_area[0] = (elt) a->buf[a->base++];
  return(RET);
}	

int array_bottom(array a)
{
  if (a->size <= 0) {
    signame = "bounds";
    return(SIG);	
  }
  retval_area[0] = (elt) a->buf[a->base];
  return(RET);
}

int array_top(array a)
{
  if (a->size <= 0) {
    signame = "bounds";
    return(SIG);	
  }
  retval_area[0] = (elt) a->buf[a->base + a->size - 1];
  return(RET);
}

int array_elements(bool init, elt **ivarp, array a)
{
  /* local var int i, sz */

  if (init) {               /* initial iterator call */
    *ivarp = (int*) malloc(ALLOC_SZ_ELM*sizeof(elt));
    (*ivarp)[1] = (elt) a;
  }
  else                     /* not initial call, goto previous yield point */
    if ((*ivarp)[0] == 1) goto yield1;

  LV_SZ_ELM = (elt) ARG_A_ELM->size;

  for (LV_I_ELM = 1; LV_I_ELM <= LV_SZ_ELM; LV_I_ELM++) {
    retval_area[0] = (elt) ARG_A_ELM->buf[ARG_A_ELM->base + LV_I_ELM - 1];
    (*ivarp)[0] = 1;
    return(RET);
  yield1:
    (0);
  }
  iter_end = TRUE;
  return(RET);
}

int array_indexes(bool init, elt **ivarp, array a)
{
  /* local var int lo, hi */

  if (init) {               /* initial iterator call */
    *ivarp = (int*) malloc(ALLOC_SZ_IND*sizeof(elt));
    (*ivarp)[1] = (elt) a;
  }
  else                     /* not initial call, goto previous yield point */
    if ((*ivarp)[0] == 1) goto yield1;

  LV_LO_IND = (elt) ARG_A_IND->low;
  LV_HI_IND = (elt) ARG_A_IND->low + ARG_A_IND->size - 1;

  while (LV_LO_IND <= LV_HI_IND) {
    retval_area[0] = (elt) LV_LO_IND;
    (*ivarp)[0] = 1;
    return(RET);
  yield1:
    LV_LO_IND++;
  }
  iter_end = TRUE;
  return(RET);
}

int array_equal(array a1, array a2)
{
  retval_area[0] = (elt) (a1 == a2);
  return(RET);
}

int array_copy1(array a1)
{
  array res;
  int i;

  res = (array) malloc(sizeof(struct array_rep));	/* make rep */
  res->low = a1->low;
  res->size = a1->size;
  res->base = a1->base;
  res->buf_sz = a1->buf_sz;
  res->buf = (elt *) malloc((res->buf_sz) * sizeof(elt));
  
  for(i= 0; i < a1->size; i++)
    res->buf[res->base + i] = a1->buf[a1->base + i];
  retval_area[0] = (elt) res;
  return(RET);
}

/*
 * print = proc (a: array[t], pst: pstream)
 *	where t has print: proctype(t, pstream)
 */

int array_print(int (*(op_list[]))(), array a, clus pst)
{
    int max;
    int i;

    if ( _cpstream_text(pst, "[") == SIG ) {
	out_handler();
	return SIG;
    }

    /* print low bound */

    array_low(a);
    if ( int_print((int) retval_area[0], pst) == SIG ) {
	out_handler();
	return SIG;
    }

    /* print high bound unless all elements will be printed  */

    if ( _cpstream_get_max_width(pst) == SIG ) {
	out_handler();
	return SIG;
    }
    if ( a->size >= (int) retval_area[0] ) {
	if ( _cpstream_text(pst, "..") == SIG ) {
	    out_handler();
	    return SIG;
	}
	if ( int_print(a->low + a->size - 1, pst) == SIG ) {
	    out_handler();
	    return SIG;
	}
    }

    if ( _cpstream_start(pst, ": ") == SIG ) {
	out_handler();
	return SIG;
    }

    /* print elements */

    for ( i = a->base; i < a->base + a->size; i++ ) {
	if ( i != a->base ) {
	    if ( _cpstream_pause(pst, ", ") == SIG ) {
		out_handler();
		return SIG;
	    }
	    bool_not((bool) retval_area[0]);
	    if ( (bool) retval_area[0] ) {
		break;
	    }
	}
	(*(op_list[0]))(a->buf[i], pst);	 /* invoke t$print */
    }
    
    if ( _cpstream_stop(pst, "]") == SIG ) {
	out_handler();
	return(SIG);
    }
    return RET;
}

/*
 * _gcd = proc(a: array[t], tab: gcd_tab) returns(int)
 *		where t has _gcd: proctype(t, gcd_tab) returns(int)
 */

/* stub version */

int array__gcd(int (*(op_list[]))(), array a, clus tab)
{
    signame = "failure";
    sigarg_area[0] = (elt) "array$_gcd: not implemented";
    return SIG;
}

/*
 * cons = proc(q: sequence[t]) returns (array[t])
 */

int array_cons(sequence q)
{
    return array_cons2(1, q);
}

/*
 * cons2 = proc (low: int, q: sequence[t]) returns (array[t])
 */

int array_cons2(int low, sequence q)
{
    array res;
    int i, sz;

    sequence_size(q);
    sz = (int) retval_area[0];
    
    res = (array) malloc(sizeof(struct array_rep));	/* make rep */
    res->low = low;
    res->size = sz;
    res->base = 0;
    res->buf_sz = sz;	
    res->buf = (elt *) malloc(sz * sizeof(elt));

    for ( i = 0; i < sz; i++ ) {
	res->buf[i] = q->buf[i]; 	/* !!! assumes rep of sequence */
    }

    retval_area[0] = (elt) res;
    return(RET);
}
