/************************************************************************
*
* snacc.c -  Compiles ASN.1 src files into an internal type tree.
*        Imported type/value references are resolved if possible.
*        Produces C or C++ encoder/decoder/print/free code and .h for 
*        data struct and prototypes.
*        Generated C can be either ANSI or old style via macros.
*        Produces values for OBJECT IDENTIFIERs, INTEGERs and BOOLEANs
*
* Mike Sample 1991/92
*
* NOTES
*
*  See the README file for compiling tips.  This should compile
*  with ANSI or non-ANSI c compilers.
*
*  each ASN.1 source file must contain a complete ASN.1 module:
*       <ModName> DEFINITIONS ::= BEGIN ... END
*
* 91/09/04 - modified to handle new data struct (ASN.1 generated)
*            for module info. MS.
*  
*
* Copyright (C) 1991, 1992 Michael Sample
*            and the University of British Columbia
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program and the associated libraries are distributed in the hope
* that they will be useful, but WITHOUT ANY WARRANTY; without even the
* implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
* PURPOSE.  See the GNU General Public License and GNU Library General
* Public License for more details.
**
************************************************************************/


#include <stdio.h>
#include <sys/types.h>
#include <sys/file.h>
#include "snacc_config.h"
#include "basetypes.h"
#include "ber.h"
#include "list.h"
#include "mem.h"
#include "asn1module.h"
#include "exports.h"
#include "parser.h"          /* for parser (ech!) globals */
#include "dependency.h"
#include "link_types.h"
#include "link_values.h"
#include "err_chk.h"
#include "print.h"
#include "recursive.h"
#include "define.h"
#include "normalize.h"
#include "do_macros.h"

#include "rules.h"       /* for c file generation */
#include "str_util.h"
#include "type_info.h"
#include "gen_c_code.h"

#include "cpp_rules.h"       /* for c++ file generation */
#include "cpp_types.h"
#include "gen_cpp_code.h"


/* prototypes for this file's routines */

Module* ParseAsn1File PROTO((char* fileName));

void GenCCode PROTO((ModuleList* allMods, long int longJmpVal, int genTypes,
                     int genEncoders, int genDecoders, int genPrinters, 
                     int genValues, int genFree));

void GenCppCode PROTO((ModuleList* allMods, long int longJmpVal, int genTypes,
                       int genEncoders, int genDecoders, int genPrinters, 
                       int genValues, int genFree));

int ModNamesUnique PROTO((ModuleList* m));

Module* usefulTypeModG = NULL;
char* versionG = "1.0";
int maxFileNameLenG = -1; /* values > 2 are considered valid */
                          /* this is used in back_ends/c_gen/str_util.c */

void
Usage PARAMS((prgName),
char* prgName)
{
    fprintf(stderr,"\nUsage: %s ",prgName);
    fprintf(stderr,"[-h] [-c] [-C] [-P] [-t] [-v] [-e] [-d] [-p] [-f]\n");
    fprintf(stderr,"            [-u <useful types ASN.1 file>] [-mf <max file name length>]\n");
    fprintf(stderr,"            [-l <neg number>] <ASN.1 file list>\n\n");
    fprintf(stderr,"  -h prints this msg\n");
    fprintf(stderr,"  -c generate C encoders and decoders (default)\n");
    fprintf(stderr,"  -C generate C++ encoders and decoders\n");
    fprintf(stderr,"  -P print the parsed ASN.1 modules to stdout from their parse trees\n");
    fprintf(stderr,"     (helpful debugging)\n");

    fprintf(stderr,"  -t generate type definitions\n");
    fprintf(stderr,"  -v generate value definitions (limited)\n");
    fprintf(stderr,"  -e generate encode routines\n");
    fprintf(stderr,"  -d generate decode routines\n");
    fprintf(stderr,"  -p generate print routines\n");
    fprintf(stderr,"  -f generate hierarchical free routines (C only)\n");
    fprintf(stderr,"     if none of -t -v -e -d -p -f are given, all are generated\n");
    
    fprintf(stderr,"  -u specify the useful types (ie PrintableString) ASN.1 definition.\n");
    fprintf(stderr,"     See the useful.asn1 file in the snacc/asn1specs/ directory\n");
    fprintf(stderr,"  -mf <num> num is maximum file name length for the generated source files\n");
    fprintf(stderr,"  -l <neg num> where to start error longjmp values decending from (obscure).\n");

    fprintf(stderr,"\n Use \"-\" as the ASN.1 source file name to parse stdin.\n\n");


    fprintf(stderr," This ASN.1 compiler produces C or C++ BER encoders and decoders.\n");

    fprintf(stderr,"\n Version %s, February 1993.\n", versionG);
    fprintf(stderr," Please send bug reports and comments to" );
    fprintf(stderr," snacc-bugs@cs.ubc.ca\n");

/*
    fprintf(stderr,"\nACKNOWLEDGMENTS\n");
    fprintf(stderr,"  This compiler was written by Mike Sample at UBC, with financial\n");
    fprintf(stderr,"  assistance from The Canadian Institute for Telecommunications\n");
    fprintf(stderr,"  Research and the Natural Sciences and Engineering Research Council\n");
    fprintf(stderr,"  of Canada.\n\n");
*/

/*
    fprintf(stderr,"  Mike Sample <msample@cs.ubc.ca> or\n");
    fprintf(stderr,"  Dr. Gerald Neufeld <neufeld@cs.ubc.ca>\n");
    fprintf(stderr,"\n  Department of Computer Science\n");
    fprintf(stderr,"  University of British Columbia\n");
    fprintf(stderr,"  6356 Agricultural Road\n");
    fprintf(stderr,"  Vancouver, British Columbia\n");
    fprintf(stderr,"  Canada  V6T 1Z2\n\n");
*/
    fprintf(stderr,"\n Copyright (C) 1993 Michael Sample and UBC\n");

    fprintf(stderr," This program is free software; you can redistribute it and/or modify\n");
    fprintf(stderr," it under the terms of the GNU General Public License as published by\n");
    fprintf(stderr," the Free Software Foundation; either version 2 of the License, or\n");
    fprintf(stderr," (at your option) any later version.\n\n");

        fprintf(stderr," This program is distributed in the hope that it will be useful,\n");
    fprintf(stderr," but WITHOUT ANY WARRANTY; without even the implied warranty of\n");
    fprintf(stderr," MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n");
    fprintf(stderr," GNU General Public License for more details.\n\n");

/*
    fprintf(stderr," You should have received a copy of the GNU General Public License\n");
    fprintf(stderr," along with this program; if not, write to the Free Software\n");
    fprintf(stderr," Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.\n");
*/
}
    

typedef struct
{
    char*      fileName;
}  SRC_FILE;



int main PARAMS((argc, argv),
int argc _AND_
char **argv)
{
    int i, j, semErr;
    int retVal;
    int fd;
    SRC_FILE* srcList;
    int       numSrcFiles;
    ModuleList* allMods;
    Module* currMod;
    Module** tmpModHndl;
    int currArg;
    int argsProcessed;
    char* usefulTypeModFileName;
    FILE* fPtr;
    int printModuleFlag = FALSE;  /* default: Don't print */
    int genTypeCode = FALSE;
    int genEncodeCode = FALSE;
    int genDecodeCode = FALSE;
    int genPrintCode = FALSE; 
    int genFreeCode = FALSE;  
    int genValueCode = FALSE;  
    int genCCode = FALSE;        /* defaults to C if neither specified */
    int genCppCode = FALSE;
    long longJmpVal = -100;

#ifdef YYDEBUG
    /* prints yacc debugging info to stdout */
    yydebug = 1;
#endif


    if (argc <= 1)
    {
        Usage(argv[0]);
        exit(1);
    }
    
    srcList = (SRC_FILE*) Malloc((argc -1) * sizeof(SRC_FILE));



    /*
     * parse cmd line args
     */
    numSrcFiles = 0;
    usefulTypeModFileName = NULL;
    for (currArg = 1; (currArg < argc); )
    { 
        if ((argv[currArg][0] == '-') && (argv[currArg][1] != '\0'))
            switch(argv[currArg][1])
            {
                case('h'):
                    Usage(argv[0]);
                    exit(1);
                break;

                case('P'):
                    printModuleFlag = TRUE;
                    currArg++;
                break;

                case('v'):
                    genValueCode = TRUE;
                    currArg++;
                break;

                case('t'):
                    genTypeCode = TRUE;
                    currArg++;
                break;

                case('e'):
                    genEncodeCode = TRUE;
                    currArg++;
                break;

                case('d'):
                    genDecodeCode = TRUE;
                    currArg++;
                break;

                case('p'):
                    genPrintCode = TRUE;
                    currArg++;
                break;

                case('f'):
                    genFreeCode = TRUE;
                    currArg++;
                break;

                case('C'):
                    genCppCode = TRUE;
                    currArg++;
                break;

                case('c'):
                    genCCode = TRUE;
                    currArg++;
                break;

                case('u'):
                    if (argv[currArg][2] != '\0')  /* no space after -u */
                    {
                        usefulTypeModFileName = &argv[currArg][2];
                        currArg++;
                    }
                    else
                    {
                        usefulTypeModFileName = argv[currArg+1];
                        currArg += 2;
                    }
                break;

                case('l'):
                    if (argv[currArg][2] != '\0')  /* no space after -l */
                    {
                        longJmpVal = atoi(&argv[currArg][2]);
                        currArg++;
                    }
                    else
                    {
                        longJmpVal = atoi(argv[currArg+1]);
                        currArg += 2;
                    }
                break;

                case('m'):
                    if (argv[currArg][2] == 'f')  
                    {
                        if (argv[currArg][3] != '\0')  /* no space after -mf */
                        {
                            maxFileNameLenG = atoi(&argv[currArg][3]);
                            currArg++;
                        }
                        else
                        {
                            maxFileNameLenG = atoi(argv[currArg+1]);
                            currArg += 2;
                        }
                        break;
                    }
                    else
                        /* fall through to default error */
                
                /* add new cmd line switches here! */
                
                default:
                    fprintf(stderr,"%s: ERROR - unknown cmd line option \"%s\"\n\n", argv[0], argv[currArg]);
                    Usage(argv[0]);
                    exit(1);
            }
        
        else /* asn1srcFileName */
            srcList[numSrcFiles++].fileName = argv[currArg++];
        
    }


    if (numSrcFiles == 0)
    {
        fprintf(stderr,"%s: ERROR - no ASN.1 source files were specified\n",
                argv[0]);
        Usage(argv[0]);
        exit(1);
    }

    /* 
     * set default options
     */
    if (!genCCode && !genCppCode)
        genCCode = TRUE;  /* default to C if neither specified */

    if (!(genTypeCode || genValueCode || genEncodeCode || genDecodeCode ||
          genFreeCode || genPrintCode))
    {
        genTypeCode = TRUE;
        genValueCode = TRUE;
        genEncodeCode = TRUE;
        genDecodeCode = TRUE;
        genFreeCode = TRUE;
        genPrintCode = TRUE;
    }

    else if (genCCode && genCppCode)
    {
        fprintf(stderr,"%s: ERROR - Choose either C or C++, not both\n", 
                argv[0]);
        Usage(argv[0]);
        exit(1);
    }



    /*
     * STEP 1
     * parse 'useful' type module for linking purposes only (if given)
     * the useful type encode,decode, print, free routines are 
     * already in the runtime library.
     */
    if (usefulTypeModFileName != NULL)
    {
        usefulTypeModG = ParseAsn1File(usefulTypeModFileName);

        if (usefulTypeModG == NULL)
            exit(1);
    }
    else
    {
        fprintf(stderr,"  Hmmm, you didn't specify a useful types ASN.1 file with the \"-u\" option.  ");
        fprintf(stderr,"I'll continue assuming your ASN.1 file(s) don't need any useful types.\n\n");
    }


    /*
     * STEP 2 - parse each ASN.1 src file
     */
    allMods = (ModuleList*) GListNew(sizeof(void*));
    for (i = 0 ; i < numSrcFiles; i++)
    {
        currMod = ParseAsn1File(srcList[i].fileName);

        if (currMod == NULL)
            exit(1);

        /*
         * insert this module at the head of the list
         * of already parsed (if any) modules
         */
        tmpModHndl = (Module**)GListAppend(allMods);
        *tmpModHndl = currMod;

    }  /* end per src file for loop */


    /*
     * Check that the module names/oids are unique.
     */
    if (!ModNamesUnique(allMods))
    {
        fprintf(stderr,"Conflicting module names, cannot proceed.\n");
        exit(1);
    }



    /*
     * STEP 3
     * Now that all files have been parsed,
     * link local and locatable import type refs
     */
    if (LinkTypeRefs(allMods) < 0)
    {
        fprintf(stderr,"Type linking errors - cannot proceed\n");
        exit(1);
    }



    /*
     * STEP 4
     * Parse constructed values now that types are all parsed 
     * and have been linked.  Need type info to be able to
     * parse values easily (elimitate ambiguity).
     */
    FOR_EACH_LIST_ELMT(currMod, allMods)
    {
        if (ParseValues(allMods, currMod) != 0)
            fprintf(stderr,"Warning: Value parsing error(s), attempting to continue\n");
    }


    /*
     * STEP 5
     * Value parsing may have defined some new values
     * so can link local and locatable import value refs now.
     */
    if (LinkValueRefs(allMods) < 0)
    {
        fprintf(stderr,"Value linking errors - cannot proceed\n");
        exit(1);
    }

        

    /*
     * STEP 6
     * process macros
     *   - adding type/value defs as nec
     *   - mark type defs with ANY DEFINED BY id if nec
     *     so they are put in the id to ANY type hash tbl.
     */
    semErr = 0;
    FOR_EACH_LIST_ELMT( currMod, allMods)
    {
        ProcessMacros(currMod);
        if (currMod->status == MOD_ERROR)
            semErr = 1;
    }
    if (semErr)
        exit(1);

    /*
     * STEP 7
     * convert silly type constructs into
     * a normal format, leaving behind pure type/value info
     * eg: expand COMPONENTS OF refs, SELECTION types.
     * boil down values into simplest rep. (eg OID -> ENC_OID)
     */
    semErr = 0;
    FOR_EACH_LIST_ELMT( currMod, allMods)
    {
        NormalizeModule(currMod);
        if (currMod->status == MOD_ERROR)
            semErr = 1;
    }
    if (semErr)
        exit(1);


    /*
     * STEP 8
     * Mark recusive types.  Currently the recursive information is
     * not used elsewhere.
     */
    FOR_EACH_LIST_ELMT(currMod, allMods)
    {
        MarkRecursiveTypes(currMod);
    }


    /*
     * STEP 9
     * Check for errors in the ASN.1 modules.
     * Check all modules and exit if errors were found
     */
    semErr = 0;
    if ( usefulTypeModG != NULL)
    {
        ErrChkModule(usefulTypeModG);
        if (usefulTypeModG->status == MOD_ERROR)
            semErr = 1;
    }

    FOR_EACH_LIST_ELMT( currMod, allMods)
    {
        ErrChkModule(currMod);
        if (currMod->status == MOD_ERROR)
            semErr = 1;
    }    


    if (semErr)
        exit(1);




    /*
     *  exit if any sundry errors occurred at any point.
     *  smallErrG is set upon finding small errors that prevent code
     *  production but should not affect the other processing/error
     *  checking steps.  This allows full display of errors.
     */
    if (smallErrG)
    {
        /*
         * for debugging show "parsed" version of ASN.1 module if
         * the print flag is set.
         * Dumps each module to stdout. Printed from Module data struct
         * print here before exiting otherwise print after sorting
         */
        if (printModuleFlag)
        {
            FOR_EACH_LIST_ELMT( currMod, allMods)
            {
                printf("\n\n");
                PrintModule( stdout, currMod);
            }
        }

        exit(1);
    }

    /*
     * STEP 10
     * Make C/C++ typenames/routine names for enc/decode.
     * Type/Value renaming will occur if name conflicts
     * arise between modules.
     * 
     * NOTE: this is done before sorting the types because
     *       the type sorting routine may use the 'isPtr'
     *       information to help order knots of recursive types.
     */
    if (genCCode)
        FillCTypeInfo( &cRulesG, allMods);

    else if (genCppCode)
        FillCppTypeInfo( &cppRulesG, allMods);


    /*
     * STEP 11
     * Sort each typedef list such that independent types are
     * before the types that depend on them
     *
     *  modules remain in same order as given on command line
     *  (cmd line file order should be 
     *      least dependent module-> most dependent module
     *      so that include file order in generated src is correct)
     *  (useful.asn1 is always considered 'first' if given)
     */
    SortAllDependencies(allMods); 

    /*
     * for debugging show "parsed" version of ASN.1 module.
     * dumps each module to stdout. Printed from Module data struct
     * Shows the results of normalization and sorting.
     */
    if (printModuleFlag)
    {
        FOR_EACH_LIST_ELMT( currMod, allMods)
        {
            printf("\n\n");
            PrintModule( stdout, currMod);
        }
    }


    /*
     * Step 12
     * Final Step: Code generation
     */
    if (genCCode)
        GenCCode(allMods, longJmpVal, genTypeCode, genValueCode, 
                 genEncodeCode, genDecodeCode, genPrintCode, genFreeCode);

    else if (genCppCode)
        GenCppCode(allMods, longJmpVal, genTypeCode, genValueCode, 
                   genEncodeCode, genDecodeCode, genPrintCode, genFreeCode);


    return(0); /* keep make happy */

} /* end main */



/*
 * Calls the yacc/lex parser given a the ASN.1 src file's filename.
 * Returns a Module* for the given ASN.1 module. If the filename is
 * "-" stdin is used.
 */
Module*
ParseAsn1File PARAMS((fileName),
char* fileName)
{
    FILE* fPtr;
    Module* retVal;    
    int parseResult;

    /*
     *  Open input file for lexical analyzer/parser
     *  Use stdin if the filename is "-"
     */
    if (strcmp(fileName,"-") == 0)
        fPtr = stdin;
    else
        fPtr = fopen(fileName,"r");
    
    if (fPtr == NULL)
    {
        fprintf(stderr,"ERROR - asn1 src file \"%s\" cannot be opened for reading\n", fileName);
        return(NULL);
    }
    
    retVal = (Module*) Malloc(sizeof(Module));
    
    /*
     * Init Parser by giving it a ptr to the Module data struct
     * to initialize/use, and the file name associtated with
     * the given FILE*, fPtr (for error reporting).
     * fPtr should be an opened FILE* to an ASN.1 source FILE
     */
    InitAsn1Parser(retVal, fileName, fPtr);
    
    
    /*
     * parse the current asn1 src file into the 
     * Module data struct
     */
    parseResult = yyparse();
    
    if ((parseResult != 0) || (retVal->status == MOD_ERROR))
    {
        /* parser will print exact err msg */
        fprintf(stderr,"Parsing errors - cannot proceed\n");
        return(NULL);
    }

    if (fPtr != stdin)
        fclose(fPtr);

    return(retVal);
    
}  /* ParseAsn1File */


/*
 * Given the list of parsed, linked, normalized, error-checked and sorted
 * modules, and some code generation flags, generates C code and
 * writes it to files derived from each modules name.  Each module
 * gets 2 source files, one .h for data struct and prototypes, the other .c
 * for the enc/dec/print/free routine code.
 */
void
GenCCode PARAMS((allMods, longJmpVal, genTypes, genValues, genEncoders, genDecoders, genPrinters, genFree),
ModuleList* allMods _AND_
long int longJmpVal _AND_
int genTypes _AND_
int genValues _AND_
int genEncoders _AND_
int genDecoders _AND_
int genPrinters _AND_
int genFree)
{  
    Module* currMod;
    char* modBaseFileName;
    FILE* cHdrFilePtr;
    FILE* cSrcFilePtr;
    DefinedObj* fNames;
    int fNameConflict = 0;

    /*
     * Make names for each module's encoder/decoder src and hdr files
     * so import references can be made via include files.
     * If file names conflict, print error msg & exit.
     */
    fNames = NewObjList();
    FOR_EACH_LIST_ELMT(currMod, allMods)
    {
        /* shorten module name if necessary (SYSV etc) */
        modBaseFileName = MakeBaseFileName(currMod->modId->name);
        currMod->cHdrFileName = MakeCHdrFileName(modBaseFileName);
        currMod->cSrcFileName = MakeCSrcFileName(modBaseFileName);

        if (ObjIsDefined(fNames, currMod->cHdrFileName, StrObjCmp) ||
            ObjIsDefined(fNames, currMod->cSrcFileName, StrObjCmp))
        {
            fprintf(stderr,"Ack! ERROR - file name conflict for generated source files with names \"%s\" and \"%s\".\n\n", 
                    currMod->cHdrFileName, currMod->cSrcFileName);
            fprintf(stderr,"This usually means the max file name length is truncating the file names.\n");
            fprintf(stderr,"Try re-naming the modules with shorter names or increasing the argument to -mf option (if you are using it).\n");
            fprintf(stderr,"This error can also be caused by 2 modules with the same names but different OBJECT IDENTIFIERs.");
            fprintf(stderr,"  Try renaming the modules to correct this.\n");
            fNameConflict = 1;
        }
        else
        {
            DefineObj(&fNames, currMod->cHdrFileName);
            DefineObj(&fNames, currMod->cSrcFileName);
        }
        Free(modBaseFileName);
    }
    if (fNameConflict)
        exit(1);

    FreeDefinedObjs(&fNames);
    /*
     * make c files
     */
    FOR_EACH_LIST_ELMT(currMod, allMods)
    {
        cHdrFilePtr = fopen(currMod->cHdrFileName, "w");
        cSrcFilePtr = fopen(currMod->cSrcFileName, "w");
        if ((cSrcFilePtr == NULL) || (cHdrFilePtr == NULL))
            perror("fopen");
        else
        {
            PrintCCode(cSrcFilePtr, cHdrFilePtr, allMods, currMod, &cRulesG,
                       longJmpVal, genTypes,  genValues, genEncoders, 
                       genDecoders, genPrinters, genFree);

            fclose(cHdrFilePtr);
            fclose(cSrcFilePtr);
        }
    }

}  /* GenCCode */


/*
 * Given the list of parsed, linked, normalized, error-checked and sorted
 * modules, and some code generation flags, generates C++ code and
 * writes it to files derived from each modules name.  Each module
 * gets 2 source files, one .h for data struct and prototypes, the other .C
 * for the enc/dec/print/free routine code.
 */
void
GenCppCode PARAMS((allMods, longJmpVal, genTypes, genValues, genEncoders, genDecoders, genPrinters, genFree),
ModuleList* allMods _AND_
long int longJmpVal _AND_
int genTypes _AND_
int genValues _AND_
int genEncoders _AND_
int genDecoders _AND_
int genPrinters _AND_
int genFree)
{  
    Module* currMod;
    char* modBaseFileName;
    FILE* hdrFilePtr;
    FILE* srcFilePtr;
    DefinedObj* fNames;
    int fNameConflict = 0;

    /*
     * Make names for each module's encoder/decoder src and hdr files
     * so import references can be made via include files
     * check for truncation --> name conflicts & exit if nec
     */
    fNames = NewObjList();
    FOR_EACH_LIST_ELMT(currMod, allMods)
    {
        /* shorten module name if necessary (SYSV etc) */
        modBaseFileName = MakeBaseFileName(currMod->modId->name);
        currMod->cppHdrFileName = MakeCppHdrFileName(modBaseFileName);
        currMod->cppSrcFileName = MakeCppSrcFileName(modBaseFileName);

        if (ObjIsDefined(fNames, currMod->cppHdrFileName, StrObjCmp) ||
            ObjIsDefined(fNames, currMod->cppSrcFileName, StrObjCmp))
        {
            fprintf(stderr,"Ack! ERROR - file name conflict for generated source files with names \"%s\" and \"%s\".\n\n", 
                    currMod->cppHdrFileName, currMod->cppSrcFileName);
            fprintf(stderr,"This usually means the max file name length is truncating the file names.\n");
            fprintf(stderr,"Try re-naming the modules with shorter names or increasing the argument to -mf option (if you are using it).\n");
            fprintf(stderr,"This error can also be caused by 2 modules have the same names but different OBJECT IDENTIFIERs.");
            fprintf(stderr,"  Try renaming the modules to correct this.\n");
            fNameConflict = 1;
        }
        else
        {
            DefineObj(&fNames, currMod->cppHdrFileName);
            DefineObj(&fNames, currMod->cppSrcFileName);
        }
        Free(modBaseFileName);
    }
    if (fNameConflict)
        exit(1);

    FreeDefinedObjs(&fNames);


    /*
     * make C++ files
     */
    FOR_EACH_LIST_ELMT(currMod, allMods)
    {
        /*
         * create and fill .h file for module's data structs
         */
        hdrFilePtr = fopen(currMod->cppHdrFileName, "w");
        srcFilePtr = fopen(currMod->cppSrcFileName, "w");
        if ((hdrFilePtr == NULL) || (srcFilePtr == NULL))
            perror("fopen");
        else
        {
            PrintCppCode( srcFilePtr, hdrFilePtr, allMods, currMod,
                         &cppRulesG, longJmpVal, genTypes, genValues, 
                         genEncoders, genDecoders, genPrinters,
                         genFree);

            fclose(hdrFilePtr);
            fclose(srcFilePtr);
        }
    }
}  /* GenCppCode */


/*
 * returns 1 if the module names and oid's are unique.
 * otherwise returns 0
 */
int ModNamesUnique PARAMS((mods),
ModuleList* mods)
{
    DefinedObj* names;
    DefinedObj* oids;
    Module* m;
    int retVal = 1;

    names = NewObjList();
    oids = NewObjList();

    FOR_EACH_LIST_ELMT( m, mods)
    {
        if (((m->modId->oid != NULL) && 
            ObjIsDefined(oids, m->modId->oid, OidObjCmp)))
        {
            /* oops, 2 modules have the same oid */
            PrintErrLoc( m->asn1SrcFileName, 1);
            fprintf(stderr,"ERROR - 2 modules have the OBJECT IDENTIFIER \"");
            PrintOid(stderr, m->modId->oid);
            fprintf(stderr,"\".\n");
            retVal = 0;
        }
        /* name is only signficant if oid is empty */
        else if ((m->modId->oid == NULL) && 
             (ObjIsDefined(names, m->modId->name, StrObjCmp)))
        {
            /* oops, 2 modules have the same name */
            PrintErrLoc( m->asn1SrcFileName, 1);
            fprintf(stderr,"ERROR - 2 modules have the name \"%s\"\n", m->modId->name);
            retVal = 0;
        }
        else
        {
            DefineObj(&names, m->modId->name);
            if (m->modId->oid != NULL)
                DefineObj(&oids, m->modId->oid);
        }
    }
    FreeDefinedObjs(&names);
    FreeDefinedObjs(&oids);
    return(retVal);
} /* ModNamesUnique */

