/*
 *  asn_octs.C - ASN.1 OCTET STRING methods for AsnOcts Class
 *
 *  Mike Sample
 *  92/07/02
 * Copyright (C) 1992 Michael Sample and the University of British Columbia
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 */

#include "asn_config.h"
#include "asn_len.h"
#include "asn_tag.h"
#include "asn_type.h"
#include "asn_bits.h" /* for TO_HEX */
#include "asn_octs.h"
#include "str_stk.h"

extern StrStk strStkG;


// Initialize the AsnOcts with a char* and length.
// copies the string str.
void AsnOcts::Set(const char* str, unsigned long int len)
{ 
    octetLen = len;
    octs = new char[octetLen];
    memcpy(octs, str, octetLen);
}

// Initialize the AsnOcts from another AsnOcts
// copies the string from o.
void AsnOcts::Set(const AsnOcts& o)
{
    if (&o != this) // avoid o = o;
    {
        octetLen = o.octetLen;
        octs = new char[octetLen];
        memcpy(octs, o.octs, octetLen);
    }
}

// Initialize the AsnOcts from a Null terminated string.
// copies the string str.
void AsnOcts::Set(const char* str) 
{
    octetLen = strlen(str);
    octs = new char[octetLen];
    strcpy(octs, str);
}


// free old octs value and tehn set the new
// octs and octetLen values with the given char* and length.
// copies the string str.
void AsnOcts::ReSet(const char* str, unsigned long int len)
{ 
    if (str != octs)
    {
        delete(octs);
        Set(str, len);
    }
}

// Free old octs value and then set the new
// octs and octetLen from the given AsnOcts
// copies the string in o.
void AsnOcts::ReSet(const AsnOcts& o)
{
    if (&o != this) // avoid s = s;
    {
        delete(octs);
        Set(o);
    }
}

// Free old octs value and then set the new
// octs and octetLen values from a null terminated string.
// copies the string str.
void AsnOcts::ReSet(const char* str) 
{
    if (str != octs)
    {
        delete(octs);
        Set(str);
    }
}


// Prints the AsnOcts to the given ostream in Value Notation.
void AsnOcts::Print(ostream& os)
{
    int i;
    os << "'";
    for(i = 0; i < octetLen; i++)
        os << TO_HEX(octs[i] >> 4) << (TO_HEX(octs[i]));

    os << "'H  -- \"";

    /* put printable parts in ASN.1 comment */
    for( i = 0; i < octetLen; i++)
    {
        if (isspace(octs[i]))
            os << " ";  /* newlines->space (so don't screw up ASN.1 comment) */
        else if (isprint(octs[i]))  
            os <<octs[i];      
        else
            os << ".";
    }
    os << "\" --";
} /* AsnOcts::Print */


// Decodes a BER OCTET STRING value and puts it in this object.
// Constructed OCTET STRINGs are always concatenated into primitive ones.
void AsnOcts::BDecContent( BUF_TYPE b, AsnTag tagId, AsnLen elmtLen,
                               AsnLen& bytesDecoded, ENV_TYPE env)
{
    /*
     * tagId is encoded tag shifted into long int.
     * if CONS bit is set then constructed octet string
     */
    if (tagId & 0x20000000)
        BDecConsOcts(b, elmtLen, bytesDecoded, env);

    else /* primitive octet string */
    {
        octetLen = elmtLen;
        octs =  Asn1Alloc(elmtLen+1);
        b.Copy(octs, elmtLen);

        if (b.ReadError())
        {
            Asn1Error("BDecOctetString: ERROR - decoded past end of data\n");
            longjmp(env, -14);
        }

        /* add null terminator - this is not included in the str's len */
        octs[elmtLen] = '\0';
        bytesDecoded += elmtLen;
    }

} /* AsnOcts::BDecContent */



/*
 * Used for concatenating constructed OCTET STRING values when decoding
 *
 * fills string stack with references to the pieces of a
 * construced octet string
 */
void
FillOctetStringStk (BUF_TYPE b, AsnLen elmtLen0, 
                  AsnLen& bytesDecoded, ENV_TYPE env)
{
    unsigned long int refdLen;
    unsigned long int totalRefdLen;
    char* strPtr;
    AsnLen totalElmtsLen1 = 0;
    unsigned long int tagId1;
    AsnLen elmtLen1;

    for ( ; (totalElmtsLen1 < elmtLen0) || (elmtLen0 == INDEFINITE_LEN);)
    {
        tagId1 = BDecTag(b, totalElmtsLen1, env);

        if ( (tagId1 == EOC_TAG_ID) && (elmtLen0 == INDEFINITE_LEN))
        {
            BDEC_2ND_EOC_OCTET(b, totalElmtsLen1, env);
            break;
        }

        elmtLen1 = BDecLen (b, totalElmtsLen1, env);
        if ( tagId1 == MAKE_TAG_ID( UNIV, PRIM, OCTETSTRING_TAG_CODE))
        {
            /*
             * primitive part of string, put references to piece(s) in
             * str stack
             */
            totalRefdLen = 0;
            refdLen = elmtLen1;
            while (1)
            {
                strPtr = b.GetSeg(&refdLen);

                strStkG.Push(strPtr, refdLen);
                totalRefdLen += refdLen;
                if (totalRefdLen == elmtLen1)
                    break; /* exit this while loop */

                if (refdLen == 0) /* end of data */
                {
                    Asn1Error("BDecConsOctetString: ERROR - attempt to decode past end of data\n");
                    longjmp(env, -15);
                }
                refdLen = elmtLen1 - totalRefdLen;
            }
            totalElmtsLen1 += elmtLen1;
        } 

    
        else if ( tagId1 == MAKE_TAG_ID( UNIV, CONS, OCTETSTRING_TAG_CODE))
        {
            /*
             * constructed octets string embedding in this constructed
             * octet string. decode it.
             */
            FillOctetStringStk( b, elmtLen1, totalElmtsLen1, env);
        }
        else  /* wrong tag */
        {
            Asn1Error("BDecConsOctetString: ERROR - decoded non-OCTET STRING tag inside a constructed OCTET STRING\n");
            longjmp(env, -16);
        }
    } /* end of for */

    bytesDecoded += totalElmtsLen1;

}  /* AsnOcts::FillOctetStringStk */



/*
 * decodes a seq of universally tagged octets until either EOC is 
 * encountered or the given len decoded.  Return them in a
 * single concatenated octet string
 */
void AsnOcts::BDecConsOcts (BUF_TYPE b, AsnLen elmtLen,
                                 AsnLen& bytesDecoded, ENV_TYPE env)
{
    strStkG.Reset();
        
    /*
     * decode each piece of the octet string, puting
     * an entry in the octet/bit string stack for each 
     */
    FillOctetStringStk(b, elmtLen, bytesDecoded, env);
    
    /* alloc single str long enough for combined bitstring */
    octetLen = strStkG.totalByteLen;

    octs = Asn1Alloc(octetLen);

    strStkG.Copy(octs);

}  /* BDecConsOcts */





