/*
 *  asn_octs.h - ASN.1 OCTET STRING type
 * 
 *  Mike Sample
 *  92/07/02
 * Copyright (C) 1992 Michael Sample and the University of British Columbia
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 */

#ifndef _asn_octs_h_
#define _asn_octs_h_

#include <string.h>

class AsnOcts : public AsnType
{
  protected:
    char* octs;
    unsigned long int octetLen;

  public:
    // constructor and Set alway copy strings so destructor can always delete
    AsnOcts(): octs(NULL), octetLen(0) { }
    AsnOcts(const char* str)  { Set(str); }
    AsnOcts(const char* str, const unsigned long int len) { Set(str,len); }
    AsnOcts(const AsnOcts& o) { Set(o); }
    ~AsnOcts() { delete octs; }

    AsnOcts& operator=(const AsnOcts& o) { ReSet(o); return *this; }
    AsnOcts& operator=(const char* str) { ReSet(str); return *this; }

    // these set the octs and octetLen values 
    void Set(const char* str, unsigned long int len);
    void Set(const AsnOcts& o);
    void Set(const char* str);

    // these free the old octs value and 
    // then reset the octs and octetLen values 
    void ReSet(const char* str, unsigned long int len);
    void ReSet(const AsnOcts& o);
    void ReSet(const char* str);

    operator char*() { return octs;}
    unsigned long int Len() {return octetLen;}

    void Print(ostream& os);
    AsnType* Clone() { return new AsnOcts; }

    operator==(const AsnOcts& o)  { return(OctsEquiv(o)); }
    operator!=(const AsnOcts& o)  { return(!OctsEquiv(o)); }

    AsnLen BEnc( BUF_TYPE b)
    {
        AsnLen l;
        l =  BEncContent(b); 
        l += BEncDefLen(b, l);
        l += BEncTag1(b, UNIV, PRIM, OCTETSTRING_TAG_CODE);
        return (l);
    }

    void BDec( BUF_TYPE b, AsnLen& bytesDecoded, ENV_TYPE env)
    {
        AsnLen elmtLen;
        AsnTag tag;

        tag = BDecTag(b, bytesDecoded, env);
        if ((tag != MAKE_TAG_ID(UNIV, PRIM, OCTETSTRING_TAG_CODE)) &&
            (tag != MAKE_TAG_ID(UNIV, CONS, OCTETSTRING_TAG_CODE)))
        {
            Asn1Error("AsnOcts::BDec: ERROR tag on OCTET STRING is wrong.\n");
            longjmp(env,-56);
        }
        elmtLen = BDecLen(b, bytesDecoded, env);
        BDecContent(b, tag, elmtLen, bytesDecoded, env);
    }

    AsnLen BEncContent( BUF_TYPE b)
    {
        b.PutSegRvs(octs, octetLen);
        return(octetLen);
    }

    void BDecContent( BUF_TYPE b, AsnTag tagId, AsnLen elmtLen,
                           AsnLen& bytesDecoded, ENV_TYPE env);

    PDU_MEMBER_MACROS

    private:

    int OctsEquiv(const AsnOcts& o)
    {
        return ( (o.octetLen == octetLen) && 
                 (memcmp(o.octs, octs, octetLen)==0));
    }


    void FillBitStringStk (BUF_TYPE b, AsnLen elmtLen0, AsnLen&
bytesDecoded, ENV_TYPE env);

    void BDecConsOcts (BUF_TYPE b, AsnLen elmtLen,
                                     AsnLen& bytesDecoded,
                                     ENV_TYPE env);

};



#endif /* conditional include */
