/*
 * Test_asn_int.C 
 */

#include <stdio.h>
#include <iostream.h>
#include "asn_incl.h"


int TestAsnBuffers();
int TestAsnTag();
int TestAsnLen();
int TestAsnBool();
int TestAsnInt();
int TestAsnReal();
int TestAsnOcts();
int TestAsnBits();
int TestAsnOid();
int TestAsnList();

const int bufSize = 256;

int main()
{
    int isErr = FALSE;

    if (!TestAsnBuffers())
    {
        cout << "Failed buffer tests, no point in proceeding ... bye!" << endl;
        return (1);
    }

    if (!TestAsnTag())
    {
        cout << "Failed Tag test." << endl;
        isErr = TRUE;
    }

    if (!TestAsnLen())
    {
        cout << "Failed Length test." << endl;
        isErr = TRUE;
    }

    if (!TestAsnBool())
    {
        cout << "Failed BOOLEAN test." << endl;
        isErr = TRUE;
    }


    if (!TestAsnInt())
    {
        cout << "Failed INTEGER test." << endl;
        isErr = TRUE;
    }

    if (!TestAsnOcts())
    {
        cout << "Failed OCTET STRING test." << endl;
        isErr = TRUE;
    }


    if (!TestAsnBits())
    {
        cout << "Failed BIT STRING test." << endl;
        isErr = TRUE;
    }


    if (!TestAsnOid())
    {
        cout << "Failed OBJECT IDENTIFIER test." << endl;
        isErr = TRUE;
    }


    if (!TestAsnReal())
    {
        cout << "Failed REAL test." << endl;
        isErr = TRUE;
    }



    if (isErr)
    {
        cout << "There are errors in the primitive type encoding/decoding" << endl;
        cout << "library for this architecture.  Time for gdb..." << endl;
    }
    else
    {
        cout << "The primitive type encoding/decoding library passed" << endl;
        cout << "simple tests. It should be safe to use..." << endl;
    }
    return(isErr);
}


/*
 * returns TRUE if passes encode/decode tests
 */
int
TestAsnBuffers()
{
    AsnBuf  b;
    char bufData[256];
    int i,j;
    int noErr = TRUE;

    // initialize buffer
    b.Init(bufData, 256);
    b.ResetInWriteRvsMode();

    // write whole range of byte (0..255)
    // remember, write works in reverse
    for(i = 0; i < 256; i++)
        b.PutByteRvs(i);

    if (b.WriteError())
    {
        cout << "Error writing to buffer." << endl;
        noErr = FALSE;
    }
    
    // read in values & verify
    b.ResetInReadMode();
    for(i = 255; i >= 0; i--)
        if (b.GetByte() != i)
        {
            cout << "Error verifying data written to buffer." << endl;
            noErr = FALSE;
        }

    if (b.ReadError())
    {
        cout << "Error reading from buffer." << endl;
        noErr = FALSE;
    }


    /* now make sure errors are detected */
    b.ResetInWriteRvsMode();

    for(i = 0; i < 257; i++) // write past end of buffer
        b.PutByteRvs(0);

    if (!b.WriteError())
    {
        cout << "Buffers failed to report buffer write overflow." << endl;
        noErr = FALSE;
    }
     
    
    b.ResetInReadMode();
    for(i = 256; i >= 0; i--)  // read past end of buffer 
        b.GetByte();

    if (!b.ReadError())
    {
        cout << "Buffers failed to report buffer read overflow." << endl;
        noErr = FALSE;
    }

    return(noErr);
}  /* TestAsnBuffers */



/*
 * returns TRUE if passes encode/decode tests
 */
int
TestAsnTag()
{
    AsnTag aTag1;
    AsnTag aTag2;
    int i, j;
    AsnLen len1;
    AsnLen len2;
    AsnTag tag;
    int noErr = TRUE;
    ENV_TYPE env;
    AsnBuf  b;
    char bufData[256];
    long int val;
    BER_CLASS tagClass;
    BER_FORM form;
    BER_UNIV_CODE code;


    /* initialize buffer */
    b.Init( bufData, 256);

    /* encode a true value and verify */
    tagClass = UNIV;
    form = PRIM;
    code = INTEGER_TAG_CODE;
    aTag1 = MAKE_TAG_ID(tagClass, form, code);
    
    for (i = 0; i < 2; i++)
    {
        b.ResetInWriteRvsMode();
        len1 = BEncTag1(b, tagClass, form, code);
        
        if (b.WriteError())
        {
            noErr = FALSE;
            cout << "Error encoding a Tag." << endl;
        }
        
        b.ResetInReadMode();
        
        aTag2 = 0;
        
        /* make sure no decode errors and that it decodes to same tag */
        len2 = 0;
        if ((val = setjmp(env)) == 0)
        {
            aTag2 = BDecTag(b, len2, env);
        }
        else
        {
            noErr = FALSE;
            cout << "Error decoding a Tag - error number " << val << endl;
        }
        if (noErr && ((aTag2 != aTag1) || (len1 != len2)))
        {
            noErr = FALSE;
            cout << "Error decoded Tag does not match encoded Tag." << endl;
        }
        /* set a new test tag value */
        tagClass = CNTX;
        form = CONS;
        code = 29;
        aTag1 = MAKE_TAG_ID(tagClass, form, code);
    }
    return(noErr);
}  /* TestAsnTag */


/*
 * returns TRUE if passes encode/decode tests
 */
int
TestAsnLen()
{
    AsnLen aLen1;
    AsnLen aLen2;
    int i,j;
    AsnLen len1;
    AsnLen len2;
    AsnTag tag;
    int noErr = TRUE;
    ENV_TYPE env;
    AsnBuf  b;
    char bufData[256];
    long int val;    

    /* initialize buffer */
    b.Init(bufData, 256);


    /* encode a true value and verify */
    aLen1 = 99999;
    for (i = 0 ; i < 2; i++)
    {
        b.ResetInWriteRvsMode();
        len1 = BEncDefLen(b, aLen1);
        
        if (b.WriteError())
        {
            noErr = FALSE;
            cout <<  "Error encoding Length." << endl;
        }
        
        b.ResetInReadMode();
        
        aLen2 = 0;
        
        /* make sure no decode errors and that it decodes to true */
        len2 = 0;
        if ((val = setjmp(env)) == 0)
        {
            aLen2 = BDecLen(b, len2, env);
        }
        else
        {
            noErr = FALSE;
            cout << "Error decoding Length - error number " << val << endl;
        }
        
        
        if (noErr && ((aLen2 != aLen1) || (len1 != len2)))
        {
            noErr = FALSE;
            cout << "Error - decoded length does not match encoded length" << endl;
        }
        aLen1 = 2;
    }
    

    /* test indef len */
    b.ResetInWriteRvsMode();
    len1 = BEncIndefLen(b);
    
    if (b.WriteError())
    {
        noErr = FALSE;
        cout << "Error encoding indefinite Length." << endl;
    }
    
    b.ResetInReadMode();
    
    aLen2 = 0;
    
    /* make sure no decode errors */
    len2 = 0;
    if ((val = setjmp(env)) == 0)
    {
        aLen2 = BDecLen(b, len2, env);
    }
    else
    {
        noErr = FALSE;
        cout << "Error decoding Length - error number " << val << endl;
    }
    
    
    if (noErr && ((aLen2 != INDEFINITE_LEN) || (len1 != len2)))
    {
        noErr = FALSE;
        cout << "Error - decoded length does not match encoded length" << endl;
    }

    /* test EOC */
    b.ResetInWriteRvsMode();
    len1 = BEncEoc(b);
    
    if (b.WriteError())
    {
        noErr = FALSE;
        cout << "Error encoding indefinite Length." << endl;
    }
    
    b.ResetInReadMode();
    
    aLen2 = 0;
    
    /* make sure no decode errors */
    len2 = 0;
    if ((val = setjmp(env)) == 0)
    {
        BDecEoc(b, len2, env);
    }
    else
    {
        noErr = FALSE;
        cout << "Error decoding Length - error number " <<  val << endl;
    }
    
    
    if (noErr && (len1 != len2))
    {
        noErr = FALSE;
        cout << "Error - decoded EOC length error" << endl;
    }

    return(noErr);
}  /* TestAsnLen */



/*
 * returns TRUE if passes encode/decode tests
 */
int
TestAsnBool()
{
    AsnBuf  b;
    char bufData[bufSize];
    AsnBool aBool1;
    AsnBool aBool2;
    int j;
    AsnLen len1;
    AsnLen len2;
    int noErr = TRUE;

    // initialize a small buffer
    b.Init(bufData, bufSize);
    b.ResetInWriteRvsMode();

    // encode a true value and verify
    aBool1 = TRUE;
     
    if (!aBool1.BEncPdu(b, len1))
    {
        noErr = FALSE;
        cout << "Error encoding TRUE BOOLEAN value." << endl;
    }

    b.ResetInReadMode();

    aBool2 = FALSE; // set to opposite of expected value
    
    // make sure no decode errors and that it decodes to true
    if (!aBool2.BDecPdu(b, len2) || !aBool2 || (len1 != len2))
    {
        noErr = FALSE;
        cout << "Error decoding TRUE BOOLEAN value." << endl;
    }

    // now encode a false value and verify
    b.ResetInWriteRvsMode();
    aBool1 = FALSE;
     
    if (!aBool1.BEncPdu(b, len1))
    {
        noErr = FALSE;
        cout << "Error encoding FALSE BOOLEAN value." << endl;
    }

    b.ResetInReadMode();

    aBool2 = TRUE; // set to opposite of expected value
    
    // make sure no decode errors and that it decodes to false
    if (!aBool2.BDecPdu(b, len2) || aBool2 || (len1 != len2))
    {
        noErr = FALSE;
        cout << "Error decoding FALSE BOOLEAN value." << endl;
    }

    return(noErr);
}  /* TestAsnBool */


/*
 * returns TRUE if passes encode/decode tests
 */
int
TestAsnInt()
{
    AsnBuf  b;
    char bufData[bufSize];
    AsnInt a1;
    AsnInt a2;
    int i,j, sign;
    AsnLen len1;
    AsnLen len2;
    int noErr = TRUE;

    // initialize a small buffer
    b.Init(bufData, bufSize);

    //
    // Encode a range of integers : negative & positive in
    //  the 1 to sizeof(long int) range 
    //

    sign = 1;
    for ( j = 0; j < 2; j++)
    {
        for (i = 0 ; i < sizeof(long int); i++)
        {
            b.ResetInWriteRvsMode();
    
            a1 = sign * (17 << (i * 8)); // 17 is a random choice
            if (!a1.BEncPdu(b, len1))
            {
                noErr = FALSE;
                cout << "Error encoding INTEGER value " << a1 << "." << endl;
            }

            b.ResetInReadMode();
            a2 = 0;
    
            // make sure no decode errors and that it decodes to the correc val
            if (!a2.BDecPdu(b, len2) || (a2 != a1) || (len1 != len2))
            {
                noErr = FALSE;
                cout << "Error decoding INTEGER value " << a1 << "." << endl;
            }
        }
        sign = -1;
    }

    return(noErr);

} /* TestAsnInt */


/*
 * returns TRUE if passes encode/decode tests
 */
int
TestAsnOcts()
{
    AsnBuf  b;
    char bufData[bufSize];
    AsnOcts a1;
    AsnOcts a2;
    int i,j;
    AsnLen len1;
    AsnLen len2;
    int noErr = TRUE;

    // initialize a small buffer
    b.Init(bufData, bufSize);

    a1 = "Hello Gumby?";
    for ( j = 0; j < 2; j++)
    {
        b.ResetInWriteRvsMode();
    
        if (!a1.BEncPdu(b, len1))
        {
            noErr = FALSE;
            cout << "Error encoding OCTET STRING value " << a1 << "." << endl;
        }

        b.ResetInReadMode();

        // make sure no decode errors and that it decodes to the correc val
        if (!a2.BDecPdu(b, len2) || (a2 != a1) || (len1 != len2))
        {
            noErr = FALSE;
            cout << "Error decoding OCTET STRING value " << a1 << "." << endl;
        }
        a1 = ""; // try an empty string
    }

    return(noErr);

} /* TestAsnOcts */



/*
 * returns TRUE if passes encode/decode tests
 */
int
TestAsnBits()
{
    AsnBuf  b;
    char bufData[bufSize];
    AsnBits a1(32);
    AsnBits a2(32);
    short bitsToSet[32] = { 0, 1, 0, 0, 1 , 1 , 0, 1,
                            0, 1, 0, 0, 1 , 1 , 0, 1,
                            0, 1, 0, 0, 1 , 1 , 0, 1,
                            0, 1, 0, 0, 1 , 1 , 0, 1};
    int i,j;
    AsnLen len1;
    AsnLen len2;
    int noErr = TRUE;

    // initialize a small buffer
    b.Init(bufData, bufSize);


    // set some bits
    for (i = 0; i < 32; i++)
    {
        if ( bitsToSet[i])
            a1.SetBit(i);
        else
            a1.ClrBit(i);
             
    }

    b.ResetInWriteRvsMode();
    if (!a1.BEncPdu(b, len1))
    {
        noErr = FALSE;
        cout << "Error encoding BIT STRING value " << a1 << "." << endl;
    }

    b.ResetInReadMode();

    // make sure no decode errors and that it decodes to the correc val
    if (!a2.BDecPdu(b, len2) || (a2 != a1) || (len1 != len2))
    {
        noErr = FALSE;
        cout << "Error decoding BIT STRING value " << a1 << "." << endl;
    }


    return(noErr);

} /* TestAsnBits */



/*
 * returns TRUE if passes encode/decode tests
 */
int
TestAsnOid()
{
    AsnBuf  b;
    char bufData[bufSize];
    AsnOid a1(0,1,2,3,4,5,6);
    AsnOid a2;
    AsnOid a3(2,38,29,40,200,10,4000);
    int i,j;
    AsnLen len1;
    AsnLen len2;
    int noErr = TRUE;

    // initialize a small buffer
    b.Init(bufData, bufSize);

    for (i = 0; i < 2; i++)
    {
        b.ResetInWriteRvsMode();

        if (!a1.BEncPdu(b, len1))
        {
            noErr = FALSE;
            cout << "Error encoding OBJECT IDENTIFIER value " << a1 << "." << endl;
        }
        
        b.ResetInReadMode();
        
        // make sure no decode errors and that it decodes to the correc val
        if (!a2.BDecPdu(b, len2) || (a2 != a1) || (len1 != len2))
        {
            noErr = FALSE;
            cout << "Error decoding OBJECT IDENTIFIER value " << a1 << "." << endl;
        }

        a1 = a3;
    }
    return(noErr);

} /* TestAsnOid */

/*
 * returns TRUE if passes encode/decode tests
 * 
 * NOT USED - nuked template design.
 */
/*
int
TestAsnList()
{
    AsnBuf  b;
    char bufData[bufSize];
    AsnList<AsnInt> intList1;
    AsnList<AsnInt> intList2;
    AsnList<AsnBool> boolList1;
    AsnList<AsnBool> boolList2;
    int i,j;
    AsnLen len1;
    AsnLen len2;
    int noErr = TRUE;

    b.Init(bufData, bufSize);

    b.ResetInWriteRvsMode();

    if (!intList1.BEncPdu(b, len1))
    {
        noErr = FALSE;
        cout << "Error encoding SEQUENCE OF value " << intList1 << "." << endl;
    }
        
    b.ResetInReadMode();
        
    if (!intList2.BDecPdu(b, len2) || (len1 != len2))
    {
        noErr = FALSE;
        cout << "Error decoding SEQUENCE OF value " << intList1 << "." << endl;
    }
    cout << "intlist 1 = "  <<  intList1 << endl;
    cout << "intlist 2 = "  <<  intList1 << endl;


    if (!boolList1.BEncPdu(b, len1))
    {
        noErr = FALSE;
        cout << "Error encoding SEQUENCE OF value " << boolList1 << "." << endl;
    }
        
    b.ResetInReadMode();
        
    if (!boolList2.BDecPdu(b, len2) ||  (len1 != len2))
    {
        noErr = FALSE;
        cout << "Error decoding SEQUENCE OF value " << boolList1 << "." << endl;
    }
    cout << "boolList 1 = "  <<  boolList1 << endl;
    cout << "boolList 2 = "  <<  boolList1 << endl;

    return(noErr);

}  TestAsnList */



/*
 * returns TRUE if passes encode/decode tests
 */
int
TestAsnReal()
{
    AsnBuf  b;
    char bufData[bufSize];
    AsnReal  a2;
    AsnReal  a[] = { 0.0, 0.8, -22.484848, PLUS_INFINITY, MINUS_INFINITY};
    int i,j;
    AsnLen len1;
    AsnLen len2;
    int noErr = TRUE;


    /*
     * if you do not have the ieee_functions in your math lib,
     * this will not link.  Comment it out and cross you fingers.
     * (or check/set the +/-infinity values for you architecture)
     */
    if (!isinf(PLUS_INFINITY) || !isinf(MINUS_INFINITY))
    {
        cout << "WARNING: PLUS_INFINITY and MINUS_INFINITY in asn_real.C are not" << endl;
        cout << "correct for this architecture.  Modify the AsnPlussInf() Routine." << endl;
    }


    // initialize a small buffer
    b.Init(bufData, bufSize);

    for (i = 0; i < 5; i++)
    {
        b.ResetInWriteRvsMode();

        if (!a[i].BEncPdu(b, len1))
        {
            noErr = FALSE;
            cout << "Error encoding REAL value " << a[i] << "." << endl;
        }
        
        b.ResetInReadMode();
        
        // make sure no decode errors and that it decodes to the correc val
        if (!a2.BDecPdu(b, len2) || (a2 != a[i]) || (len1 != len2))
        {
            noErr = FALSE;
            cout << "Error decoding REAL value " << a[i] << "." << endl;
        }
    }

    return (noErr);

} /* TestAsnReal */

