/* {{{ Log info, include files and prototypes */

/*
 * Vpane.c,v 2.1 1992/06/23 00:29:05 pete Exp
 * Vpane.c,v
 * Revision 2.1  1992/06/23  00:29:05  pete
 * Changed interface to _XoMenuNew and _XoMenuDone.
 *
 * Revision 2.0  1992/04/23  02:52:06  ware
 * First public release.
 *
 * Revision 1.11  1992/04/23  02:18:52  ware
 * Added several classes.  Worked on geometry management
 *
 * Revision 1.10  1992/02/27  14:30:29  ware
 * Compiled with GCC 2.0 and very strict checks.  Fixed Warnings
 *
 * Revision 1.9  1992/02/24  01:19:31  ware
 * Changed scale to user_height.  This allows the user to interactively
 * set the height of a widget.  When this is non-zero, this is used as
 * the preferred height of the widget instead of querying for the value.
 *
 * Revision 1.8  1992/02/22  21:31:45  ware
 * Got grips to move back to the original position and to avoid flutter.
 * The earlier problems were caused by not calculating a new edge location
 * when a prev'ious or next grip is moved -- which might have changed the
 * position of this widget.
 *
 * Also worked on getting the grip lines to draw correctly.  Need to add
 * width to the lines.  Need to make the grips actually resize children.
 *
 * Revision 1.7  92/02/21  10:47:16  ware
 * Got the grips so they move the other grips on collision.  Also does
 * not allow any grips to move off the screen.  Now I need to work
 * on getting them to move back to the original position after a
 * collision.
 * .l
 * 
 * Revision 1.6  1992/02/20  15:11:09  ware
 * Applied new indentation
 *
 * Revision 1.5  1992/02/04  21:22:46  pete
 * Release 44
 *
 * Revision 1.4  1991/12/01  16:29:04  pete
 * Removed an empty ClassPartInit().  Added some comments to delimit
 * the various class methods.
 *
 * Revision 1.3  1991/09/12  09:50:30  pete
 * Made it a subclass of Column.  Wrote a bunch of geometry management
 * functions but ended up moving them to column.
 *
 * Revision 1.2  91/08/26  11:58:08  pete
 * Use XoProto() for conditional prototypes.  Working on getting traversals
 * and menus to work more efficiently.  Changed to following naming
 * conventions.
 *
 * Revision 1.1  91/07/19  00:58:43  pete
 * Initial revision
 *
 */

#include <stdio.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xo/VpaneP.h>
#include <X11/Xo/Grip.h>
#include <X11/Xo/dbug.h>

#include <X11/Xo/VpaneRec.h>

XoProto (static void, MoveGrip, (Widget gw, XtPointer client_data, XtPointer call_data));
XoProto (static void, NotifyGrip, (Widget gw, XtPointer client_data, XtPointer call_data));
XoProto (static void, GripAssociate, (Widget gw, Widget before, Widget after, int indx));
XoProto (static void, GripAdd, (Widget gw));
XoProto (static void, GripPlace, (Widget gw, Widget grip, Widget match));
XoProto (static void, GripDrawLine, (Widget gw, Widget grip, GC gc));
XoProto (static void, GripMove, (Widget parent, Widget grip, int delta));
XoProto (static void, GripUpdate, (Widget gw));
XoProto (static void, GripCalcScale, (Widget gw, Widget child, int delta));
XoProto (static void, CalcGC, (Widget gw));

/* }}} */
/* {{{ Core Class Methods */

/*
 *----------------------------------------------------------------------
 * Core Class Methods
 *----------------------------------------------------------------------
 */

/* {{{ Initialize */

static void
Initialize (request, new, arglist, num_args)
	Widget          request;	/* as first created */
	Widget          new;		/* after other parent classes */
	ArgList         arglist;	/* list of arguments */
	Cardinal       *num_args;	/* how many */
{
	XoVpaneWidget   w = (XoVpaneWidget) new;

	w->vpane.total_stretch = 0;
	w->vpane.total_shrink = 0;
	w->vpane.start_y = 0;
	w->vpane.moving_grip = NULL;
	w->vpane.grips = NULL;
	w->vpane.grip_count = 0;
	w->vpane.layout_grips = True;
	w->vpane.gc = NULL;
	w->vpane.move_gc = NULL;
	CalcGC (new);
}

/* }}} */
/* {{{ Realize */

static void
Realize (gw, valueMask, attrs)
	Widget          gw;
	XtValueMask    *valueMask;
	XSetWindowAttributes *attrs;
{
	XoVpaneWidget   w = (XoVpaneWidget) gw;
	int             i;

	/*
	 * Use the parent class's realize
	 */
	(*MySuperClass.core_class.realize) (gw, valueMask, attrs);

	/*
	 * Realize all the grips last so they are above the other children in
	 * the window stack.
	 */
	for (i = 0; i < w->composite.num_children; i++)
	{
		if (XtIsManaged (w->composite.children[i]) &&
		    !SKIPCHILD (w->composite.children[i]))
			XtRealizeWidget (w->composite.children[i]);
	}
	for (i = 0; i < w->vpane.grip_count; i++)
	{
		XtRealizeWidget (w->vpane.grips[i]);
	}
}

/* }}} */
/* {{{ Resize */

static void
Resize (gw)
	Widget          gw;
{
	XoVpaneWidget   w = (XoVpaneWidget) gw;
	XoVpaneConstraint c;		/* constraints child */
	Widget          match;
	int             i;

	/*
	 * Use the parent class's resize
	 */
	(*MySuperClass.core_class.resize) (gw);
	for (i = 0; i < w->vpane.grip_count; i++)
	{
		c = VPCONS (w->vpane.grips[i]);
		match = c->vpane.before;
		GripPlace ((Widget) w, w->vpane.grips[i], match);
		/*GripDrawLine ((Widget) w, w->vpane.grips[i], w->vpane.gc);*/
	}
}

/* }}} */
/* {{{ Redisplay */

static void
Redisplay (gw, event, region)
	Widget          gw;
	XEvent         *event;
	Region          region;
{
	XoVpaneWidget   w = (XoVpaneWidget) gw;
	int             i;

	if (region)
		XSetRegion (XtDisplay ((Widget) w), w->vpane.gc, region);
	for (i = 0; i < w->vpane.grip_count; i++)
	{
		GripDrawLine ((Widget) w, w->vpane.grips[i], w->vpane.gc);
	}
	XSetClipMask (XtDisplay ((Widget) w), w->vpane.gc, (Pixmap) None);
}

/* }}} */
/* {{{ SetValues */

static Boolean
SetValues (current, request, new, args, num_args)
	Widget          current;	/* widget before the XtSetValues() */
	Widget          request;	/* after args applied, no set_values */
	Widget          new;		/* the allowed changes */
	ArgList         args;		/* list of arguments */
	Cardinal       *num_args;	/* how many arguments */
{
	XoVpaneWidget   w = (XoVpaneWidget) new;
	XoVpaneWidget   cur = (XoVpaneWidget) current;
	Boolean         redisplay = False;

	if (w->cons_base.fg != cur->cons_base.fg)
	{
		CalcGC (new);
		redisplay = True;
	}
	return redisplay;
}

/* }}} */
/* {{{ Destroy */

static void
Destroy (gw)
	Widget          gw;
{
	XoVpaneWidget   w = (XoVpaneWidget) gw;

	if (w->vpane.gc)
		XtReleaseGC (gw, w->vpane.gc);
	if (w->vpane.move_gc)
		XtReleaseGC (gw, w->vpane.move_gc);
	XtFree ((char *) w->vpane.grips);
	w->vpane.grips = NULL;
}

/* }}} */

/* }}} */
/* {{{ Composite Class Methods */

/*
 *----------------------------------------------------------------------
 * Composite Class Methods
 *----------------------------------------------------------------------
 */

/*
 * InsertChild - Called when a new child is added.  Delete this method
 */
static void
InsertChild (child)
	Widget          child;
{
	XoVpaneWidget   w;
	XoVpaneConstraint c;		/* constraints for this child */

	w = (XoVpaneWidget) XtParent (child);
	c = VPCONS (child);
	(*MySuperClass.composite_class.insert_child) (child);
}

/*
 * ChangeManaged - Called when one or more children are managed, unmanaged,
 *		or destroyed.  Use's Column's change_managed to take
 *		care of the "normal" children and then places each of the
 *		grips appropriately.
 */
static void
ChangeManaged (gw)
	Widget          gw;
{
	XoVpaneWidget   w = (XoVpaneWidget) gw;
	int             i;
	int             next_grip;	/* next grip to allocate */
	Widget          child;
	Widget		last;		/* the previous widget */

	if (!w->vpane.layout_grips || !w->cons_base.do_layout)
		return;
	w->vpane.layout_grips = False;
	(*MySuperClass.composite_class.change_managed) (gw);

	/*
	 * Avoid doing any of the layout work again as grips are
	 * managed/unmanaged/destroyed.
	 */
	w->cons_base.do_layout = False;
	next_grip = 0;
	last = NULL;
	for (i = 0; i < w->composite.num_children; i++)
	{
		child = w->composite.children[i];
		if (XtIsManaged (child) && !SKIPCHILD (child))
		{
			if (last)
				GripAssociate (gw, last, child, next_grip++);
			last = child;
		}
	}
	/*
	 * Now get rid of any unused Grip widgets.
	 */
	for (i = next_grip; i < w->vpane.grip_count; i++)
	{
		/*
		 * We do not end up back here because we
		 * set layout_grips to False when first entering
		 */
		XtDestroyWidget (w->vpane.grips[next_grip]);
		w->vpane.grips[next_grip] = NULL;
	}
	w->vpane.grip_count = next_grip;
	w->cons_base.do_layout = True;
	w->vpane.layout_grips = True;
	_XoRedisplaySelf (gw);
}

/* }}} */
/* {{{ Constraint Class Methods */

/*
 *----------------------------------------------------------------------
 * Constraint Class Methods
 *----------------------------------------------------------------------
 */

static void
ConsInitialize (request, new, arglist, num_args)
	Widget          request;	/* as first created */
	Widget          new;		/* after other parent classes */
	ArgList         arglist;	/* list of arguments */
	Cardinal       *num_args;	/* how many */
{
	XoVpaneConstraint c;		/* constraints for this child */

	c = VPCONS (new);
	c->vpane.before = NULL;
	c->vpane.after = NULL;
}

/* }}} */
/* {{{ ConsBase Class Methods */

/*
 *----------------------------------------------------------------------
 * ConsBase Class Methods
 *----------------------------------------------------------------------
 */

/* }}} */
/* {{{ Column Class Methods */

/*
 *----------------------------------------------------------------------
 * Column Class Methods
 *----------------------------------------------------------------------
 */

/* }}} */
/* {{{ Vpane Class Methods */

/*
 *----------------------------------------------------------------------
 * Vpane Class Methods
 *----------------------------------------------------------------------
 */

/* }}} */
/* {{{ Private functions */

/*
 *----------------------------------------------------------------------
 * Private functions
 *----------------------------------------------------------------------
 */

/*
 * MoveGrip	- Callback added to each grip widget that is responsible
 *		  for physically moving the grip.  Also insures
 *		  the grip is on the top of the window stack
 */

static void
MoveGrip (gw, client_data, call_data)
	Widget          gw;
	XtPointer       client_data;
	XtPointer       call_data;
{
	XoVpaneWidget   w = (XoVpaneWidget) XtParent (gw);
	XoMoveCBdata   *d;
	XWindowChanges  values;

	DBUG_ENTER ("MoveGrip");
	d = (XoMoveCBdata *) call_data;
	/*
	 * If this is the beginning motion of the grip, then make sure it is
	 * above all the other widgets.
	 */
	if (!w->vpane.moving_grip)
	{
		w->vpane.moving_grip = gw;
		if (d)
			w->vpane.start_y = d->mcbd_y;
		if (XtIsRealized (gw))
		{
			values.stack_mode = Above;
			XConfigureWindow (XtDisplay (gw), XtWindow (gw),
					  CWStackMode, &values);
		}
	}
	if (d)
	{
		DBUG_PRINT ("grip", ("==========="));
		GripMove ((Widget) w, gw,
			  (int) (d->mcbd_y - (gw->core.y+gw->core.height/2)));
		DBUG_PRINT ("grip", ("==========="));
	}
	DBUG_VOID_RETURN;
}

/*
 * NotifyGrip - Callback added to each grip that is executed when the
 *		grip is finished moving.  Responsible for resizing children
 *		to new height, setting moving_grip to NULL.
 */

static void
NotifyGrip (gw, client_data, call_data)
	Widget          gw;
	XtPointer       client_data;
	XtPointer       call_data;
{
	XoVpaneWidget   w = (XoVpaneWidget) XtParent (gw);
	XoMoveCBdata   *d;

	d = (XoMoveCBdata *) call_data;
	if (d)
	{
		GripMove ((Widget) w, gw,
			  (int) (d->mcbd_y - (gw->core.y+gw->core.height/2)));
	}
	w->vpane.moving_grip = NULL;

	GripUpdate ((Widget) w);
	_XoRedisplaySelf ((Widget) w);
	(*ThisClass((Widget) w).core_class.resize) ((Widget) w);
}

static void
GripAssociate (gw, before, after, indx)
	Widget          gw;		/* the vpane */
	Widget          before;		/* child before the grip */
	Widget		after;		/* child after the grip */
	int             indx;		/* which grip to associate */
{
	XoVpaneWidget   w = (XoVpaneWidget) gw;
	XoVpaneConstraint c;		/* constraints for this child */
	Widget          grip;

	if (indx < 0 || indx > w->vpane.grip_count)
	{
		_XoWarn (gw, "badArgument", "Vpane.GripAssociate",
		"Internal error.  Trying to allocate too many grip widgets");
		return;
	}
	/*
	 * See if we need to create a new widget or can just reuse one.
	 */
	if (indx == w->vpane.grip_count || !w->vpane.grips[indx])
	{
		GripAdd (gw);
	}
	grip = w->vpane.grips[indx];
	c = VPCONS (grip);
	c->vpane.before = before;
	c->vpane.after = after;
	GripPlace ((Widget) w, grip, c->vpane.before);
}

static void
GripPlace (gw, grip, match)
	Widget          gw;
	Widget          grip;
	Widget          match;
{
	XoVpaneWidget   w = (XoVpaneWidget) gw;
	XoVpaneConstraint	cons;	/* constraint information */

	cons = VPCONS (grip);
	cons->vpane.orig_y = match->core.y + match->core.height + w->column.vspace / 2
		- grip->core.height / 2;
	XtMoveWidget (grip, w->core.width
		      - grip->core.width - 2 * grip->core.border_width,
		      cons->vpane.orig_y);
}


static void
GripAdd (gw)
	Widget          gw;
{
	XoVpaneWidget   w = (XoVpaneWidget) gw;
	Widget          grip;
	XWindowChanges  values;
	Arg             args[10];
	Cardinal        cnt;

	++w->vpane.grip_count;
	if (!w->vpane.grips)
	{
		w->vpane.grips = (Widget *)
			XtMalloc (sizeof (Widget) * (unsigned) w->vpane.grip_count);
	}
	else
	{
		w->vpane.grips = (Widget *)
			XtRealloc ((char *) w->vpane.grips,
				   sizeof (Widget) * (unsigned) w->vpane.grip_count);
	}
	cnt = 0;
	XtSetArg (args[cnt], XtNchildType, XoGRIP);
	++cnt;
	/*
	 * Do not add a destroy callback to remove this from the list of
	 * grips -- it is cleaned up in ChangeManaged.
	 */
	grip = XtCreateManagedWidget ("grip", w->vpane.grip_class,
				      (Widget) w, args, cnt);
	SKIPCHILD (grip) = True;
	XtAddCallback (grip, XtNmoveCallback, MoveGrip, (XtPointer) NULL);
	XtAddCallback (grip, XtNcallback, NotifyGrip, (XtPointer) NULL);
	w->vpane.grips[w->vpane.grip_count - 1] = grip;
	if (XtIsRealized (grip))
	{
		values.stack_mode = Above;
		XConfigureWindow (XtDisplay (grip), XtWindow (grip),
				  CWStackMode, &values);
	}
}

static void
GripDrawLine (gw, grip, gc)
	Widget          gw;
	Widget          grip;
	GC		gc;		/* how to draw th eline */
{
	int             y;

	if (!XtIsRealized (gw))
		return;
	y = grip->core.y + grip->core.height / 2;
	XDrawLine (XtDisplay (gw), XtWindow (gw), gc,
		   0, y, grip->core.x, y);
}

/*
 * GripMove - Move a grip up or down by delta pixels.  Maintains
 *	      each grip from overlapping with it's nearest neighbor.  If
 *	      a neighbor is moved, the GripMove() is called recursively
 *	      to move any nearer neighbors.  It also keeps any grip from
 *	      being pushed off the window.
 */
static void
GripMove (parent, grip, delta)
	Widget		parent;		/* the vpane */
	Widget		grip;		/* grip being moved */
	int		delta;		/* amount to move it */
{
	XoVpaneWidget	w = (XoVpaneWidget) parent;
	Widget		prev;		/* the grip previous to grip */
	Widget		next;		/* the grip after grip */
	int		i;		/* which grip this is */
	int		grip_top;	/* top of the grip */
	int		grip_bottom;	/* bottom of the grip */
	int		edge;		/* edge of next or previous grip */
	int		move;		/* amount other grip moves */
	XoVpaneConstraint	cons;	/* constraint information */

	DBUG_ENTER ("Vpane.GripMove");
	if (delta == 0)
		DBUG_VOID_RETURN;

	/*
	 * Find this grip so we can determine the next and previous grips
	 */
	for (i = 0; i < w->vpane.grip_count; i++)
	{
		if (grip == w->vpane.grips[i])
			break;
	}
	if (i >= w->vpane.grip_count)
	{
		_XoWarn (parent, "badArgument", "Vpane.GripMove",
			 "Internal error.  Grip not really a grip");
		DBUG_VOID_RETURN;
	}

	/*
	 * Set the prev'ious and next widgets
	 */
	if (i > 0)
		prev = w->vpane.grips[i-1];
	else
		prev = NULL;
	if (i + 1 < w->vpane.grip_count)
		next = w->vpane.grips[i+1];
	else
		next = NULL;
	grip_top = _XoEdgeTop (grip);
	grip_bottom = _XoEdgeBottom (grip);
	/*
	 * Make sure we don't move grip above the parent window or below it
	 */
	if (grip_top + delta < 0)
		delta = -grip->core.y + (int) grip->core.border_width;
	if (grip_bottom + delta > (int) w->core.height)
		delta = w->core.height - grip_bottom;
	if (delta < 0)			/* moving up */
	{
		if (prev)
		{
			/*
			 * We need to move the previous grip up if its
			 * bottom edge overlaps with where we will
			 * move this grip.  The prev'ious may be
			 * unable to move all the way in which case we
			 * adjust delta to insure we don't go past it.
			 */
			edge = _XoEdgeBottom (prev);
			if (edge >= grip_top + delta)
			{
				DBUG_PRINT ("grip", ("Moving previous up by %d",grip_top + delta - edge));
				GripMove (parent, prev, grip_top + delta - edge);
				grip_top = _XoEdgeTop (grip);
				delta = _XoEdgeBottom (prev) - grip_top + 1;
			}
		}
	}
	else				/* moving down */
	{
		if (next)
		{
			/*
			 * Similar to above except we are looking at the next
			 * grip and check it's top edge
			 */
			edge = _XoEdgeTop (next);
			if (edge <= grip_bottom + delta)
			{
				DBUG_PRINT ("grip", ("Moving next down by %d",grip_bottom + delta - edge));
				GripMove (parent, next,
					  grip_bottom + delta - edge);
				grip_bottom = _XoEdgeBottom (grip);
				delta = _XoEdgeTop (next) - grip_bottom - 1;
			}
		}
	}
	if (delta != 0)
	{
		GripDrawLine ((Widget) w, grip, w->vpane.move_gc);
		XtMoveWidget (grip, grip->core.x, grip->core.y + delta);
		GripDrawLine ((Widget) w, grip, w->vpane.move_gc);
	}
	/*
	 * See if we can move the next widget up towards it
	 * original position but only if we are moving up.
	 */
	if (next && delta < 0)
	{
		cons = VPCONS (next);
		edge = next->core.y;
		if (edge > cons->vpane.orig_y)
		{
			if (edge + delta > cons->vpane.orig_y)
				move = delta;
			else
				move = cons->vpane.orig_y - edge;
			DBUG_PRINT ("grip", ("Adjust next up %d towards original", move));
			GripMove (parent, next, move);
		}
	}

	/*
	 * See if we can move the previous grip down towards its
	 * original position, but again only if we are moving down.
	 */
	if (prev && delta > 0)
	{
		cons = VPCONS (prev);
		edge = prev->core.y;
		if (edge < cons->vpane.orig_y)
		{
			DBUG_PRINT ("grip", ("Adjust previous towards original"));
			if (edge + delta < cons->vpane.orig_y)
				move = delta;
			else
				move = cons->vpane.orig_y - edge;
			GripMove (parent, prev, move);
			DBUG_PRINT ("grip", ("Adjust prev down %d towards original", move));
		}
	}
	DBUG_VOID_RETURN;
}

/*
 * GripUpdate	- Based on the current position of all the grips, update
 *		  the scale of each child.  Still need to update the
 *		  next child of each grip, too.
 */
static void
GripUpdate (gw)
Widget		gw;			/* the vpane */
{
	XoVpaneWidget	w = (XoVpaneWidget) gw;
	Widget		grip;		/* current grip */
	XoVpaneConstraint	c;
	int		i;
	int		delta;		/* amount grip moved by */

	for (i = 0; i < w->vpane.grip_count; i++)
	{
		grip = w->vpane.grips[i];
		c = VPCONS (grip);
		if (c->vpane.orig_y != grip->core.y)
		{
			/*
			 * Erase whatever it has drawn.
			 */
			GripDrawLine (gw, grip, w->vpane.move_gc);
			delta = grip->core.y - c->vpane.orig_y;
			GripCalcScale ((Widget) w, c->vpane.before, delta);
			GripCalcScale ((Widget) w, c->vpane.after, -delta);
		}
	}
}

static void
GripCalcScale (gw, child, delta)
	Widget		gw;
	Widget		child;
	int		delta;
{
	XoVpaneConstraint	c;

	if (!child)
		return;
	c = VPCONS (child);
	c->column.user_height = child->core.height + delta;
}

static void
CalcGC (gw)
	Widget          gw;
{
	XoVpaneWidget   w = (XoVpaneWidget) gw;
	XGCValues       attr;

	if (w->vpane.gc)
		XtReleaseGC (gw, w->vpane.gc);
	if (w->vpane.move_gc)
		XtReleaseGC (gw, w->vpane.move_gc);

	attr.line_style = LineSolid;
	attr.subwindow_mode = IncludeInferiors;
	attr.function = GXxor;
	attr.foreground = w->cons_base.fg ^ w->core.background_pixel;
	w->vpane.move_gc = XtGetGC ((Widget) w, GCForeground | GCSubwindowMode |
			       GCLineStyle | GCFunction, &attr);
	attr.line_style = LineSolid;
	attr.foreground = w->cons_base.fg;
	w->vpane.gc = XtGetGC ((Widget) w, GCForeground|GCLineStyle,
				    &attr);
}

/* }}} */
