/*
 * Scrollbar.c,v 2.0 1992/04/23 02:51:57 ware Exp
 * Scrollbar.c,v
 * Revision 2.0  1992/04/23  02:51:57  ware
 * First public release.
 *
 * Revision 1.4  1992/04/23  02:18:48  ware
 * Added several classes.  Worked on geometry management
 *
 * Revision 1.3  1992/03/05  05:14:01  ware
 * *** empty log message ***
 *
 * Revision 1.2  1992/03/05  01:14:07  ware
 * Reached point of Grip following mouse.
 *
 * Revision 1.1  1992/03/03  17:20:30  ware
 * Initial revision
 *
 */

#include <stdio.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xo/ScrollbarP.h>
#include <X11/Xo/Grip.h>
#include <X11/Xo/ObjArrow.h>
#include <X11/Xo/dbug.h>

#include <X11/Xo/ScrollbRec.h>

XoProto (static double, GripTopCalc, (Widget gw, int delta));
XoProto (static void, GripAddCallbacks, (Widget gw, Widget grip));
XoProto (static double, GripPlace, (Widget gw, double top));
XoProto (static void, GripMove, (Widget gw, double top));
XoProto (static void, ScrollerPlace, (Widget gw, Widget scroller, int flag));
XoProto (static void, ScrollerAddCallback, (Widget gw, Widget scroller, int flag));
XoProto (static int, ScrollbarLength, (Widget gw));
XoProto (static void, GripCallbackMove, (Widget gw, XtPointer client_data, XtPointer call_data));
XoProto (static void, ScrollerCallbackMove, (Widget gw, XtPointer client_data, XtPointer call_data));

/*
 *----------------------------------------------------------------------
 * Core Class Methods
 *----------------------------------------------------------------------
 */

static void
ClassInit ()
{
	_XoRegisterConverters ();
}

static void
Initialize (request, new, arglist, num_args)
	Widget          request;	/* as first created */
	Widget          new;		/* after other parent classes */
	ArgList         arglist;	/* list of arguments */
	Cardinal       *num_args;	/* how many */
{
	XoScrollbarWidget  w = (XoScrollbarWidget) new;
	Arg             args[10];
	Cardinal        cnt;

	w->scrollbar.direction = False;
	w->scrollbar.grip_widget = NULL;
	if (w->scrollbar.grip_class)
	{
		cnt = 0;
		w->scrollbar.grip_widget =
			XtCreateWidget ("grip", w->scrollbar.grip_class,
					(Widget) w, args, cnt);
		_XoInheritedSet (new, w->scrollbar.grip_widget,
				 arglist, num_args);
	}
	w->scrollbar.reverse_scroller = NULL;
	w->scrollbar.forward_scroller = NULL;
	if (w->scrollbar.scroller_class)
	{
		/*
		 * FIX: This won't work for vertical scrollbars.
		 *	Secondly, we should be using the appropriate
		 *	object for drawing an arrowhead.
		 */
		cnt = 0;
		if (w->scrollbar.scroller_draw)
		{
			XtSetArg (args[cnt], XtNdrawClass, w->scrollbar.scroller_draw);
			++cnt;
		}
		if (w->scrollbar.orientation == XoVERTICAL)
			XtSetArg (args[cnt], XtNdirection, XoDOWN);
		else
			XtSetArg (args[cnt], XtNdirection, XoRIGHT);
		++cnt;
		w->scrollbar.forward_scroller =
			XtCreateWidget ("forward", w->scrollbar.scroller_class,
					(Widget) w, args, cnt);
		_XoInheritedSet (new, w->scrollbar.forward_scroller,
				 arglist, num_args);
		cnt = 0;
		if (w->scrollbar.scroller_draw)
		{
			XtSetArg (args[cnt], XtNdrawClass, w->scrollbar.scroller_draw);
			++cnt;
		}
		if (w->scrollbar.orientation == XoVERTICAL)
			XtSetArg (args[cnt], XtNdirection, XoUP);
		else
			XtSetArg (args[cnt], XtNdirection, XoLEFT);
		++cnt;
		w->scrollbar.reverse_scroller =
			XtCreateWidget ("reverse", w->scrollbar.scroller_class,
					(Widget) w, args, cnt);
		_XoInheritedSet (new, w->scrollbar.forward_scroller,
				 arglist, num_args);
	}
	GripAddCallbacks ((Widget) w, w->scrollbar.grip_widget);
	ScrollerAddCallback ((Widget) w, w->scrollbar.forward_scroller, False);
	ScrollerAddCallback ((Widget) w, w->scrollbar.reverse_scroller, True);

	/*
	 * Make sure the fields contain reasonable values
	 */
	if (w->scrollbar.size == 0)
	{
		w->scrollbar.size = 1;
	}
	if (w->scrollbar.size > 1.0)
	{
		w->scrollbar.size = 1.0;
	}
	if (w->scrollbar.min_grip == 0)
	{
		w->scrollbar.min_grip = 1;
	}
	if (w->scrollbar.increment > 1.0)
	{
		w->scrollbar.increment = 1.0;
	}
	if (w->scrollbar.increment < 0.0)
	{
		w->scrollbar.increment = 0.0;
	}
	if (w->scrollbar.top + w->scrollbar.size > 1.0)
	{
		w->scrollbar.top = 1.0 - w->scrollbar.size;
	}
	if (w->scrollbar.top < 0.0)
	{
		w->scrollbar.top = 0.0;
	}
	ScrollerPlace ((Widget) w, w->scrollbar.reverse_scroller, False);
	ScrollerPlace ((Widget) w, w->scrollbar.forward_scroller, True);
	(void) GripPlace ((Widget) w, w->scrollbar.top);
	w->scrollbar.orig_width = w->core.width;
	w->scrollbar.orig_height = w->core.height;
}

static Boolean
SetValues (current, request, new, args, num_args)
	Widget          current;	/* widget before the XtSetValues() */
	Widget          request;	/* args applied, not set_values */
	Widget          new;		/* the allowed changes */
	ArgList         args;		/* list of arguments */
	Cardinal       *num_args;	/* how many arguments */
{
	XoScrollbarWidget  w = (XoScrollbarWidget) new;
	XoScrollbarWidget  cur = (XoScrollbarWidget) current;
	Boolean         redisplay = False;

	/*
	 * If the application changes the width or height with XtSetValues
	 * then adopt the new dimension as the preferred width or height.
	 */
	if (w->core.width != cur->core.width)
	{
		w->scrollbar.orig_width = w->core.width;
	}
	if (w->core.height != cur->core.height)
	{
		w->scrollbar.orig_height = w->core.height;
	}
	if (w->scrollbar.forward_scroller != cur->scrollbar.forward_scroller)
	{
		if (cur->scrollbar.forward_scroller)
			XtDestroyWidget (cur->scrollbar.forward_scroller);
		ScrollerAddCallback ((Widget) w, w->scrollbar.forward_scroller, False);
		ScrollerPlace ((Widget) w, w->scrollbar.forward_scroller, False);
	}
	if (w->scrollbar.reverse_scroller != cur->scrollbar.reverse_scroller)
	{
		if (cur->scrollbar.reverse_scroller)
			XtDestroyWidget (cur->scrollbar.reverse_scroller);
		ScrollerAddCallback ((Widget) w, w->scrollbar.reverse_scroller, True);
		ScrollerPlace ((Widget) w, w->scrollbar.reverse_scroller, True);
	}
	if (w->scrollbar.grip_widget != cur->scrollbar.grip_widget)
	{
		if (cur->scrollbar.grip_widget)
		{
			XtDestroyWidget (cur->scrollbar.grip_widget);
		}
		GripAddCallbacks ((Widget) w, w->scrollbar.grip_widget);
		(void) GripPlace ((Widget) w, w->scrollbar.top);
	}
	if (w->scrollbar.min_grip != cur->scrollbar.min_grip)
	{
		if (w->scrollbar.min_grip == 0)
			w->scrollbar.min_grip = 1;
		(void) GripPlace ((Widget) w, w->scrollbar.top);
	}
	if (w->scrollbar.size != cur->scrollbar.size)
	{
		(void) GripPlace ((Widget) w, w->scrollbar.top);
	}
	if (w->scrollbar.top != cur->scrollbar.top)
	{
		GripMove ((Widget) w,  w->scrollbar.top);
	}
	return redisplay;
}

/*
 * Realize	- Use the superclass realize method for the scrollbar
 *		  but we need to realize the grip, and scroller widgets
 *		  since the intrinsics don't know about them (i.e.
 *		  this is not a sublcass of Composite).
 */

static void
Realize (gw, valueMask, attrs)
	Widget          gw;
	XtValueMask    *valueMask;
	XSetWindowAttributes *attrs;
{
	XoScrollbarWidget  w = (XoScrollbarWidget) gw;

	(*MySuperClass.core_class.realize) (gw, valueMask, attrs);
	if (w->scrollbar.grip_widget)
		XtRealizeWidget (w->scrollbar.grip_widget);
	if (w->scrollbar.forward_scroller)
		XtRealizeWidget (w->scrollbar.forward_scroller);
	if (w->scrollbar.reverse_scroller)
		XtRealizeWidget (w->scrollbar.reverse_scroller);
	XMapSubwindows (XtDisplay (gw), XtWindow (gw));
}

static void
Resize (gw)
	Widget		gw;
{
	XoScrollbarWidget  w = (XoScrollbarWidget) gw;

	ScrollerPlace (gw, w->scrollbar.reverse_scroller, False);
	ScrollerPlace (gw, w->scrollbar.forward_scroller, True);
	w->scrollbar.top = GripPlace (gw, w->scrollbar.top);
}

/*
 * Destroy	- Clean up anything allocated by this widget.  In particular
 *		  the grip and scroller widgets need to be destroyed since
 *		  this is not a composite widget and so the intrinsics don't
 *		  know anything about them.
 */

static void
Destroy (gw)
	Widget		gw;
{
	XoScrollbarWidget  w = (XoScrollbarWidget) gw;

	if (w->scrollbar.grip_widget)
		XtDestroyWidget (w->scrollbar.grip_widget);
	if (w->scrollbar.forward_scroller)
		XtDestroyWidget (w->scrollbar.forward_scroller);
	if (w->scrollbar.reverse_scroller)
		XtDestroyWidget (w->scrollbar.reverse_scroller);
}

/*
 * GeometryQuery - Returns the preferred size of this widget.  This is
 *		   calculated as being as close to the original size
 *		   as possible.
 */

static XtGeometryResult
GeometryQuery (gw, proposed, desired)
	Widget          gw;
	XtWidgetGeometry *proposed;
	XtWidgetGeometry *desired;
{
	XoScrollbarWidget   w = (XoScrollbarWidget) gw;
	XtGeometryResult result;

	DBUG_ENTER ("Scrollbar.GeometryQuery");
	if (!proposed->request_mode)
	{
		/*
		 * I'm being asked my ideal size
		 */
		desired->width = w->scrollbar.orig_width;
		desired->height = w->scrollbar.orig_height;
		desired->request_mode = CWWidth | CWHeight;
		if (desired->width == w->core.width
		    && desired->height == w->core.height)
			result = XtGeometryNo;
		else
			result = XtGeometryAlmost;
	}
	else if (XoGeoDimFlags (proposed) == CWWidth)
	{
		/*
		 * I'm being told what my height is and return with a
		 * preferred width.
		 */
		desired->height = w->scrollbar.orig_height;
		desired->request_mode = CWHeight;
		result = XtGeometryAlmost;
	}
	else if (XoGeoDimFlags (proposed) == CWHeight)
	{
		/*
		 * I'm being told what my width is and return with a
		 * preferred height
		 */
		desired->width = w->scrollbar.orig_width;
		desired->request_mode = CWWidth;
		result = XtGeometryAlmost;
	}
	else
	{
		/*
		 * Parent is specifying both width and height, so just return
		 * if it's what we'd like to be
		 */
		if (proposed->width == w->scrollbar.orig_width
		    && proposed->height == w->scrollbar.orig_height)
			result = XtGeometryYes;
		else
			result = XtGeometryNo;
	}
	DBUG_RETURN (result);
}

/*
 *----------------------------------------------------------------------
 * Simple Class Methods
 *----------------------------------------------------------------------
 */

/*
 *----------------------------------------------------------------------
 * Scrollbar Class Methods
 *----------------------------------------------------------------------
 */

/*
 *----------------------------------------------------------------------
 * Private Procedures
 *----------------------------------------------------------------------
 */

/*
 * GripTopCalc - return the value of "top" based on moving the grip
 *		delta pixels.  Negative is up/left movement.
 */
static double
GripTopCalc (gw, delta)
	Widget		gw;
	int		delta;
{
	XoScrollbarWidget	w = (XoScrollbarWidget) gw;
	double		top;
	int		length;
	int		orig_top;
	int		new_top;

	length = ScrollbarLength ((Widget) w);
	orig_top = w->scrollbar.top * length;
	if (w->scrollbar.orientation == XoVERTICAL)
	{
		new_top = orig_top + delta;
	}
	else
	{
		new_top = orig_top + delta;
	}
	top = new_top/(double)length;
	if (top + w->scrollbar.size > 1.0)
		top = 1.0 - w->scrollbar.size;
	else if (top < 0.0)
		top = 0.0;
	return (top);
}

/*
 * GripCallbackMove	- Callback added to each grip widget that is
 *		responsible for physically moving the grip.
 */

static void
GripCallbackMove (gw, client_data, call_data)
	Widget          gw;
	XtPointer       client_data;
	XtPointer       call_data;
{
	XoScrollbarWidget	w = (XoScrollbarWidget) client_data;
	XoMoveCBdata   *d;
	int		delta;		/* change in movement */
	double		top;

	if ((d = (XoMoveCBdata *) call_data) == NULL || w == NULL)
		return;
	if (w->scrollbar.orientation == XoVERTICAL)
		delta = d->mcbd_y - d->mcbd_y_first;
	else
		delta = d->mcbd_x - d->mcbd_x_first;
	top = GripPlace ((Widget) w, GripTopCalc ((Widget) w, delta));
	XtCallCallbackList ((Widget) w, w->scrollbar.callbacks,
			    (XtPointer) &top);
}


static void
GripCallbackDone (gw, client_data, call_data)
	Widget          gw;
	XtPointer       client_data;
	XtPointer       call_data;
{
	XoScrollbarWidget	w = (XoScrollbarWidget) client_data;
	XoMoveCBdata   *d;
	double		top;
	int		delta;		/* change in movement */

	if ((d = (XoMoveCBdata *) call_data) == NULL || w == NULL)
		return;
	if (w->scrollbar.orientation == XoVERTICAL)
		delta = d->mcbd_y - d->mcbd_y_first;
	else
		delta = d->mcbd_x - d->mcbd_x_first;
	top = GripPlace ((Widget) w, GripTopCalc ((Widget) w, delta));
	GripMove ((Widget) w, top);
}

static void
GripAddCallbacks (gw, grip)
	Widget		gw;
	Widget		grip;
{
	if (!grip)
		return;
	XtAddCallback (grip, XtNmoveCallback, GripCallbackMove, (XtPointer) gw);
	XtAddCallback (grip, XtNcallback, GripCallbackDone, (XtPointer) gw);
}

/*
 * GripGotoAction - Called to move the grip to this location.
 */

static void
GripGoto (gw, event, params, num_params)
	Widget          gw;
	XEvent         *event;
	String         *params;
	Cardinal       *num_params;
{
	XoScrollbarWidget w = (XoScrollbarWidget) gw;
	Position	x,y;
	int		edge;		/* edge of grip widget */
	int		delta;		/* amount to move */

	if (event)
		XoEventLocation (event, &x, &y);
	if (w->scrollbar.orientation == XoVERTICAL)
	{
		/*
		 * Decide which edge (top or bottom) is closer and then
		 * calculate the distance that edge needs to move to
		 * get to the event location.
		 */
		if (y <= _XoEdgeTop (w->scrollbar.grip_widget))
		{
			w->scrollbar.direction = False;
			edge = _XoEdgeTop (w->scrollbar.grip_widget);
		}
		else if (y >= _XoEdgeBottom (w->scrollbar.grip_widget))
		{
			w->scrollbar.direction = True;
			edge = _XoEdgeBottom (w->scrollbar.grip_widget);
		}
		else if (w->scrollbar.direction)
		{
			/* event is over grip and we were moving forward */
			edge = _XoEdgeBottom (w->scrollbar.grip_widget);
		}
		else
		{
			/* event is over grip and we were moving reverse */
			edge = _XoEdgeTop (w->scrollbar.grip_widget);
		}
		delta = y - edge;
	}
	else
	{
		/*
		 * Decide which edge (left or right) is closer and then
		 * calculate the distance that edge needs to move to
		 * get to the event location.
		 */
		if (x <= _XoEdgeLeft (w->scrollbar.grip_widget))
		{
			w->scrollbar.direction = False;
			edge = _XoEdgeLeft (w->scrollbar.grip_widget);
		}
		else if (x >= _XoEdgeRight (w->scrollbar.grip_widget))
		{
			w->scrollbar.direction = True;
			edge = _XoEdgeRight (w->scrollbar.grip_widget);
		}
		else if (w->scrollbar.direction)
		{
			edge = _XoEdgeRight (w->scrollbar.grip_widget);
		}
		else
		{
			edge = _XoEdgeLeft (w->scrollbar.grip_widget);
		}
		delta = x - edge;
	}
	GripMove (gw, GripTopCalc (gw, delta));
}

#define CHILDOFFSET 1			/* whitespace to show */

static double
GripPlace (gw, top)
	Widget		gw;
	double		top;
{
	XoScrollbarWidget w = (XoScrollbarWidget) gw;
	int		x, y;
	Dimension	width, height;
	Widget		grip;
	Dimension	grip_width;
	Dimension	grip_height;
	int		begin;		/* inner edge of reverse scroller */
	int		end;		/* inner edge of forward scroller */

	DBUG_ENTER ("Scrollbar.GripPlace");
	if ((grip = w->scrollbar.grip_widget) == NULL)
		DBUG_RETURN (top);
	if (w->scrollbar.orientation == XoVERTICAL)
	{
		grip_width = w->core.width - 2*grip->core.border_width;
		x = 0;
		height = ScrollbarLength (gw);
		/*
		 * Calculate the "desired" size of the grip instead
		 * of the actual size since the two may be different
		 */
		grip_height =  height * w->scrollbar.size;
		if (grip_height < w->scrollbar.min_grip)
			grip_height = w->scrollbar.min_grip;
		if (w->scrollbar.reverse_scroller)
			begin = _XoEdgeBottom(w->scrollbar.reverse_scroller);
		else
			begin = 0;
		if (w->scrollbar.forward_scroller)
			end = _XoEdgeTop (w->scrollbar.forward_scroller);
		else
			end = w->core.height;
		y = begin + (height * top);
		/*
		 * Make sure we don't put the grip past the bottom or before
		 * the two scrollers.
		 */
		if (y + grip_height > end)
			y = end - grip_height;
		if (y < begin)
			y = begin;
		/*
		 * Calculate the new value of top.
		 */
		top = (y-begin)/(double)height;
	}
	else				/* horizontal */
	{
		grip_height = w->core.height
			- 2*(grip->core.border_width + CHILDOFFSET);
		y = CHILDOFFSET;
		width = ScrollbarLength (gw);
		grip_width =  width * w->scrollbar.size;
		if (grip_width < w->scrollbar.min_grip)
			grip_width = w->scrollbar.min_grip;
		if (w->scrollbar.reverse_scroller)
			begin = _XoEdgeRight (w->scrollbar.reverse_scroller);
		else
			begin = 0;
		if (w->scrollbar.forward_scroller)
			end = _XoEdgeLeft (w->scrollbar.forward_scroller);
		else
			end = w->core.width;
		x = begin + (width * top);
		/*
		 * Now insure we don't put the grip either past or before
		 * the two scrollers
		 */
		if (x + grip_width > end)
		{
			DBUG_PRINT ("scroll",
				    ("Grip to far right (%d + %d > %d)",
				     x, grip_width+grip->core.border_width,
				     end));
			x = end - grip_width;
		}
		if (x < begin)
			x = begin;
		grip_width -= 2*grip->core.border_width;
		/*
		 * Calculate the new value of top.
		 */
		top = (x-begin)/(double)width;
		DBUG_PRINT ("scroll", ("Setting value of top to %f", top));
	}
	XoConfWidget (grip, x, y, (unsigned) grip_width,
		      (unsigned) grip_height,
		      (unsigned) grip->core.border_width);
	DBUG_RETURN (top);
}


static void
GripMove (gw, top)
	Widget		gw;
	double		top;		/* where top of grip should be */
{
	XoScrollbarWidget w = (XoScrollbarWidget) gw;

	w->scrollbar.top = GripPlace (gw, top);
	XtCallCallbackList (gw, w->scrollbar.callbacks,
			    (XtPointer) &top);
}

static void
ScrollerCallbackMove (gw, client_data, call_data)
	Widget		gw;		/* scroller widget */
	XtPointer	client_data;
	XtPointer	call_data;
{
	XoScrollbarWidget w = (XoScrollbarWidget) XtParent (gw);
	int		flag = (int) client_data;
	int		delta;

	/*
	 * FIX: Need to account for fraction of a pixel movement.
	 */
	if (flag)
	{
		/* forward scroller (bottom or right) */
		delta = -w->scrollbar.increment*ScrollbarLength ((Widget) w);
	}
	else
	{
		/* reverse scroller (top or left) */
		delta = w->scrollbar.increment*ScrollbarLength ((Widget) w);
	}
	GripMove ((Widget) w, GripTopCalc ((Widget) w, delta));
}


static void
ScrollerAddCallback (gw, scroller, flag)
	Widget		gw;
	Widget		scroller;
	int		flag;
{
	if (!scroller)
		return;
	XtAddCallback (scroller, XtNcallback, ScrollerCallbackMove, 
			(XtPointer) flag);
}

static void
ScrollerPlace (gw, child, flag)
	Widget		gw;
	Widget		child;
	int		flag;
{
	XoScrollbarWidget w = (XoScrollbarWidget) gw;
	Dimension	width;
	Dimension	height;
	int		x,y;

	DBUG_ENTER ("Scrollbar.ScrollerPlace");
	if (!child)
		DBUG_VOID_RETURN;
	if (w->scrollbar.orientation == XoVERTICAL)
	{
		width = w->core.width
			- 2 * (child->core.border_width + CHILDOFFSET);
		height = width;
		if (flag)
		{
			/* the forward scroller (i.e. on bottom) */
			y = w->core.height - height - CHILDOFFSET
				- 2*child->core.border_width;
		}
		else
		{
			/* the backward scroller (i.e. on top) */
			y = CHILDOFFSET;
		}
		x = CHILDOFFSET;
	}
	else
	{
		height = w->core.height
			- 2 * (child->core.border_width + CHILDOFFSET);
		width = height;
		if (flag)
		{
			/* the forward scroller (i.e. on the right side) */
			x = w->core.width - width - CHILDOFFSET
				- 2 * child->core.border_width;;
		}
		else
		{
			/* the backward scroller (i.e. on the left side) */
			x = CHILDOFFSET;
		}
		y = CHILDOFFSET;
	}
	XoConfWidget (child, x, y, (unsigned) width, (unsigned) height,
		      (unsigned) child->core.border_width);
	DBUG_VOID_RETURN;
}

static int
ScrollbarLength (gw)
	Widget	gw;
{
	XoScrollbarWidget	w = (XoScrollbarWidget) gw;
	Widget		scroller;
	int		length;

	length = 0;
	if (w->scrollbar.orientation == XoVERTICAL)
	{
		if ((scroller = w->scrollbar.forward_scroller) != NULL)
			length += scroller->core.height
				+ 2*scroller->core.border_width;
		if ((scroller = w->scrollbar.reverse_scroller) != NULL)
			length += scroller->core.height
				+ 2*scroller->core.border_width;
		length = w->core.height - length;
	}
	else
	{
		if ((scroller = w->scrollbar.forward_scroller) != NULL)
			length += scroller->core.width
				+ 2*scroller->core.border_width;
		if ((scroller = w->scrollbar.reverse_scroller) != NULL)
			length += scroller->core.width
				+ 2*scroller->core.border_width;
		length = w->core.width - length;
	}
	if (length < 0)
		length = 0;
	return length;
}

