/*
 * MenuShell	- Implements a popup shell for walking menus.  In order to
 *		  improve performance, it turns out to be too slow to keep
 *		  popping down the old menu and then popuping up a new
 *		  menu (both are override shells).  Instead, we just create
 *		  one shell and allow it to manage multiple children.  Each
 *		  pulldown or submenu is a child of this shell.
 *		  Unmanage the old child than manage the new one.
 *		  One should also
 *		  specify the new x & y values at the same time to
 *		  insure maximum speed.
 * MenuShell.c,v 2.3 1992/07/22 16:03:18 pete Exp
 * MenuShell.c,v
 * Revision 2.3  1992/07/22  16:03:18  pete
 * (CallbackMenu): Removed redundant call to XoMenuNew() when
 * cb->pi_submenu == NULL.  The same thing happens in
 * MenuShell.SetValues().  Work on getting Menu to display the proper
 * size.  Finally the initial menu is placed correctly.  (Resize): Added
 * to handle changing border widget size properly.
 *
 * Revision 2.2  1992/07/16  17:21:13  pete
 * Use utility function XoFindChild() instead of private method.
 *
 * Revision 2.1  1992/06/23  00:29:01  pete
 * Changed interface to _XoMenuNew and _XoMenuDone.
 *
 * Revision 2.0  1992/04/23  02:51:27  ware
 * First public release.
 *
 * Revision 1.8  1992/02/27  14:30:29  ware
 * Compiled with GCC 2.0 and very strict checks.  Fixed Warnings
 *
 * Revision 1.7  1992/02/20  15:11:09  ware
 * Applied new indentation
 *
 * Revision 1.6  92/02/04  21:22:46  pete
 * Release 44
 *
 * Revision 1.5  1991/09/12  09:46:36  pete
 * Added a border widget to outline the edges.  Needed several changes to
 * fix geometry calculations because of this.
 *
 * Revision 1.4  91/08/30  17:39:48  pete
 * Changed to use the new XtNCallbackMenu callback protocol.  Changed
 * Menu to be a subclass of Column.
 *
 * Revision 1.3  91/08/27  09:39:37  pete
 * Changed SetValues/ChangeManaged so that XtNcurrent just indicates the
 * child that should be on top.  Kept so that setting XtNcurrent causes
 * an implicit XtManageChild() but removed any implicit XtUnmanageChild().
 *
 * ChangeManage() negotiates with the parent to request a new size based
 * on XtNcurrent.
 *
 * Revision 1.2  91/08/26  11:57:58  pete
 * Use XoProto() for conditional prototypes.  Working on getting traversals
 * and menus to work more efficiently.  Changed to following naming
 * conventions.
 *
 * Revision 1.1  1991/07/19  00:58:43  pete
 * Initial revision
 *
 */

#include <stdio.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xo/Menu.h>
#include <X11/Xo/MenuShellP.h>
#include <X11/Xo/dbug.h>

#include <X11/Xo/MenuSRec.h>

XoProto (static int, BWidth, (XoMenuShellWidget w));

/*
 *----------------------------------------------------------------------
 * Core Class Methods
 *----------------------------------------------------------------------
 */

/*
 * ClassPartInit - Inherit some class methods appropriately.
 */

static void
ClassPartInit (class)
	WidgetClassRec *class;
{
	XoMenuShellClassRec *c = (XoMenuShellClassRec *) class;
	XoMenuShellClassRec *super;

	DBUG_ENTER ("MenuShell.ClassPartInit");
	super = (XoMenuShellClassRec *) c->core_class.superclass;
	if (c->menu_shell_class.callback_menu == XtInheritCallbackMenu)
		c->menu_shell_class.callback_menu =
			super->menu_shell_class.callback_menu;
	if (c->menu_shell_class.traverse == XtInheritTraversalProc)
		c->menu_shell_class.traverse =
			super->menu_shell_class.traverse;
	DBUG_VOID_RETURN;
}

/*
 * Initialize
 */

static void
Initialize (request, new, arglist, num_args)
	Widget          request;	/* as first created */
	Widget          new;		/* after other parent classes */
	ArgList         arglist;	/* list of arguments */
	Cardinal       *num_args;	/* how many */
{
	XoMenuShellWidget w = (XoMenuShellWidget) new;
	Arg             args[10];
	Cardinal        cnt;

	DBUG_ENTER ("MenuShell.Initialize");
	if (w->menu_shell.border_class)
	{
		/*
		 * FIX: Need to pass appropriate args along to the widget.
		 * Probably need to create it (so the border widget has a
		 * chance to indicate it's resoruces and then set them
		 * appropriately.
		 */
		cnt = 0;
		w->menu_shell.border_widget =
			XtCreateWidget ("border", w->menu_shell.border_class,
					(Widget) w, args, cnt);
		_XoInheritedSet (new, w->menu_shell.border_widget,
				 arglist, num_args);
	}

	/*
	 * Add a callback for children to execute when they want to pop up or
	 * down a submenu.  This allows us to popdown any other submenus that
	 * may have been displayed.
	 */
	if (ThisClass (new).menu_shell_class.callback_menu)
	{
		XtAddCallback (new, XtNcallbackMenu,
			     ThisClass (new).menu_shell_class.callback_menu,
			       (XtPointer) new);
	}
	if (ThisClass (new).menu_shell_class.traverse
	    && XtHasCallbacks (new, XtNtraverse) != XtCallbackHasSome)
	{
		XtAddCallback (new, XtNtraverse,
			       ThisClass (new).menu_shell_class.traverse,
			       (XtPointer) NULL);
	}
	DBUG_VOID_RETURN;
}

static Boolean
SetValues (current, request, new, args, num_args)
	Widget          current;	/* widget before the XtSetValues() */
	Widget          request;	/* after args applied but no
					 * set_values */
	Widget          new;		/* the allowed changes */
	ArgList         args;		/* list of arguments */
	Cardinal       *num_args;	/* how many arguments */
{
	XoMenuShellWidget w = (XoMenuShellWidget) new;
	XoMenuShellWidget cur = (XoMenuShellWidget) current;
	Boolean         redisplay = False;

	DBUG_ENTER ("MenuShell.SetValues");
	if (w->menu_shell.current && _XoChildFind (new, w->menu_shell.current)<0)
	{
		w->menu_shell.current = NULL;
		_XoWarn (new, "badChild", "MenuShell",
			 "Cannot set current to a widget that is not a child of the MenuShell.");
	}
	if (w->menu_shell.current != cur->menu_shell.current)
	{
		_XoMenuNew (cur->menu_shell.current, (Widget) NULL,
			    w->menu_shell.popup_starter,
			    XoNORTHWEST,
			    (XEvent *) NULL, XoPOPUP_NEW);
		if (!w->menu_shell.current)
		{
			XtPopdown (new);
			if (w->menu_shell.start)
				XtUngrabPointer (new, CurrentTime);
			XtUnmanageChild (cur->menu_shell.current);
		}
		else
		{
			/*
			 * if current == NULL it means we were not already
			 * popped up.
			 */
			if (!cur->menu_shell.current)
			{
				if (w->menu_shell.start)
				{
					XtPopupSpringLoaded (new);
					(void) XtGrabPointer (XtParent (new), True, EnterWindowMask | LeaveWindowMask | ButtonReleaseMask,
							      GrabModeAsync, GrabModeAsync, None, None,
							      CurrentTime);
					XtAddGrab (XtParent (new), False, False);
				}
				else
				{
					XtPopup (new, XtGrabNonexclusive);
				}
			}
			if (w->menu_shell.border_widget)
			{
				_XoObjRedraw (w->menu_shell.border_widget, (XEvent *) NULL, (Region) NULL);
			}
		}
	}
	DBUG_RETURN (redisplay);
}

static void
Resize (gw)
	Widget		gw;
{
	XoMenuShellWidget w = (XoMenuShellWidget) gw;
	int		i;
	Widget		child;

	DBUG_ENTER ("MenuShell.Resize");
	XoConfWidget (w->menu_shell.border_widget, 0, 0,
			   w->core.width,
			   w->core.height,
			   BWidth (w));
	for (i = 0; i < w->composite.num_children; i++)
	{
		child = w->composite.children[i];
		if (XtIsManaged (child))
		{
			XtResizeWidget (child,
					w->core.width - 2*BWidth (w),
					w->core.height - 2*BWidth (w),
					child->core.border_width);
		}
	}
	w->menu_shell.resize_flag = True;
	DBUG_VOID_RETURN;
}

static void
Redisplay (gw, event, region)
	Widget          gw;
	XEvent         *event;
	Region          region;
{
	XoMenuShellWidget w = (XoMenuShellWidget) gw;

	DBUG_ENTER ("MenuShell.Redisplay");
	_XoObjRedraw (w->menu_shell.border_widget, event, region);
	DBUG_VOID_RETURN;
}

/*
 *----------------------------------------------------------------------
 * Composite Class Methods
 *----------------------------------------------------------------------
 */

static void
ChangeManaged (gw)
	Widget          gw;
{
	XoMenuShellWidget w = (XoMenuShellWidget) gw;
	XtWidgetGeometry req;
	Widget          current;
	Dimension       border_width;
	Dimension       width;
	Dimension       height;
	XtGeometryResult res;
	int             i;

	DBUG_ENTER ("MenuShell.ChangeManaged");
	current = NULL;
	for (i = 0; i < w->composite.num_children; i++)
	{
		if (XtIsManaged (w->composite.children[i]))
		{
			if (current)
			{
				_XoWarn (gw, "number managed", "MenuShell",
					 "Too many children are managed.");
			}
			current = w->composite.children[i];
		}
	}
	if (current)
	{
		/*
		 * Set a flag so we know if the Resize Method has
		 * been called.  It is set to True by Resize().  If
		 * it is still False, then call Resize() by hand.
		 */

		w->menu_shell.resize_flag = False;
		border_width = BWidth (w);
		width = current->core.width + 2 * border_width + 2 * current->core.border_width;
		height = current->core.height + 2 * border_width + 2 * current->core.border_width;
		/*
		 * The intent _was_ to give the window manager several
		 * tries to size the window.  The reality is since
		 * this is an override shell the request is always
		 * granted.  I left it in just as an example.
		 */
		i = 0;
		do
		{
			req.request_mode = 0;
			if (w->core.width != width)
			{
				req.width = width;
				req.request_mode |= CWWidth;
			}
			if (w->core.height != height)
			{
				req.height = height;
				req.request_mode |= CWHeight;
			}
			res = XtMakeGeometryRequest (gw, &req, &req);
		} while (i++ < 5 && res != XtGeometryYes);
		DBUG_PRINT ("menugeometry", ("Took %d tries [%d,%d] vs [%d,%d]",
					     i, width, height, req.width, req.height));
		if (res == XtGeometryAlmost)
			res = XtMakeGeometryRequest (gw, &req, &req);
		XtMoveWidget (current, (Position) border_width,
			      (Position) border_width);
		if (!w->menu_shell.resize_flag)
		{
			ThisClass(w).core_class.resize (gw);
		}
	}
	DBUG_VOID_RETURN;
}

/*
 *----------------------------------------------------------------------
 * Shell Class methods
 *----------------------------------------------------------------------
 */

/*
 *----------------------------------------------------------------------
 * OverrideShell Class methods
 *----------------------------------------------------------------------
 */

/*
 *----------------------------------------------------------------------
 * MenuShell methods
 *----------------------------------------------------------------------
 */

static void
CallbackMenu (gw, client_data, call_data)
	Widget          gw;
	XtPointer       client_data;
	XtPointer       call_data;
{
	XoMenuShellWidget w = (XoMenuShellWidget) gw;
	XoPopupInfo    *cb = (XoPopupInfo *) call_data;
	Cardinal        cnt;
	Arg             args[10];

	DBUG_ENTER ("MenuShell.CallbackMenu");
	if (!cb)
		DBUG_VOID_RETURN;
	if (cb->pi_action == XoPOPUP_NONE)
	{
		cnt = 0;
		XtSetArg (args[cnt], XtNpopupStarter, NULL); ++cnt;
		XtSetArg (args[cnt], XtNcurrent, NULL); ++cnt;
		XtSetValues (gw, args, cnt);
		_XoMenuDone (XtParent (gw), cb->pi_submenu, cb->pi_start,
			     cb->pi_where, cb->pi_event, XoPOPUP_NONE);
	}
	else if (!cb->pi_submenu)
	{
		cnt = 0;
		XtSetArg (args[cnt], XtNpopupStarter, NULL); ++cnt;
		XtSetArg (args[cnt], XtNcurrent, cb->pi_submenu); ++cnt;
		XtSetValues (gw, args, cnt);
		/* _XoMenuNew (w->menu_shell.current, cb->pi_submenu, cb->pi_start,
			     cb->pi_where, cb->pi_event, XoPOPUP_NEW);*/
	}
	else
	{
		/*
		 * Undisplay the current child
		 */
		if (w->menu_shell.current && w->menu_shell.current != cb->pi_submenu)
			XtUnmanageChild (w->menu_shell.current);
		/*
		 * Display the new one at the suggested x & y coordinates
		 */
		XtManageChild (cb->pi_submenu);
		XoPlacePopup (gw, cb->pi_root_x, cb->pi_root_y);
		cnt = 0;
		XtSetArg (args[cnt], XtNpopupStarter, cb->pi_start); ++cnt;
		XtSetArg (args[cnt], XtNcurrent, cb->pi_submenu); ++cnt;
		XtSetValues (gw, args, cnt);
	}
	DBUG_VOID_RETURN;
}

static void
Traverse (gw, client_data, call_data)
	Widget          gw;
	XtPointer       client_data;	/* data from the application */
	XtPointer       call_data;	/* data from the widget */
{
	XoMenuShellWidget w = (XoMenuShellWidget) gw;
	XoTraverseInfo *info;

	DBUG_ENTER ("MenuShell.Traverse");
	if (!(info = (XoTraverseInfo *) call_data))
	{
		_XoWarn (gw, "Traverse", "badArgument", "call data is null");
		DBUG_VOID_RETURN;
	}
	/*
	 * Tell our parent that some other menu needs to be popped up. The
	 * only difficulty is specifying relative to what.  So we rely on
	 * being told the widget that was responsible for popping us up.
	 */
	XoTraverse (w->menu_shell.popup_starter, info->t_direction, info->t_time);
	DBUG_VOID_RETURN;
}


/*
 *----------------------------------------------------------------------
 * Actions
 *----------------------------------------------------------------------
 */

static void
XoMenuPopdown (gw, event, params, num_params)
	Widget          gw;
	XEvent         *event;
	String         *params;
	Cardinal       *num_params;
{
	XoMenuShellWidget w = (XoMenuShellWidget) gw;

	DBUG_ENTER ("XoMenuPopdown");

	_XoMenuDone (w->menu_shell.current, (Widget) NULL, (Widget) NULL,
			    _XoGravityAction (params, num_params, XoNORTHWEST),
			    event, XoPOPUP_NONE);
	DBUG_VOID_RETURN;
}


/*
 *----------------------------------------------------------------------
 * Private Utilities
 *----------------------------------------------------------------------
 */


/*
 * Return the border width of this widget
 */

static int
BWidth (w)
	XoMenuShellWidget w;
{
	if (!w || !w->menu_shell.border_widget)
		return 0;
	else
		return w->menu_shell.border_widget->core.border_width;
}

/*
 *----------------------------------------------------------------------
 * Global utilities.
 *----------------------------------------------------------------------
 */

/*
 * Returns the popup for walking menus.  Retrieves the value of
 * XtNcurrentPopup.  If it is NULL, then a new child of the specified
 * class and the arguments is created.  If the XtNcurrentPopup already
 * exists, it returns that value and the class and argument list is ignored.
 */

Widget
XoMenuMake (name, class, parent, al, ac)
	String          name;		/* name of new widget */
	WidgetClass     class;		/* class of new widget */
	Widget          parent;		/* the parent */
	Arg            *al;
	Cardinal        ac;
{
	Arg             args[10];
	Cardinal        cnt;
	Widget          gw = NULL;

	cnt = 0;
	XtSetArg (args[cnt], XtNpopup, &gw);
	++cnt;
	XtGetValues (parent, args, cnt);
	/*
	 * If there is no such child then go ahead and create it
	 */
	if (!gw)
	{
		gw = XtCreatePopupShell (name, class, parent, al, ac);
		cnt = 0;
		XtSetArg (args[cnt], XtNpopup, gw);
		++cnt;
		XtSetValues (parent, args, cnt);
	}
	return gw;
}

/*
 * Same as XoMenuMake except the class is defaulted to xoMenuShellWidgetClass
 * and the name has "popup_" prepended to it.
 */

Widget
XoMenuMakeUtil (name, parent, al, ac)
	String          name;		/* base name */
	Widget          parent;
	Arg            *al;
	Cardinal        ac;
{
	char            buf[256];

	sprintf (buf, "popup_%s", name ? name : "made");
	return XoMenuMake (buf, xoMenuShellWidgetClass, parent, al, ac);
}

void
_XoMenuNew (gw, submenu, start, where, event, action)
	Widget          gw;
	Widget		submenu;
	Widget		start;
	XoGravity	where;
	XEvent		*event;
	XoPopupAction	action;
{
	XoPopupInfo    p;

	DBUG_ENTER ("_XoMenuNew");
	if (!gw)
	{
		DBUG_PRINT ("menu", ("NULL widget"));
		DBUG_VOID_RETURN;
	}
	p.pi_submenu = submenu;
	p.pi_start = start;
	p.pi_event = event;
	p.pi_action = action;
	p.pi_where = where;
	if (start)
	{
		XoWidgetGravityPoint (start, where, &p.pi_root_x, &p.pi_root_y);
		XtTranslateCoords (start, p.pi_root_x, p.pi_root_y,
				   &p.pi_root_x, &p.pi_root_y);
	}
	DBUG_PRINT ("menu", ("Placing %s at (%d, %d)", XoName (submenu),
			     p.pi_root_x, p.pi_root_y));
	if (XtHasCallbacks (gw, XtNcallbackMenu) == XtCallbackHasSome)
	{
		XtCallCallbacks (gw, XtNcallbackMenu, (XtPointer) &p);
	}
	DBUG_VOID_RETURN;
}

void
_XoMenuDone (gw, submenu, start, where, event, action)
	Widget          gw;
	Widget		submenu;
	Widget		start;
	XoGravity	where;
	XEvent		*event;
	XoPopupAction	action;
{
	DBUG_ENTER ("_XoMenuDone");
	_XoMenuNew (gw, submenu, start, where, event, action);
	DBUG_VOID_RETURN;
}
