/*LINTLIBRARY*/

#include "udposix.h"
#include <stddef.h>		/* for size_t (for <udalloc.h>) and NULL */
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <stdlib.h>
#include "udalloc.h"
#include "valuestring.h"
#include "uderrmsg.h"

#ifndef lint
    static char	rcsid[]	= "$Id: valuestring.c,v 2.5 1991/04/10 14:30:16 steve Exp $";
    static char	afsid[]	= "$__Header$";
#endif

static char	FieldSepChar	= ',';
static char	EscapeChar	= '\\';


/*
 * WHAT:   Free a list of substring pointers.
 *
 * INPUT:  The number of substrings and the array of substring pointers.
 */
void
VsFree(nsub, subs)
    int		nsub;		/* Number of substrings */
    char	**subs;		/* Substring pointers */
{
    if (nsub > 0 && subs != NULL) {
	if (*subs != NULL)
	    (void)FREE(*subs);
	(void)FREE(subs);
    }
}


/*
 * Copy a substring -- handling escaped field separators as necessary.
 */
static int
CopySub(to, from, SepPtr)
    char	*to, *from, *SepPtr;
{
    char	*cp	= to;

    assert(SepPtr >= from);

    while (from < SepPtr) {
	if (from[0] == EscapeChar && from[1] == FieldSepChar)
	    ++from;
	*cp++	= *from++;
    }
    *cp++	= 0;

    return cp - to;
}


/*
 * WHAT	   Split a value-string into substrings.
 *
 * INPUT   A value-string.  It will be untouched.
 *
 * OUTPUT  An array of pointers to the individual substrings (which have been
 *	   copied to their own storage) or a nil pointer on failure.
 *
 * HOW	   Count the number of substrings.  Allocate memory for the substring
 *	   pointer array.  Allocate contiguous memory sufficient for all the 
 *	   substrings.  For each substring in the value-string, add the current
 *	   location in the substring buffer to the substring pointer array and
 *	   copy the substring into the substring buffer (removing any
 *	   escape characters used to escape field delimiter characters).
 */
char**
VsSplit(string, numsub)
    char	*string;
    int		*numsub;		/* Number of substrings */
{
    int		nsub;			/* Number of substrings */
    char	*SepPtr;		/* Points to field-delimiter */
    char	**subs;			/* Substring pointers */
    int		success	= 0;		/* Routine status = failure */
    static char	me[]	= "VsSplit";	/* This routine's name */

    assert(string != NULL);

    /* Count the number of substrings */
    for (nsub = 1, SepPtr = string;
	(SepPtr = strchr(SepPtr, FieldSepChar)) != NULL;
	++SepPtr) {

	/* If not an escaped field-delimiter */
	if (!(SepPtr > string && SepPtr[-1] == EscapeChar))
	    ++nsub;
    }

    /* Allocate memory for the substring pointers */
    if ((subs = UD_ALLOC(nsub, char*)) == NULL) {
	udadvise("%s: Couldn't allocate %u-element substring-pointer vector.",
	    me, (unsigned)nsub);
    } else {
	/* Allocate memory for the substrings */
	char	*to	= UD_ALLOC(strlen(string)+1, char);

	if (to == NULL) {
	    udadvise("%s: Couldn't allocate memory for substrings.", me);
	} else {
	    int		isub	= 0;

	    /* For each field delimiter: */
	    for (SepPtr = string; 
		(SepPtr = strchr(SepPtr, FieldSepChar)) != NULL;
		++SepPtr) {

		/* If not an escaped delimiter */
		if (SepPtr == string || SepPtr[-1] != EscapeChar) {

		    /* Add substring-pointer to list */
		    assert(isub < nsub);
		    subs[isub++]	= to;

		    /* Copy substring */
		    to	+= CopySub(to, string, SepPtr);

		    /* Advance input-string pointer to next field */
		    string	= SepPtr + 1;
		}
	    }

	    /* Handle last substring */
	    assert(isub == nsub - 1);
	    (void)CopySub(subs[isub] = to, string, string + strlen(string));

	    *numsub	= nsub;
	    success	= 1;
	}

	if (!success)
	    (void)FREE(subs);
    }

    return subs;
}


/*
 * WHAT	   Join substrings into a single value-string (adding field 
 *	   delimiters as needed).
 *
 * INPUT   The (possibly zero) number of substrings and the (possibly nil)
 *	   array of substring pointers.  A nil substring pointer is equivalent
 *	   to an empty substring.
 *
 * OUTPUT  A pointer to a malloc'ed value-string, or a nil pointer if 1) the 
 *	   number of substrings is zero; or 2) storage for the value-string 
 *	   couldn't be allocated.
 *
 * HOW	   Determine the size for the value-string and allocate memory.  Take
 *	   into account the need to escape any field delimiting characters in
 *	   the substrings.  Append the substrings to the value-string buffer --
 *	   separating them with field delimiters and escaping any included
 *	   delimiters.
 */
char*
VsJoin(nsub, subs)
    int		nsub;		/* Number of substrings */
    char	**subs;		/* Pointers to substrings */
{
    int		isub;			/* Substring index */
    char	*string	= NULL;		/* Returned pointer to value-string */
    static char	me[]	= "VsJoin";	/* This routine's name */

    assert(nsub >= 0);

    if (nsub > 0) {
	int	nchr	= 0;		/* Number of value-string characters --
					 * excluding delimiters and terminator
					 */

	assert(subs != NULL);

	/* Determine the required size of the value-string buffer */
	for (isub = 0; isub < nsub; ++isub) {
	    char	*from	= subs[isub];

	    if (from != NULL) {
		for (; *from != 0; ++from) {
		    if (*from == FieldSepChar)
			++nchr;
		}
		nchr	+= from - subs[isub];
	    }
	}

	/* Allocate storage for the value-string */
	if ((string = UD_ALLOC(nchr + nsub - 1 + 1, char)) == NULL) {
	    udadvise("%s: Couldn't allocate value-string buffer.", me);

	} else {
	    char	*to	= string;

	    /* Join the substrings together in the value-string buffer */
	    for (isub = 0; isub < nsub; ++isub) {
		char	*from	= subs[isub];

		if (from != NULL) {
		    while (*from != 0) {
			if (*from == FieldSepChar)
			    *to++	= EscapeChar;
			*to++	= *from++;
		    }
		}
		*to++	= FieldSepChar;
	    }

	    /* Terminate the vaue-string */
	    *--to	= 0;
	}
    }

    return string;
}
