/*
 * <time.h> implementation for the UNICOS 5.1.9 version of the Unidata
 * UDPOSIX environment.
 */

/*LINTLIBRARY*/

#include "udposix.h"
#include <stddef.h>		/* for size_t and NULL */
#include <stdio.h>		/* for sprintf() */
#include <string.h>		/* for strcpy() */
#include <time.h>		/* for the obvious reason */

#ifndef lint
    static char	rcsid[]	= "$Id: time.c,v 1.1 1991/07/22 20:40:41 steve Exp $";
#endif


/*
 * strftime.c
 *
 * Public-domain relatively quick-and-dirty implementation of
 * ANSI library routine.
 *
 * The code for %c, %x, and %X is my best guess as to what's "appropriate".
 * This version ignores LOCALE information.
 * It also doesn't worry about multi-byte characters.
 * So there.
 *
 * Arnold Robbins
 * January, February, 1991
 *
 * Fixes from ado@elsie.nci.nih.gov
 * February 1991
 *
 * Adapted to UDPOSIX conventions, April 1991 --Steve Emmerson.
 */

/*
 * weeknumber --- figure how many weeks into the year.
 *
 * With thanks and tip of the hat to ado@elsie.nci.nih.gov.
 */
    static int
weeknumber(timeptr, firstweekday)
    const struct tm    *timeptr;
    int			firstweekday;
{
    return (timeptr->tm_yday + 7 - (
		firstweekday == 0
		    ? timeptr->tm_wday
		    : timeptr->tm_wday == 0
			? 6
			: timeptr->tm_wday - 1
		)
	   ) / 7;
}


/*
 * strftime --- produce formatted time.
 */
    size_t
strftime(s, maxsize, format, timeptr)
    char               *s;
    size_t		maxsize;
    const char         *format;
    const struct tm    *timeptr;
{
    char		*endp	= s + maxsize;
    char		*start	= s;
    char		tbuf[100];
    int			i;
    static short	first	= 1;

    /* various tables, useful in North America */
    static char *days_a[] = {
	"Sun", "Mon", "Tue", "Wed",
	"Thu", "Fri", "Sat",
    };
    static char *days_l[] = {
	"Sunday", "Monday", "Tuesday", "Wednesday",
	"Thursday", "Friday", "Saturday",
    };
    static char *months_a[] = {
	"Jan", "Feb", "Mar", "Apr", "May", "Jun",
	"Jul", "Aug", "Sep", "Oct", "Nov", "Dec",
    };
    static char *months_l[] = {
	"January", "February", "March", "April",
	"May", "June", "July", "August", "September",
	"October", "November", "December",
    };
    static char *ampm[] = { "AM", "PM", };

    if (s == NULL || format == NULL || timeptr == NULL || maxsize == 0)
	return 0;

    if (strchr(format, '%') == NULL && strlen(format) + 1 >= maxsize)
	return 0;

    if (first) {
	tzset();
	first = 0;
    }

    for (; *format && s < endp - 1; format++) {
	tbuf[0] = '\0';

	if (*format != '%') {
	    *s++ = *format;
	    continue;
	}

	switch (*++format) {
	case '\0':
	    *s++ = '%';
	    goto out;

	case '%':
	    *s++ = '%';
	    continue;

	case 'a':	/* abbreviated weekday name */
	    if (timeptr->tm_wday < 0 || timeptr->tm_wday > 6) {
		(void) strcpy(tbuf, "?");
	    } else {
		(void) strcpy(tbuf, days_a[timeptr->tm_wday]);
	    }
	    break;

	case 'A':	/* full weekday name */
	    if (timeptr->tm_wday < 0 || timeptr->tm_wday > 6) {
		(void) strcpy(tbuf, "?");
	    } else {
		(void) strcpy(tbuf, days_l[timeptr->tm_wday]);
	    }
	    break;

	case 'b':	/* abbreviated month name */
	    if (timeptr->tm_mon < 0 || timeptr->tm_mon > 11) {
		(void) strcpy(tbuf, "?");
	    } else {
		(void) strcpy(tbuf, months_a[timeptr->tm_mon]);
	    }
	    break;

	case 'B':	/* full month name */
	    if (timeptr->tm_mon < 0 || timeptr->tm_mon > 11) {
		(void) strcpy(tbuf, "?");
	    } else {
		(void) strcpy(tbuf, months_l[timeptr->tm_mon]);
	    }
	    break;

	case 'c':	/* appropriate date and time representation */
	    (void) sprintf(tbuf, "%s %s %2d %02d:%02d:%02d %d",
			   days_a[timeptr->tm_wday],
			   months_a[timeptr->tm_mon],
			   timeptr->tm_mday,
			   timeptr->tm_hour,
			   timeptr->tm_min,
			   timeptr->tm_sec,
			   timeptr->tm_year + 1900);
	    break;

	case 'd':	/* day of the month, 01 - 31 */
	    (void) sprintf(tbuf, "%02d", timeptr->tm_mday);
	    break;

	case 'H':	/* hour, 24-hour clock, 00 - 23 */
	    (void) sprintf(tbuf, "%02d", timeptr->tm_hour);
	    break;

	case 'I':	/* hour, 12-hour clock, 01 - 12 */
	    (void) sprintf(tbuf, "%02d", (timeptr->tm_hour + 11) % 12 + 1);
	    break;

	case 'j':	/* day of the year, 001 - 366 */
	    (void) sprintf(tbuf, "%03d", timeptr->tm_yday + 1);
	    break;

	case 'm':	/* month, 01 - 12 */
	    (void) sprintf(tbuf, "%02d", timeptr->tm_mon + 1);
	    break;

	case 'M':	/* minute, 00 - 59 */
	    (void) sprintf(tbuf, "%02d", timeptr->tm_min);
	    break;

	case 'p':	/* am or pm based on 12-hour clock */
	    if (timeptr->tm_hour < 12) {
		(void) strcpy(tbuf, ampm[0]);
	    } else {
		(void) strcpy(tbuf, ampm[1]);
	    }
	    break;

	case 'S':	/* second, 00 - 61 */
	    (void) sprintf(tbuf, "%02d", timeptr->tm_sec);
	    break;

	case 'U':	/* week of year, Sunday is first day of week */
	    (void) sprintf(tbuf, "%d", weeknumber(timeptr, 0));
	    break;

	case 'w':	/* weekday, Sunday == 0, 0 - 6 */
	    (void) sprintf(tbuf, "%d", timeptr->tm_wday);
	    break;

	case 'W':	/* week of year, Monday is first day of week */
	    (void) sprintf(tbuf, "%d", weeknumber(timeptr, 1));
	    break;

	case 'x':	/* appropriate date representation */
	    (void) sprintf(tbuf, "%s %s %2d %d",
		           days_a[timeptr->tm_wday],
		           months_a[timeptr->tm_mon],
		           timeptr->tm_mday,
		           timeptr->tm_year + 1900);
	    break;

	case 'X':	/* appropriate time representation */
	    (void) sprintf(tbuf, "%02d:%02d:%02d",
		           timeptr->tm_hour,
		           timeptr->tm_min,
		           timeptr->tm_sec);
	    break;

	case 'y':	/* year without a century, 00 - 99 */
	    (void) sprintf(tbuf, "%d", timeptr->tm_year % 100);
	    break;

	case 'Y':	/* year with century */
	    (void) sprintf(tbuf, "%d", 1900 + timeptr->tm_year);
	    break;

	case 'Z':	/* time zone name or abbrevation */
	    (void) strcpy(tbuf, timeptr->tm_zone);
	    break;

	default:
	    tbuf[0] = '%';
	    tbuf[1] = *format;
	    tbuf[2] = '\0';
	    break;
	}

	i = strlen(tbuf);
	if (i) {
	    if (s + i < endp - 1) {
		(void) strcpy(s, tbuf);
		s += i;
	    } else {
		return 0;
	    }
	}
    }						/* format-string loop */

out:
    if (s < endp && *format == '\0') {
	*s = '\0';
	return (s - start);
    } else {
	return 0;
    }
}
